#include "lite.h"

#include <drive/telematics/protocol/crc.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/stream/buffer.h>
#include <util/system/hostname.h>

void NDrive::TCommonDistributedTaskMetaInfo::Serialize(NProto::TTelematicsTask& proto) const {
    proto.SetAuthor(Author);
    proto.SetIMEI(IMEI);
    proto.SetCreated(Created.Seconds());
    proto.SetDeadline(Deadline.Seconds());
    proto.SetCallbackDeadline(CallbackDeadline.Seconds());
    proto.SetRandomId(RandomId);
}

bool NDrive::TCommonDistributedTaskMetaInfo::Deserialize(const NProto::TTelematicsTask& proto) {
    Author = proto.GetAuthor();
    IMEI = proto.GetIMEI();
    Created = TInstant::Seconds(proto.GetCreated());
    Deadline = TInstant::Seconds(proto.GetDeadline());
    CallbackDeadline = TInstant::Seconds(proto.GetCallbackDeadline());
    RandomId = proto.GetRandomId();
    return true;
}

bool NDrive::TCommonDistributedTaskMetaInfo::IsExpired(const TDuration additionalWaiting) const {
    return !!Deadline && Deadline + additionalWaiting < Now();
}

TString NDrive::TCommonDistributedTaskMetaInfo::GenerateId() const {
    return
        IMEI + '-' +
        ToString(Created.Seconds()) + '-' +
        ToString(RandomId) + '-' +
        Author + ";fake_lock=1";
}

NDrive::TCommonDistributedData::TCommonDistributedData(const TString& type, const TCommonDistributedTaskMetaInfo& info, TDuration lifetime /*= TDuration::Days(7)*/)
    : IDistributedData(TDataConstructionContext{ type, info.GenerateId() })
    , MetaInfo(info)
{
    SetDeadline(MetaInfo.GetCreatedTime() + lifetime);
}

bool NDrive::TCommonDistributedData::IsExpired(const TDuration additionalWaiting) const {
    return MetaInfo.IsExpired(additionalWaiting) || TBase::IsExpired(additionalWaiting);
}

void NDrive::TCommonDistributedData::OnHeartbeat(const TString& host) {
    auto now = Now();
    if (!Started) {
        Started = now;
    }
    Heartbeat = now;
    Host = host ? host : HostName();
}

NJson::TJsonValue NDrive::TCommonDistributedData::DoGetInfo(const TCgiParameters* /*cgi*/) const {
    NJson::TJsonValue result;
    result["status"] = NProto::TTelematicsTask::EStatus_Name(Status);
    result["started"] = Started.MicroSeconds();
    result["heartbeat"] = Heartbeat.MicroSeconds();
    result["message"] = Message;
    result["host"] = Host;
    for (auto&& id : HandlerIds) {
        result["handlers"].AppendValue(id);
    }
    NDrive::NProto::TTelematicsTask proto;
    MetaInfo.Serialize(proto);
    Serialize(proto);
    NProtobufJson::Proto2Json(proto, result["details"]);
    return result;
}

bool NDrive::TCommonDistributedData::Deserialize(const TBlob& data) {
    NDrive::NProto::TTelematicsTask proto;
    if (!proto.ParseFromArray(data.Data(), data.Size())) {
        return false;
    }
    if (!MetaInfo.Deserialize(proto)) {
        return false;
    }
    HandlerIds = { proto.GetHandlerId().begin(), proto.GetHandlerId().end() };
    Status = proto.GetStatus();
    Started = TInstant::Seconds(proto.GetStarted());
    Heartbeat = TInstant::Seconds(proto.GetHeartbeat());
    Message = proto.GetMessage();
    Host = proto.GetHost();
    return Deserialize(proto);
}

TBlob NDrive::TCommonDistributedData::Serialize() const {
    NDrive::NProto::TTelematicsTask proto;
    MetaInfo.Serialize(proto);
    proto.SetStatus(Status);
    if (Started) {
        proto.SetStarted(Started.Seconds());
    }
    if (Heartbeat) {
        proto.SetHeartbeat(Heartbeat.Seconds());
    }
    if (Message) {
        proto.SetMessage(Message);
    }
    if (Host) {
        proto.SetHost(Host);
        proto.SetHost(Host);
    }
    for (auto&& id : HandlerIds) {
        proto.AddHandlerId(id);
    }
    Serialize(proto);

    TBufferOutput stream;
    CHECK_WITH_LOG(proto.SerializeToArcadiaStream(&stream));
    return TBlob::FromBuffer(stream.Buffer());
}

bool NDrive::TSendCommandDistributedData::Deserialize(const NProto::TTelematicsTask& proto) {
    if (!proto.HasSendCommandContext()) {
        return false;
    }

    const auto& context = proto.GetSendCommandContext();
    Command = static_cast<NDrive::NVega::ECommandCode>(context.GetCommand());
    Result = static_cast<NDrive::NVega::TCommandResponse::EResult>(context.GetResult());
    Prelinger = TDuration::MicroSeconds(context.GetPrelinger());
    Postlinger = TDuration::MicroSeconds(context.GetPostlinger());
    if (context.HasGenericArguments()) {
        const auto& argument = context.GetGenericArguments();
        TStringInput si(argument);
        GenericArgument.ConstructInPlace();
        GenericArgument->Load(&si);
    }
    if (context.HasGetParameterArguments()) {
        const auto& argument = context.GetGetParameterArguments();
        SetId(argument.GetId());
        SetSubId(argument.GetSubId());
        if (argument.HasIntegerValue()) {
            SetValue<ui64>(argument.GetIntegerValue());
        } else if (argument.HasFloatValue()) {
            SetValue<double>(argument.GetFloatValue());
        } else if (argument.HasStringValue()) {
            SetValue(argument.GetStringValue());
        } else if (argument.HasBinaryValue()) {
            const auto& value = argument.GetBinaryValue();
            SetValue(TBuffer(value.data(), value.size()));
        }
    }
    if (context.HasMoveToCoordParameterArguments()) {
        const auto& argument = context.GetMoveToCoordParameterArguments();
        SetMoveToCoordArgument(argument.GetLon(), argument.GetLat());
    }
    if (context.HasPanicArguments()) {
        const auto& argument = context.GetPanicArguments();
        NDrive::NVega::TCommandRequest::TPanic panic;
        panic.Type = argument.GetType();
        panic.Time = argument.GetTime();
        SetPanicArgument(std::move(panic));
    }
    if (context.HasElectricCarCommandArguments()) {
        const auto& argument = context.GetElectricCarCommandArguments();
        NDrive::NVega::TCommandRequest::TElectricCarCommand command;
        command.Type = argument.GetType();
        SetElectricCarCommandArgument(std::move(command));
    }
    if (context.HasWarmingArguments()) {
        const auto& argument = context.GetWarmingArguments();
        NDrive::NVega::TCommandRequest::TWarming warming;
        warming.Time = argument.GetTime();
        SetWarmingArgument(std::move(warming));
    }
    return true;
}

void NDrive::TSendCommandDistributedData::Serialize(NProto::TTelematicsTask& proto) const {
    auto context = proto.MutableSendCommandContext();
    CHECK_WITH_LOG(context);
    context->SetCommand(Command);
    context->SetResult(Result);
    context->SetPrelinger(Prelinger.MicroSeconds());
    context->SetPostlinger(Postlinger.MicroSeconds());
    if (GenericArgument) {
        auto argument = context->MutableGenericArguments();
        TStringOutput so(*Yensured(argument));
        GenericArgument->Save(&so);
    }
    if (ParameterArgument) {
        auto argument = context->MutableGetParameterArguments();
        CHECK_WITH_LOG(argument);
        argument->SetId(ParameterArgument->Id);
        argument->SetSubId(ParameterArgument->SubId);

        NDrive::TSensorRef value = ParameterArgument->GetValue();
        if (std::holds_alternative<ui64>(value)) {
            argument->SetIntegerValue(std::get<ui64>(value));
        } else if (std::holds_alternative<double>(value)) {
            argument->SetFloatValue(std::get<double>(value));
        } else if (std::holds_alternative<TStringBuf>(value)) {
            const auto& str = std::get<TStringBuf>(value);
            argument->SetStringValue(str.data(), str.size());
        } else if (std::holds_alternative<TConstArrayRef<char>>(value)) {
            const auto& buffer = std::get<TConstArrayRef<char>>(value);
            argument->SetBinaryValue(buffer.data(), buffer.size());
        }
    }
    if (MoveToCoordArgument) {
        auto argument = context->MutableMoveToCoordParameterArguments();
        CHECK_WITH_LOG(argument);
        argument->SetLon(MoveToCoordArgument->Lon);
        argument->SetLat(MoveToCoordArgument->Lat);
    }
    if (PanicArgument) {
        auto argument = Checked(context->MutablePanicArguments());
        argument->SetType(PanicArgument->Type);
        argument->SetTime(PanicArgument->Time);
    }
    if (ElectricCarCommandArgument) {
        auto argument = Checked(context->MutableElectricCarCommandArguments());
        argument->SetType(ElectricCarCommandArgument->Type);
    }
    if (WarmingArgument) {
        auto argument = Checked(context->MutableWarmingArguments());
        argument->SetTime(WarmingArgument->Time);
    }
}

bool NDrive::TCommonFileDistributedData::Deserialize(const NProto::TTelematicsTask& proto) {
    if (!proto.HasCommonFileContext()) {
        return false;
    }

    const auto& context = proto.GetCommonFileContext();
    Name = context.GetName();

    const TString& data = context.GetData();
    Data.Assign(data.data(), data.size());

    const ui32 crc32required = context.GetCrc32();
    const ui32 crc32actual = NDrive::VegaCrc32(Data.Begin(), Data.End());
    if (crc32actual != crc32required) {
        ERROR_LOG << GetIdentifier() << ": CRC32 mismatch " << crc32actual << " " << crc32required << Endl;
        return false;
    }

    return true;
}

void NDrive::TCommonFileDistributedData::Serialize(NProto::TTelematicsTask& proto) const {
    auto context = proto.MutableCommonFileContext();
    context->SetName(Name);
    context->SetData(Data.Data(), Data.Size());
    context->SetCrc32(NDrive::VegaCrc32(Data.Begin(), Data.End()));
}

bool NDrive::TInterfaceCommunicationDistributedData::Deserialize(const NProto::TTelematicsTask& proto) {
    if (!proto.HasInterfaceCommunicationContext()) {
        return false;
    }

    const auto& context = proto.GetInterfaceCommunicationContext();
    Interface = static_cast<NDrive::NVega::TInterfaceData::EInterface>(context.GetInterface());
    if (context.HasInput()) {
        const auto& input = context.GetInput();
        Input.Assign(input.data(), input.size());
    }
    if (context.HasOutput()) {
        const auto& output = context.GetOutput();
        Output.Assign(output.data(), output.size());
    }
    return true;
}

void NDrive::TInterfaceCommunicationDistributedData::Serialize(NProto::TTelematicsTask& proto) const {
    auto context = proto.MutableInterfaceCommunicationContext();
    context->SetInterface(Interface);
    context->SetInput(Input.Data(), Input.Size());
    context->SetOutput(Output.Data(), Output.Size());
}

bool NDrive::TCanRequestDistributedData::Deserialize(const NProto::TTelematicsTask& proto) {
    if (!proto.HasCanRequestContext()) {
        return false;
    }

    const auto& context = proto.GetCanRequestContext();
    CanId = context.GetId();
    CanIndex = context.GetIndex();
    if (context.HasData()) {
        const auto& data = context.GetData();
        Data.Assign(data.data(), data.size());
    }
    Duration = TDuration::MicroSeconds(context.GetDuration());
    Interval = TDuration::MicroSeconds(context.GetInterval());
    return true;
}

void NDrive::TCanRequestDistributedData::Serialize(NProto::TTelematicsTask& proto) const {
    auto context = proto.MutableCanRequestContext();
    context->SetId(CanId);
    context->SetIndex(CanIndex);
    context->SetData(Data.data(), Data.size());
    context->SetDuration(Duration.MicroSeconds());
    context->SetInterval(Interval.MicroSeconds());
}

IDistributedData::TFactory::TRegistrator<NDrive::TPingDistributedData> PingData("Ping");
IDistributedData::TFactory::TRegistrator<NDrive::TSendCommandDistributedData> SendCommandData("SendCommand");
IDistributedData::TFactory::TRegistrator<NDrive::TDownloadFileDistributedData> DownloadFileData("DownloadFile");
IDistributedData::TFactory::TRegistrator<NDrive::TUploadFileDistributedData> UploadFileData("UploadFile");
IDistributedData::TFactory::TRegistrator<NDrive::TInterfaceCommunicationDistributedData> InterfaceCommunicationData("Interface");
IDistributedData::TFactory::TRegistrator<NDrive::TCanRequestDistributedData> CanRequestData("CanRequest");
