#pragma once

#include <drive/telematics/server/protos/tasks.pb.h>

#include <drive/telematics/protocol/vega.h>

#include <rtline/library/executor/executor.h>

#include <util/random/random.h>

namespace NDrive {
    struct TCommonDistributedTaskMetaInfo {
    public:
        TCommonDistributedTaskMetaInfo() = default;
        TCommonDistributedTaskMetaInfo(const TString& author, const TString& imei, TDuration taskTimeout = TDuration::Max(), TDuration callbackTimeout = TDuration::Zero())
            : Author(author)
            , IMEI(imei)
            , Created(Now())
            , Deadline(Created + taskTimeout)
            , RandomId(RandomNumber<ui16>())
            , CallbackDeadline((callbackTimeout == TDuration::Zero()) ? Deadline : (Created + callbackTimeout))
        {
        }

        void Serialize(NProto::TTelematicsTask& proto) const;
        bool Deserialize(const NProto::TTelematicsTask& proto);

        bool IsExpired(const TDuration additionalWaiting) const;

        const TString& GetAuthor() const {
            return Author;
        }
        const TString& GetIMEI() const {
            return IMEI;
        }
        TInstant GetCreatedTime() const {
            return Created;
        }
        TInstant GetDeadline() const {
            return Deadline;
        }
        TDuration GetTimeout() const {
            return Deadline - Created;
        }
        TInstant GetCallbackDeadline() const {
            return CallbackDeadline;
        }
        TDuration GetCallbackTimeout() const {
            return CallbackDeadline - Created;
        }

        TString GenerateId() const;

    private:
        TString Author;
        TString IMEI;
        TInstant Created;
        TInstant Deadline;
        ui16 RandomId;
        TInstant CallbackDeadline;
    };

    class TCommonDistributedData: public IDistributedData {
    private:
        using TBase = IDistributedData;
    public:

        using IDistributedData::IDistributedData;

        TCommonDistributedData(const TString& type, const TCommonDistributedTaskMetaInfo& info, TDuration lifetime = TDuration::Days(7));

        virtual bool IsExpired(const TDuration additionalWaiting) const override;
        const TCommonDistributedTaskMetaInfo& GetMetaInfo() const {
            return MetaInfo;
        }
        const TString& GetMessage() const {
            return Message;
        }
        const TString& GetHost() const {
            return Host;
        }
        NProto::TTelematicsTask::EStatus GetStatus() const {
            return Status;
        }
        TInstant GetStarted() const {
            return Started;
        }
        TInstant GetHeartbeat() const {
            return Heartbeat;
        }
        const TVector<TString>& GetHandlers() const {
            return HandlerIds;
        }
        size_t GetHandlersCount() const {
            return HandlerIds.size();
        }
        size_t GetRetriesCount() const {
            return RetriesCount;
        }

        void AddHandlerId(const TString& id) {
            HandlerIds.push_back(id);
        }
        void SetStatus(NProto::TTelematicsTask::EStatus value, const TString& message = Default<TString>()) {
            Status = value;
            Message = message;
        }
        void OnHeartbeat(const TString& host = {});

        virtual bool Deserialize(const TBlob& data) override;
        virtual TBlob Serialize() const override;

    protected:
        virtual NJson::TJsonValue DoGetInfo(const TCgiParameters* /*cgi*/) const override;

        virtual bool Deserialize(const NProto::TTelematicsTask& proto) = 0;
        virtual void Serialize(NProto::TTelematicsTask& proto) const = 0;

    private:
        TCommonDistributedTaskMetaInfo MetaInfo;
        TVector<TString> HandlerIds;
        NProto::TTelematicsTask::EStatus Status = NProto::TTelematicsTask::NEW;
        TString Message;
        TString Host;
        TInstant Started;
        TInstant Heartbeat;
        size_t RetriesCount = 3;
    };

    class TPingDistributedData: public TCommonDistributedData {
    public:
        using TCommonDistributedData::TCommonDistributedData;
        TPingDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonDistributedData("Ping", info)
        {
        }

    protected:
        virtual bool Deserialize(const NProto::TTelematicsTask& /*proto*/) override { return true; }
        virtual void Serialize(NProto::TTelematicsTask& /*proto*/) const override {}
    };

    class TSendCommandDistributedData: public TCommonDistributedData {
    public:
        using TCommonDistributedData::TCommonDistributedData;
        TSendCommandDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonDistributedData("SendCommand", info)
        {
        }

        NDrive::NVega::ECommandCode GetCommand() const {
            return Command;
        }
        void SetCommand(NDrive::NVega::ECommandCode value) {
            Command = value;
        }

        const NDrive::NVega::TArgument* GetGenericArgument() const {
            return GenericArgument.Get();
        }
        template <class T>
        void SetGenericArgumentFrom(T&& value) {
            GenericArgument.ConstructInPlace();
            GenericArgument->Set(std::forward<T>(value));
        }
        void SetGenericArgument(const NDrive::NVega::TArgument& argument) {
            if (!argument.IsNull()) {
                GenericArgument = argument;
            } else {
                GenericArgument.Clear();
            }
        }
        void SetGenericArgument(NDrive::NVega::TArgument&& argument) {
            if (!argument.IsNull()) {
                GenericArgument = std::move(argument);
            } else {
                GenericArgument.Clear();
            }
        }

        const NDrive::NVega::TIdSubIdValue* GetParameter() const {
            return ParameterArgument.Get();
        }
        void SetId(ui16 value) {
            if (!ParameterArgument) {
                ParameterArgument.ConstructInPlace();
            }
            ParameterArgument->Id = value;
        }
        void SetSubId(ui16 value) {
            if (!ParameterArgument) {
                ParameterArgument.ConstructInPlace();
            }
            ParameterArgument->SubId = value;
        }
        template <class T>
        void SetValue(T value) {
            if (!ParameterArgument) {
                ParameterArgument.ConstructInPlace();
            }
            ParameterArgument->SetValue(value);
        }

        const NDrive::NVega::TCommandRequest::TMoveToCoordParameter* MoveToCoordParameter() const {
            return MoveToCoordArgument.Get();
        }
        void SetMoveToCoordArgument(const double lon, const double lat) {
            if (!MoveToCoordArgument) {
                MoveToCoordArgument.ConstructInPlace();
            }
            MoveToCoordArgument->Lon = lon;
            MoveToCoordArgument->Lat = lat;
        }

        const NDrive::NVega::TCommandRequest::TPanic* GetPanicArgument() const {
            return PanicArgument.Get();
        }
        template <class T>
        void SetPanicArgument(T&& value) {
            PanicArgument = std::forward<T>(value);
        }

        const NDrive::NVega::TCommandRequest::TElectricCarCommand* GetElectricCarCommandArgument() const {
            return ElectricCarCommandArgument.Get();
        }
        template <class T>
        void SetElectricCarCommandArgument(T&& value) {
            ElectricCarCommandArgument = std::forward<T>(value);
        }

        const NDrive::NVega::TCommandRequest::TWarming* GetWarmingArgument() const {
            return WarmingArgument.Get();
        }
        template <class T>
        void SetWarmingArgument(T&& value) {
            WarmingArgument = std::forward<T>(value);
        }

        TDuration GetPrelinger() const {
            return Prelinger;
        }
        TDuration GetPostlinger() const {
            return Postlinger;
        }
        void SetPrelinger(TDuration value) {
            Prelinger = value;
        }
        void SetPostlinger(TDuration value) {
            Postlinger = value;
        }

    protected:
        virtual bool Deserialize(const NProto::TTelematicsTask& proto) override;
        virtual void Serialize(NProto::TTelematicsTask& proto) const override;

    private:
        NDrive::NVega::ECommandCode Command = NDrive::NVega::ECommandCode::UNKNOWN;
        NDrive::NVega::TCommandResponse::EResult Result = NDrive::NVega::TCommandResponse::UNKNOWN;
        TMaybe<NDrive::NVega::TArgument> GenericArgument;
        TMaybe<NDrive::NVega::TIdSubIdValue> ParameterArgument;
        TMaybe<NDrive::NVega::TCommandRequest::TMoveToCoordParameter> MoveToCoordArgument;
        TMaybe<NDrive::NVega::TCommandRequest::TPanic> PanicArgument;
        TMaybe<NDrive::NVega::TCommandRequest::TWarming> WarmingArgument;
        TMaybe<NDrive::NVega::TCommandRequest::TElectricCarCommand> ElectricCarCommandArgument;
        TDuration Prelinger;
        TDuration Postlinger;
    };

    class TCommonFileDistributedData: public TCommonDistributedData {
    public:
        using TCommonDistributedData::TCommonDistributedData;


        const TString& GetName() const {
            return Name;
        }
        const TBuffer& GetData() const {
            return Data;
        }
        TBuffer ReleaseData() {
            return std::move(Data);
        }

        void SetName(const TString& name) {
            Name = name;
        }
        void SetData(TBuffer&& data) {
            Data = std::move(data);
        }

    protected:
        virtual bool Deserialize(const NProto::TTelematicsTask& proto) override;
        virtual void Serialize(NProto::TTelematicsTask& proto) const override;

    private:
        TString Name;
        TBuffer Data;
    };

    class TDownloadFileDistributedData: public TCommonFileDistributedData {
    public:
        using TCommonFileDistributedData::TCommonFileDistributedData;
        TDownloadFileDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonFileDistributedData("DownloadFile", info)
        {
        }
    };

    class TUploadFileDistributedData: public TCommonFileDistributedData {
    public:
        using TCommonFileDistributedData::TCommonFileDistributedData;
        TUploadFileDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonFileDistributedData("UploadFile", info)
        {
        }
    };

    class TInterfaceCommunicationDistributedData: public TCommonDistributedData {
    public:
        using TCommonDistributedData::TCommonDistributedData;
        TInterfaceCommunicationDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonDistributedData("Interface", info)
        {
        }

        NDrive::NVega::TInterfaceData::EInterface GetInterface() const {
            return Interface;
        }
        const TBuffer& GetInput() const {
            return Input;
        }
        const TBuffer& GetOutput() const {
            return Output;
        }
        TBuffer ReleaseOutput() {
            return std::move(Output);
        }

        void SetInterface(NDrive::NVega::TInterfaceData::EInterface value) {
            Interface = value;
        }
        void SetInput(TBuffer&& value) {
            Input = std::move(value);
        }
        void SetOutput(TBuffer&& value) {
            Output = std::move(value);
        }

    protected:
        virtual bool Deserialize(const NProto::TTelematicsTask& proto) override;
        virtual void Serialize(NProto::TTelematicsTask& proto) const override;

    private:
        NDrive::NVega::TInterfaceData::EInterface Interface = NDrive::NVega::TInterfaceData::UNKNOWN;
        TBuffer Input;
        TBuffer Output;
    };

    class TCanRequestDistributedData: public TCommonDistributedData {
    public:
        using TCommonDistributedData::TCommonDistributedData;
        TCanRequestDistributedData(const TCommonDistributedTaskMetaInfo& info)
            : TCommonDistributedData("CanRequest", info)
        {
        }

        ui32 GetCanId() const {
            return CanId;
        }
        ui8 GetCanIndex() const {
            return CanIndex;
        }
        const TBuffer& GetData() const {
            return Data;
        }
        TDuration GetDuration() const {
            return Duration;
        }
        TDuration GetInterval() const {
            return Interval;
        }

        void SetCanId(ui32 value) {
            CanId = value;
        }
        void SetCanIndex(ui8 value) {
            CanIndex = value;
        }
        void SetData(TBuffer&& value) {
            Data = std::move(value);
        }
        void SetRepetition(TDuration duration, TDuration interval) {
            Duration = duration;
            Interval = interval;
        }

    protected:
        virtual bool Deserialize(const NProto::TTelematicsTask& proto) override;
        virtual void Serialize(NProto::TTelematicsTask& proto) const override;

    private:
        ui32 CanId = 0;
        ui8 CanIndex = 0;
        TBuffer Data;

        TDuration Duration = TDuration::Zero();
        TDuration Interval = TDuration::Zero();
    };

    template <class TBase = IDistributedTask>
    class TPingDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TPingDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "Ping", info.GenerateId() })
        {
        }
    };

    class TPingDistributedTaskLite: public TPingDistributedTaskBase<> {
    public:
        using TPingDistributedTaskBase::TPingDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };

    template <class TBase = IDistributedTask>
    class TSendCommandDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TSendCommandDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "SendCommand", info.GenerateId() })
        {
        }
    };

    class TSendCommandDistributedTaskLite: public TSendCommandDistributedTaskBase<> {
    public:
        using TSendCommandDistributedTaskBase::TSendCommandDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };

    template <class TBase = IDistributedTask>
    class TDownloadFileDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TDownloadFileDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "DownloadFile", info.GenerateId() })
        {
        }
    };

    class TDownloadFileDistributedTaskLite: public TDownloadFileDistributedTaskBase<> {
    public:
        using TDownloadFileDistributedTaskBase::TDownloadFileDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };

    template <class TBase = IDistributedTask>
    class TUploadFileDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TUploadFileDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "UploadFile", info.GenerateId() })
        {
        }
    };

    class TUploadFileDistributedTaskLite: public TUploadFileDistributedTaskBase<> {
    public:
        using TUploadFileDistributedTaskBase::TUploadFileDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };

    template <class TBase = IDistributedTask>
    class TInterfaceCommunicationDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TInterfaceCommunicationDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "Interface", info.GenerateId() })
        {
        }
    };

    class TInterfaceCommunicationDistributedTaskLite: public TInterfaceCommunicationDistributedTaskBase<> {
    public:
        using TInterfaceCommunicationDistributedTaskBase::TInterfaceCommunicationDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };

    template <class TBase = IDistributedTask>
    class TCanRequestDistributedTaskBase: public TBase {
    public:
        using TBase::TBase;

        TCanRequestDistributedTaskBase(const TCommonDistributedTaskMetaInfo& info)
            : TBase(TTaskConstructionContext{ "CanRequest", info.GenerateId() })
        {
        }
    };

    class TCanRequestDistributedTaskLite: public TCanRequestDistributedTaskBase<> {
    public:
        using TCanRequestDistributedTaskBase::TCanRequestDistributedTaskBase;

    protected:
        virtual bool DoExecute(IDistributedTask::TPtr /*self*/) noexcept override {
            FAIL_LOG("unimplemented");
        }
    };
}
