#include "pusher.h"

#include <drive/telematics/server/common/signals.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/location/proto/heartbeat.pb.h>
#include <drive/telematics/server/location/proto/location.pb.h>
#include <drive/telematics/server/pusher/common.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/url/url.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/network/neh.h>

#include <functional>

namespace {
    NJson::TJsonValue GetHeartbeatArrayItem(const TString& imei, const NDrive::THeartbeat& heartbeat) {
        NJson::TJsonValue arrayItem;
        arrayItem["data_type"] = "heartbeat";
        arrayItem["data_proto"] = heartbeat.ToProto().SerializeAsString();
        arrayItem["unix_timestamp"] = heartbeat.Timestamp.Seconds();
        arrayItem["imei"] = imei;
        // assuming that heartbeat's name is always empty
        return arrayItem;
    }

    NJson::TJsonValue GetLocationArrayItem(const TString& imei, const NDrive::TLocation& location) {
        NJson::TJsonValue arrayItem;
        arrayItem["data_type"] = "location";
        arrayItem["data_proto"] = location.ToProto().SerializeAsString();
        arrayItem["unix_timestamp"] = location.Timestamp.Seconds();
        arrayItem["imei"] = imei;
        arrayItem["location_name"] = location.Name;
        return arrayItem;
    }

    NJson::TJsonValue GetSensorArrayItem(const TString& imei, const NDrive::TSensor& sensor) {
        NJson::TJsonValue arrayItem;
        arrayItem["data_type"] = "sensor";
        arrayItem["data_proto"] = sensor.ToProto().SerializeAsString();
        arrayItem["unix_timestamp"] = sensor.Timestamp.Seconds();
        arrayItem["imei"] = imei;
        NJson::TJsonValue sensorIdJson = NJson::JSON_MAP;
        sensorIdJson["id"] = sensor.Id;
        sensorIdJson["sub_id"] = sensor.SubId;
        arrayItem["sensor_id"] = std::move(sensorIdJson);
        return arrayItem;
    }
}

THolder<NDrive::IPusher> NDrive::TTelematicsCacheApiPusherOptions::BuildPusher(const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm) const {
    Y_UNUSED(tvm);
    return MakeHolder<TTelematicsCacheApiPusher>(*this);
}

TSet<NTvmAuth::TTvmId> NDrive::TTelematicsCacheApiPusherOptions::GetDestinationClientIds() const {
  return {}; // Service has no authorization yet
}

void NDrive::TTelematicsCacheApiPusherOptions::Init(const TYandexConfig::Section& section) {
    const auto& directives = section.GetDirectives();
    Endpoint = directives.Value("Endpoint", Endpoint);
    TrafficPercent = directives.Value("TrafficPercent", TrafficPercent);
}

void NDrive::TTelematicsCacheApiPusherOptions::Print(IOutputStream& os) const {
    os << "Endpoint: " << Endpoint << Endl;
    os << "TrafficPercent: " << TrafficPercent << Endl;
}

NDrive::TTelematicsCacheApiPusher::TTelematicsCacheApiPusher(const NDrive::TTelematicsCacheApiPusherOptions& options)
    : Client(MakeHolder<NNeh::THttpClient>(options.Endpoint))
    , Handler(NUrl::SplitUrlToHostAndPath(options.Endpoint).path)
    , TrafficPercent(options.TrafficPercent)
{
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsCacheApiPusher::Push(const TString& imei, const IHandlerDescription& handler, TInstant deadline) {
    // Implement me later
    Y_UNUSED(deadline, handler, imei);
    return NThreading::MakeFuture(TPushResult{ /* Written= */ true, /* Message= */ "" });
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsCacheApiPusher::Push(const TString& imei, const THeartbeat& heartbeat, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TPushResult>(GetHeartbeatArrayItem, imei, NContainer::Scalar(heartbeat), deadline);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsCacheApiPusher::Push(const TString& imei, const TLocation& location, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TPushResult>(GetLocationArrayItem, imei, NContainer::Scalar(location), deadline);
}

NThreading::TFuture<NDrive::IPusher::TPushResult> NDrive::TTelematicsCacheApiPusher::Push(const TString& imei, const TSensor& sensor, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TPushResult>(GetSensorArrayItem, imei, NContainer::Scalar(sensor), deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TTelematicsCacheApiPusher::BulkPush(const TString& imei,const TConstArrayRef<TTelematicsHandlerPtr> handlersPtrs, TInstant deadline) {
    // Implement me later
    Y_UNUSED(deadline, imei);
    return NThreading::MakeFuture(TBulkPushResult{ /* Written= */ handlersPtrs.size(), /* Messages= */ {} });
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TTelematicsCacheApiPusher::BulkPush(const TString& imei, const TConstArrayRef<THeartbeat> heartbeats, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TBulkPushResult>(GetHeartbeatArrayItem, imei, heartbeats, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TTelematicsCacheApiPusher::BulkPush(const TString& imei, const TConstArrayRef<TLocation> locations, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TBulkPushResult>(GetLocationArrayItem, imei, locations, deadline);
}

NThreading::TFuture<NDrive::IPusher::TBulkPushResult> NDrive::TTelematicsCacheApiPusher::BulkPush(const TString& imei, const TConstArrayRef<TSensor> sensors, TInstant deadline) {
    return PushGeneric<NDrive::IPusher::TBulkPushResult>(GetSensorArrayItem, imei, sensors, deadline);
}

template <typename TGenericPushResult, typename TValue, typename TFunction>
NThreading::TFuture<TGenericPushResult> NDrive::TTelematicsCacheApiPusher::PushGeneric(TFunction getArrayItemFunction, const TString& imei, const TConstArrayRef<TValue> values, TInstant deadline) const {
    if (values.empty()) {
        return NThreading::MakeFuture(TGenericPushResult{ /* Written= */ 0, /* Messages= */ {} });
    }
    if (std::hash<TString>{}(imei) % 100 >= TrafficPercent) {
        if constexpr (std::is_same_v<TGenericPushResult, NDrive::IPusher::TBulkPushResult>) {
            return NThreading::MakeFuture(TGenericPushResult{ /* Written= */ values.size(), /* Messages= */ {} });
        }
        return NThreading::MakeFuture(TGenericPushResult{ /* Written= */ static_cast<bool>(values.size()), /* Message= */ {} });
    }
    NNeh::THttpRequest request;
    request.SetUri(Handler);
    NJson::TJsonValue requestBody = NJson::JSON_MAP;
    NJson::TJsonValue& dataArray = requestBody.InsertValue("data_array", NJson::JSON_ARRAY);
    for (const auto& value : values) {
        dataArray.AppendValue(getArrayItemFunction(imei, value));
    }
    request.SetPostData(requestBody.GetStringRobust());

    const auto asyncResponse = Yensured(Client)->SendAsync(request, deadline);
    return ProcessResponse<TGenericPushResult>(asyncResponse);
}

template <typename TGenericPushResult>
NThreading::TFuture<TGenericPushResult> NDrive::TTelematicsCacheApiPusher::ProcessResponse(const NThreading::TFuture<NNeh::THttpReply>& reply) const {
    return reply.Apply([](const NThreading::TFuture<NNeh::THttpReply>& r) {
        if (r.HasException()) {
            NDrive::TTelematicsUnistatSignals::Get().TelematicsCacheApiPusherException.Signal(1);
            return TGenericPushResult{ /* Written= */ 0, /* Messags= */ {NThreading::GetExceptionMessage(r)} };
        }
        const auto& response = r.GetValue();
        NJson::TJsonValue replyJson;
        if (!response.IsSuccessReply() || !NJson::ReadJsonFastTree(response.Content(), &replyJson)) {
            NDrive::TTelematicsUnistatSignals::Get().TelematicsCacheApiPusherFailure.Signal(1);
            return TGenericPushResult{ /* Written= */ 0, /* Messages= */ {response.Serialize().GetStringRobust()}};
        };
        NDrive::TTelematicsUnistatSignals::Get().TelematicsCacheApiPusherSuccess.Signal(1);
        const auto written = replyJson["written"].GetInteger();
        if constexpr (std::is_same_v<TGenericPushResult, NDrive::IPusher::TBulkPushResult>) {
            return TGenericPushResult{ /* Written= */ static_cast<size_t>(written), /* Messages= */ {} };
        }
        return TGenericPushResult{ /* Written= */ static_cast<bool>(written), /* Message= */ {} };
    });
}
