#include <drive/telematics/server/ut/library/helper.h>

#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/location/beacon_recognizer.h>
#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/pusher/pusher.h>
#include <drive/telematics/server/tasks/lite.h>

#include <drive/telematics/client/car_emulator/handlers.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/util/types/uuid.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(ExternalTelematicsServerSuite) {
    Y_UNIT_TEST(NetPusher) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();

        NDrive::TPusherOptions pusherOptions;
        pusherOptions.Host = "saas-indexerproxy-maps-prestable.yandex.net";
        pusherOptions.Port = 80;
        pusherOptions.Token = "ac6ac7f7a6544f336202c0b058104374";

        server->SetPusher(MakeHolder<NDrive::TPusher>(pusherOptions));

        NRTLine::TNehSearchClient searchClient("drive_cache", "saas-searchproxy-maps-prestable.yandex.net", 17000);
        NDrive::TSensorApi sensorApi("test", searchClient);

        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());

        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.AddHandler(heartbeat);
            auto setParameter = MakeAtomicShared<NDrive::TSetParamHandler>(client, context);
            client.AddHandler(setParameter);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            auto blackbox = Blackbox();

            auto handler = NDrive::NProtocol::Handle(client, MakeAtomicShared<NDrive::TBlackboxHandler>(std::move(blackbox)), TDuration::Seconds(10));
            Y_ENSURE(handler->GetResult() == NDrive::NVega::TBlackboxRecordsAck::Received, "cannot send Blackbox message");

            blackbox = Blackbox();
            TMaybe<NDrive::TLocation> location;
            for (size_t i = 0; i < 100; ++i) {
                location = sensorApi.GetLocation(imei).ExtractValueSync();
                if (location && abs(location->Longitude - blackbox.back().Longitude) < 1e-3 && abs(location->Latitude - blackbox.back().Lattitude) < 1e-3) {
                    break;
                }
            }
            UNIT_ASSERT(location);
            UNIT_ASSERT(abs(location->Longitude - blackbox.back().Longitude) < 1e-3);
            UNIT_ASSERT(abs(location->Latitude - blackbox.back().Lattitude) < 1e-3);
            // 5 * 4 m
            constexpr double precision = 20;
            UNIT_ASSERT(abs(location->Precision - precision) < 1e-3);

            TMaybe<NDrive::THeartbeat> hb;
            for (size_t i = 0; i < 100; ++i) {
                hb = sensorApi.GetHeartbeat(imei).ExtractValueSync();
                if (hb && hb->Timestamp.Seconds() > 0) {
                    break;
                } else {
                    Sleep(TDuration::Seconds(1));
                }
            }
            UNIT_ASSERT(hb);
            UNIT_ASSERT(hb->Timestamp.Seconds() > 0);

            ui16 id = CAN_ODOMETER_KM;
            double value = 1000;

            auto handle = api.SetParameter<double>(imei, id, 0, value);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(context.GetOdometerKm(), value);

            TMaybe<NDrive::TSensor> sensor;
            for (size_t i = 0; i < 100; ++i) {
                sensor = sensorApi.GetSensor(imei, id).ExtractValueSync();
                if (sensor && std::holds_alternative<double>(sensor->Value) && (std::abs(std::get<double>(sensor->Value) - value) < 0.001)) {
                    break;
                } else {
                    Sleep(TDuration::MilliSeconds(100));
                }
            }
            UNIT_ASSERT(sensor);
            UNIT_ASSERT(std::holds_alternative<double>(sensor->Value));
            UNIT_ASSERT_DOUBLES_EQUAL(std::get<double>(sensor->Value), value, 1e-3);
        }
    }

    Y_UNIT_TEST(LocatorGeocoded) {
        NDrive::TLocator::TOptions options;
        options.GeocoderHost = "addrs-testing.search.yandex.net";
        NDrive::TLocator locator(options);

        TString imei = "861108036391661";
        NDrive::TSensorsCache sensors(16);

        TGeoCoord coordinate = { 37.65732956, 55.88800812 };
        TInstant timestamp = TInstant::Seconds(1536583420);
        double course = 42;

        sensors.Add(VEGA_LAT, 0, coordinate.Y, timestamp);
        sensors.Add(VEGA_LON, 0, coordinate.X, timestamp);
        sensors.Add(VEGA_DIR, 0, course, timestamp);

        auto locations = locator.LocateAll(imei, sensors);
        UNIT_ASSERT_VALUES_EQUAL(locations.size(), 2);

        auto regular = locations[0].GetValueSync();
        auto geocoded = locations[1].GetValueSync();
        UNIT_ASSERT_VALUES_EQUAL(regular.Name, "");
        UNIT_ASSERT_VALUES_EQUAL(geocoded.Name, "geocoded");
        UNIT_ASSERT(!geocoded.Content.empty());
    }

    Y_UNIT_TEST(LocatorLinked) {
        NDrive::TLocator::TOptions options;
        options.LinkerHost = "saas-searchproxy-maps.yandex.net";
        options.TracksHost = "saas-searchproxy-maps-prestable.yandex.net";
        options.LinkTrackTimeout = TDuration::Seconds(2);
        NDrive::TLocator locator(options);

        TString imei = "867962043095507";
        NDrive::TSensorsCache sensors(16);

        TGeoCoord coordinate = { 37.65732956, 55.88800812 };
        TInstant timestamp = TInstant::Seconds(Now().Seconds()) - TDuration::Hours(1);
        double course = 42;

        sensors.Add(VEGA_LAT, 0, coordinate.Y, timestamp);
        sensors.Add(VEGA_LON, 0, coordinate.X, timestamp);
        sensors.Add(VEGA_DIR, 0, course, timestamp);

        TInstant brokenTimestamp = timestamp + TDuration::Seconds(20);
        sensors.Add(VEGA_LAT, 0, 0.0, brokenTimestamp);
        sensors.Add(VEGA_LON, 0, 0.0, brokenTimestamp);
        sensors.Add(VEGA_DIR, 0, 0.0, brokenTimestamp);

        auto linkedF = locator.Link(imei, sensors);
        NDrive::TLocation linked = linkedF.ExtractValueSync();

        UNIT_ASSERT_VALUES_EQUAL(linked.Name, "linked");
        UNIT_ASSERT_VALUES_EQUAL(linked.Type, NDrive::TLocation::LinkedPrevious);
        UNIT_ASSERT_VALUES_EQUAL(linked.Timestamp, timestamp);
        UNIT_ASSERT(linked.Base);
        UNIT_ASSERT_DOUBLES_EQUAL(linked.Base->Latitude, coordinate.Y, 0.00001);
        UNIT_ASSERT_DOUBLES_EQUAL(linked.Base->Longitude, coordinate.X, 0.00001);
        UNIT_ASSERT(linked.Course >= 0);
        UNIT_ASSERT(linked.Course < 360);
        UNIT_ASSERT(linked.Precision > 0);
        UNIT_ASSERT(linked.Precision <= 1000);

        auto all = locator.LocateAll(imei, sensors);
        UNIT_ASSERT_VALUES_EQUAL(all.size(), 2);
    }

    Y_UNIT_TEST(LocatorLBS) {
        NDrive::TLocator::TOptions options;
        options.LBSToken = GetEnv("LBS_TOKEN", "AIL6tlwBAAAAy7bXegMCk3rkvAcJHQuMZOFnkpH_AYCTlcoAAAAAAAAAAABu7F9F-i6Kqh-vAd2V8M0t5cPOcw==");
        UNIT_ASSERT(options.LBSToken);
        NDrive::TLocator locator(options);

        TString imei = "861108038340062";
        NDrive::TSensorsCache sensors(16);

        TInstant timestamp = TInstant::Seconds(1536583420);
        sensors.Add(VEGA_MCC, 0, ui64(250), timestamp);
        sensors.Add(VEGA_MNC, 0, ui64(2), timestamp);
        sensors.Add(VEGA_LAC, 0, ui64(9731), timestamp);
        sensors.Add(VEGA_CELLID, 0, ui64(198059542), timestamp);
        sensors.Add(VEGA_GSM_SIGNAL_LEVEL, 0, ui64(31), timestamp);

        auto lbs1F = locator.LBS(imei, sensors);
        auto lbs1 = lbs1F.GetValueSync();
        INFO_LOG << lbs1.ToJson().GetStringRobust() << Endl;

        UNIT_ASSERT_VALUES_EQUAL(lbs1.Name, "lbs");
        UNIT_ASSERT_VALUES_EQUAL(lbs1.Type, NDrive::TLocation::LBS);
        UNIT_ASSERT_VALUES_EQUAL(lbs1.Timestamp, timestamp);
        UNIT_ASSERT_DOUBLES_EQUAL(lbs1.Course, 0, 0.00001);
        UNIT_ASSERT(lbs1.Latitude > 0);
        UNIT_ASSERT(lbs1.Longitude > 0);
        UNIT_ASSERT(lbs1.Precision > 0);

        TInstant timestamp2 = timestamp + TDuration::Seconds(10);
        NDrive::TSensorsCache sensors2(16);
        sensors2.Add(VEGA_EXT_SERVING_CELL_INF, 0, TString("3,0\n0,\"250,02,2613,08ad,38,47\"\n1,\",,0000,0000,00,00\"\n2,\",,0000,0000,00,00\"\n3,\",,0000,0000,00,00\"\n4,\",,0000,0000,00,00\"\n5,\",,0000,0000,00,00\"\n6,\",,0000,0000,00,00\"\n\0002,2613,8e06,00,22\"\n\00022\"\n\00022\"\n"), timestamp2);

        auto lbs2F = locator.LBS(imei, sensors2);
        auto lbs2 = lbs2F.GetValueSync();
        INFO_LOG << lbs2.ToJson().GetStringRobust() << Endl;

        UNIT_ASSERT_VALUES_EQUAL(lbs2.Name, "lbs");
        UNIT_ASSERT_VALUES_EQUAL(lbs2.Type, NDrive::TLocation::LBS);
        UNIT_ASSERT_VALUES_EQUAL(lbs2.Timestamp, timestamp2);
        UNIT_ASSERT_DOUBLES_EQUAL(lbs2.Course, 0, 0.00001);
        UNIT_ASSERT(lbs2.Latitude > 0);
        UNIT_ASSERT(lbs2.Longitude > 0);
        UNIT_ASSERT(lbs2.Precision > 0);
    }

    Y_UNIT_TEST(LocatorProjection) {
        NDrive::TLocator::TOptions options;
        options.LinkerHost = "saas-searchproxy-maps.yandex.net";
        options.TracksHost = "saas-searchproxy-maps-prestable.yandex.net";
        options.LinkEnableProjection = true;
        options.LinkTrackTimeout = TDuration::Seconds(2);
        NDrive::TLocator locator(options);

        TString imei = "fake";
        NDrive::TSensorsCache sensors(16);

        TGeoCoord coordinate = { 37.65732956, 55.88800812 };
        TInstant timestamp = TInstant::Seconds(Now().Seconds()) - TDuration::Hours(1);
        double course = 42;

        sensors.Add(VEGA_LAT, 0, coordinate.Y, timestamp);
        sensors.Add(VEGA_LON, 0, coordinate.X, timestamp);
        sensors.Add(VEGA_DIR, 0, course, timestamp);

        auto linkedF = locator.Link(imei, sensors);
        NDrive::TLocation linked = linkedF.ExtractValueSync();

        UNIT_ASSERT_VALUES_EQUAL(linked.Name, "linked");
        UNIT_ASSERT_VALUES_EQUAL(linked.Type, NDrive::TLocation::Projection);
        UNIT_ASSERT_VALUES_EQUAL(linked.Timestamp, timestamp);
        UNIT_ASSERT(linked.Base);
        UNIT_ASSERT_DOUBLES_EQUAL(linked.Base->Latitude, coordinate.Y, 0.00001);
        UNIT_ASSERT_DOUBLES_EQUAL(linked.Base->Longitude, coordinate.X, 0.00001);
        UNIT_ASSERT(linked.Course >= 0);
        UNIT_ASSERT(linked.Course < 360);
        UNIT_ASSERT(linked.Precision > 0);
        UNIT_ASSERT(linked.Precision <= 1000);
    }

    Y_UNIT_TEST(BeaconRecognizer) {
        {
            NDrive::NVega::TBeaconInfo info;
            info.MutableRawUUID()[0] = 1;
            info.MutableRawUUID()[1] = 2;
            info.MutableRawUUID()[2] = 4;
            info.MutableRawUUID()[3] = 8;
            info.MutableRawMajor() = 256 * 2 + 1;
            info.MutableRawMinor() = 256 * 3 + 1;
            TString expected = "uuid:" + TGUID({1<<24, 2<<24, 4<<24, 8<<24}).AsUuidString() + ":major:258:minor:259";
            UNIT_ASSERT_VALUES_EQUAL(NDrive::TDynamicBeaconRecognizer::ConvertToKey(info), expected);
        }
        {
            const TString jsonValueStr = R"({"key1":{"name":"my_location",
                                              "latitude":11,
                                              "longitude":21},
                                    "uuid:00000000-0000-0000-0000-000000000001:major:0:minor:0":{"name":"my_cool_location",
                                                                                                        "latitude":50,
                                                                                                        "longitude":49}
                                    })";
            NJson::TJsonValue value;
            UNIT_ASSERT(NJson::ReadJsonFastTree(jsonValueStr, &value));
            TMap<TString, NDrive::IBeaconRecognizer::TLocationData> locationDataByKey;
            UNIT_ASSERT(NDrive::TryFromJson(value, locationDataByKey));
            UNIT_ASSERT_VALUES_EQUAL(locationDataByKey.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(locationDataByKey["key1"].Name, "my_location");
            UNIT_ASSERT_DOUBLES_EQUAL(locationDataByKey["key1"].Latitude, 11, 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(locationDataByKey["key1"].Longitude, 21, 1e-9);

            NDrive::NVega::TBeaconInfo info;
            info.MutableRawUUID() = {0, 0, 0, 1 << 24};
            info.MutableRawMajor() = 0;
            info.MutableRawMinor() = 0;
            auto key = NDrive::TDynamicBeaconRecognizer::ConvertToKey(info);
            Cerr << key << Endl;
            UNIT_ASSERT_VALUES_EQUAL(key, "uuid:00000000-0000-0000-0000-000000000001:major:0:minor:0");
            UNIT_ASSERT_VALUES_EQUAL(locationDataByKey[key].Name, "my_cool_location");
            UNIT_ASSERT_DOUBLES_EQUAL(locationDataByKey[key].Latitude, 50, 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(locationDataByKey[key].Longitude, 49, 1e-9);
        }
    }
}
