#include "helper.h"

#include <drive/telematics/server/library/config.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/pusher/local.h>
#include <drive/telematics/server/pusher/pusher.h>

#include <drive/telematics/client/library/client.h>
#include <drive/telematics/client/library/handlers.h>

#include <rtline/library/executor/ut/helpers/config.h>
#include <rtline/library/storage/ut/library/helper.h>

namespace {
    struct TTelematicServerPorts {
        ui16 TcpPort;
        ui16 HttpPort;

        TTelematicServerPorts(ui16 index = 0)
            : TcpPort(Singleton<TPortManager>()->GetPortsRange(12345 + index * 10, 2))
            , HttpPort(TcpPort + 1)
        {
        }
    };
}

ui16 TTelematicServerBuilder::GetTcpPort() {
    return Singleton<TTelematicServerPorts>()->TcpPort;
}

ui16 TTelematicServerBuilder::GetHttpPort() {
    return Singleton<TTelematicServerPorts>()->HttpPort;
}

TAtomicSharedPtr<NDrive::TTelematicsConfig> TTelematicServerBuilder::CreateConfig() {
    ui16 tcpPort = Singleton<TTelematicServerPorts>()->TcpPort;
    ui16 httpPort = Singleton<TTelematicServerPorts>()->HttpPort;
    if (Index) {
        TTelematicServerPorts ports(Index);
        tcpPort = ports.TcpPort;
        httpPort = ports.HttpPort;
    }

    auto config = MakeAtomicShared<NDrive::TTelematicsConfig>();
    config->SetPingerInterval(TDuration::Seconds(10));
    config->SetClientServerPort(httpPort);
    config->SetTelematicsServerPort(tcpPort);
    config->SetFastDataInterval(TDuration::Seconds(100));
    config->SetLogLoaderInterval(TDuration::Seconds(5));
    config->SetSimSwitcherInterval(TDuration::Seconds(10));
    config->SetPusherOptions(MakeHolder<TLocalPusherOptions>());
    if (StorageType) {
        config->SetTaskExecutorConfig(BuildExecutorConfig(StorageType));
        config->SetOffsetStorageOptions(BuildStorageOptions(StorageType));
    }
    if (UseRemotePusher) {
        auto pusherOptions = MakeHolder<NDrive::TPusherOptions>();
        pusherOptions->Host = "saas-indexerproxy-maps-prestable.yandex.net";
        pusherOptions->Port = 80;
        pusherOptions->Token = "ac6ac7f7a6544f336202c0b058104374";
        config->SetPusherOptions(std::move(pusherOptions));
    }

    return config;
}

TAtomicSharedPtr<NDrive::TTelematicsServer> TTelematicServerBuilder::CreateServer(TAtomicSharedPtr<NDrive::TTelematicsConfig> config) {
    CHECK_WITH_LOG(config);
    return MakeAtomicShared<NDrive::TTelematicsServer>(*config);
}

TTelematicServerBuilder::TTelematicServerBuilder(ui16 index, const TString& storageType /*= "LOCAL"*/, bool useRemotePusher /*= false*/)
    : StorageType(storageType)
    , UseRemotePusher(useRemotePusher)
    , Index(index)
    , Config(CreateConfig())
    , API(storageType ? MakeAtomicShared<NDrive::TTelematicsApi>(Config->GetTaskExecutorConfig()) : nullptr)
{
    NDrive::TLocatorOptions locatorOptions;
    locatorOptions.SpeedThreshold = 300 * 1000 * 1000;
    Config->SetLocatorOptions(locatorOptions);
}

TTelematicServerBuilder::~TTelematicServerBuilder() {
    if (Server) {
        Server->Stop(0);
    }
}

bool TTelematicServerBuilder::IsRunning() const {
    return Server ? Server->IsRunning() : false;
}

void TTelematicServerBuilder::Run() {
    GetServer()->Run();
}

TAtomicSharedPtr<NDrive::TCarEmulator> TTelematicServerBuilder::BuildEmulator(const TString& imei, TAtomicSharedPtr<NDrive::TTelematicsClientContext> context) {
    if (context) {
        return MakeAtomicShared<NDrive::TCarEmulator>(imei, "localhost", Config->GetTelematicsServerOptions().Port, std::move(context));
    } else {
        return MakeAtomicShared<NDrive::TCarEmulator>(imei, "localhost", Config->GetTelematicsServerOptions().Port);
    }
}

TAtomicSharedPtr<NDrive::TTelematicsConfig> TTelematicServerBuilder::GetConfig() {
    return Config;
}

TAtomicSharedPtr<NDrive::TTelematicsServer> TTelematicServerBuilder::GetServer() {
    if (!Server) {
        Server = CreateServer(Config);
    }
    return Server;
}

TAtomicSharedPtr<NDrive::TTelematicsApi> TTelematicServerBuilder::GetAPI() {
    if (!API) {
        API = MakeAtomicShared<NDrive::TTelematicsApi>(Config->GetTaskExecutorConfig());
    }
    return API;
}

bool TTelematicServerBuilder::SetSensorDoubleValue(const NDrive::TCarEmulator& emulator, const ui16 sensorId, const double value) {
    auto handle = API->SetParameter<double>(emulator.GetIMEI(), sensorId, 0, value);
    while (!API->Await(handle)) {
        Sleep(TDuration::MilliSeconds(100));
    }
    CHECK_WITH_LOG(API->GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
    return true;
}

NDrive::NProtocol::TArray<NDrive::NVega::TBlackboxRecords::TRecord, ui16> Blackbox() {
    using namespace NDrive::NVega;

    NDrive::NProtocol::TArray<TBlackboxRecords::TRecord, ui16> records;
    auto& record = records.emplace_back();
    record.Longitude = 37.1;
    record.Lattitude = 55.5;
    record.Course = 5;
    record.Height = -1;
    record.Satelites = 14;
    record.Speed = 7;
    record.Timestamp = Seconds();

    {
        auto& param = record.Parameters.Mutable().emplace_back();
        param.Id = 1234;
        param.SetValue<ui16>(4242);
    }
    {
        auto& param = record.Parameters.Mutable().emplace_back();
        param.Id = 42;
        param.SetValue<TStringBuf>("fakefake");
    }
    {
        auto& param = record.Parameters.Mutable().emplace_back();
        param.Id = VEGA_HDOP;
        param.SetValue<double>(5);
    }

    return records;
}

NDrive::NNavTelecom::TRecord NavTelecomBlackbox() {
    using namespace NDrive::NNavTelecom;

    TRecord record;
    auto& parameters = record.Parameters;

    parameters.emplace_back(PI_TIMESTAMP, 0, ui64(1648553411));
    parameters.emplace_back(PI_LATITUDE, 0, i64(33897737));
    parameters.emplace_back(PI_LONGITUDE, 0, i64(50976672));
    parameters.emplace_back(PI_HEIGHT, 0, double(1129));
    parameters.emplace_back(PI_SPEED, 0, double(47.9));
    parameters.emplace_back(PI_COURSE, 0, double(74));
    parameters.emplace_back(PI_GPS_STATUS, 0, ui64(48));

    return record;
}

NDrive::NWialon::TShortData WialonBlackbox() {
    using namespace NDrive::NWialon;

    TShortData record;

    record.DateTime = TInstant::Seconds(1648553411);
    record.Lattitude.Set(TCoordinate(37.1));
    record.Longitude.Set(TCoordinate(55.5));
    record.Course.Set(5);
    record.Height.Set(-1);
    record.Satellites.Set(14);
    record.Speed.Set(7);

    return record;
}
