#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/sensors/cache.h>

#include <drive/telematics/api/client.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/threading/future/future.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/cast.h>

Y_UNIT_TEST_SUITE(TelematicsServerLocation) {
    void AddCoordinate(NDrive::TSensorsCache& sensors, TGeoCoord coordinate, TInstant timestamp, double course = 42) {
        sensors.Add(VEGA_LAT, 0, coordinate.Y, timestamp);
        sensors.Add(VEGA_LON, 0, coordinate.X, timestamp);
        sensors.Add(VEGA_DIR, 0, course, timestamp);
    }

    void CheckLocation(const NDrive::TLocation& location, const TGeoCoord& coordinate) {
        UNIT_ASSERT_DOUBLES_EQUAL(location.Latitude, coordinate.Y, 0.0001);
        UNIT_ASSERT_DOUBLES_EQUAL(location.Longitude, coordinate.X, 0.0001);
    }

    Y_UNIT_TEST(LocatorSimple) {
        TString imei = "42";
        NDrive::TLocator locator;
        NDrive::TSensorsCache sensors(16);

        TGeoCoord coordinate = { 27.526108, 53.891758 };
        TInstant timestamp = TInstant::Seconds(1536583420);
        AddCoordinate(sensors, coordinate, timestamp);

        auto expectedType = NDrive::TLocation::EType::GPSCurrent;
        auto checker = [&](const NDrive::TLocation& location) {
            UNIT_ASSERT_DOUBLES_EQUAL(location.Latitude, coordinate.Y, 0.0001);
            UNIT_ASSERT_DOUBLES_EQUAL(location.Longitude, coordinate.X, 0.0001);
            UNIT_ASSERT_VALUES_EQUAL(location.Timestamp, timestamp);
            UNIT_ASSERT_VALUES_EQUAL(location.Type, expectedType);
        };

        auto location = locator.Locate(imei, sensors).ExtractValueSync();
        checker(location);
        UNIT_ASSERT(!location.Base);

        TInstant timestamp1 = timestamp + TDuration::Seconds(42);
        sensors.Add(VEGA_LAT, 0, double(0), timestamp1);
        sensors.Add(VEGA_LON, 0, double(0), timestamp1);
        sensors.Add(VEGA_DIR, 0, double(0), timestamp1);

        expectedType = NDrive::TLocation::EType::GPSPrevious;
        auto location1 = locator.Locate(imei, sensors).ExtractValueSync();
        checker(location1);
        UNIT_ASSERT(location1.Base);
        UNIT_ASSERT_VALUES_EQUAL(location1.Base->Timestamp, timestamp1);
    }

    Y_UNIT_TEST(LocatorVKO) {
        TString imei = "42";
        NDrive::TLocator locator;
        NDrive::TSensorsCache sensors(16);

        TGeoCoord first = { 37.587679, 55.733296 };
        TGeoCoord firstAndHalf = { 37.587379, 55.733496 };
        TGeoCoord second = { 37.587180, 55.733614 };
        TInstant timestamp = TInstant::Seconds(1536666696);
        AddCoordinate(sensors, first, timestamp);
        AddCoordinate(sensors, second, timestamp + TDuration::Seconds(5));
        {
            auto location = locator.Locate(imei, sensors).ExtractValueSync();
            CheckLocation(location, second);
        }

        TGeoCoord third = { 37.276182, 55.601048 };
        AddCoordinate(sensors, third, timestamp + TDuration::Seconds(15));
        {
            auto location = locator.Locate(imei, sensors).ExtractValueSync();
            CheckLocation(location, second);
            UNIT_ASSERT(location.Base);
            UNIT_ASSERT_DOUBLES_EQUAL(location.Base->Latitude, third.Y, 0.0001);
            UNIT_ASSERT_DOUBLES_EQUAL(location.Base->Longitude, third.X, 0.0001);
            UNIT_ASSERT_VALUES_EQUAL(location.Base->Timestamp, timestamp + TDuration::Seconds(15));
        }

        NDrive::TSensorsCache sensors2(16);
        AddCoordinate(sensors2, third, timestamp);
        AddCoordinate(sensors2, first, timestamp + TDuration::Seconds(5));
        AddCoordinate(sensors2, firstAndHalf, timestamp + TDuration::Seconds(10));
        AddCoordinate(sensors2, second, timestamp + TDuration::Seconds(243));
        {
            auto location = locator.Locate(imei, sensors2).ExtractValueSync();
            CheckLocation(location, second);
            UNIT_ASSERT_VALUES_EQUAL(location.Type, NDrive::TLocation::EType::GPSCurrent);
        }
    }

    Y_UNIT_TEST(LocatorVKO2) {
        TString imei = "42";
        NDrive::TLocator locator;
        NDrive::TSensorsCache sensors(16);

        TInstant start = TInstant::Seconds(1537864616);
        TGeoCoord correct0 = { 55.6911, 55.6911 };
        TInstant ct0 = TInstant::Seconds(1537867181);
        TGeoCoord correct1 = { 55.6914, 55.6914 };
        TInstant ct1 = TInstant::Seconds(1537869018);
        TGeoCoord zero0;
        TInstant zt0 = TInstant::Seconds(1537869059);
        TGeoCoord vko0 = { 55.5986, 55.5986 };
        TInstant vt0 = TInstant::Seconds(1537870875);
        TGeoCoord zero1;
        TInstant zt1 = TInstant::Seconds(1537870915);
        TGeoCoord vko1 = { 55.5989, 55.5989 };
        TInstant vt1 = TInstant::Seconds(1537871905);

        AddCoordinate(sensors, correct0, start);
        AddCoordinate(sensors, correct0, ct0);
        AddCoordinate(sensors, correct1, ct1);
        AddCoordinate(sensors, zero0, ct1 + TDuration::Seconds(10));
        AddCoordinate(sensors, zero0, zt0);
        AddCoordinate(sensors, vko0, zt0 + TDuration::Seconds(10));
        AddCoordinate(sensors, vko0, vt0);
        AddCoordinate(sensors, zero1, vt0 + TDuration::Seconds(10));
        AddCoordinate(sensors, zero1, zt1);
        AddCoordinate(sensors, vko1, zt1 + TDuration::Seconds(10));
        AddCoordinate(sensors, vko1, vt1);

        NJson::TJsonValue d;
        sensors.Fill(d);

        auto location = locator.Locate(imei, sensors).GetValueSync();
        UNIT_ASSERT_VALUES_EQUAL(location.Type, NDrive::TLocation::EType::GPSPrevious);
    }

    Y_UNIT_TEST(LocationRestrictedAreas) {
        TString imei = "42";
        NDrive::TLocator::TOptions options;
        TVector<TGeoCoord> coordinates;
        UNIT_ASSERT(TGeoCoord::DeserializeVector(
            "37.39682328484823 55.967731624463596 37.433301327450785 55.97129362593212 37.427636502011325 55.98111149239805 37.395020840390224 55.97774289866315 37.39682328484823 55.967731624463596",
            coordinates
        ));
        options.RestrictedAreas.emplace_back(coordinates);
        NDrive::TLocator locator(options);
        NDrive::TSensorsCache sensors(16);

        TInstant start = TInstant::Seconds(1537864616);
        TGeoCoord correct = { 37.405449269040126, 55.96128067559328 };

        TInstant finish = start + TDuration::Seconds(100);
        TGeoCoord bad = { 37.412299537236954, 55.973550671162265 };

        AddCoordinate(sensors, correct, start);
        {
            auto location = locator.Locate(imei, sensors).GetValueSync();
            UNIT_ASSERT_VALUES_EQUAL(location.Type, NDrive::TLocation::EType::GPSCurrent);
        }

        AddCoordinate(sensors, bad, finish);
        {
            auto location = locator.Locate(imei, sensors).GetValueSync();
            UNIT_ASSERT_VALUES_EQUAL(location.Type, NDrive::TLocation::EType::GPSPrevious);
        }
    }

    Y_UNIT_TEST(LocationSerialization) {
        NDrive::TLocation original;
        original.MutableBase().Latitude = 42;
        original.MutableBase().Name = "bbb";
        UNIT_ASSERT(original.Base);

        auto serialized = NJson::ToJson(original);
        auto deserialized = NJson::FromJson<NDrive::TLocation>(serialized);

        UNIT_ASSERT(deserialized.Base);
        UNIT_ASSERT_DOUBLES_EQUAL(deserialized.Base->Latitude, 42, 0.001);
        UNIT_ASSERT_VALUES_EQUAL(deserialized.Base->Name, "bbb");
    }

    Y_UNIT_TEST(LatLonTimestampSplit) {
        TString imei = "42";
        NDrive::TLocator::TOptions locatorOptions;
        locatorOptions.EnableClusterization = false;
        NDrive::TLocator locator(locatorOptions);
        NDrive::TSensorsCache sensors(16);
        TInstant timestamp = TInstant::Seconds(1582046422);
        sensors.Add(VEGA_LAT, 0, 55.1, timestamp);
        sensors.Add(VEGA_LON, 0, 37.1, timestamp);
        sensors.Add(VEGA_DIR, 0, 42.0, timestamp);

        sensors.Add(VEGA_LAT, 0, 0.0, timestamp + TDuration::Minutes(1));
        sensors.Add(VEGA_LON, 0, 0.0, timestamp + TDuration::Minutes(2));

        auto asyncLocation = locator.Locate(imei, sensors);
        auto location = asyncLocation.ExtractValueSync();
        UNIT_ASSERT_VALUES_EQUAL(location.Timestamp, timestamp + TDuration::Minutes(1));
    }
}
