#include <drive/telematics/api/server/sd.h>
#include <drive/telematics/api/server/client.h>
#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/library/config.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>


namespace {
    class TMessageHandler: public NDrive::TCommonHandler {
    public:
        TMessageHandler(NDrive::NNavTelecom::EMessageType messageType)
            : MessageType_(messageType)
        {
        }

        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& /*connection*/, const NDrive::NProtocol::IMessage& message) override {
            auto protocol = message.GetProtocolType();
            auto type = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();
            Clog << "receive message " << ToString(protocol) << " " << ToString(type) << Endl;

            if (protocol == NDrive::NProtocol::PT_NAVTELECOM && type == MessageType_) {
                Signal();
                return NDrive::TTelematicsTestClient::EHandlerStatus::Finish;
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }

    private:
        NDrive::NNavTelecom::EMessageType MessageType_;
    };

    TAtomicSharedPtr<TMessageHandler> SendMessage(
        NDrive::TTelematicsTestClient& client,
        THolder<NDrive::NProtocol::IMessage> message,
        NDrive::NNavTelecom::EMessageType answerType
    ) {
        auto handler = MakeAtomicShared<TMessageHandler>(answerType);
        client.AddMessageHandler(handler);
        client.SendMessage(std::move(message));
        return std::move(handler);
    }

    inline constexpr size_t DataSize = 122;

    NDrive::NNavTelecom::TBitField GenerateBitField() {
        NDrive::NNavTelecom::TBitField result;
        result.Reserve(DataSize);

        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_ID));
        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_EVENT_ID));
        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_TIMESTAMP));
        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_DEVICE_STATUS));
        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_GSM_LEVEL));
        result.Set(static_cast<size_t>(NDrive::NNavTelecom::PI_MAIN_VOLTAGE));

        while (result.Size() < DataSize) {
            result.Push(false);
        }

        return result;
    }

    THolder<NDrive::NNavTelecom::THandShakeRequest> GenerateHandshake(std::string_view imei) {
        auto result = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::THandShakeRequest>();
        result->IMEI.Set(imei);
        return std::move(result);
    }

    THolder<NDrive::NNavTelecom::TProtocolSettingRequest> GenerateSettings() {
        auto result = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TProtocolSettingRequest>();

        result->Protocol = 0xB0;
        result->ProtocolVersion = NDrive::NNavTelecom::EProtocolVersion::Second;
        result->StructVersion = NDrive::NNavTelecom::EStructVersion::Second;
        result->DataSize = DataSize;
        result->BitField = GenerateBitField();

        return std::move(result);
    }

    NDrive::NNavTelecom::TRecord GenerateBlackBoxRecord() {
        NDrive::NNavTelecom::TRecord record;
        record.Parameters = {
            { NDrive::NNavTelecom::PI_ID, 0, ui64(23) },
            { NDrive::NNavTelecom::PI_EVENT_ID, 0, ui64(12) },
            { NDrive::NNavTelecom::PI_TIMESTAMP, 0, Now().Seconds() },
            { NDrive::NNavTelecom::PI_DEVICE_STATUS, 0, ui64(1) },
            { NDrive::NNavTelecom::PI_GSM_LEVEL, 0, ui64(18) },
            { NDrive::NNavTelecom::PI_MAIN_VOLTAGE, 0, ui64(11) }
        };
        return record;
    }

    THolder<NDrive::NNavTelecom::TBlackBoxRequest> GenerateBlackBox() {
        auto result = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TBlackBoxRequest>(GenerateBitField());
        result->Records.emplace_back(GenerateBlackBoxRecord());
        return std::move(result);
    }

    THolder<NDrive::NNavTelecom::TICCIDAnswer> GenerateICCIDAnswer() {
        auto result = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TICCIDAnswer>();
        result->ICCID = "11112345555566677789";
        return std::move(result);
    }

    void SendMessageAndWait(
        NDrive::TTelematicsTestClient& client,
        THolder<NDrive::NProtocol::IPayload>&& payload,
        NDrive::NNavTelecom::EMessageType answer
    ) {
        auto message = MakeHolder<NDrive::NNavTelecom::TMessage>(std::move(payload));
        auto handler = SendMessage(client, std::move(message), answer);
        UNIT_ASSERT(handler);
        handler->Wait();
        UNIT_ASSERT_C(!handler->WasDropped(), ToString(answer));
    }

    template <class F, typename... Args>
    void SendMessage(NDrive::TTelematicsTestClient& client, NDrive::NNavTelecom::EMessageType answer, F&& generator, Args&&... args) {
        SendMessageAndWait(client, generator(std::forward<Args>(args)...), answer);
    }

    void DefaultState(NDrive::TTelematicsTestClient& client, std::string_view imei) {
        SendMessage(client, NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER, GenerateHandshake, imei);
        SendMessage(client, NDrive::NNavTelecom::MT_PROTOCOL_SETTING_ANSWER, GenerateSettings);
        SendMessage(client, NDrive::NNavTelecom::MT_ICCID_REQUEST, GenerateBlackBox);
        SendMessage(client, NDrive::NNavTelecom::MT_BLACKBOX_ANSWER, GenerateICCIDAnswer);
    }

    template <ui8 N>
    NDrive::NVega::TCommand GenerateDigitalOutputCommand(bool state) {
        NDrive::NVega::TCommandRequest::TSetParameter setParameter;
        setParameter.Id = NDrive::NVega::DigitalOutput<N>();
        setParameter.SetValue(ui8(state));
        NDrive::NVega::TArgument argument;

        argument.Set(setParameter);
        return { NDrive::NVega::SET_PARAM, std::move(argument) };
    }

    class TDigitalOutputHandler: public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            auto protocol = message.GetProtocolType();
            auto type = message.GetMessageTypeAs<NDrive::NNavTelecom::EMessageType>();

            if (protocol == NDrive::NProtocol::PT_NAVTELECOM && type == NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_COMMAND) {
                auto receive = message.As<NDrive::NNavTelecom::TDigitalOutputCommandRequest>();
                auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TDigitalOutputCommandAnswer>(GenerateBitField());
                payload->Record = GenerateBlackBoxRecord();

                auto message = MakeHolder<NDrive::NNavTelecom::TMessage>(std::move(payload));
                connection.SendMessage(std::move(message));
                return NDrive::TTelematicsTestClient::EHandlerStatus::Finish;
            }

            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    };

    TAtomicSharedPtr<TMessageHandler> SendMessage(
        NDrive::TTelematicsTestClient& client,
        std::string_view data,
        NDrive::NNavTelecom::EMessageType answerMessageType
    ) {
        auto handler = MakeAtomicShared<TMessageHandler>(answerMessageType);

        client.AddMessageHandler(handler);
        client.Send(data);

        return std::move(handler);
    }
}

Y_UNIT_TEST_SUITE(TelematicsServerNavtelecomSuite) {
    Y_UNIT_TEST(RealData) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetTraceInputTraffic(true);
        config->SetProtocolType(NDrive::NProtocol::PT_AUTODETECT);

        tmBuilder.Run();

        const TString imei = "100001000010000";
        NDrive::TTelematicsTestClient client(imei);
        client.SetProtocol(NDrive::NProtocol::PT_NAVTELECOM);
        client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
        UNIT_ASSERT(client.Alive());

        TString handshake = HexDecode("404E54432A0100000000000013004E6F2A3E533A383636383534303531313434313036");
        TString settings = HexDecode("404E54432A010000000000001A00EAC22A3E464C4558B014147AFFFE2008080000000200000000000000");
        TString blackbox = HexDecode(
            "7E4102EE9E0200041715CFC66200A980124B15CFC6623E93040215560A034206" \
            "0000ECA3854283007A969046983680709431000609EF9E0200041750CFC66200" \
            "A980125350CFC66210850402DF720A03C20600007B5A7B4283007F989046A336" \
            "80AB9431000508A4"
        );

        TMap<NDrive::NNavTelecom::EMessageType, TString> states = {
            { NDrive::NNavTelecom::MT_HANDSHAKE_ANSWER, handshake },
            { NDrive::NNavTelecom::MT_PROTOCOL_SETTING_ANSWER, settings},
            { NDrive::NNavTelecom::MT_ICCID_REQUEST, blackbox}
        };

        for (const auto& state : states) {
            auto handler = SendMessage(client, state.second, state.first);
            UNIT_ASSERT(handler);
            handler->Wait();
            UNIT_ASSERT(!handler->WasDropped());
        }
    }

    Y_UNIT_TEST(MainState) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetTraceInputTraffic(true);
        config->SetProtocolType(NDrive::NProtocol::PT_AUTODETECT);

        tmBuilder.Run();

        const TString imei = "100001000010000";
        NDrive::TTelematicsTestClient client(imei);
        client.SetAuthorized(true);
        client.SetProtocol(NDrive::NProtocol::PT_NAVTELECOM);
        client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
        UNIT_ASSERT(client.Alive());

        DefaultState(client, imei);
    }

    Y_UNIT_TEST(SetParamHandler) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();

        config->SetProtocolType(NDrive::NProtocol::PT_AUTODETECT);
        config->SetTraceInputTraffic(true);
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient api(sd);
        {
            const TString imei = "100001000010000";
            NDrive::TTelematicsTestClient client(imei);
            client.SetAuthorized(true);
            client.SetProtocol(NDrive::NProtocol::PT_NAVTELECOM);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            UNIT_ASSERT(client.Alive());

            DefaultState(client, imei);

            {
                auto handler = MakeAtomicShared<TDigitalOutputHandler>();
                client.AddHandler(handler);
            }

            auto handler = api.Command(imei, GenerateDigitalOutputCommand<1>(true));
            SendMessage(client, NDrive::NNavTelecom::MT_DIGITAL_OUTPUT_COMMAND, GenerateBlackBox);
            handler.GetFuture().Wait();

            auto status = handler.GetStatus();
            UNIT_ASSERT_EQUAL_C(status, NDrive::TTelematicsClient::EStatus::Success, handler.GetMessage());
        }
    }
}
