#include <drive/telematics/server/ut/library/helper.h>

#include <drive/telematics/server/data/blackbox.h>
#include <drive/telematics/server/library/handlers.h>
#include <drive/telematics/server/library/logging.h>
#include <drive/telematics/server/library/pin_resetter.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/library/tasks.h>
#include <drive/telematics/server/location/beacon_recognizer.h>
#include <drive/telematics/server/location/locator.h>
#include <drive/telematics/server/metadata/interface.h>

#include <drive/telematics/api/client.h>
#include <drive/telematics/api/sensor/local.h>
#include <drive/telematics/api/server/client.h>
#include <drive/telematics/client/car_emulator/handlers.h>
#include <drive/telematics/client/library/client.h>
#include <drive/telematics/client/library/context.h>
#include <drive/telematics/client/library/proxy.h>
#include <drive/telematics/protocol/actions.h>
#include <drive/telematics/protocol/errors.h>
#include <drive/telematics/protocol/navtelecom.h>
#include <drive/telematics/protocol/wialon.h>

#include <drive/library/cpp/yt/node/cast.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/unistat/unistat.h>
#include <library/cpp/yson/node/node.h>
#include <library/cpp/yson/node/node_io.h>

#include <rtline/library/executor/ut/helpers/config.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>

namespace {
    class TExecutorStarter {
    public:
        TExecutorStarter(TTaskExecutor& executor)
            : Executor(executor)
        {
            Executor.Start();
        }
        ~TExecutorStarter() {
            Executor.Stop();
        }

    private:
        TTaskExecutor& Executor;
    };

    class TDroppedConnection: public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& /*connection*/, const NDrive::NProtocol::IMessage& /*message*/) override {
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    };

    class TPingCounter: public NDrive::TCommonHandler {
    public:
        ui32 GetCount() const {
            return Count;
        }

        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& /*connection*/, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::PING_REQUEST) {
                    Count++;
                    if (Count >= 10) {
                        Signal();
                        return NDrive::TTelematicsTestClient::EHandlerStatus::Finish;
                    }
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }

    private:
        ui32 Count = 0;
    };

    class TCommandFailer: public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::COMMAND_REQUEST) {
                    auto ack = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::COMMAND_RESPONSE);
                    const auto& request = message.As<NDrive::NVega::TCommandRequest>();
                    auto& response = ack->As<NDrive::NVega::TCommandResponse>();
                    response.Id = request.Id;
                    response.Result = response.ERROR;
                    connection.SendMessage(std::move(ack));
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    };

    class TCommandBusyRestart: public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::COMMAND_REQUEST) {
                    const auto& request = message.As<NDrive::NVega::TCommandRequest>();
                    if (request.Code == NDrive::NVega::ECommandCode::RESTART) {
                        auto ack = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::COMMAND_RESPONSE);
                        auto& response = ack->As<NDrive::NVega::TCommandResponse>();
                        response.Id = request.Id;
                        if (Count % 2) {
                            response.Result = response.PROCESSED;
                        } else {
                            response.Result = response.BUSY;
                        }
                        Count++;
                        connection.SendMessage(std::move(ack));
                    }
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }

    private:
        ui32 Count = 0;
    };

    class TCommandSlowRestart: public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::COMMAND_REQUEST) {
                    const auto& request = message.As<NDrive::NVega::TCommandRequest>();
                    if (request.Code == NDrive::NVega::ECommandCode::RESTART) {
                        auto ack = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::COMMAND_RESPONSE);
                        auto& response = ack->As<NDrive::NVega::TCommandResponse>();
                        response.Id = request.Id;
                        response.Result = response.PROCESSED;
                        if (Count % 2) {
                            connection.SendMessage(std::move(ack));
                        } else {
                            INFO_LOG << "ignoring RESTART " << Count << Endl;
                        }
                        Count++;
                    }
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }

    private:
        ui32 Count = 0;
    };

    class TParameterHandler: public NDrive::TCommonHandler {
    public:
        TParameterHandler(ui16 id, ui16 value)
            : Id(id)
            , Value(value)
            , Got(0)
            , Set(0)
        {
        }

        ui64 GetGot() const {
            return Got;
        }
        ui64 GetSet() const {
            return Set;
        }

        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::COMMAND_REQUEST) {
                    const auto& request = message.As<NDrive::NVega::TCommandRequest>();
                    if (request.Code == NDrive::NVega::ECommandCode::GET_PARAM) {
                        const auto& argument = request.Argument.Get<NDrive::NVega::TCommandRequest::TGetParameter>();
                        if (argument.Id == Id) {
                            auto ack = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::COMMAND_RESPONSE);
                            auto& response = ack->As<NDrive::NVega::TCommandResponse>();
                            response.Id = request.Id;
                            response.Result = response.PROCESSED;

                            NDrive::NVega::TCommandResponse::TGetParameter parameter;
                            parameter.Id = Id;
                            parameter.SetValue<ui64>(Value);
                            response.Argument.Set(parameter);

                            connection.SendMessage(std::move(ack));
                            Got += 1;
                            INFO_LOG << "replying " << Id << ":" << Value << Endl;
                            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
                        }
                    }
                    if (request.Code == NDrive::NVega::ECommandCode::SET_PARAM) {
                        const auto& argument = request.Argument.Get<NDrive::NVega::TCommandRequest::TSetParameter>();
                        if (argument.Id == Id) {
                            auto ack = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::COMMAND_RESPONSE);
                            auto& response = ack->As<NDrive::NVega::TCommandResponse>();
                            response.Id = request.Id;

                            auto value = argument.GetValue();
                            if (!std::holds_alternative<ui64>(value)) {
                                response.Result = response.INCORRECT;
                            } else if (std::get<ui64>(value) > Max<decltype(Value)>()) {
                                response.Result = response.INCORRECT;
                            } else {
                                Value = std::get<ui64>(value);
                                response.Result = response.PROCESSED;
                            }

                            connection.SendMessage(std::move(ack));
                            Set += 1;
                            INFO_LOG << "replying " << Id << ":" << Value << Endl;
                            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
                        }
                    }
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }

        ui16 GetValue() const {
            return Value;
        }
        void SetValue(ui16 value) {
            Value = value;
        }

    private:
        ui16 Id;
        ui16 Value;
        ui64 Got;
        ui64 Set;
    };

    class TInterfaceHandler : public NDrive::TCommonHandler {
    public:
        NDrive::TTelematicsTestClient::EHandlerStatus OnMessage(NDrive::NVega::IConnection& connection, const NDrive::NProtocol::IMessage& message) override {
            if (message.GetProtocolType() == NDrive::NProtocol::PT_VEGA) {
                if (message.GetMessageType() == NDrive::NVega::INTERFACE_REQUEST) {
                    const auto& request = message.As<NDrive::NVega::TInterfaceRequest>();
                    auto response = MakeHolder<NDrive::NVega::TMessage>(NDrive::NVega::INTERFACE_RESPONSE);
                    auto& payload = response->As<NDrive::NVega::TInterfaceResponse>();
                    payload.Interface = request.Interface;
                    payload.Data = request.Data;
                    connection.SendMessage(std::move(response));
                }
            }
            return NDrive::TTelematicsTestClient::EHandlerStatus::Continue;
        }
    };

    NJson::TJsonValue Fetch(const TString& request, TStringBuf data = {}) {
        NNeh::TMessage message = NNeh::TMessage::FromString(request);
        NNeh::NHttp::MakeFullRequest(message, {}, data);
        const NNeh::THandleRef handle = NNeh::Request(message);
        UNIT_ASSERT(handle);
        const NNeh::TResponseRef response = handle->Wait();
        UNIT_ASSERT(response);
        UNIT_ASSERT(!response->IsError());

        INFO_LOG << request << " : " << response->Data << Endl;
        NJson::TJsonValue result;
        NJson::ReadJsonFastTree(response->Data, &result, true);
        return result;
    }

    NJson::TJsonValue GetUnistat() {
        TString dump = TUnistat::Instance().CreateJsonDump(Min<int>(), true);
        INFO_LOG << "unistat : " << dump << Endl;
        NJson::TJsonValue result;
        NJson::ReadJsonFastTree(dump, &result, true);
        return result;
    }

    double GetSignal(const NJson::TJsonValue& unistat, const TString& name) {
        UNIT_ASSERT(unistat.IsArray());
        for (auto&& s : unistat.GetArraySafe()) {
            UNIT_ASSERT(s.IsArray());
            const auto& fields = s.GetArraySafe();
            UNIT_ASSERT_VALUES_EQUAL(fields.size(), 2);
            if (fields[0].GetStringSafe() == name) {
                return fields[1].GetDoubleRobust();
            }
        }
        ythrow yexception() << "signal " << name << " is not found";
    }

    class TDummyLogValue {
    public:
        void Set(TStringBuf data) {
            Data = data;
        }
        TStringBuf Get() const {
            return Data;
        }
        void Clear() {
            Data.clear();
        }

    private:
        TString Data;
    };

    class TDummyLogBackend: public TLogBackend {
    public:
        TDummyLogBackend() = default;
        virtual ~TDummyLogBackend() = default;

        void WriteData(const TLogRecord& rec) override {
            TStringBuf data{ rec.Data, rec.Len };
            Singleton<TDummyLogValue>()->Set(data);
        }

        void ReopenLog() override {
        }
    };

    class TDummyLogBackendKeeper {
    public:
        TDummyLogBackendKeeper() {
            auto backend = MakeHolder<TDummyLogBackend>();
            TLoggerOperator<NDrive::TTelematicsLog>::Set(new NDrive::TTelematicsLog(std::move(backend)));
        }

        virtual ~TDummyLogBackendKeeper() {
            TLoggerOperator<NDrive::TTelematicsLog>::Set(nullptr);
            Singleton<TDummyLogValue>()->Clear();
        }
    };
}

Y_UNIT_TEST_SUITE(TelematicsServerSuite) {
    Y_UNIT_TEST(TCPServer) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            NDrive::TTelematicsConnectionPtr connection;
            for (size_t i = 0; !connection && (i < 3); ++i) {
                INFO_LOG << "probing connection" << Endl;
                connection = server->GetConnection(imei);
                Sleep(TDuration::Seconds(1));
            }
            UNIT_ASSERT(connection);

            auto unistat = GetUnistat();
            double established = GetSignal(unistat, "connection-established_dmmm");
            double registered = GetSignal(unistat, "connection-registered_dmmm");
            double total = GetSignal(unistat, "connection-total_ammv");
            UNIT_ASSERT_DOUBLES_EQUAL(established, 1.0, 0.001);
            UNIT_ASSERT_DOUBLES_EQUAL(registered, 1.0, 0.001);
            UNIT_ASSERT_DOUBLES_EQUAL(total, 1.0, 0.001);
        }
        auto unistat = GetUnistat();
        double established = GetSignal(unistat, "connection-established_dmmm");
        double destroyed = GetSignal(unistat, "connection-destroyed_dmmm");
        double deregistered = GetSignal(unistat, "connection-deregistered_dmmm");
        double registered = GetSignal(unistat, "connection-registered_dmmm");
        double total = GetSignal(unistat, "connection-total_ammv");
        UNIT_ASSERT_DOUBLES_EQUAL(established, 1.0, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(destroyed, 1.0, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(deregistered, 1.0, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(registered, 1.0, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(total, 0.0, 0.001);
    }

    Y_UNIT_TEST(Pinger) {
        TTelematicServerBuilder tmBuilder;

        auto config = tmBuilder.GetConfig();
        config->SetPingerInterval(TDuration::MilliSeconds(100));
        auto server = tmBuilder.GetServer();

        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            NDrive::TTelematicsTestClient client(imei);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            auto counter = MakeAtomicShared<TPingCounter>();
            client.AddHandler(counter);
            counter->Wait();
            UNIT_ASSERT(!counter->WasDropped());
            UNIT_ASSERT_VALUES_EQUAL(counter->GetCount(), 10);
        }
    }

    Y_UNIT_TEST(PingByTask) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        TTaskExecutor local(config->GetTaskExecutorConfig(), nullptr);
        local.SetSelector(new TNoTasksSelector(nullptr));
        TExecutorStarter es(local);
        {
            const TString imei = "100001000010000";

            NDrive::TCommonDistributedTaskMetaInfo info("me", imei);
            NDrive::TPingDistributedData data(info);
            NDrive::TPingDistributedTaskLite task(info);

            local.StoreData2(&data);
            local.StoreTask2(&task);
            local.EnqueueTask(task.GetIdentifier(), data.GetIdentifier());

            NDrive::TTelematicsTestClient client(imei);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            THolder<IDistributedData> d;
            for (ui32 i = 0; i < 10; ++i) {
                d = local.RestoreDataInfo(data.GetIdentifier());
                UNIT_ASSERT(d);
                if (d->GetIsFinished()) {
                    break;
                }
                Sleep(TDuration::Seconds(1));
            }
            UNIT_ASSERT(d);
            UNIT_ASSERT(d->GetIsFinished());

            const auto dd = dynamic_cast<const NDrive::TPingDistributedData*>(d.Get());
            UNIT_ASSERT(dd);
            UNIT_ASSERT_VALUES_EQUAL(static_cast<ui32>(dd->GetStatus()), static_cast<ui32>(NDrive::NProto::TTelematicsTask::SUCCESS));
        }
    }

    Y_UNIT_TEST(ExpelConnection) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            const auto heartbeat1 = MakeAtomicShared<NDrive::TOnHeartbeat>();
            const auto droppedConnection = MakeAtomicShared<TDroppedConnection>();
            NDrive::TTelematicsTestClient client1(imei, nullptr, "1");
            client1.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            client1.AddHandler(heartbeat1);
            client1.AddHandler(droppedConnection);
            heartbeat1->Wait();
            UNIT_ASSERT(!heartbeat1->WasDropped());
            UNIT_ASSERT(client1.Alive());

            const auto heartbeat2 = MakeAtomicShared<NDrive::TOnHeartbeat>();
            NDrive::TTelematicsTestClient client2(imei, nullptr, "2");
            client2.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            client2.AddHandler(heartbeat2);
            heartbeat2->Wait();
            UNIT_ASSERT(!heartbeat2->WasDropped());
            UNIT_ASSERT(client2.Alive());

            droppedConnection->Wait();
            UNIT_ASSERT(droppedConnection->WasDropped());
            UNIT_ASSERT(!client1.Alive());
            UNIT_ASSERT_DOUBLES_EQUAL(GetSignal(GetUnistat(), "connection-expelled_dmmm"), 1, 0.001);
        }
    }

    Y_UNIT_TEST(SimpleCommands) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            NDrive::TTelematicsTestClient client(imei);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            client.AddHandler(MakeAtomicShared<TCommandFailer>());

            NDrive::TTelematicsConnectionPtr connection;
            for (size_t i = 0; !connection && (i < 3); ++i) {
                INFO_LOG << "probing connection" << Endl;
                connection = server->GetConnection(imei);
                Sleep(TDuration::Seconds(1));
            }
            UNIT_ASSERT(connection);

            const TString taskId = "fakefake";
            connection->AddHandler(
                server->CreateTask<NDrive::TSendCommandTask>(
                    taskId,
                    NDrive::NVega::ECommandCode::YADRIVE_WARMING,
                    Default<NDrive::NVega::TArgument>()
                )
            );
            auto task = server->GetTask(taskId);
            UNIT_ASSERT(task);
            task->Wait();
            UNIT_ASSERT(task->GetStatus() == NDrive::TCommonTask::EStatus::Failure);
        }
    }

    Y_UNIT_TEST(GetParameter) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            const ui16 id = VEGA_TRIP_COUNTER;
            const ui16 value = 4242;
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            const auto getParameterHandler = MakeAtomicShared<TParameterHandler>(id, value);
            client.AddHandler(getParameterHandler);

            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);
            auto connectionList = Fetch(hostPort + "/connection/list/");
            UNIT_ASSERT(connectionList.Has("connections"));
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().at(0)["imei"].GetStringRobust(), imei);

            NDrive::NVega::TCommand command;
            command.Code = NDrive::NVega::ECommandCode::GET_PARAM;
            NDrive::NVega::TCommandRequest::TGetParameter parameter;
            parameter.Id = id;
            command.Argument.Set(parameter);

            auto tasks = Fetch(
                hostPort + "/connection/command/?imei=" + imei,
                NJson::ToJson(command).GetStringRobust()
            );
            UNIT_ASSERT(tasks.Has("id"));
            const TString& taskId = tasks["id"].GetStringSafe();
            UNIT_ASSERT(taskId);

            auto taskResult = Fetch(hostPort + "/task/wait/?id=" + taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["id"].GetStringSafe(), taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["status"].GetStringSafe(), "success");
            UNIT_ASSERT_VALUES_EQUAL(taskResult["result"]["sensor"]["id"].GetUIntegerRobust(), id);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["result"]["sensor"]["value"].GetUIntegerRobust(), value);

            auto sensorsResult = Fetch(hostPort + "/connection/sensors/?imei=" + imei);
            UNIT_ASSERT(sensorsResult.IsArray());
            UNIT_ASSERT(!sensorsResult.GetArraySafe().empty());

            bool found = false;
            for (auto&& sensor : sensorsResult.GetArraySafe()) {
                const ui64 i = sensor["id"].GetUIntegerSafe();
                if (i == id) {
                    found = true;
                    UNIT_ASSERT_VALUES_EQUAL(sensor["values"].GetArraySafe().size(), 2);
                    UNIT_ASSERT_VALUES_EQUAL(sensor["values"].GetArraySafe()[0]["value"].GetUIntegerSafe(), value);
                }
            }
            UNIT_ASSERT(found);
        }
    }

    Y_UNIT_TEST(GetParameterByTask) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        TTaskExecutor local(config->GetTaskExecutorConfig(), nullptr);
        local.SetSelector(new TNoTasksSelector(nullptr));
        TExecutorStarter es(local);
        {
            const TString imei = "100001000010000";
            const ui16 id = VEGA_TRIP_COUNTER;
            const ui16 value = 4242;
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            TInstant start = Now();
            NDrive::TCommonDistributedTaskMetaInfo info("me", imei);
            NDrive::TSendCommandDistributedData data(info);
            data.SetPostlinger(TDuration::Seconds(30));
            data.SetCommand(NDrive::NVega::ECommandCode::GET_PARAM);

            NDrive::NVega::TCommandRequest::TGetParameter parameter;
            parameter.Id = id;
            data.SetGenericArgumentFrom(parameter);

            NDrive::TSendCommandDistributedTask task(info);

            local.StoreData2(&data);
            local.StoreTask2(&task);
            local.EnqueueTask(task.GetIdentifier(), data.GetIdentifier());

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            const auto getParameterHandler = MakeAtomicShared<TParameterHandler>(id, value);
            client.AddHandler(getParameterHandler);

            THolder<IDistributedData> d;
            for (ui32 i = 0; i < 10; ++i) {
                d = local.RestoreDataInfo(data.GetIdentifier());
                UNIT_ASSERT(d);
                if (d->GetIsFinished()) {
                    break;
                }
                Sleep(TDuration::Seconds(5));
            }
            UNIT_ASSERT(d);
            UNIT_ASSERT(d->GetIsFinished());
            TInstant finish = Now();
            TDuration duration = finish - start;
            INFO_LOG << duration.Seconds() << Endl;
            UNIT_ASSERT(finish - start > data.GetPostlinger());

            const auto dd = dynamic_cast<const NDrive::TSendCommandDistributedData*>(d.Get());
            UNIT_ASSERT(dd);
            UNIT_ASSERT_VALUES_EQUAL(static_cast<ui32>(dd->GetStatus()), static_cast<ui32>(NDrive::NProto::TTelematicsTask::SUCCESS));
            UNIT_ASSERT(dd->GetParameter());
            UNIT_ASSERT(std::holds_alternative<ui64>(dd->GetParameter()->GetValue()));
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(dd->GetParameter()->GetValue()), value);
        }
    }

    Y_UNIT_TEST(GetParameterByApi) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            const ui16 id = VEGA_TRIP_COUNTER;
            const ui16 value = 42;
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient::TOptions opts;
            opts.ReadTimeout = TDuration::Minutes(1);
            NDrive::TTelematicsTestClient client(imei, nullptr, "clnt", opts);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            const auto getParameterHandler = MakeAtomicShared<TParameterHandler>(id, value);
            client.AddHandler(getParameterHandler);

            NDrive::TTelematicsApi::THandlers handles;
            while (!api.Await(handles, 100,
                [&] {
                    INFO_LOG << "requesting " << id << " from " << imei << Endl;
                    return api.GetParameter(imei, id);
                },
                [&] (const NDrive::TTelematicsApi::THandler& handler) {
                    auto value = api.GetValue<ui64>(handler);
                    INFO_LOG << "got " << value << Endl;
                    return value > 50;
                }
            )) {
                getParameterHandler->SetValue(getParameterHandler->GetValue() + 1);
                Sleep(TDuration::MilliSeconds(10));
            }
            UNIT_ASSERT(api.GetStatus(handles) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT(api.GetValue<ui64>(handles) > 50);
        }
    }

    Y_UNIT_TEST(SetParameterByApi) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            const ui16 id = VEGA_SETTING_GPS_MOVE;
            ui16 value = 1;
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient::TOptions opts;
            opts.ReadTimeout = TDuration::Minutes(1);
            NDrive::TTelematicsTestClient client(imei, nullptr, "clnt", opts);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            const auto parameterHandler = MakeAtomicShared<TParameterHandler>(id, value);
            client.AddHandler(parameterHandler);

            {
                auto handle = api.PoliteSetParameter<ui64>(imei, id, 0, value);
                UNIT_ASSERT(api.Wait(handle));
                UNIT_ASSERT_C(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success, api.GetMessage(handle));
                UNIT_ASSERT_VALUES_EQUAL(parameterHandler->GetGot(), 1);
                UNIT_ASSERT_VALUES_EQUAL(parameterHandler->GetSet(), 0);
            }
            {
                auto handle = api.PoliteSetParameter<ui64>(imei, id, 0, value);
                UNIT_ASSERT(api.Wait(handle));
                UNIT_ASSERT_C(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success, api.GetMessage(handle));
                UNIT_ASSERT_VALUES_EQUAL(parameterHandler->GetGot(), 1);
                UNIT_ASSERT_VALUES_EQUAL(parameterHandler->GetSet(), 0);
            }

            value = 42;
            auto handle = api.SetParameter<ui64>(imei, id, 0, value);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(parameterHandler->GetValue(), value);

            NDrive::TTelematicsApi::THandler handle2;
            try {
                handle2 = api.SetParameter<ui64>(imei, id, 0, Max<ui32>());
                while (!api.Await(handle2)) {
                    Sleep(TDuration::MilliSeconds(100));
                }
            } catch (const std::exception& e) {
                INFO_LOG << "Expected exception: " << FormatExc(e) << Endl;
            }

            NJson::TJsonValue subtasksInfo = api.GetSubTasksInfo(handle);
            INFO_LOG << subtasksInfo.GetStringRobust() << Endl;
            UNIT_ASSERT(subtasksInfo.IsArray());
            UNIT_ASSERT_VALUES_EQUAL(subtasksInfo.GetArraySafe().size(), 1);
            UNIT_ASSERT(subtasksInfo.GetArraySafe()[0].Has("created"));

            auto taskId = subtasksInfo[0]["id"].GetString();
            UNIT_ASSERT(taskId);
            auto asyncHandlerDescription = Yensured(server->GetMetadataClient())->GetHandler(taskId);
            auto handlerDescription = asyncHandlerDescription.ExtractValueSync();
            UNIT_ASSERT_C(handlerDescription, taskId);
            INFO_LOG << handlerDescription->Data.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(handlerDescription->IMEI, imei);
            UNIT_ASSERT_VALUES_EQUAL(handlerDescription->Id, taskId);
            UNIT_ASSERT(handlerDescription->Finished);
            UNIT_ASSERT(handlerDescription->Data.IsDefined());
            UNIT_ASSERT(handlerDescription->Timestamp > TInstant::Zero());
        }
    }

    Y_UNIT_TEST(OpenDrDoor) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";

            auto emulator = tmBuilder.BuildEmulator(imei);

            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);
            auto connectionList = Fetch(hostPort + "/connection/list/");
            UNIT_ASSERT(connectionList.Has("connections"));
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().at(0)["imei"].GetStringRobust(), imei);

            NDrive::NVega::TCommand command;
            command.Code = NDrive::NVega::ECommandCode::OPEN_DR_DOOR;
            auto tasks = Fetch(
                hostPort + "/connection/command/?imei=" + imei,
                NJson::ToJson(command).GetStringRobust()
            );

            UNIT_ASSERT(tasks.Has("id"));
            TString taskId = tasks["id"].GetStringSafe();
            UNIT_ASSERT(taskId);

            auto taskResult = Fetch(hostPort + "/task/wait/?id=" + taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["id"].GetStringSafe(), taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["status"].GetStringSafe(), "success");

            UNIT_ASSERT(emulator->GetContext().GetDrDoorOpened());

            tasks = Fetch(
                hostPort + "/connection/command/?imei=" + imei,
                NJson::ToJson(command).GetStringRobust()
            );

            UNIT_ASSERT(tasks.Has("id"));
            taskId = tasks["id"].GetStringSafe();
            UNIT_ASSERT(taskId);

            taskResult = Fetch(hostPort + "/task/wait/?id=" + taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["id"].GetStringSafe(), taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["status"].GetStringSafe(), "failure");
        }
    }

    Y_UNIT_TEST(SimpleCommandsHandler) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            auto multiHandler = MakeAtomicShared<NDrive::TSimpleCommandsHandler>(client);
            client.AddHandler(multiHandler);

            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);
            auto connectionList = Fetch(hostPort + "/connection/list/");
            UNIT_ASSERT(connectionList.Has("connections"));
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().at(0)["imei"].GetStringRobust(), imei);

            NDrive::NVega::TCommand command;
            command.Code = NDrive::NVega::ECommandCode::TRUNK;

            auto tasks = Fetch(
                hostPort + "/connection/command/?imei=" + imei,
                NJson::ToJson(command).GetStringRobust()
            );
            UNIT_ASSERT(tasks.Has("id"));
            const TString& taskId = tasks["id"].GetStringSafe();
            UNIT_ASSERT(taskId);

            auto taskResult = Fetch(hostPort + "/task/wait/?id=" + taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["id"].GetStringSafe(), taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["status"].GetStringSafe(), "success");
        }
    }

    Y_UNIT_TEST(GetParamHandler) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            ui16 id = CAN_FUEL_LEVEL_P;
            const ui8 fuelPercent = 55;

            auto emulator = tmBuilder.BuildEmulator(imei);
            emulator->GetContext().SetFuelPercent(fuelPercent);

            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);
            auto connectionList = Fetch(hostPort + "/connection/list/");
            UNIT_ASSERT(connectionList.Has("connections"));
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(connectionList["connections"].GetArraySafe().at(0)["imei"].GetStringRobust(), imei);

            NDrive::NVega::TCommand command;
            command.Code = NDrive::NVega::ECommandCode::GET_PARAM;
            NDrive::NVega::TCommandRequest::TGetParameter parameter;
            parameter.Id = id;
            command.Argument.Set(parameter);

            auto tasks = Fetch(
                hostPort + "/connection/command/?imei=" + imei,
                NJson::ToJson(command).GetStringRobust()
            );
            UNIT_ASSERT(tasks.Has("id"));
            const TString& taskId = tasks["id"].GetStringSafe();
            UNIT_ASSERT(taskId);

            auto taskResult = Fetch(hostPort + "/task/wait/?id=" + taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["id"].GetStringSafe(), taskId);
            UNIT_ASSERT_VALUES_EQUAL(taskResult["status"].GetStringSafe(), "success");
            UNIT_ASSERT_VALUES_EQUAL(taskResult["result"]["sensor"]["value"], fuelPercent);
        }
    }

    Y_UNIT_TEST(RetryCommand) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);

            auto handler = MakeAtomicShared<TCommandBusyRestart>();
            client.AddHandler(handler);

            auto handle = api.Command(imei, NDrive::NVega::ECommandCode::RESTART);
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
        }
    }

    Y_UNIT_TEST(HedgeCommand) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetPingerInterval(TDuration::Seconds(10));
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei, nullptr, "TCL");
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);

            auto handler = MakeAtomicShared<TCommandSlowRestart>();
            client.AddHandler(handler);

            auto handle = api.Command(imei, NDrive::NVega::ECommandCode::RESTART, TDuration::Minutes(1));
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
        }
    }

    Y_UNIT_TEST(ProxyList) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        {
            NDrive::TTelematicsProxy proxy("localhost", config->GetTelematicsServerOptions().Port);
            auto devices = proxy.List();
            UNIT_ASSERT_VALUES_EQUAL(devices.size(), 1);
        }
    }

    Y_UNIT_TEST(Scenario) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);
            client.AddHandler(MakeAtomicShared<NDrive::TGetParamHandler>(client, context));
            UNIT_ASSERT(context.TrySetEngineStarted(true));
            auto refreshEngineIsOn = api.GetParameter(imei, CAN_ENGINE_IS_ON);
            UNIT_ASSERT(refreshEngineIsOn);
            UNIT_ASSERT(api.Wait(refreshEngineIsOn));
            UNIT_ASSERT(api.GetStatus(refreshEngineIsOn) == NDrive::TTelematicsApi::EStatus::Success);

            auto stopWarming = MakeAtomicShared<NDrive::TOnStopWarming>(client, context);
            client.AddHandler(stopWarming);

            auto openDoors = MakeAtomicShared<NDrive::TOnOpenDoors>(client, context);
            client.AddHandler(openDoors);

            auto handle = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_STOP_WARMING_AND_OPEN_DOORS);
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT(stopWarming->Wait());
            UNIT_ASSERT(openDoors->Wait());
        }
    }

    Y_UNIT_TEST(ConditionalScenario) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);
            client.AddHandler(MakeAtomicShared<NDrive::TGetParamHandler>(client, context));
            auto refreshEngineIsOn = api.GetParameter(imei, CAN_ENGINE_IS_ON);
            UNIT_ASSERT(refreshEngineIsOn);
            UNIT_ASSERT(api.Wait(refreshEngineIsOn));
            UNIT_ASSERT(api.GetStatus(refreshEngineIsOn) == NDrive::TTelematicsApi::EStatus::Success);

            auto openDoors = MakeAtomicShared<NDrive::TOnOpenDoors>(client, context);
            client.AddHandler(openDoors);

            auto handle = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_STOP_WARMING_AND_OPEN_DOORS);
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT(openDoors->Wait());
        }
    }

    Y_UNIT_TEST(LonghornScenario) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);

            auto handler = MakeAtomicShared<TParameterHandler>(NDrive::NVega::DigitalOutput<3>(), ui16(0));
            client.AddHandler(handler);
            UNIT_ASSERT_VALUES_EQUAL(handler->GetValue(), 0);

            auto enable = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_ENABLE_LONGHORN);
            UNIT_ASSERT(enable);
            while (!api.Await(enable)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(enable) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(handler->GetValue(), 1);

            auto disable = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_DISABLE_LONGHORN);
            UNIT_ASSERT(disable);
            while (!api.Await(disable)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(disable) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(handler->GetValue(), 0);
        }
    }

    Y_UNIT_TEST(PoliteForcedEndOfLease) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            auto emulator = tmBuilder.BuildEmulator(imei);
            NDrive::TTelematicsClientContext& ctx = emulator->GetContext();

            UNIT_ASSERT(ctx.TrySetSpeed(100));
            ctx.StartLeasing();
            UNIT_ASSERT(ctx.GetLeasingState());

            auto refreshSpeed = api.GetParameter(imei, VEGA_SPEED);
            UNIT_ASSERT(refreshSpeed);
            UNIT_ASSERT(api.Wait(refreshSpeed));
            UNIT_ASSERT(api.GetStatus(refreshSpeed) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_DOUBLES_EQUAL(api.GetValue<double>(refreshSpeed), 100, 0.001);

            auto cmd = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE);
            UNIT_ASSERT(cmd);
            UNIT_ASSERT(api.Wait(cmd));
            UNIT_ASSERT(api.GetStatus(cmd) == NDrive::TTelematicsApi::EStatus::Failure);
            INFO_LOG << api.GetMessage(cmd) << Endl;
            UNIT_ASSERT(ctx.GetLeasingState());

            UNIT_ASSERT(ctx.TrySetSpeed(0));
            refreshSpeed = api.GetParameter(imei, VEGA_SPEED);
            UNIT_ASSERT(refreshSpeed);
            UNIT_ASSERT(api.Wait(refreshSpeed));
            UNIT_ASSERT(api.GetStatus(refreshSpeed) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_DOUBLES_EQUAL(api.GetValue<double>(refreshSpeed), 0, 0.001);

            cmd = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE);
            UNIT_ASSERT(cmd);
            UNIT_ASSERT(api.Wait(cmd));
            UNIT_ASSERT(api.GetStatus(cmd) == NDrive::TTelematicsApi::EStatus::Failure);
            INFO_LOG << api.GetMessage(cmd) << Endl;
            UNIT_ASSERT(ctx.GetLeasingState());

            Sleep(TDuration::Seconds(30));
            cmd = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE);
            UNIT_ASSERT(cmd);
            UNIT_ASSERT(api.Wait(cmd));
            UNIT_ASSERT(api.GetStatus(cmd) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT(!ctx.GetLeasingState());
        }
    }

    Y_UNIT_TEST(Ble) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            auto emulator = tmBuilder.BuildEmulator(imei);
            NDrive::TTelematicsClientContext& ctx = emulator->GetContext();

            auto before = ctx.GetBlePasskey();
            auto sessionKeyBefore = ctx.GetBleSessionKey();
            INFO_LOG << "BLE passkey before: " << before << Endl;
            INFO_LOG << "BLE sessionKey before: " << sessionKeyBefore << Endl;

            auto cmd = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_RESET);
            UNIT_ASSERT(cmd);
            UNIT_ASSERT(api.Wait(cmd));
            UNIT_ASSERT(api.GetStatus(cmd) == NDrive::TTelematicsApi::EStatus::Success);

            auto notChanged = ctx.GetBlePasskey();
            auto sessionKeyNotChanged = ctx.GetBleSessionKey();
            UNIT_ASSERT_VALUES_EQUAL(notChanged, before);
            UNIT_ASSERT_VALUES_EQUAL(sessionKeyNotChanged, sessionKeyBefore);

            auto bleMac = api.GetParameter(imei, BLE_EXT_BOARD_MAC);
            UNIT_ASSERT(bleMac);
            UNIT_ASSERT(api.Wait(bleMac));
            UNIT_ASSERT_VALUES_EQUAL(api.GetStatus(bleMac), NDrive::TTelematicsApi::EStatus::Success);

            auto cmd2 = api.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_RESET);
            UNIT_ASSERT(cmd2);
            UNIT_ASSERT(api.Wait(cmd2));
            UNIT_ASSERT(api.GetStatus(cmd2) == NDrive::TTelematicsApi::EStatus::Success);

            auto after = ctx.GetBlePasskey();
            auto sessionKeyAfter = ctx.GetBleSessionKey();
            INFO_LOG << "BLE passkey after: " << after << Endl;
            INFO_LOG << "BLE sessionKey after: " << sessionKeyAfter << Endl;
            UNIT_ASSERT(before != after);
            UNIT_ASSERT(sessionKeyBefore != sessionKeyAfter);
            UNIT_ASSERT_VALUES_EQUAL(after.size(), 6);
            UNIT_ASSERT_VALUES_EQUAL(sessionKeyAfter.size(), 32);
        }
    }

    Y_UNIT_TEST(LocalPusher) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();

        TLocalSensorStorage::TPtr localSensorStorage = new TLocalSensorStorage();
        server->SetPusher(MakeHolder<TLocalPusher>(localSensorStorage));

        TLocalSensorApi sensorApi(localSensorStorage);

        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            auto blackbox = Blackbox();

            auto handler = NDrive::NProtocol::Handle(client, MakeAtomicShared<NDrive::TBlackboxHandler>(std::move(blackbox)), TDuration::Seconds(10));
            Y_ENSURE(handler->GetResult() == NDrive::NVega::TBlackboxRecordsAck::Received, "cannot send Blackbox message");

            blackbox = Blackbox();
            auto location = sensorApi.GetLocation(imei).ExtractValueSync();
            UNIT_ASSERT(location);
            UNIT_ASSERT(abs(location->Longitude - blackbox.back().Longitude) < 1e-3);
            UNIT_ASSERT(abs(location->Latitude - blackbox.back().Lattitude) < 1e-3);
            // 5 * 4 m
            constexpr double precision = 20;
            UNIT_ASSERT(abs(location->Precision - precision) < 1e-3);

            auto hb = sensorApi.GetHeartbeat(imei).ExtractValueSync();
            UNIT_ASSERT(hb);
            UNIT_ASSERT(hb->Timestamp.Seconds() > 0);
        }
    }

    Y_UNIT_TEST(SetParamHandler) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();

        TLocalSensorStorage::TPtr localSensorStorage = new TLocalSensorStorage();
        server->SetPusher(MakeHolder<TLocalPusher>(localSensorStorage));

        TLocalSensorApi sensorApi(localSensorStorage);

        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            ui16 id = CAN_ODOMETER_KM;
            double value = 1000;
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            NDrive::TTelematicsTestClient::THandlerPtr setParameter = MakeAtomicShared<NDrive::TSetParamHandler>(client, context);
            client.AddHandler(setParameter);

            auto handle = api.SetParameter<double>(imei, id, 0, value);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }
            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(context.GetOdometerKm(), value);

            TMaybe<NDrive::TSensor> sensor;
            for (size_t i = 0; i < 10; ++i) {
                sensor = sensorApi.GetSensor(imei, id).ExtractValueSync();
                if (sensor) {
                    break;
                } else {
                    Sleep(TDuration::Seconds(10));
                }
            }
            UNIT_ASSERT(sensor);
            UNIT_ASSERT(std::holds_alternative<double>(sensor->Value));
            UNIT_ASSERT(abs(std::get<double>(sensor->Value) - value) < 1e-3);
        }
    }

    Y_UNIT_TEST(GetFileHandler) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();

        TLocalSensorStorage::TPtr localSensorStorage = new TLocalSensorStorage();
        server->SetPusher(MakeHolder<TLocalPusher>(localSensorStorage));

        tmBuilder.Run();
        {
            const TString imei = "100001000010000";
            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();

            NDrive::TTelematicsTestClient client(imei);
            client.AddHandler(heartbeat);
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);

            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            TAtomicSharedPtr<NDrive::TGetFileHandler> getFile = MakeAtomicShared<NDrive::TGetFileHandler>(client);
            client.AddHandler(getFile);

            while (getFile->GetCount() == 0) {
                sleep(1);
            }

            auto connection = server->GetConnection(imei);
            const auto& logs = connection->GetLogRecords();
            UNIT_ASSERT_VALUES_EQUAL(logs.size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(logs[0].Message, "abc");
            UNIT_ASSERT_VALUES_EQUAL(logs[2].Message, "123");
        }
    }

    Y_UNIT_TEST(DerivedSensors) {
        TMap<ui16, NDrive::TSensorPushOptions> sensorPushOptions;
        sensorPushOptions[CAN_FUEL_LEVEL_P].Policy = NDrive::TSensorPushOptions::EPolicy::Realtime;
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetSensorPushOptions(sensorPushOptions);
        auto server = tmBuilder.GetServer();

        TLocalSensorStorage::TPtr localSensorStorage = new TLocalSensorStorage();
        server->SetPusher(MakeHolder<TLocalPusher>(localSensorStorage));
        TLocalSensorApi sensorApi(localSensorStorage);

        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            NDrive::TTelematicsTestClient client(imei);

            const auto heartbeat = MakeAtomicShared<NDrive::TOnHeartbeat>();
            client.AddHandler(heartbeat);

            const ui16 engineId = CAN_ENGINE_IS_ON;
            const ui16 engineValue = 1;
            const auto engine = MakeAtomicShared<TParameterHandler>(engineId, engineValue);
            client.AddHandler(engine);

            const ui16 fuelId = CAN_FUEL_LEVEL_P;
            const ui16 fuelValue = 42;
            const auto fuel = MakeAtomicShared<TParameterHandler>(fuelId, fuelValue);
            client.AddHandler(fuel);

            client.Connect("localhost", config->GetTelematicsServerOptions().Port, false);
            heartbeat->Wait();
            UNIT_ASSERT(!heartbeat->WasDropped());

            auto engineGet = api.GetParameter(imei, engineId);
            UNIT_ASSERT(engineGet);
            UNIT_ASSERT(api.Wait(engineGet));
            auto fuelGet = api.GetParameter(imei, fuelId);
            UNIT_ASSERT(fuelGet);
            UNIT_ASSERT(api.Wait(fuelGet));

            auto original = sensorApi.GetSensor(imei, fuelId).ExtractValueSync();
            UNIT_ASSERT(original);
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(original->Value), fuelValue);

            auto derived = sensorApi.GetSensor(imei, { fuelId, engineId }).ExtractValueSync();
            UNIT_ASSERT(derived);
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(derived->Value), fuelValue);

            const ui64 newEngineValue = 0;
            auto engineSet = api.SetParameter(imei, engineId, 0, newEngineValue);
            UNIT_ASSERT(engineSet);
            UNIT_ASSERT(api.Wait(engineSet));

            const ui64 newFuelValue = 24;
            auto fuelSet = api.SetParameter(imei, fuelId, 0, newFuelValue);
            UNIT_ASSERT(fuelSet);
            UNIT_ASSERT(api.Wait(fuelSet));

            original = sensorApi.GetSensor(imei, fuelId).ExtractValueSync();
            UNIT_ASSERT(original);
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(original->Value), newFuelValue);

            derived = sensorApi.GetSensor(imei, { fuelId, engineId }).ExtractValueSync();
            UNIT_ASSERT(derived);
            UNIT_ASSERT_VALUES_EQUAL(std::get<ui64>(derived->Value), fuelValue);
        }
    }

    Y_UNIT_TEST(DownloadFile) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetLogLoaderInterval(TDuration::Zero());
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());

        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            auto downloadHandle = api.Download(imei, "LOG");
            UNIT_ASSERT(downloadHandle);
            UNIT_ASSERT(api.Wait(downloadHandle, TDuration::Seconds(20)));
            UNIT_ASSERT_VALUES_EQUAL(api.GetStatus(downloadHandle), NDrive::TTelematicsApi::EStatus::Success);
            auto content = api.GetValue<TString>(downloadHandle);
            UNIT_ASSERT(!content.empty());

            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);
            {
                const NJson::TJsonValue log = Fetch(hostPort + "/connection/log/?imei=" + imei);
                INFO_LOG << log.GetStringRobust() << Endl;
                const auto& records = log["records"].GetArraySafe();
                UNIT_ASSERT(!records.empty());
            }

            {
                NDrive::TTelematicsClient::TOptions options;
                options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
                auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(hostPort);
                NDrive::TTelematicsClient client(sd, options);

                auto handle = client.Download(imei, "LOG");
                handle.GetFuture().Wait();
                Cerr << NJson::ToJson(handle.GetEvents()).GetStringRobust() << Endl;
                auto response = handle.GetResponse<NDrive::TTelematicsClient::TDownloadFileResponse>();
                UNIT_ASSERT(response);
                Cerr << response->GetStatus() << Endl;
                UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);
                Cerr << "log file bytes: " << HexEncode(response->Data.Data(), response->Data.Size()) << Endl;
                UNIT_ASSERT(!response->Data.Empty());
            }
        }
    }

    Y_UNIT_TEST(UnknownSensor) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            auto handle = api.GetParameter(imei, CAN_DRIVER_DOOR);
            UNIT_ASSERT(handle);
            UNIT_ASSERT(api.Wait(handle, TDuration::Seconds(20)));
            UNIT_ASSERT_VALUES_EQUAL(api.GetStatus(handle), NDrive::TTelematicsApi::EStatus::Success);
            auto value = api.GetValue<ui64>(handle);
            UNIT_ASSERT(value == 0 || value == 1);
        }
    }

    Y_UNIT_TEST(Interface) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);
            client.AddHandler(MakeAtomicShared<TInterfaceHandler>());

            TString payload = "surprise mothertruckers";
            TBuffer input(payload.data(), payload.size());
            auto handle = api.Interface(imei, NDrive::NVega::TInterfaceData::RS232_1, std::move(input));
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }

            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
            UNIT_ASSERT_VALUES_EQUAL(api.GetValue<TString>(handle), payload);
        }
    }

    Y_UNIT_TEST(CanRequest) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";

            NDrive::TTelematicsTestClient client(imei);
            NDrive::TTelematicsClientContext context;
            client.Connect("localhost", config->GetTelematicsServerOptions().Port, true);
            client.AddHandler(MakeAtomicShared<TInterfaceHandler>());

            ui32 canId = 0x7A0;
            ui8 canIndex = 0;
            std::array<char, 8> d = { 0x02, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00 };
            TBuffer input(d.data(), d.size());
            auto handle = api.CanRequest(imei, canId, canIndex, std::move(input));
            UNIT_ASSERT(handle);
            while (!api.Await(handle)) {
                Sleep(TDuration::MilliSeconds(100));
            }

            UNIT_ASSERT(api.GetStatus(handle) == NDrive::TTelematicsApi::EStatus::Success);
        }
    }

    Y_UNIT_TEST(ConnectionCategories) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            auto mcc = api.GetParameter(imei, VEGA_MCC);
            auto mnc = api.GetParameter(imei, VEGA_MNC);
            UNIT_ASSERT(api.Wait(mcc, TDuration::Seconds(20)));
            UNIT_ASSERT(api.Wait(mnc, TDuration::Seconds(20)));
            const TString& hostPort = "http://localhost:" + ToString(config->GetClientServerOptions().Port);

            ui64 signalValue = 0;
            for (size_t i = 0; i < 10; i++) {
                NJson::TJsonValue tass = Fetch(hostPort + "/tass");
                for (auto&& i : tass.GetArraySafe()) {
                    auto fields = i.GetArraySafe();
                    auto name = fields.at(0).GetStringSafe();
                    if (name.find("hni") != TString::npos) {
                        INFO_LOG << name << Endl;
                        signalValue = fields.at(1).GetUIntegerRobust();
                        if (signalValue) {
                            break;
                        }
                    }
                }
                if (signalValue) {
                    break;
                }
                Sleep(TDuration::Seconds(5));
            }
            UNIT_ASSERT_VALUES_EQUAL(signalValue, 0);
        }
    }

    Y_UNIT_TEST(Api2) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        TTelematicServerBuilder tsb2(1);
        auto config2 = tsb2.GetConfig();
        tsb2.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const TString shard2 = "localhost:" + ToString(config2->GetClientServerOptions().Port);
        const TString shard3 = "localhost:10101";
        UNIT_ASSERT(shard != shard2);

        auto shards = TVector{ shard, shard2, shard3 };
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(std::move(shards));
        NDrive::TTelematicsClient client(sd);
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);

            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::BLINKER_FLASH);
            auto blink = client.Command(imei, command);
            blink.GetFuture().Wait();
            UNIT_ASSERT(!blink.GetShard().empty());
            UNIT_ASSERT_VALUES_EQUAL(blink.GetStatus(), NDrive::TTelematicsClient::EStatus::Success);
            auto blinkResponse = blink.GetResponse<NDrive::TTelematicsClient::TCommandResponse>();
            UNIT_ASSERT(blinkResponse);
            auto events = blink.GetEvents();
            for (auto&& ev : events) {
                Cout << ev.Timestamp.MicroSeconds() << ' ' << ev.Data.GetStringRobust() << Endl;
            }

            auto blinkId = blink.GetId();
            auto blinkRestored = client.Restore(blinkId);
            blinkRestored.GetFuture().Wait();
            UNIT_ASSERT_VALUES_EQUAL(blinkRestored.GetShard(), blink.GetShard());
            UNIT_ASSERT_VALUES_EQUAL(blinkRestored.GetStatus(), blink.GetStatus());

            auto notFound = client.Command("fakeimei", command);
            notFound.GetFuture().Wait();
            UNIT_ASSERT(notFound.GetShard().empty());
            UNIT_ASSERT_VALUES_EQUAL(notFound.GetStatus(), NDrive::TTelematicsClient::EStatus::ConnectionNotFound);
            auto notFoundMessage = notFound.GetMessage();
            UNIT_ASSERT(NDrive::IsError(notFoundMessage));
            UNIT_ASSERT_VALUES_EQUAL(NDrive::ParseError(notFoundMessage), NDrive::ETelematicsNotification::ConnectionNotFound);

            auto notRestored = client.Restore("fakeid");
            notRestored.GetFuture().Wait();
            UNIT_ASSERT(notRestored.GetShard().empty());
            UNIT_ASSERT_VALUES_EQUAL(notRestored.GetStatus(), NDrive::TTelematicsClient::EStatus::Lost);

            auto mcc = client.Command(imei, NDrive::NVega::TCommand::GetParameter(VEGA_MCC));
            mcc.GetFuture().Wait();
            UNIT_ASSERT_VALUES_EQUAL(mcc.GetStatus(), NDrive::TTelematicsClient::EStatus::Success);
            auto mccResponse = mcc.GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            UNIT_ASSERT(mccResponse);
            UNIT_ASSERT(mccResponse->Events.size() > 0);
            UNIT_ASSERT_VALUES_EQUAL(mccResponse->Sensor.Id, VEGA_MCC);
            UNIT_ASSERT_VALUES_EQUAL(mccResponse->Sensor.ConvertTo<ui64>(), 250);
        }
    }

    Y_UNIT_TEST(Api2Conditionals) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd);
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            UNIT_ASSERT(emulator->GetContext().TrySetSpeed(0));
            client.Command(imei, NDrive::NVega::TCommand::GetParameter(VEGA_SPEED)).GetFuture().Wait();
            Sleep(TDuration::Seconds(30));

            auto pfeol = client.Command(imei, NDrive::NVega::ECommandCode::SCENARIO_POLITE_FORCED_END_OF_LEASE);
            pfeol.GetFuture().Wait();
            UNIT_ASSERT_VALUES_EQUAL_C(pfeol.GetStatus(), NDrive::TTelematicsClient::EStatus::Success, pfeol.GetMessage());
        }
    }

    Y_UNIT_TEST(Api2ConnectionList) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd);
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            auto acl = client.ListConnections();
            acl.Wait();
            UNIT_ASSERT_VALUES_EQUAL(acl.GetValue().ShardedConnections.size(), 1);
            auto connections = acl.GetValue().ShardedConnections.at(shard);
            UNIT_ASSERT_VALUES_EQUAL(connections.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(connections.at(0).IMEI, imei);
            UNIT_ASSERT_VALUES_EQUAL(acl.GetValue().UnansweredShards.size(), 0);
            UNIT_ASSERT(acl.GetValue().Context);

            UNIT_ASSERT_VALUES_EQUAL(client.GetCachedShard(imei), shard);
        }
    }

    Y_UNIT_TEST(Api2Optimistic) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString imei = "100001000010000";
        auto emulator = tmBuilder.BuildEmulator(imei);

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd);

        TString cachedShard;
        for (size_t i = 0; i < 10; ++i) {
            cachedShard = client.GetCachedShard(imei);
            if (cachedShard) {
                break;
            }
            Sleep(TDuration::Seconds(10));
        }
        UNIT_ASSERT_VALUES_EQUAL(cachedShard, shard);

        {
            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::BLINKER_FLASH);
            auto blink = client.Command(imei, command);
            blink.GetFuture().Wait();
            UNIT_ASSERT(blink.GetOptimisticRequestSucceeded());
        }
    }

    Y_UNIT_TEST(Api2Terminated) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        InitGlobalLog2Console(TLOG_DEBUG);

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        const auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd);
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);

            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::DISCONNECT);
            auto disconnect = client.Command(imei, command);
            disconnect.WaitAndEnsureSuccess();
            UNIT_ASSERT(disconnect.GetTerminationProcessed());
        }
    }

    Y_UNIT_TEST(Api2WaitConnection) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        double mileage = 424242;
        const auto command = NDrive::NVega::TCommand::SetParameter(CAN_ODOMETER_KM, mileage);
        const TString imei = "100001000010000";
        {
            auto start = Now();
            auto handle = client.Command(imei, command);
            handle.GetFuture().Wait();
            auto finish = Now();
            auto duration = finish - start;
            UNIT_ASSERT(duration > options.DefaultWaitConnectionTimeout);
        }
        {
            auto handle = client.Command(imei, command);
            UNIT_ASSERT(!handle.GetFuture().Wait(TDuration::Seconds(1)));
            auto emulator = tmBuilder.BuildEmulator(imei);
            UNIT_ASSERT(emulator);
            handle.WaitAndEnsureSuccess();
            UNIT_ASSERT_DOUBLES_EQUAL(emulator->GetContext().GetOdometerKm(), mileage, 1);
        }
    }

    Y_UNIT_TEST(SimSwitcher) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        NDrive::TTelematicsApi api(config->GetTaskExecutorConfig());
        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            const auto& context = emulator->GetContext();
            ui32 mncSource = 99;
            ui32 mncTarget = 2;
            UNIT_ASSERT_VALUES_EQUAL(context.GetMNC(), mncSource);
            auto storage = server->GetStorage();
            UNIT_ASSERT(storage);
            UNIT_ASSERT(storage->SetValue("mnc:" + ToString(mncTarget) + ":fraction", "100"));
            UNIT_ASSERT(storage->SetValue("sim_switcher:switch_timeout", "0"));
            for (size_t i = 0; i < 10; ++i) {
                auto mnc = context.GetMNC();
                if (mnc == mncTarget) {
                    break;
                }
                Sleep(TDuration::Seconds(10));
            }
            UNIT_ASSERT_VALUES_EQUAL(context.GetMNC(), mncTarget);
        }
    }

    Y_UNIT_TEST(QueryBeaconsScenario) {
        NDrive::NVega::TBeaconInfos fakeBeaconInfos;
        for (size_t i = 0; i < fakeBeaconInfos.Elements.size(); ++i) {
            fakeBeaconInfos.Elements[i].MutableRawUUID() = {0, 0, 0, 0};
            fakeBeaconInfos.Elements[i].MutableRawMajor() = 0;
            fakeBeaconInfos.Elements[i].MutableRawMinor() = 0;
        }
        TMap<TString, NDrive::IBeaconRecognizer::TLocationData> keyToLocationData;
        keyToLocationData[NDrive::IBeaconRecognizer::ConvertToKey(fakeBeaconInfos.Elements[0])] = NDrive::IBeaconRecognizer::TLocationData{"my_location", 45., 23.};
        auto beaconRecognizer = MakeHolder<NDrive::TStaticBeaconRecognizer>(std::move(keyToLocationData));
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetBeaconsRefreshInterval(TDuration::Zero());
        auto server = tmBuilder.GetServer();
        server->GetLocator().SetBeaconRecognizer(std::move(beaconRecognizer));
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        {
            const TString imei = "100001000010000";
            auto context = MakeHolder<NDrive::TTelematicsClientContext>();
            constexpr int beaconSensorsNumber = 10;
            TVector<NDrive::NVega::TBeaconInfos> beaconInfos(beaconSensorsNumber, fakeBeaconInfos);
            context->SetBeaconsInfos(std::move(beaconInfos));
            auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));
            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::SCENARIO_QUERY_BEACONS);
            auto handle = client.Command(imei, command);
            handle.GetFuture().Wait();
            auto response = handle.GetResponse<NDrive::TTelematicsClient::TCommonResponse>();
            UNIT_ASSERT(response);
            UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);
            auto location = TLocalSensorStorage::Instance()->GetLocation(imei, "beacon");
            UNIT_ASSERT(location);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Latitude, 45., 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Longitude, 23., 1e-9);
            UNIT_ASSERT_VALUES_EQUAL(location->Content, "my_location");
        }
    }

    Y_UNIT_TEST(QueryBeaconsScenarioNoWait) {
        NDrive::NVega::TBeaconInfos fakeBeaconInfos;
        for (size_t i = 0; i < fakeBeaconInfos.Elements.size(); ++i) {
            fakeBeaconInfos.Elements[i].MutableRawUUID() = {0, 0, 0, 0};
            fakeBeaconInfos.Elements[i].MutableRawMajor() = 0;
            fakeBeaconInfos.Elements[i].MutableRawMinor() = 0;
        }
        TMap<TString, NDrive::IBeaconRecognizer::TLocationData> keyToLocationData;
        keyToLocationData[NDrive::IBeaconRecognizer::ConvertToKey(fakeBeaconInfos.Elements[0])] = NDrive::IBeaconRecognizer::TLocationData{"my_location", 45., 23.};
        auto beaconRecognizer = MakeHolder<NDrive::TStaticBeaconRecognizer>(std::move(keyToLocationData));
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetBeaconsRefreshInterval(TDuration::Zero());
        auto server = tmBuilder.GetServer();
        server->GetLocator().SetBeaconRecognizer(std::move(beaconRecognizer));
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        {
            const TString imei = "100001000010000";
            auto context = MakeHolder<NDrive::TTelematicsClientContext>();
            constexpr int beaconSensorsNumber = 10;
            TVector<NDrive::NVega::TBeaconInfos> beaconInfos(beaconSensorsNumber, fakeBeaconInfos);
            context->SetBeaconsInfos(std::move(beaconInfos));
            auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));
            NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::SCENARIO_QUERY_BEACONS_NO_WAIT);
            auto handle = client.Command(imei, command);
            handle.GetFuture().Wait();
            auto response = handle.GetResponse<NDrive::TTelematicsClient::TCommonResponse>();
            UNIT_ASSERT(response);
            UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);

            TMaybe<NDrive::TLocation> location;
            constexpr int iterationsNumber = 20;
            int iteration = 0;
            do {
                if (iteration) {
                    Sleep(TDuration::Seconds(1));
                }
                location = TLocalSensorStorage::Instance()->GetLocation(imei, "beacon");
                ++iteration;
            } while (iteration < iterationsNumber && !location);

            UNIT_ASSERT(location);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Latitude, 45., 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Longitude, 23., 1e-9);
            UNIT_ASSERT_VALUES_EQUAL(location->Content, "my_location");
        }
    }

    Y_UNIT_TEST(BeaconsRefresher) {
        NDrive::NVega::TBeaconInfos fakeBeaconInfos;
        for (size_t i = 0; i < fakeBeaconInfos.Elements.size(); ++i) {
            fakeBeaconInfos.Elements[i].MutableRawUUID() = {0, 0, 0, 0};
            fakeBeaconInfos.Elements[i].MutableRawMajor() = 0;
            fakeBeaconInfos.Elements[i].MutableRawMinor() = 0;
        }
        TMap<TString, NDrive::IBeaconRecognizer::TLocationData> keyToLocationData;
        keyToLocationData[NDrive::IBeaconRecognizer::ConvertToKey(fakeBeaconInfos.Elements[0])] = NDrive::IBeaconRecognizer::TLocationData{"my_location", 45., 23.};
        auto beaconRecognizer = MakeHolder<NDrive::TStaticBeaconRecognizer>(std::move(keyToLocationData));
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        server->GetLocator().SetBeaconRecognizer(std::move(beaconRecognizer));
        tmBuilder.Run();

        {
            const TString imei = "100001000010000";
            auto context = MakeHolder<NDrive::TTelematicsClientContext>();
            constexpr int beaconSensorsNumber = 10;
            TVector<NDrive::NVega::TBeaconInfos> beaconInfos(beaconSensorsNumber, fakeBeaconInfos);
            context->SetBeaconsInfos(std::move(beaconInfos));
            // UNIT_ASSERT(context.TrySetEngineStarted(true));
            // UNIT_ASSERT(context.TrySetSpeed(20.0));
            auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));
            TMaybe<NDrive::TLocation> location;
            constexpr int iterationsNumber = 20;
            int iteration = 0;
            do {
                if (iteration) {
                    Sleep(TDuration::Seconds(2));
                }
                location = TLocalSensorStorage::Instance()->GetLocation(imei, "beacon");
                ++iteration;
            } while (iteration < iterationsNumber && !location);

            UNIT_ASSERT(location);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Latitude, 45., 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Longitude, 23., 1e-9);
            UNIT_ASSERT_VALUES_EQUAL(location->Content, "my_location");
        }
    }

    Y_UNIT_TEST(BeaconsRefresherWithSetParameter) {
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto config = tmBuilder.GetConfig();
        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        NDrive::NVega::TBeaconInfos fakeBeaconInfos;
        for (size_t i = 0; i < fakeBeaconInfos.Elements.size(); ++i) {
            fakeBeaconInfos.Elements[i].MutableRawUUID() = {1, 1, 23, 0};
            fakeBeaconInfos.Elements[i].MutableRawMajor() = 0;
            fakeBeaconInfos.Elements[i].MutableRawMinor() = 0;
        }

        TMap<TString, NDrive::IBeaconRecognizer::TLocationData> keyToLocationData;
        keyToLocationData[NDrive::IBeaconRecognizer::ConvertToKey(fakeBeaconInfos.Elements[0])] = NDrive::IBeaconRecognizer::TLocationData{"my_location", 45., 23.};
        auto beaconRecognizer = MakeHolder<NDrive::TStaticBeaconRecognizer>(std::move(keyToLocationData));
        auto server = tmBuilder.GetServer();
        server->GetLocator().SetBeaconRecognizer(std::move(beaconRecognizer));

        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            TString str;
            TStringOutput output(str);
            fakeBeaconInfos.Save(&output);
            TBuffer buf(str.data(), str.size());
            const auto command = NDrive::NVega::TCommand::SetParameter(BLE_EXT_BOARD_BEACONS_INFO3, buf);
            auto handler = client.Command(imei, command);
            handler.GetFuture().Wait();
            auto response = handler.GetResponse<NDrive::TTelematicsClient::TCommonResponse>();
            UNIT_ASSERT(response);
            UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);
            TMaybe<NDrive::TLocation> location;
            constexpr int iterationsNumber = 20;
            int iteration = 0;
            do {
                if (iteration) {
                    Sleep(TDuration::Seconds(2));
                }
                location = TLocalSensorStorage::Instance()->GetLocation(imei, "beacon");
                ++iteration;
            } while (iteration < iterationsNumber && !location);

            UNIT_ASSERT(location);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Latitude, 45., 1e-9);
            UNIT_ASSERT_DOUBLES_EQUAL(location->Longitude, 23., 1e-9);
            UNIT_ASSERT_VALUES_EQUAL(location->Content, "my_location");
        }
    }

    Y_UNIT_TEST(TelematicsClientUpload) {
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto config = tmBuilder.GetConfig();
        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        {
            const TString imei = "100001000010000";
            auto emulator = tmBuilder.BuildEmulator(imei);
            TBuffer data("abacaba", 7);
            auto handle = client.Upload(imei, "FIRMWARE", std::move(data));
            // Cerr << TDuration::Minutes(1);
            handle.GetFuture().Wait();
            Cerr << NJson::ToJson(handle.GetEvents()).GetStringRobust() << Endl;
            auto response = handle.GetResponse<NDrive::TTelematicsClient::TCommonResponse>();
            UNIT_ASSERT(response);
            Cerr << response->GetStatus() << Endl;
            UNIT_ASSERT(response->GetStatus() == NDrive::TTelematicsClient::EStatus::Success);
        }
    }

    Y_UNIT_TEST(ObdForwardConfig) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        const TString imei = "100001000010000";
        auto emulator = tmBuilder.BuildEmulator(imei);
        NJson::TJsonValue serializedCommand;
        serializedCommand["command"] = ToString(NDrive::NVega::ECommandCode::OBD_FORWARD_CONFIG);
        serializedCommand["value"] = 0x123;
        serializedCommand["mask"] = 0x1;
        auto command = NJson::FromJson<NDrive::NVega::TCommand>(serializedCommand);
        {
            auto firmware = client.Command(imei, NDrive::NVega::TCommand::GetParameter(VEGA_MCU_FIRMWARE_VERSION));
            firmware.WaitAndEnsureSuccess();
            auto handle = client.Command(imei, command);
            handle.WaitAndEnsureSuccess();
        }
        NJson::TJsonValue serializedCanRequest;
        serializedCanRequest["command"] = ToString(NDrive::NVega::ECommandCode::SCENARIO_OBD_REQUEST);
        serializedCanRequest["id"] = 0x715;
        serializedCanRequest["index"] = 1;
        serializedCanRequest["data"] = NJson::ToJson({
            0x03, 0x22, 0x23, 0x64, 0x00, 0x00, 0x00, 0x00
        });
        auto canRequest = NJson::FromJson<NDrive::NVega::TCommand>(serializedCanRequest);
        {
            auto handle = client.Command(imei, canRequest);
            handle.WaitAndEnsureSuccess();
        }
    }

    Y_UNIT_TEST(CanRequest2) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString shard = "localhost:" + ToString(config->GetClientServerOptions().Port);
        NDrive::TTelematicsClient::TOptions options;
        options.DefaultWaitConnectionTimeout = TDuration::Seconds(20);
        auto sd = MakeAtomicShared<NDrive::TStaticServiceDiscovery>(shard);
        NDrive::TTelematicsClient client(sd, options);

        const TString imei = "100001000010000";
        auto emulator = tmBuilder.BuildEmulator(imei);
        auto firmware = client.Command(imei, NDrive::NVega::TCommand::GetParameter(VEGA_MCU_FIRMWARE_VERSION));
        firmware.WaitAndEnsureSuccess();

        NJson::TJsonValue serializedCanRequest;
        serializedCanRequest["command"] = ToString(NDrive::NVega::ECommandCode::SCENARIO_OBD_REQUEST);
        serializedCanRequest["id"] = 0x715;
        serializedCanRequest["index"] = 1;
        serializedCanRequest["data"] = NJson::ToJson({
            0x03, 0x22, 0x23, 0x64, 0x00, 0x00, 0x00, 0x00
        });
        serializedCanRequest["response_id"] = 0x716;
        auto canRequest = NJson::FromJson<NDrive::NVega::TCommand>(serializedCanRequest);
        auto handle = client.Command(imei, canRequest);
        auto canResponse = handle.WaitAndEnsureSuccess().GetResponse<NDrive::TTelematicsClient::TCanResponse>();
        UNIT_ASSERT(canResponse);
        UNIT_ASSERT_VALUES_EQUAL(canResponse->Frames.size(), 1);
    }

    Y_UNIT_TEST(PinResetter) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetPinResetterInterval(TDuration::Seconds(20));
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        const TString imei = "100001000010000";
        auto emulator = tmBuilder.BuildEmulator(imei);
        const auto& context = emulator->GetContext();
        UNIT_ASSERT_VALUES_UNEQUAL(context.GetServerPin(), NDrive::TPinResetter::DefaultPin);
        UNIT_ASSERT_VALUES_UNEQUAL(context.GetWiredPin(), NDrive::TPinResetter::DefaultPin);
        auto storage = server->GetStorage();
        UNIT_ASSERT(storage);
        NJson::TJsonValue imeiWhiteListJson = NJson::TArrayBuilder(imei);
        UNIT_ASSERT(storage->SetValue(NDrive::TPinResetter::IMEIWhitelistSettingName, imeiWhiteListJson.GetStringRobust()));
        for (size_t i = 0; i < 6; ++i) {
            auto pin = context.GetServerPin();
            auto wiredPin = context.GetWiredPin();
            if (pin == NDrive::TPinResetter::DefaultPin && wiredPin == NDrive::TPinResetter::DefaultPin) {
                break;
            }
            Sleep(TDuration::Seconds(10));
        }
        UNIT_ASSERT_VALUES_EQUAL(context.GetServerPin(), NDrive::TPinResetter::DefaultPin);
        UNIT_ASSERT_VALUES_EQUAL(context.GetWiredPin(), NDrive::TPinResetter::DefaultPin);
    }

    Y_UNIT_TEST(CarEngineIsOnDerivation) {
        TTelematicServerBuilder tmBuilder;
        auto config = tmBuilder.GetConfig();
        config->SetPinResetterInterval(TDuration::Seconds(20));
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();
        const TString imei = "100001000010000";
        auto context = MakeHolder<NDrive::TTelematicsClientContext>();
        {
            NDrive::TSensor customRPMSensor{};
            customRPMSensor.Id = 2802;
            customRPMSensor.Timestamp = Now();
            customRPMSensor.Since = customRPMSensor.Timestamp;
            customRPMSensor.Value = 2.5;
            context->SetCustomSensors({ customRPMSensor });
        }
        auto emulator = tmBuilder.BuildEmulator(imei, std::move(context));

        const auto &api = tmBuilder.GetServer()->GetSensors();
        bool initialized = false;
        for (int i = 0; i < 5 && !initialized; ++i) {
            auto sensorsF = api->GetSensors(TVector<TString>{ imei }, { CAN_ENGINE_IS_ON, CAN_ENGINE_RPM });
            auto imeiToSensors = sensorsF.GetValue();
            const auto& sensors = imeiToSensors[imei];
            if (sensors.size() != 2) {
                Sleep(TDuration::Seconds(2));
                continue;
            }
            bool sensorsInitialized = true;
            for (const auto& s : sensors) {
                if (s.Id == CAN_ENGINE_IS_ON && !s.ConvertTo<bool>()) {
                    sensorsInitialized = false;
                } else if (s.Id == CAN_ENGINE_RPM && s.ConvertTo<double>() != 2.5) {
                    sensorsInitialized = false;
                }
            }
            initialized = sensorsInitialized;
        }
        UNIT_ASSERT(initialized);
        {
            NDrive::TSensor customIgnitionSensor{};
            customIgnitionSensor.Id = 2806;
            customIgnitionSensor.Timestamp = Now() + TDuration::Seconds(3);
            customIgnitionSensor.Since = customIgnitionSensor.Timestamp;
            customIgnitionSensor.Value = static_cast<double>(0);
            auto customSensors = emulator->GetContext().GetCustomSensors();
            customSensors.push_back(customIgnitionSensor);
            emulator->GetContext().SetCustomSensors(std::move(customSensors));
        }
        initialized = false;
        for (int i = 0; i < 5 && !initialized; ++i) {
            auto sensorsF = api->GetSensors(TVector<TString>{ imei }, { CAN_ENGINE_IS_ON, CAN_IGNITION });
            auto imeiToSensors = sensorsF.GetValue();
            const auto& sensors = imeiToSensors[imei];
            if (sensors.size() != 2) {
                Sleep(TDuration::Seconds(5));
                continue;
            }
            bool sensorsInitialized = true;
            for (const auto& s : sensors) {
                if (s.Id == CAN_ENGINE_IS_ON && s.ConvertTo<bool>()) {
                    sensorsInitialized = false;
                } else if (s.Id == CAN_IGNITION && s.ConvertTo<double>() != 0) {
                    sensorsInitialized = false;
                }
            }
            initialized = sensorsInitialized;
        }
        UNIT_ASSERT(initialized);
    }

    Y_UNIT_TEST(LoggingProtocol) {
        TTelematicServerBuilder tmBuilder;
        auto server = tmBuilder.GetServer();
        tmBuilder.Run();

        auto connection = NDrive::TTelematicsConnection::Create(server.Get(), nullptr);

        auto checkNode = [] (const NYT::TNode& node) {
            UNIT_ASSERT(node.HasKey("data"));
            UNIT_ASSERT(node.HasKey("type"));
            UNIT_ASSERT_EQUAL(node["type"].AsString(), ToString(NDrive::NVega::BLACKBOX_RECORDS));
        };

        {
            TDummyLogBackendKeeper keeper;

            NDrive::NVega::TMessage message(NDrive::NVega::BLACKBOX_RECORDS);
            auto& payload = message.As<NDrive::NVega::TBlackboxRecords>();
            payload.Id = 1;
            payload.Records = Blackbox();

            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Incoming, connection.Get(), message);

            auto node = NYT::NodeFromJsonString(Singleton<TDummyLogValue>()->Get());
            checkNode(node);
            UNIT_ASSERT_NO_EXCEPTION(NYT::FromNode<NDrive::TBlackboxRecords>(node["data"]));
        }

        {
            TDummyLogBackendKeeper keeper;

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NNavTelecom::TSingleBlackBoxRequest>(NDrive::NNavTelecom::TBitField());
            payload->Record = NavTelecomBlackbox();
            NDrive::NNavTelecom::TMessage message(std::move(payload));

            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Incoming, connection.Get(), message);

            auto node = NYT::NodeFromJsonString(Singleton<TDummyLogValue>()->Get());
            checkNode(node);
            UNIT_ASSERT_NO_EXCEPTION(NYT::FromNode<NDrive::TBlackboxRecords>(node["data"]));
        }

        {
            TDummyLogBackendKeeper keeper;

            auto payload = NDrive::NProtocol::IPayload::Create<NDrive::NWialon::TShortDataRequest>();
            payload->ShortData = WialonBlackbox();
            NDrive::NWialon::TMessage message(std::move(payload));
            NDrive::TTelematicsLog::Log(NDrive::TTelematicsLog::Incoming, connection.Get(), message);

            auto node = NYT::NodeFromJsonString(Singleton<TDummyLogValue>()->Get());
            checkNode(node);
            UNIT_ASSERT_NO_EXCEPTION(NYT::FromNode<NDrive::TBlackboxRecords>(node["data"]));
        }
    }
}
