#include "features.h"

#include <drive/backend/offers/ranking/features.h>
#include <drive/backend/offers/ranking/model.h>
#include <drive/library/cpp/threading/future.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/yson/node/node_io.h>

#include <mapreduce/yt/client/client.h>

#include <rtline/api/indexing_client/neh_client.h>
#include <rtline/library/json/adapters.h>

#include <util/digest/fnv.h>
#include <util/draft/date.h>
#include <util/string/builder.h>
#include <util/string/join.h>

#include <deque>

namespace {
    template <class T>
    TVector<T> GetValues(const NYT::TNode& node) {
        TVector<T> result;
        if (node.IsList()) {
            for (auto&& i : node.AsList()) {
                result.push_back(i.ConvertTo<T>());
            }
        } else {
            result.push_back(node.ConvertTo<T>());
        }
        return result;
    }

    template <class TReaderType, class TWriterType = TReaderType>
    void MapLocal(NYT::IClientBasePtr client,
        const NYT::TMapOperationSpec& spec,
        TIntrusivePtr<NYT::IMapper<NYT::TTableReader<TReaderType>, NYT::TTableWriter<TWriterType>>> mapper,
        const NYT::TOperationOptions& options = NYT::TOperationOptions()
    ) {
        Y_UNUSED(options);
        auto writer = client->CreateTableWriter<TWriterType>(spec.Outputs_.front());
        mapper->Start(writer.Get());
        for (auto&& input : spec.Inputs_) {
            auto reader = client->CreateTableReader<TReaderType>(input);
            mapper->Do(reader.Get(), writer.Get());
        }
        mapper->Finish(writer.Get());
    }

    class TOfferCreatedMapper: public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TOfferCreatedMapper() = default;

        void Do(TReader* reader, TWriter* writer) override {
            for (; reader && reader->IsValid(); reader->Next()) {
                const NYT::TNode& row = reader->GetRow();
                try {
                    const auto ev = row["event"].ConvertTo<TString>();
                    if (ev != "OfferCreated") {
                        continue;
                    }
                    const NYT::TNode& offer = row["data"];
                    const auto type = offer["InstanceType"].ConvertTo<TString>();
                    if (type != "standart_offer") {
                        continue;
                    }
                    const auto objectId = offer["ObjectId"].ConvertTo<TString>();
                    const auto userId = offer["UserId"].ConvertTo<TString>();
                    const auto timestamp = offer["Timestamp"].ConvertTo<ui64>();
                    const NYT::TNode& features = offer["StandartOffer"]["Features"];

                    NYT::TNode result;
                    result["object_id"] = objectId;
                    result["user_id"] = userId;
                    result["timestamp"] = timestamp;
                    result["features"] = features;
                    writer->AddRow(result);
                } catch (const std::exception& e) {
                    NYT::TNode error = row;
                    error["error"] = FormatExc(e);
                    writer->AddRow(error, 1);
                }
            }
        }
    };

    class TBookingMapper: public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TBookingMapper() = default;

        void Do(TReader* reader, TWriter* writer) override {
            for (; reader && reader->IsValid(); reader->Next()) {
                const NYT::TNode& row = reader->GetRow();
                try {
                    const auto tag = row["tag"].ConvertTo<TString>();
                    if (tag != "old_state_reservation") {
                        continue;
                    }
                    const auto action = row["history_action"].ConvertTo<TString>();
                    if (action != "set_performer") {
                        continue;
                    }
                    const auto objectId = row["object_id"].ConvertTo<TString>();
                    const auto timestamp = row["history_timestamp"].ConvertTo<ui64>();

                    NYT::TNode result;
                    result["object_id"] = objectId;
                    result["timestamp"] = timestamp;
                    writer->AddRow(result);
                } catch (const std::exception& e) {
                    NYT::TNode error = row;
                    error["error"] = FormatExc(e);
                    writer->AddRow(error, 1);
                }
            }
        }
    };

    class TCreatePoolMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TCreatePoolMapper() = default;

        void Do(TReader* reader, TWriter* writer) override {
            for (; reader && reader->IsValid(); reader->Next()) {
                const NYT::TNode& row = reader->GetRow();
                try {
                    const auto target = row["idle"].ConvertTo<ui64>();
                    NDrive::TOfferFeatures features;
                    {
                        Y_ENSURE(row["features"]["Float"].IsList());
                        const auto& floats = row["features"]["Float"].AsList();
                        Y_ENSURE(floats.size() <= features.Floats.size());
                        for (size_t i = 0; i < floats.size(); ++i) {
                            features.Floats[i] = floats[i].ConvertTo<double>();
                        }
                    }
                    {
                        Y_ENSURE(row["features"]["Category"].IsList());
                        const auto& categories = row["features"]["Category"].AsList();
                        Y_ENSURE(categories.size() <= features.Categories2.size());
                        for (size_t i = 0; i < categories.size(); ++i) {
                            features.Categories2[i] = categories[i].ConvertTo<TString>();
                        }
                    }

                    NYT::TNode result;
                    result["key"] = ToString(target);
                    result["value"] = Join("\t"
                        , JoinRange("\t", features.Floats.begin(), features.Floats.end())
                        , JoinRange("\t", features.Categories2.begin(), features.Categories2.end())
                    );
                    writer->AddRow(result);
                    if (target * FnvHash<ui32>(features.Categories2[NDriveOfferCatFactors2::FI_MODEL]) % 100 >= 10) {
                        writer->AddRow(result, 1);
                    } else {
                        writer->AddRow(result, 2);
                    }
                } catch (const std::exception& e) {
                    NYT::TNode error = row;
                    error["error"] = FormatExc(e);
                    writer->AddRow(error, 3);
                }
            }
        }
    };

    class TIdleTimeReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TIdleTimeReducer() = default;

        void Do(TReader* reader, TWriter* writer) override {
            TString id;
            TVector<TInstant> bookings;
            TMap<TInstant, NYT::TNode> features;
            for (; reader && reader->IsValid(); reader->Next()) {
                NYT::TNode row = reader->MoveRow();
                auto objectId = row["object_id"].ConvertTo<TString>();
                Y_ENSURE(!id || id == objectId, id << ' ' << objectId);
                id = objectId;

                auto timestamp = TInstant::Seconds(row["timestamp"].ConvertTo<ui64>());

                if (row.HasKey("features")) {
                    features[timestamp] = std::move(row["features"]);
                } else {
                    bookings.push_back(timestamp);
                }
            }
            std::sort(bookings.begin(), bookings.end());
            auto booking = bookings.begin();
            for (auto&&[timestamp, feature] : features) {
                while (booking != bookings.end() && timestamp > *booking) {
                    ++booking;
                }
                if (booking == bookings.end()) {
                    break;
                }
                TDuration idle = *booking - timestamp;
                if (idle > MaxIdleTime) {
                    continue;
                }

                NYT::TNode result;
                result["idle"] = idle.Seconds();
                result["features"] = std::move(feature);
                result["timestamp"] = timestamp.Seconds();
                writer->AddRow(result);
            }
        }

        Y_SAVELOAD_JOB(
            MaxIdleTime
        );

    public:
        TDuration MaxIdleTime = TDuration::Days(1);
    };
}

REGISTER_MAPPER(TOfferCreatedMapper);
REGISTER_MAPPER(TBookingMapper);
REGISTER_MAPPER(TCreatePoolMapper);
REGISTER_REDUCER(TIdleTimeReducer);

int main_apply_model(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('m', "meta", "model meta description").RequiredArgument("JSON").Required();
    options.AddLongOption('d', "data", "model data").RequiredArgument("FILE").Optional();
    options.AddLongOption('f', "features", "serialized features").RequiredArgument("JSON").Required();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    auto description = NJson::ReadJsonFastTree(res.Get("meta"));
    auto original = description;
    if (res.Has("data")) {
        description["data"] = TIFStream(res.Get("data")).ReadAll();
    }
    auto features = NJson::FromJson<NDrive::TOfferFeatures>(NJson::ReadJsonFastTree(res.Get("features")));

    auto model = NDrive::IOfferModel::Construct(description);
    Y_ENSURE(model, "cannot construct model from " << original.GetStringRobust());
    Cout << model->Calc(features) << Endl;
    return EXIT_SUCCESS;
}

int main_DRIVEANALYTICS_155(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-destination", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-local", "run map operation locally").NoArgument().Optional();
    options.AddLongOption("yt-keep-temps", "keep temporary tables").NoArgument().Optional();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN");
    options.AddLongOption("date-from", "source date FROM").RequiredArgument("YYYYMMDD").Required();
    options.AddLongOption("date-to", "source date TO").RequiredArgument("YYYYMMDD").Required();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& server = res.Get("yt-server");
    const TString& token = res.Get("yt-token");
    const TString& destination = res.Get("yt-destination");
    const bool local = res.Has("yt-local");
    const bool keepTemps = res.Has("yt-keep-temps");

    const auto from = TDate(res.Get("date-from"));
    const auto to = TDate(res.Get("date-to"));

    TVector<TString> dates;
    for (auto i = from; i <= to; ++i) {
        dates.push_back(i.ToStroka("%Y-%m-%d"));
    }

    NYT::TCreateClientOptions cco;
    cco.Token(token);
    auto client = NYT::CreateClient(server, cco);
    auto tx = client->StartTransaction();

    auto now = TInstant::Now();
    auto offersTable = "//tmp/svshevtsov/offers-" + ToString(now.MicroSeconds());
    {
        NYT::TMapOperationSpec spec;
        for (auto&& date : dates) {
            spec.AddInput<NYT::TNode>("//logs/carsharing-backend-events-log/1d/" + date);
        }
        spec.AddOutput<NYT::TNode>(offersTable);
        spec.AddOutput<NYT::TNode>(offersTable + ".errors");
        auto mapper = MakeIntrusive<TOfferCreatedMapper>();
        if (!local) {
            tx->Map(spec, mapper);
        } else {
            MapLocal<NYT::TNode, NYT::TNode>(tx, spec, mapper);
        }
    }
    auto bookingsTable = "//tmp/svshevtsov/bookings-" + ToString(now.MicroSeconds());
    {
        NYT::TMapOperationSpec spec;
        for (auto&& date : dates) {
            spec.AddInput<NYT::TNode>("//home/carsharing/production/car_tags_history/" + date);
        }
        spec.AddOutput<NYT::TNode>(bookingsTable);
        spec.AddOutput<NYT::TNode>(bookingsTable + ".errors");
        auto mapper = MakeIntrusive<TBookingMapper>();
        if (!local) {
            tx->Map(spec, mapper);
        } else {
            MapLocal<NYT::TNode, NYT::TNode>(tx, spec, mapper);
        }
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(offersTable);
        spec.Output(offersTable);
        spec.SortBy({ "object_id" });
        tx->Sort(spec);
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(bookingsTable);
        spec.Output(bookingsTable);
        spec.SortBy({ "object_id" });
        tx->Sort(spec);
    }
    auto raw = destination + "/" + "raw";
    {
        NYT::TReduceOperationSpec spec;
        spec.AddInput<NYT::TNode>(offersTable);
        spec.AddInput<NYT::TNode>(bookingsTable);
        spec.AddOutput<NYT::TNode>(raw);
        spec.ReduceBy({ "object_id" });
        tx->Reduce(spec, new TIdleTimeReducer);
    }
    auto all = destination + "/" + "all";
    auto learn = destination + "/" + "learn";
    auto test = destination + "/" + "test";
    {
        NYT::TMapOperationSpec spec;
        spec.AddInput<NYT::TNode>(raw);
        spec.AddOutput<NYT::TNode>(all);
        spec.AddOutput<NYT::TNode>(learn);
        spec.AddOutput<NYT::TNode>(test);
        spec.AddOutput<NYT::TNode>(all + ".errors");
        auto mapper = MakeIntrusive<TCreatePoolMapper>();
        if (!local) {
            tx->Map(spec, mapper);
        } else {
            MapLocal<NYT::TNode, NYT::TNode>(tx, spec, mapper);
        }
    }
    auto fi = destination + "/" + "pool.cd";
    {
        auto writer = tx->CreateTableWriter<NYT::TNode>(fi);
        size_t index = 0;
        {
            NYT::TNode row;
            row["key"] = ToString(index);
            row["value"] = "Label";
            writer->AddRow(row);
            ++index;
        }

        auto floats = NDrive::GetOfferFactorsInfo();
        for (size_t i = 0; i < floats->GetFactorCount(); ++i) {
            NYT::TNode row;
            row["key"] = ToString(index);
            row["value"] = TStringBuilder() << "Num" << '\t' << floats->GetFactorName(i);
            writer->AddRow(row);
            ++index;
        }

        auto categs = NDrive::GetCatOfferFactorsInfo2();
        for (size_t i = 0; i < categs->GetFactorCount(); ++i) {
            NYT::TNode row;
            row["key"] = ToString(index);
            row["value"] = TStringBuilder() << "Categ" << '\t' << categs->GetFactorName(i);
            writer->AddRow(row);
            ++index;
        }

        writer->Finish();
    }
    if (!keepTemps) {
        tx->Remove(offersTable);
        tx->Remove(bookingsTable);
    }
    tx->Commit();

    return EXIT_SUCCESS;
}

int main_DRIVEANALYTICS_331(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('o', "output", "output file").RequiredArgument("PATH").Required();
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-source", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    options.AddLongOption("min-price", "minimal allowed price in rubles")
           .RequiredArgument("MIN_PRICE")
           .DefaultValue(3.0);
    options.AddLongOption("max-price", "maximal allowed price in rubles")
           .RequiredArgument("MAX_PRICE")
           .DefaultValue(20.0);
    options.AddLongOption("min-price-multiplier", "minimal allowed price multiplier")
           .RequiredArgument("MIN_MULTIPLIER")
           .DefaultValue(0.25);
    options.AddLongOption("max-price-multiplier", "maximal allowed price multiplier")
           .RequiredArgument("MAX_MULTIPLIER")
           .DefaultValue(4);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& output = res.Get("output");
    const TString& server = res.Get("yt-server");
    const TString& source = res.Get("yt-source");

    const double minPrice = FromString<double>(res.Get("min-price"));
    const double maxPrice = FromString<double>(res.Get("max-price"));
    const double minPriceMultiplier = FromString<double>(res.Get("min-price-multiplier"));
    const double maxPriceMultiplier = FromString<double>(res.Get("max-price-multiplier"));

    TMaybe<TString> token;
    if (res.Has("yt-token")) {
        token = res.Get("yt-token");
    }

    NYT::TCreateClientOptions cco;
    if (token) {
        cco.Token(*token);
    }
    auto client = NYT::CreateClient(server, cco);

    NDrive::TGeoLocalModel::TElements elements;
    TMaybe<bool> absolutes;
    TMaybe<bool> isFridayWeekend;
    TMaybe<NDrive::TGeoLocalModel::EVersion> version;
    for (auto reader = client->CreateTableReader<NYT::TNode>(source); reader && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        if (!absolutes) {
            absolutes = row.HasKey("riding_price");
        }
        if (!isFridayWeekend) {
            isFridayWeekend = row.HasKey("is_weekend_or_friday");
        }
        if (!version) {
            version = row.HasKey("weekday") ? NDrive::TGeoLocalModel::EVersion::V2 : NDrive::TGeoLocalModel::EVersion::V1;
        }

        double value = 0.0;
        if (*absolutes) {
            value = 0.01 * row["riding_price"].ConvertTo<double>();
            Y_VERIFY((value >= minPrice) && (value <= maxPrice));
        } else {
            value = row["multiplier"].ConvertTo<double>();
            Y_VERIFY((value >= minPriceMultiplier) && (value <= maxPriceMultiplier));
        }

        switch (*version) {
        case NDrive::TGeoLocalModel::EVersion::V1:
        {
            bool isBaseModel = row["is_base_model"].ConvertTo<bool>();
            bool isWeekend = *isFridayWeekend ? row["is_weekend_or_friday"].ConvertTo<bool>() : row["is_weekend"].ConvertTo<bool>();
            auto hour = row["hour"].ConvertTo<ui8>();
            auto street = row["street"].ConvertTo<TString>();
            auto hash = NDrive::TGeoLocalModel::CalcHash(isBaseModel, isWeekend, hour, street);

            NDrive::TGeoLocalModel::TElement element(hash, value);
            elements.push_back(element);
            break;
        }
        case NDrive::TGeoLocalModel::EVersion::V2:
        {
            auto weekdays = GetValues<ui8>(row["weekday"]);
            auto hours = GetValues<ui8>(row["hour"]);
            auto models = GetValues<TString>(row["model"]);
            auto geos = GetValues<TString>(row["geo"]);
            for (auto&& weekday : weekdays) {
                for (auto&& hour : hours) {
                    for (auto&& model : models) {
                        for (auto&& geo : geos) {
                            auto hash = NDrive::TGeoLocalModel::CalcHash2(model, weekday, hour, geo);
                            NDrive::TGeoLocalModel::TElement element(hash, value);
                            elements.push_back(element);
                        }
                    }
                }
            }
        }
        }
    }
    {
        TOFStream file(output);
        Save(&file, elements);
    }
    {
        NJson::TJsonValue meta;
        meta["name"] = "NAME";
        meta["type"] = NDrive::TGeoLocalModel::Type();
        if (*absolutes) {
            meta["polynom"] = "K0010000000V3";
        }
        if (*isFridayWeekend) {
            meta["is_friday_weekend"] = *isFridayWeekend;
        }
        if (version) {
            meta["version"] = ToString(*version);
        }
        Cout << meta.GetStringRobust() << Endl;
    }
    return EXIT_SUCCESS;
}

int main_DRIVEANALYTICS_336(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("saas-host", "SaaS host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps-prestable.yandex.net");
    options.AddLongOption("saas-port", "SaaS port").RequiredArgument("PORT").DefaultValue(80);
    options.AddLongOption("saas-token", "SaaS token").RequiredArgument("TOKEN");
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-source", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    options.AddLongOption("timestamp", "document timestamp").RequiredArgument("UNIXTIMESTAMP").Optional();
    options.AddLongOption("deadline", "document deadline").RequiredArgument("UNIXTIMESTAMP").Optional();
    options.AddLongOption("max-in-flight", "max in flight").RequiredArgument("NUM").DefaultValue(100);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& host = res.Get("saas-host");
    const auto port = FromString<ui16>(res.Get("saas-port"));
    const TString& serviceToken = res.Get("saas-token");
    const auto maxInFlight = FromString<ui64>(res.Get("max-in-flight"));

    auto indexingClient = NRTLine::TNehIndexingClient(serviceToken, host, port);

    const TString& server = res.Get("yt-server");
    const TString& source = res.Get("yt-source");
    TMaybe<TString> token;
    if (res.Has("yt-token")) {
        token = res.Get("yt-token");
    }

    NYT::TCreateClientOptions cco;
    if (token) {
        cco.Token(*token);
    }
    auto client = NYT::CreateClient(server, cco);

    TMaybe<TInstant> timestamp;
    if (res.Has("timestamp")) {
        timestamp = TInstant::Seconds(FromString<ui64>(res.Get("timestamp")));
    }
    TMaybe<TInstant> deadline;
    if (res.Has("deadline")) {
        deadline = TInstant::Seconds(FromString<ui64>(res.Get("deadline")));
    }

    std::deque<NThreading::TFuture<NRTLine::TSendResult>> results;
    for (auto reader = client->CreateTableReader<NYT::TNode>(source); reader && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        const TString& userId = row["user_id"].AsString();
        const auto averageSessionCarsCount = row["average_session_cars_count"].ConvertTo<double>();
        const auto averageSessionClicksCount = row["average_session_clicks_count"].ConvertTo<double>();

        NRTLine::TAction action;
        NRTLine::TDocument& document = action.GetDocument();
        document.SetUrl(TStringBuilder() << "user_id:" << userId << ":features");
        document.AddProperty("average_session_cars_count", averageSessionCarsCount);
        document.AddProperty("average_session_clicks_count", averageSessionClicksCount);
        if (timestamp) {
            document.SetTimestamp(timestamp->Seconds());
        }
        if (deadline) {
            document.SetDeadline(*deadline);
        }

        results.push_back(indexingClient.Send(action));
        if (results.size() > maxInFlight) {
            while (results.front().HasValue()) {
                const auto& reply = results.front().GetValue();
                if (!reply.IsSucceeded()) {
                    Cerr << reply.GetCode() << '\t' << reply.GetMessage() << Endl;
                }
                results.pop_front();
            }
            while (results.front().HasException()) {
                Cerr << NThreading::GetExceptionMessage(results.front());
                results.pop_front();
            }
            if (results.size() > maxInFlight) {
                results.front().Wait();
            }
        }
    }

    return EXIT_SUCCESS;
}

int main_DRIVEANALYTICS_470(int argc, const char** argv) {
    Y_UNUSED(argc);
    Y_UNUSED(argv);
    NJson::TJsonValue result;
    result["name"] = "DA-470";
    result["type"] = NDrive::TTimeScheduleModel::Type();
    result["config"]["time_shift"] = 10800;
    NJson::TJsonValue& segments = result["config"]["segments"];
    float coeff = 0.97;
    auto addSegment = [&coeff, &segments](TDuration start, float price) {
        NJson::TJsonValue segment;
        segment["start"] = start.Seconds();
        segment["value"] = static_cast<ui32>(100 * coeff * price);
        segments.AppendValue(std::move(segment));
    };
    for (ui32 day = 0; day < 7; ++day) {
        if (day == 0) {
            addSegment(TDuration::Zero(), 8);
        }
        if (day < 6) {
            addSegment(TDuration::Days(day) + TDuration::Hours(5), 5.5);
        }
        addSegment(TDuration::Days(day) + TDuration::Hours(11), 7);
        addSegment(TDuration::Days(day) + TDuration::Hours(18), 8);
        addSegment(TDuration::Days(day) + TDuration::Hours(21), 7);
        addSegment(TDuration::Days(day) + TDuration::Hours(23), 8);
    }
    Cout << result.GetStringRobust() << Endl;
    return EXIT_SUCCESS;
}
