#include "features.h"
#include "sessions.h"
#include "telematics.h"
#include "lua_tester.h"

#include <drive/backend/offers/ranking/model.h>
#include <drive/library/cpp/common/status.h>
#include <drive/library/cpp/compression/simple.h>
#include <drive/library/cpp/parking/api/client.h>
#include <drive/library/cpp/parking/api/money.h>
#include <drive/library/cpp/parking/inhouse/client.h>
#include <drive/library/cpp/schedule/sd.h>
#include <drive/library/cpp/taxi/suggest/client.h>
#include <drive/library/cpp/telematics/data.h>
#include <drive/library/cpp/timeline/timeline.h>
#include <drive/library/cpp/trace/action.h>
#include <drive/library/cpp/tracks/client.h>
#include <drive/library/cpp/tvm/logger.h>
#include <drive/library/cpp/yt/node/cast.h>
#include <drive/telematics/protocol/vega.h>
#include <drive/telematics/server/data/blackbox.h>
#include <drive/telematics/server/data/factors.h>

#include <kernel/catboost/catboost_calcer.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/base64/base64.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/yson/node/node_io.h>

#include <mapreduce/yt/client/client.h>
#include <mapreduce/yt/client/init.h>

#include <rtline/api/indexing_client/neh_client.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/json/cast.h>

#include <util/draft/date.h>
#include <util/stream/str.h>
#include <util/stream/zlib.h>
#include <util/string/vector.h>

namespace {
    template <class TReaderType, class TWriterType = TReaderType>
    void MapLocal(NYT::IClientBasePtr client,
        const NYT::TMapOperationSpec& spec,
        THolder<NYT::IMapper<NYT::TTableReader<TReaderType>, NYT::TTableWriter<TWriterType>>>&& mapper,
        const NYT::TOperationOptions& options = NYT::TOperationOptions()
    ) {
        Y_UNUSED(options);
        auto writer = client->CreateTableWriter<TWriterType>(spec.Outputs_.front());
        mapper->Start(writer.Get());
        for (auto&& input : spec.Inputs_) {
            auto reader = client->CreateTableReader<TReaderType>(input);
            mapper->Do(reader.Get(), writer.Get());
        }
        mapper->Finish(writer.Get());
    }

    class TFindSessionMapper: public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TFindSessionMapper() = default;
        TFindSessionMapper(const TString& sessionId)
            : SessionId(sessionId)
        {
        }

        Y_SAVELOAD_JOB(
            SessionId
        );

        void Do(TReader* reader, TWriter* writer) override {
            if (!SessionId) {
                return;
            }

            for (; reader && reader->IsValid(); reader->Next()) {
                const NYT::TNode& row = reader->GetRow();
                auto sessionId = row["session_id"].ConvertTo<TString>();
                if (sessionId == SessionId) {
                    writer->AddRow(row);
                    return;
                }
            }
        }

    private:
        TString SessionId;
    };

    class TReindexCleanup: public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TReindexCleanup() = default;
        TReindexCleanup(const TString& host, ui16 port, const TString& token)
            : Host(host)
            , Port(port)
            , Token(token)
        {
        }

        void Do(TReader* reader, TWriter* writer) override {
            NRTLine::TNehIndexingClient client(Token, Host, Port);

            TMaybe<TString> url;
            double latitude;
            double longitude;
            for (; reader && reader->IsValid(); reader->Next()) {
                const auto& row = reader->GetRow();
                url = row["url"].AsString();
                latitude = row["latitude"].AsDouble();
                longitude = row["longitude"].AsDouble();
                break;
            }
            if (url) {
                NRTLine::TAction action;
                action.SetActionType(NRTLine::TAction::atDelete);
                NRTLine::TDocument& document = action.GetDocument();
                document.SetUrl(*url);
                NRTLine::TGeoData& geoData = document.AddGeoData();
                geoData.AddCoordinate({ longitude, latitude });
                geoData.SetType("DRIVE_RIDING");
                for (size_t i = 0; i < 5; ++i) {
                    auto asyncResult = client.Send(action);
                    auto result = asyncResult.ExtractValueSync();
                    if (result.IsSucceeded()) {
                        break;
                    }
                    NYT::TNode error;
                    error["url"] = *url;
                    error["code"] = result.GetHttpCode();
                    error["message"] = result.GetMessage();
                    writer->AddRow(error);
                    Sleep(i * TDuration::Seconds(1));
                }
            }
        }

        Y_SAVELOAD_JOB(
            Host,
            Port,
            Token
        );

    private:
        TString Host;
        ui16 Port;
        TString Token;
    };

    class TReindexMapper: public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TReindexMapper() = default;
        TReindexMapper(const TString& host, ui16 port, const TString& token)
            : Host(host)
            , Port(port)
            , Token(token)
        {
        }

        void Do(TReader* reader, TWriter* writer) override {
            NRTLine::TNehIndexingClient client(Token, Host, Port);
            for (; reader && reader->IsValid(); reader->Next()) {
                const auto& row = reader->GetRow();
                const auto& actionNode = row["action"];
                const auto actionData = NYT::FromNode<NJson::TJsonValue>(actionNode);
                NRTLine::TAction action;
                action.ParseFromJson(actionData);
                for (size_t i = 0; i < 5; ++i) {
                    auto asyncResult = client.Send(action);
                    auto result = asyncResult.ExtractValueSync();
                    if (result.IsSucceeded()) {
                        break;
                    }
                    NYT::TNode error = row;
                    error["code"] = result.GetHttpCode();
                    error["message"] = result.GetMessage();
                    writer->AddRow(error);
                    Sleep(i * TDuration::Seconds(1));
                }
            }
        }

        Y_SAVELOAD_JOB(
            Host,
            Port,
            Token
        );

    private:
        TString Host;
        ui16 Port;
        TString Token;
    };

    class TReindexReducer: public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TReindexReducer() = default;
        TReindexReducer(const TString& host, const TString& token, const TString& extraCgi)
            : Host(host)
            , Token(token)
            , ExtraCgi(extraCgi)
        {
        }
        TReindexReducer(NDrive::TNewTimeline&& timeline)
            : ExternalTimeline(std::move(timeline))
        {
        }

        template <class TContainer>
        void SetModels(TContainer&& container) {
            Models = { container.begin(), container.end() };
        }
        void SetIMEI(const TString& imei) {
            IMEI = imei;
        }
        void SetSessionId(const TString& sessionId) {
            SessionId = sessionId;
        }

        void Do(TReader* reader, TWriter* writer) override {
            auto requester = Host ? MakeAtomicShared<NDrive::TSessionRequester>(Host, ExtraCgi, Token) : nullptr;

            NDrive::TNewTimelines::TOptions options;
            options.Preload = false;
            options.EnableGlobalUpdates = false;
            options.UpdateLocalDepth = TDuration::Days(30);
            NDrive::TNewTimelines timelines(requester, options);

            if (ExternalTimeline) {
                auto copy = *ExternalTimeline;
                timelines.AddTimeline(std::move(copy));
            }

            for (; reader && reader->IsValid(); reader->Next()) {
                const NYT::TNode& row = reader->GetRow();
                try {
                    const NYT::TNode& data = row["data"];
                    const TString& type = row["type"].AsString();
                    if (type != "data" && type != "BLACKBOX_RECORDS") {
                        continue;
                    }
                    const auto imei = row["imei"].ConvertTo<ui64>();
                    if (IMEI && ToString(imei) != IMEI) {
                        continue;
                    }

                    auto timeline = timelines.GetTimeline(imei);
                    Y_ENSURE(timeline);

                    if (!Models.empty()) {
                        const auto& car = timeline->GetCar();
                        if (!Models.contains(car.Model)) {
                            continue;
                        }
                    }

                    if (SessionId) {
                        auto timestamp = NDrive::GetTimestamp(data);
                        auto status = timeline->GetStatus(timestamp);
                        if (status.GetSessionId() != SessionId) {
                            continue;
                        }
                    }

                    auto timestamp = NDrive::GetTimestamp(data);
                    auto action = NDrive::CreateAction(imei, data, timeline, "new");
                    if (action) {
                        NYT::TNode result;
                        TToJsonContext context(TToJsonContext::COMMON_JSON);
                        result["action"] = NYT::ToNode(action->ToJson(context));
                        result["url"] = action->GetDocument().GetUrl();

                        result["imei"] = imei;
                        result["direction"] = NDrive::GetCourse(data);
                        result["latitude"] = NDrive::GetLatitude(data);
                        result["longitude"] = NDrive::GetLongitude(data);
                        result["speed"] = NDrive::GetSpeed(data);
                        result["timestamp"] = timestamp.Seconds();

                        auto status = timeline->GetStatus(timestamp);
                        auto carStatus = NDrive::GetStatus(status.GetCarStatus());
                        result["free"] = NDrive::IsFree(carStatus);
                        result["moving"] = carStatus == NDrive::ECarStatus::csRide;
                        writer->AddRow(result);
                    }
                } catch (const std::exception& e) {
                    NYT::TNode error = row;
                    error["error"] = FormatExc(e);
                    writer->AddRow(error, 1);
                }
            }
        }

        Y_SAVELOAD_JOB(
            Host,
            Token,
            ExtraCgi,
            ExternalTimeline,
            Models,
            IMEI,
            SessionId
        );

    private:
        TString Host;
        TString Token;
        TString ExtraCgi;

        TMaybe<NDrive::TNewTimeline> ExternalTimeline;

        TSet<TString> Models;
        TString IMEI;
        TString SessionId;
    };

    class TApplyPredictorMapper : public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TApplyPredictorMapper() = default;
        TApplyPredictorMapper(const NCatboostCalcer::TCatboostCalcer& calcer)
            : Calcer(calcer)
        {
        }

        void Do(TReader* reader, TWriter* writer) override {
            for (; reader && reader->IsValid(); reader->Next()) {
                NYT::TNode row = reader->MoveRow();
                NDrive::TOfferFeatures features;

                const auto& floats = row["vec_float"].AsList();
                for (size_t i = 0; i < std::min(features.Floats.size(), floats.size()); ++i) {
                    features.Floats[i] = floats[i].ConvertTo<double>();
                }

                const auto& categories = row["vec_categ"].AsList();
                for (size_t i = 0; i < std::min(features.Categories2.size(), categories.size()); ++i) {
                    features.Categories2[i] = categories[i].AsString();
                }

                row["features"] = NYT::NodeFromJsonValue(NJson::ToJson(features));
                auto prediction = Calcer.CalcRelev(features.FloatsView(), features.CategoriesView2());
                row["prediction"] = prediction;
                row["probability"] = Sigmoid(prediction);
                if (writer) {
                    writer->AddRow(row);
                }
            }
        }

        Y_SAVELOAD_JOB(
            Calcer
        );

    private:
        NCatboostCalcer::TCatboostCalcer Calcer;
    };

    struct TElement {
        NDrive::TSupplyDemandStat Stat;
        TDuration Offset;
        TDuration Duration;
    };

    class TTemporalScheduleReducer : public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        void Do(TReader* reader, TWriter* writer) override {
            ui64 cid = 0;
            NDrive::TSupplyDemandHistory::TTemporal schedule;

            constexpr auto defaultDuration = TDuration::Minutes(10);

            TVector<TElement> elements;
            for (; reader && reader->IsValid(); reader->Next()) {
                const auto& row = reader->GetRow();
                const auto minutes = row["day_minutes"].ConvertTo<ui64>();
                const auto days = row["weekday"].ConvertTo<ui64>();
                const auto id = row["id"].ConvertTo<ui64>();
                if (!cid) {
                    cid = id;
                } else {
                    Y_ENSURE(cid == id, "mismatch " << cid << ' ' << id);
                }
                const auto offset = TDuration::Days(days) + TDuration::Minutes(minutes);
                const auto demand = row["user_count"].ConvertTo<double>();
                const auto supply = row["car_count"].ConvertTo<double>();
                if (demand / std::max(supply, 1.0) < 0.7) {
                    continue;
                }

                TElement element;
                element.Duration = defaultDuration;
                element.Offset = offset;
                element.Stat.Demand = demand;
                element.Stat.Supply = supply;
                elements.push_back(element);
            }
            std::sort(elements.begin(), elements.end(), [](const TElement& left, const TElement& right) {
                return left.Offset < right.Offset;
            });

            TMaybe<TElement> element;
            for (auto&& i : elements) {
                if (!element) {
                    element = i;
                } else if (element->Offset + element->Duration == i.Offset && std::abs(element->Stat.Demand - i.Stat.Demand) < 0.001 && std::abs(element->Stat.Supply - i.Stat.Supply) < 0.001) {
                    element->Duration += i.Duration;
                } else {
                    schedule.Add(element->Offset, element->Duration, element->Stat);
                    element = i;
                }
            }
            if (element) {
                schedule.Add(element->Offset, element->Duration, element->Stat);
                element.Clear();
            }
            if (cid) {
                TStringStream temporal;
                ::Save(&temporal, schedule);
                NYT::TNode row;
                row["id"] = cid;
                row["temporal"] = temporal.Str();
                row["temporal_json"] = NJson::ToJson(schedule).GetStringRobust();
                if (writer) {
                    writer->AddRow(row);
                }
            }
        }
    };
}

REGISTER_MAPPER(TApplyPredictorMapper);
REGISTER_MAPPER(TFindSessionMapper);
REGISTER_MAPPER(TReindexMapper);
REGISTER_REDUCER(TReindexCleanup);
REGISTER_REDUCER(TReindexReducer);
REGISTER_REDUCER(TTemporalScheduleReducer);

template <class TClient>
int main_get_fitdev_balance(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();;
    options.AddLongOption('t', "token", "fitdev API token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& token = res.Get("token");

    TClient client(token);
    auto balance = client.GetBalance();
    Cout << balance.Amount << Endl;

    return EXIT_SUCCESS;
}

template <class TClient>
int main_start_fitdev_sessions(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('d', "duration", "Parking duration").RequiredArgument("DUR");
    options.AddLongOption('p', "parking-id", "Parking ID").RequiredArgument("ID");
    options.AddLongOption('t', "token", "fitdev API token").RequiredArgument("TOKEN");
    options.AddLongOption('v', "vrp", "Vehicle registration plate").RequiredArgument("AAA000");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TDuration duration = TDuration::Parse(res.Get("duration"));
    const TString& parkingId = res.Get("parking-id");
    const TString& token = res.Get("token");
    const TString& vrp = res.Get("vrp");

    typename TClient::TParking parking;
    parking.ParkingId = parkingId;

    typename TClient::TVehicle vehicle;
    vehicle.LicensePlate = vrp;

    TClient client(token);
    auto offer = client.GetOffer(parking, vehicle, duration);
    Cout << offer.ToJson().GetStringRobust() << Endl;
    auto session = client.StartParkingRobust(offer, duration);
    Cout << session.ToJson().GetStringRobust() << Endl;

    return EXIT_SUCCESS;
}

int main_get_timeline(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('i', "imei", "IMEI").RequiredArgument("IMEI").Optional();
    options.AddLongOption("drive-host", "Drive host").RequiredArgument("HOST").DefaultValue("prestable.carsharing.yandex.net");
    options.AddLongOption("drive-extra", "Drive extra params").RequiredArgument("CGI").DefaultValue("");
    options.AddLongOption("drive-token", "Drive token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& driveHost = res.Get("drive-host");
    const TString& driveExtra = res.Get("drive-extra");
    const TString& driveToken = res.Has("drive-token") ? res.Get("drive-token") : TString();
    const auto imei = FromString<ui64>(res.Get("imei"));

    auto requester = MakeAtomicShared<NDrive::TSessionRequester>(driveHost, driveExtra, driveToken);
    NDrive::TNewTimelines::TOptions opts;
    opts.Preload = false;
    opts.EnableGlobalUpdates = false;
    opts.UpdateLocalDepth = TDuration::Days(30);
    NDrive::TNewTimelines timelines(requester, opts);

    auto timeline = timelines.GetTimeline(imei);
    Y_ENSURE(timeline);

    auto statuses = timeline->GetAllStatuses();
    for (auto&& status : statuses) {
        Cout << status << Endl;
    }
    return EXIT_SUCCESS;
}

int main_get_tracks(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('s', "session", "SessionId").RequiredArgument("UUID").Required();
    options.AddLongOption("tracks-host", "Drive tracks storage host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps-prestable.yandex.net");
    options.AddLongOption("tracks-port", "Drive tracks storage port").RequiredArgument("PORT").DefaultValue(17000);
    options.AddLongOption("tracks-service", "Drive tracks storage service").RequiredArgument("SERVICE").DefaultValue("drive_graph");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& tracksHost = res.Get("tracks-host");
    const auto tracksPort = FromString<ui16>(res.Get("tracks-port"));
    const TString& tracksService = res.Get("tracks-service");
    NDrive::TTracksClient tracksClient(tracksService, tracksHost, tracksPort);

    NDrive::TTracksLinker::TConfig tracksLinkerConfig;
    NDrive::TTracksLinker tracksLinker(tracksLinkerConfig);

    NDrive::TTrackQuery query;
    query.SessionId = res.Get("session");
    auto tracks = tracksClient.GetTracks(query, TDuration::Seconds(10));
    auto linked = tracksLinker.Link(std::move(tracks));
    linked.Wait();
    for (auto&& result : linked.GetValue()) {
        Cout << result.Track.Name << ":" << Endl;
        for (auto&& segment : result.Segments) {
            Cout << PrintRoute(segment.Linked) << Endl;
        }
    }

    return EXIT_SUCCESS;
}

int main_restore_tracks(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('i', "imei", "IMEI").RequiredArgument("IMEI").Optional();
    options.AddLongOption('s', "session", "SessionId").RequiredArgument("UUID").Optional();
    options.AddLongOption('m', "models", "Models, comma-separated").RequiredArgument("MODEL").Optional();
    options.AddLongOption("drive-host", "Drive host").RequiredArgument("HOST").DefaultValue("prestable.carsharing.yandex.net");
    options.AddLongOption("drive-extra", "Drive extra params").RequiredArgument("CGI").DefaultValue("");
    options.AddLongOption("drive-token", "Drive token").RequiredArgument("TOKEN");
    options.AddLongOption("tracks-host", "Drive tracks storage host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps.yandex.net");
    options.AddLongOption("tracks-port", "Drive tracks storage port").RequiredArgument("PORT").DefaultValue(80);
    options.AddLongOption("tracks-token", "Drive tracks storage token").RequiredArgument("TOKEN").DefaultValue("af22bf1205f4af06b6a713d7c0458a0c");
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-src-table", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-dst-table", "destination YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-job-count", "YT threads").RequiredArgument("NUM").Optional();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& server = res.Get("yt-server");
    const TString& destination = res.Get("yt-dst-table");
    const auto jobCount = res.Has("yt-job-count") ? FromString<ui32>(res.Get("yt-job-count")) : 0;

    TVector<TString> sources;
    if (const NLastGetopt::TOptParseResult* parsed = res.FindLongOptParseResult("yt-src-table")) {
        for (auto&& value : parsed->Values()) {
            sources.push_back(value);
        }
    }

    const TString& driveHost = res.Get("drive-host");
    const TString& driveExtra = res.Get("drive-extra");
    const TString& driveToken = res.Has("drive-token") ? res.Get("drive-token") : TString();

    const TString& tracksHost = res.Get("tracks-host");
    const auto tracksPort = FromString<ui16>(res.Get("tracks-port"));
    const TString& tracksToken = res.Get("tracks-token");

    auto reducer = MakeHolder<TReindexReducer>(driveHost, driveToken, driveExtra);
    if (res.Has("imei")) {
        reducer->SetIMEI(res.Get("imei"));
    }
    if (res.Has("session")) {
        reducer->SetSessionId(res.Get("session"));
    }
    if (res.Has("models")) {
        auto models = SplitString(res.Get("models"), ",");
        reducer->SetModels(models);
    }

    NYT::TCreateClientOptions cco;
    if (res.Has("yt-token")) {
        const auto& token = res.Get("yt-token");
        cco.Token(token);
    }
    auto client = NYT::CreateClient(server, cco);
    auto sorted = destination + ".sorted_source";
    {
        NYT::TSortOperationSpec spec;
        for (auto&& source : sources) {
            spec.AddInput(source);
        }
        spec.Output(sorted);
        spec.SortBy({ "imei" });
        client->Sort(spec);
    }
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(sorted);
        spec.AddOutput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".errors");
        spec.ReduceBy({"imei"});
        spec.EnableKeyGuarantee(false);
        client->Reduce(spec, reducer.Release());
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(destination);
        spec.Output(destination);
        spec.SortBy({ "url" });
        client->Sort(spec);
    }
    auto cleanup = MakeIntrusive<TReindexCleanup>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".cleanup_errors");
        spec.ReduceBy({"url"});
        client->Reduce(spec, cleanup);
    }
    auto mapper = MakeIntrusive<TReindexMapper>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TMapOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".indexing_errors");
        client->Map(spec, mapper);
    }
    client->Remove(sorted);

    return EXIT_SUCCESS;
}

int main_restore_tracks_2(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('s', "session", "SessionId").RequiredArgument("UUID").Required();
    options.AddLongOption("drive-host", "Drive host").RequiredArgument("HOST").DefaultValue("admin.carsharing.yandex.net");
    options.AddLongOption("drive-extra", "Drive extra params").RequiredArgument("CGI").DefaultValue("");
    options.AddLongOption("drive-token", "Drive token").RequiredArgument("TOKEN");
    options.AddLongOption("tracks-host", "Drive tracks storage host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps.yandex.net");
    options.AddLongOption("tracks-port", "Drive tracks storage port").RequiredArgument("PORT").DefaultValue(80);
    options.AddLongOption("tracks-token", "Drive tracks storage token").RequiredArgument("TOKEN").DefaultValue("af22bf1205f4af06b6a713d7c0458a0c");
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-telematics-table-prefix", "telematics YT table prefix").RequiredArgument("TABLE").DefaultValue("//logs/carsharing-telematics-events-log/1d/");
    options.AddLongOption("yt-tmp-table-prefix", "destination YT table").RequiredArgument("TABLE").DefaultValue("//tmp/carsharing/");
    options.AddLongOption("yt-job-count", "YT threads").RequiredArgument("NUM").Optional();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& sessionId = res.Get("session");

    const TString& server = res.Get("yt-server");
    const TString& destination = res.Get("yt-tmp-table-prefix") + sessionId;
    const auto jobCount = res.Has("yt-job-count") ? FromString<ui32>(res.Get("yt-job-count")) : 0;

    const TString& driveHost = res.Get("drive-host");
    const TString& driveExtra = res.Get("drive-extra");
    const TString& driveToken = res.Has("drive-token") ? res.Get("drive-token") : TString();

    const TString& tracksHost = res.Get("tracks-host");
    const auto tracksPort = FromString<ui16>(res.Get("tracks-port"));
    const TString& tracksToken = res.Get("tracks-token");

    NDrive::TSessionRequester requester(driveHost, driveExtra, driveToken);
    auto session = requester.GetSession(sessionId);
    INFO_LOG << "car_id:\t" << session.Car.Id << Endl;
    INFO_LOG << "imei:\t" << session.Car.IMEI << Endl;

    NDrive::TNewTimeline timeline(nullptr, session.Car, TDuration::Zero(), TDuration::Max());
    TInstant now = Now();
    TInstant since = TInstant::Max();
    TInstant until = TInstant::Zero();
    INFO_LOG << "session_id:\t" << sessionId << Endl;
    for (auto&& status : session.Statuses) {
        INFO_LOG << "event:\t" << status << Endl;
        since = std::min(since, status.GetStart());
        until = std::max(until, status.GetStart());
        timeline.AddStatus(std::move(status), now);
    }
    INFO_LOG << "since:\t" << since << Endl;
    INFO_LOG << "until:\t" << until << Endl;

    auto reducer = MakeHolder<TReindexReducer>(std::move(timeline));
    reducer->SetIMEI(ToString(session.Car.IMEI));
    reducer->SetSessionId(sessionId);

    const TString& telematicsTablePrefix = res.Get("yt-telematics-table-prefix");
    TSet<TString> sources;
    for (auto timestamp = since; timestamp <= until; timestamp += TDuration::Days(1)) {
        auto shifted = timestamp + TDuration::Hours(3);
        sources.insert(telematicsTablePrefix + shifted.FormatGmTime("%Y-%m-%d"));
    }
    {
        auto shifted = until + TDuration::Hours(3);
        sources.insert(telematicsTablePrefix + shifted.FormatGmTime("%Y-%m-%d"));
    }
    for (auto&& source : sources) {
        INFO_LOG << "source:\t" << source << Endl;
    }

    NYT::TCreateClientOptions cco;
    if (res.Has("yt-token")) {
        const auto& token = res.Get("yt-token");
        cco.Token(token);
    }
    auto client = NYT::CreateClient(server, cco);
    auto sorted = destination + ".sorted_source";
    {
        NYT::TSortOperationSpec spec;
        for (auto&& source : sources) {
            spec.AddInput(source);
        }
        spec.Output(sorted);
        spec.SortBy({ "imei" });
        client->Sort(spec);
    }
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(sorted);
        spec.AddOutput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".errors");
        spec.ReduceBy({"imei"});
        spec.EnableKeyGuarantee(false);
        client->Reduce(spec, reducer.Release());
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(destination);
        spec.Output(destination);
        spec.SortBy({ "url" });
        client->Sort(spec);
    }
    auto cleanup = MakeIntrusive<TReindexCleanup>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".cleanup_errors");
        spec.ReduceBy({"url"});
        client->Reduce(spec, cleanup);
    }
    auto mapper = MakeIntrusive<TReindexMapper>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TMapOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".indexing_errors");
        client->Map(spec, mapper);
    }
    client->Remove(sorted);

    return EXIT_SUCCESS;
}

int main_restore_tracks_offline(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('i', "imei", "IMEI").RequiredArgument("IMEI").Optional();
    options.AddLongOption('d', "date", "Date of the session").RequiredArgument("YYYY-MM-DD").Required();
    options.AddLongOption('s', "session", "SessionId").RequiredArgument("UUID").Required();
    options.AddLongOption("tracks-host", "Drive tracks storage host").RequiredArgument("HOST").DefaultValue("saas-indexerproxy-maps.yandex.net");
    options.AddLongOption("tracks-port", "Drive tracks storage port").RequiredArgument("PORT").DefaultValue(80);
    options.AddLongOption("tracks-token", "Drive tracks storage token").RequiredArgument("TOKEN").DefaultValue("af22bf1205f4af06b6a713d7c0458a0c");
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-orders-table-prefix", "orders YT table prefix").RequiredArgument("TABLE").DefaultValue("//home/carsharing/production/orders_export/");
    options.AddLongOption("yt-telematics-table-prefix", "telematics YT table prefix").RequiredArgument("TABLE").DefaultValue("//logs/carsharing-telematics-events-log/1d/");
    options.AddLongOption("yt-dst-table", "destination YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-job-count", "YT threads").RequiredArgument("NUM").Optional();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& sessionId = res.Get("session");

    const TString& tracksHost = res.Get("tracks-host");
    const auto tracksPort = FromString<ui16>(res.Get("tracks-port"));
    const TString& tracksToken = res.Get("tracks-token");

    const TString& server = res.Get("yt-server");
    const TString& token = res.Get("yt-token");
    const TString& destination = res.Get("yt-dst-table");
    const auto jobCount = res.Has("yt-job-count") ? FromString<ui32>(res.Get("yt-job-count")) : 0;

    NYT::TCreateClientOptions cco;
    cco.Token(token);
    auto client = NYT::CreateClient(server, cco);

    TString imei;
    if (res.Has("imei")) {
        imei = res.Get("imei");
    } else {
        ythrow yexception() << "unimplemented";
    }

    TVector<TString> dates;
    if (res.Has("date")) {
        dates.push_back(res.Get("date"));
    } else {
        ythrow yexception() << "unimplemented";
    }

    const TString& orderTablePrefix = res.Get("yt-orders-table-prefix");
    TVector<TString> orderTables;
    for (auto&& date : dates) {
        orderTables.push_back(orderTablePrefix + date);
    }

    const TString& sessionTable = destination + ".order";
    {
        NYT::TMapOperationSpec spec;
        for (auto&& table : orderTables) {
            spec.AddInput<NYT::TNode>(table);
        }
        spec.AddOutput<NYT::TNode>(sessionTable);

        auto finder = MakeIntrusive<TFindSessionMapper>(sessionId);
        client->Map(spec, finder);
    }

    THolder<NDrive::TNewTimeline> timeline;
    TInstant since = TInstant::Max();
    TInstant until = TInstant::Zero();
    for (auto reader = client->CreateTableReader<NYT::TNode>(sessionTable); reader && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        const TString& currentSessionId = row["session_id"].AsString();
        Y_ENSURE(currentSessionId == sessionId, currentSessionId << ' ' << sessionId);

        const TString& objectId = row["object_id"].AsString();
        const TString& userId = row["user_id"].AsString();
        NDrive::TSessionRequester::TCar car;
        car.Id = objectId;
        car.IMEI = FromString<ui64>(imei);
        timeline = MakeHolder<NDrive::TNewTimeline>(nullptr, car, TDuration::Zero(), TDuration::Max());

        TVector<NDrive::TStatus> statuses;
        const auto& events = row["events_details"].AsList();
        for (auto&& ev : events) {
            const TString& action = ev["action"].AsString();
            const TString& name = ev["name"].AsString();
            const TInstant timestamp = TInstant::Seconds(ev["instant"].ConvertTo<ui64>());

            since = std::min(since, timestamp);
            until = std::max(until, timestamp);

            constexpr TStringBuf oldStatePrefix = "old_state_";
            const TString& carStatus = name.StartsWith(oldStatePrefix) ? name.substr(oldStatePrefix.size()) : name;

            if (action == "evolve" && userId == "robot-frontend") {
                continue;
            }
            if (action == "update_data") {
                continue;
            }
            if (action != "set_performer" && !statuses.empty() && statuses.back().GetSessionId() == sessionId) {
                statuses.back().SetFinish(timestamp);
            }
            if (action != "drop_performer") {
                const TString& actionId = ToString(timestamp.Seconds()) + "-" + action + "-" + sessionId;
                NDrive::TStatus status(carStatus, actionId, objectId, sessionId, userId, timestamp);
                statuses.push_back(std::move(status));
            }
        }

        auto now = Now();
        for (auto&& status : statuses) {
            timeline->AddStatus(std::move(status), now);
        }
        break;
    }
    Y_ENSURE(timeline);

    const TString& telematicsTablePrefix = res.Get("yt-telematics-table-prefix");
    TSet<TString> sources;
    for (auto timestamp = since; timestamp <= until; timestamp += TDuration::Days(1)) {
        auto shifted = timestamp + TDuration::Hours(3);
        sources.insert(telematicsTablePrefix + shifted.FormatGmTime("%Y-%m-%d"));
    }
    {
        auto shifted = until + TDuration::Hours(3);
        sources.insert(telematicsTablePrefix + shifted.FormatGmTime("%Y-%m-%d"));
    }

    auto reducer = MakeHolder<TReindexReducer>(std::move(*timeline));
    reducer->SetIMEI(imei);
    reducer->SetSessionId(sessionId);
    auto sorted = destination + ".sorted_source";
    {
        NYT::TSortOperationSpec spec;
        for (auto&& source : sources) {
            spec.AddInput(source);
        }
        spec.Output(sorted);
        spec.SortBy({ "imei" });
        client->Sort(spec);
    }
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(sorted);
        spec.AddOutput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".errors");
        spec.ReduceBy({ "imei" });
        spec.EnableKeyGuarantee(false);
        client->Reduce(spec, reducer.Release());
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(destination);
        spec.Output(destination);
        spec.SortBy({ "url" });
        client->Sort(spec);
    }
    auto cleanup = MakeIntrusive<TReindexCleanup>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TReduceOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".cleanup_errors");
        spec.ReduceBy({"url"});
        client->Reduce(spec, cleanup);
    }
    auto mapper = MakeIntrusive<TReindexMapper>(tracksHost, tracksPort, tracksToken);
    {
        NYT::TMapOperationSpec spec;
        if (jobCount) {
            spec.JobCount(jobCount);
        }
        spec.AddInput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".indexing_errors");
        client->Map(spec, mapper);
    }

    return EXIT_SUCCESS;
}

int main_get_parking_balance(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('p', "parking", "Parking duration").RequiredArgument("JSON");
    options.AddLongOption('t', "token", "yoomoney.ru/api/parking token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NDrive::TParkingPaymentClient3::TParking parking;
    parking.FromJson(NJson::ReadJsonFastTree(res.Get("parking")));
    const TString& token = res.Get("token");

    NDrive::TParkingPaymentClient3 client(token);
    auto balance = client.GetBalance(parking);
    Cout << balance.ToJson().GetStringRobust() << Endl;

    return EXIT_SUCCESS;
}

int main_list_parkings(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('a', "application-id", "ApplicationId").RequiredArgument("ID");
    options.AddLongOption('r', "radius", "Parking search radius").RequiredArgument("METERS").DefaultValue(ToString(250));
    options.AddLongOption('t', "token", "yoomoney.ru/api/parking token").RequiredArgument("TOKEN");
    options.AddLongOption("router-host", "Router host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps.yandex.net");
    options.AddLongOption("router-port", "Router port").RequiredArgument("PORT").DefaultValue("17000");
    options.AddLongOption("router-service", "Router service").RequiredArgument("SERVICE").DefaultValue("drive_router");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& applicationId = res.Get("application-id");
    const TString& token = res.Get("token");
    const TString& routerHost = res.Get("router-host");
    const TString& routerService = res.Get("router-service");
    const auto radius = FromString<double>(res.Get("radius"));
    const auto routerPort = FromString<ui16>(res.Get("router-port"));

    NDrive::TParkingPaymentClient3 cl(token, applicationId);
    auto vehicles = cl.GetVehicles();
    auto vehicle = vehicles.at(0);

    NRTLine::TNehSearchClient searchClient(routerService, routerHost, routerPort);
    NDrive::TParkingListClient client(searchClient);

    TString location;
    TMap<TString, ui32> parkingPrices;
    ui32 totalPrice = 0;
    while (Cin.ReadLine(location)) {
        NJson::TJsonValue parsed;
        TGeoCoord coordinate;
        try {
            if (NJson::ReadJsonFastTree(location, &parsed)) {
                coordinate.X = parsed["lon"].GetDoubleSafe();
                coordinate.Y = parsed["lat"].GetDoubleSafe();
            } else {
                coordinate = FromString<TGeoCoord>(location);
            }
        } catch (const std::exception& e) {
            ERROR_LOG << location << ' ' << e.what() << Endl;
            continue;
        }

        auto parkings = client.GetParkings(coordinate.Y, coordinate.X, radius).ExtractValueSync();
        TSet<ui32> prices;
        NJson::TJsonValue parks;
        for (auto&& parking : parkings) {
            parks.AppendValue(parking.ToJson());
            auto p = parkingPrices.find(parking.ParkingId);
            if (p != parkingPrices.end()) {
                prices.insert(p->second);
            } else {
                for (size_t i = 0; i < 3; ++i) {
                    try {
                        auto offer = cl.GetOffer(parking, vehicle, TDuration::Hours(1));
                        auto price = offer.Cost.Amount;
                        prices.insert(price);
                        parkingPrices.emplace(parking.ParkingId, price);
                        break;
                    } catch (const std::exception& e) {
                        ERROR_LOG << FormatExc(e) << Endl;
                    }
                }
            }
        }
        auto sumPrice = std::accumulate(prices.begin(), prices.end(), 0);
        totalPrice += sumPrice;
        Cout
            << "lat=" << coordinate.Y << '\t'
            << "lon=" << coordinate.X << '\t'
            << "radius=" << radius << '\t'
            << "unique_prices=" << prices.size() << '\t'
            << "sum_price=" << sumPrice << '\t'
            << "parkings=" << parks.GetStringRobust() << '\t'
            << "count=" << parkings.size() << Endl;
    }
    NOTICE_LOG << totalPrice << Endl;

    return EXIT_SUCCESS;
}

int main_list_parking_sessions(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('t', "token", "yoomoney.ru/api/parking token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& token = res.Get("token");

    NDrive::TParkingPaymentClient3 client(token);
    auto sessions = client.GetSessions();
    for (auto&& session : sessions) {
        Cout << session.ToJson().GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_start_parking_sessions(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('a', "application-id", "ApplicationId").RequiredArgument("ID");
    options.AddLongOption('d', "duration", "Parking duration").RequiredArgument("DUR");
    options.AddLongOption('p', "parking", "Parking JSON object").RequiredArgument("JSON");
    options.AddLongOption('t', "token", "yoomoney.ru/api/parking token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    NDrive::TParkingPaymentClient3::TParking parking;
    parking.FromJson(NJson::ReadJsonFastTree(res.Get("parking")));
    const TString& applicationId = res.Get("application-id");
    const TString& token = res.Get("token");
    const TDuration duration = TDuration::Parse(res.Get("duration"));

    NDrive::TParkingPaymentClient3 client(token, applicationId);
    auto vehicles = client.GetVehicles();
    auto vehicle = vehicles.at(0);

    auto offer = client.GetOffer(parking, vehicle, duration);
    Cout << offer.ToJson().GetStringRobust() << Endl;
    auto session = client.StartParkingRobust(offer, duration);
    Cout << session.ToJson().GetStringRobust() << Endl;

    return EXIT_SUCCESS;
}

int main_stop_parking_sessions(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('s', "session-id", "Session to be stopped").RequiredArgument("ID");
    options.AddLongOption('t', "token", "yoomoney.ru/api/parking token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& sessionId = res.Get("session-id");
    const TString& token = res.Get("token");

    NDrive::TParkingPaymentClient3 client(token);
    auto sessions = client.GetSessions();
    for (auto&& session : sessions) {
        if (session.Id == sessionId) {
            auto refund = client.StopParking(session.Id);
            Cout << "stopped session: " << refund.ToJson().GetStringRobust() << Endl;
        }
    }

    return EXIT_SUCCESS;
}

int main_yandex_money_balance(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('a', "agent-id", "AgentId").RequiredArgument("ID");
    options.AddLongOption('h', "host", "Deposition Host").RequiredArgument("URL").DefaultValue("https://deposit.yookassa.ru");
    options.AddLongOption('p', "port", "Deposition port").RequiredArgument("PORT").DefaultValue(ToString(9094));
    options.AddLongOption("certificate", "CERT file").RequiredArgument("FILE");
    options.AddLongOption("private-key", "private.key").RequiredArgument("FILE");
    options.AddLongOption("private-password", "private.key password").RequiredArgument("PASSWORD");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& depositionHost = res.Get("host");
    const auto depositionPort = FromString<ui16>(res.Get("port"));
    const TString& certificate = res.Get("certificate");
    const TString& privateKey = res.Get("private-key");
    const TString& privatePassword = res.Get("private-password");

    NDrive::TYandexMoneyClient::TOptions ymco;
    ymco.AgentId = res.Get("agent-id");
    ymco.DepositionHost = depositionHost;
    ymco.DepositionPort = depositionPort;
    ymco.CertificateFile = certificate;
    ymco.PrivateKeyFile = privateKey;
    ymco.PrivateKeyPassword = privatePassword;
    NDrive::TYandexMoneyClient ymc(ymco);

    auto balance = ymc.BalanceSafe(Seconds());
    Y_ENSURE(balance.Code == NDrive::TYandexMoneyClient::ECode::Success, "cannot MakeDeposition " << int(balance.Code) << ' ' << balance.Error);
    Cout << balance.Value << Endl;
    return EXIT_SUCCESS;
}

int main_ranker(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-src-table", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-dst-table", "destination YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN");
    options.AddLongOption("yt-local", "run map operation locally").NoArgument().Optional();

    options.AddLongOption("model", "Catboost model").RequiredArgument("FILE");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& server = res.Get("yt-server");
    const TString& token = res.Get("yt-token");
    const TString& destination = res.Get("yt-dst-table");
    const bool local = res.Has("yt-local");

    TVector<TString> sources;
    if (const NLastGetopt::TOptParseResult* parsed = res.FindLongOptParseResult("yt-src-table")) {
        for (auto&& value : parsed->Values()) {
            sources.push_back(value);
        }
    }

    const TString& modelFilename = res.Get("model");
    NCatboostCalcer::TCatboostCalcer calcer;
    {
        TIFStream file(modelFilename);
        calcer.Load(&file);
    }

    NYT::TCreateClientOptions cco;
    cco.Token(token);
    auto client = NYT::CreateClient(server, cco);
    {
        auto mapper = MakeHolder<TApplyPredictorMapper>(calcer);
        NYT::TMapOperationSpec spec;
        for (auto&& source : sources) {
            spec.AddInput<NYT::TNode>(source);
        }
        spec.AddOutput<NYT::TNode>(destination);
        if (local) {
            MapLocal<NYT::TNode, NYT::TNode>(client, spec, std::move(mapper));
        } else {
            client->Map(spec, mapper.Release());
        }
    }

    return EXIT_SUCCESS;
}

int main_geotimetable(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-src-table", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& server = res.Get("yt-server");
    const TString& token = res.Get("yt-token");
    const TString& source = res.Get("yt-src-table");

    NYT::TCreateClientOptions cco;
    cco.Token(token);
    auto client = NYT::CreateClient(server, cco);
    TString sorted = "//tmp/svshevtsov/sorted-source";
    TString tmp = "//tmp/svshevtsov/temporal";
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(source);
        spec.Output(sorted);
        spec.SortBy({ "id", "weekday", "day_minutes" });
        client->Sort(spec);
    }
    {
        NYT::TReduceOperationSpec spec;
        spec.AddInput<NYT::TNode>(sorted);
        spec.AddOutput<NYT::TNode>(tmp);
        spec.ReduceBy({ "id" });
        client->Reduce(spec, MakeIntrusive<TTemporalScheduleReducer>());
    }
    {
        NDrive::TSupplyDemandHistory schedule;
        auto reader = client->CreateTableReader<NYT::TNode>(tmp);
        for (; reader && reader->IsValid(); reader->Next()) {
            const auto& node = reader->GetRow();
            const auto id = node["id"].ConvertTo<ui64>();
            const TString& data = node["temporal"].ConvertTo<TString>();
            TStringInput input(data);
            auto temporal = MakeAtomicShared<NDrive::TSupplyDemandHistory::TTemporal>();
            Load(&input, *temporal);

            auto i = (id >> 10) & 1023;
            auto j = id & 1023;

            double dLat = 9.0 * 1000 / 1e6;
            double dLon = dLat / std::cos(55.45 * M_PI / 180);

            TGeoCoord from = {
                36.58 + j * dLon,
                55.14 + i * dLat
            };
            TGeoCoord to = {
                36.58 + (j + 1) * dLon,
                55.14 + (i + 1) * dLat
            };
            TGeoRect rect = { from, to };
            schedule.Add(rect, temporal);
        }
        TOFStream file("./model.bin");
        schedule.Save(&file);
    }
    return EXIT_SUCCESS;
}

int main_compress(int /*argc*/, const char** /*argv*/) {
    TString data = Cin.ReadAll();
    Cout.Write(NDrive::Compress(data).GetRef());
    return EXIT_SUCCESS;
}

int main_taxi_suggest(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('u', "uid", "Yandex.Passport UID").RequiredArgument("ID").Required();
    options.AddLongOption("lat", "Latitude").RequiredArgument("LAT").Required();
    options.AddLongOption("lon", "Longitude").RequiredArgument("LON").Required();
    options.AddLongOption("tvm-client-id", "TVM client id").RequiredArgument("ID").Required();
    options.AddLongOption("tvm-secret", "TVM secret").RequiredArgument("SECRET").Required();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const auto uid = res.Get("uid");
    const auto lat = FromString<double>(res.Get("lat"));
    const auto lon = FromString<double>(res.Get("lon"));
    const auto selfClientId = FromString<ui32>(res.Get("tvm-client-id"));
    const auto selfSecret = res.Get("tvm-secret");

    NDrive::TTaxiSuggestClient::TOptions suggestOptions;

    NTvmAuth::NTvmApi::TClientSettings settings;
    settings.SetSelfTvmId(selfClientId);
    settings.EnableServiceTicketsFetchOptions(selfSecret, { suggestOptions.DestinationClientId });
    auto tvm = MakeAtomicShared<NTvmAuth::TTvmClient>(settings, CreateGlobalTvmLogger());
    auto client = MakeAtomicShared<NDrive::TTaxiSuggestClient>(suggestOptions, tvm);

    auto suggest = client->GetZeroSuggest(lat, lon, uid).ExtractValueSync();
    Cout << NJson::ToJson(suggest.Elements).GetStringRobust() << Endl;
    return EXIT_SUCCESS;
}

int main_telematics_history(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('i', "imei", "IMEI").RequiredArgument("IMEI");
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-src-table", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    TMap<ui64, NDrive::TTelematicsHistoryPtr> data;

    const auto IMEI = FromString<ui64>(res.Get("imei"));
    const TString& server = res.Get("yt-server");
    const TString& source = res.Get("yt-src-table");

    NYT::TCreateClientOptions cco;
    if (res.Has("yt-token")) {
        const auto& token = res.Get("yt-token");
        cco.Token(token);
    }

    auto client = NYT::CreateClient(server, cco);
    auto reader = client->CreateTableReader<NYT::TNode>(source);
    for (; reader->IsValid(); reader->Next()) {
        const auto& node = reader->GetRow();
        auto type = node["type"].AsString();
        if (type != "BLACKBOX_RECORDS") {
            continue;
        }
        auto imei = node["imei"].ConvertTo<ui64>();
        if (imei != IMEI) {
            continue;
        }
        auto p = data.find(imei);
        if (p == data.end()) {
            auto history = MakeAtomicShared<NDrive::TTelematicsHistory>();
            p = data.emplace(imei, history).first;
        }
        auto history = p->second;
        CHECK_WITH_LOG(history) << imei << Endl;
        auto records = NYT::FromNode<NDrive::TBlackboxRecords>(node["data"]);
        history->Add(std::move(records));

        NDrive::TFeaturesCalculationContext context;
        NDrive::TThresholdDurationCounterOptions options;
        options.Debug = true;
        options.Threshold = TDuration::Seconds(5);
        options.MinValue = 1900;
        NDrive::TThresholdDurationCounterResult result = NDrive::Calc(NDrive::MakeSensorValueCalcer(CAN_ENGINE_RPM), *history, options, &context);
        Cout << imei << ": " << result.Ranges.size() << ' ' << NJson::ToJson(result.Values).GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}

int main(int argc, const char** argv) {
    NYT::Initialize(argc, argv);
    DoInitGlobalLog("cerr", TLOG_INFO, false, false);

    TModChooser modChooser;
    modChooser.AddMode("compress", main_compress, "ZLib+Compress");
    modChooser.AddMode("get_parkingkzn_balance", main_get_fitdev_balance<NDrive::TFitDevParkingPaymentClient>, "Get balance of parkingkzn parking account");
    modChooser.AddMode("start_parkingkzn_sessions", main_start_fitdev_sessions<NDrive::TFitDevParkingPaymentClient>, "Start parkingkzn sessions on a token");
    modChooser.AddMode("get_mosparking_balance", main_get_fitdev_balance<NDrive::TMosParkingPaymentClient>, "Get balance of parking.mos parking account");
    modChooser.AddMode("start_mosparking_sessions", main_start_fitdev_sessions<NDrive::TMosParkingPaymentClient>, "Start parking.mos sessions on a token");
    modChooser.AddMode("get_spbparking_balance", main_get_fitdev_balance<NDrive::TSpbParkingPaymentClient>, "Get balance of parking.spb parking account");
    modChooser.AddMode("start_spbparking_sessions", main_start_fitdev_sessions<NDrive::TSpbParkingPaymentClient>, "Start parking.spb sessions on a token");
    modChooser.AddMode("get_insurance_info", main_get_insurance_info, "Get insurance info");
    modChooser.AddMode("get_timeline", main_get_timeline, "Get timeline by IMEI");
    modChooser.AddMode("get_tracks", main_get_tracks, "Get telematics tracks");
    modChooser.AddMode("restore_tracks", main_restore_tracks, "Restore telematics tracks");
    modChooser.AddMode("restore_tracks_2", main_restore_tracks_2, "Restore telematics tracks");
    modChooser.AddMode("restore_tracks_offline", main_restore_tracks_offline, "Restore telematics tracks");
    modChooser.AddMode("yandex_money_balance", main_yandex_money_balance, "Request Yandex.Money balance");
    modChooser.AddMode("get_parking_balance", main_get_parking_balance, "Get balance of a parking account");
    modChooser.AddMode("list_parkings", main_list_parkings, "List parkings near a location");
    modChooser.AddMode("list_parking_sessions", main_list_parking_sessions, "List parking sessions on a token");
    modChooser.AddMode("start_parking_sessions", main_start_parking_sessions, "Start parking sessions on a token");
    modChooser.AddMode("stop_parking_sessions", main_stop_parking_sessions, "Stop parking sessions on a token");
    modChooser.AddMode("ranker", main_ranker, "test ranker");
    modChooser.AddMode("geotimetable", main_geotimetable, "test geotimetable");
    modChooser.AddMode("apply_model", main_apply_model, "apply offer model");
    modChooser.AddMode("taxi_suggest", main_taxi_suggest, "invoke taxi suggest");
    modChooser.AddMode("telematics_history", main_telematics_history, "build telematics history");
    modChooser.AddMode("DRIVEANALYTICS-155", main_DRIVEANALYTICS_155, "execute routine for DRIVEANALYTICS-155");
    modChooser.AddMode("DRIVEANALYTICS-331", main_DRIVEANALYTICS_331, "prepare model for DRIVEANALYTICS-331");
    modChooser.AddMode("DRIVEANALYTICS-336", main_DRIVEANALYTICS_336, "index data for DRIVEANALYTICS-336");
    modChooser.AddMode("DRIVEANALYTICS-470", main_DRIVEANALYTICS_470, "prepare model for DRIVEANALYTICS-470");
    modChooser.AddMode("DRIVEANALYTICS-676", main_DRIVEANALYTICS_676, "calc stats for DRIVEANALYTICS-676");
    modChooser.AddMode("test_lua", main_test_lua_model, "test lua model");
    try {
        return modChooser.Run(argc, argv);
    } catch (const std::exception& e) {
        Cerr << "An exception has occurred: " << FormatExc(e) << Endl;
        return EXIT_FAILURE;
    }
}
