#include "sessions.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/insurance/task/builder.h>
#include <drive/library/cpp/timeline/requester.h>
#include <drive/library/cpp/tracks/client.h>

#include <rtline/library/json/field.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/protobuf/json/json2proto.h>

namespace {
    TAtomicSharedPtr<TBillingSession> MakeBillingSession(const NDrive::TSessionRequester::TSession& session) {
        auto result = MakeAtomicShared<TBillingSession>();
        auto performer = TString();
        for (auto&& tagEvent : session.TagEvents) {
            auto tag = MakeAtomicShared<TChargableTag>();
            tag->SetName(tagEvent.TagName);

            auto action = FromString<EObjectHistoryAction>(tagEvent.HistoryAction);
            if (action == EObjectHistoryAction::SetTagPerformer && session.Offer) {
                NDrive::NProto::TOffer proto;
                NProtobufJson::Json2Proto(session.Offer->Raw, proto);
                auto offer = ICommonOffer::ConstructFromProto<IOffer>(proto);
                Y_ENSURE(offer, proto.GetInstanceType());
                tag->SetOffer(offer);
                performer = tagEvent.HistoryUserId;
            }
            if (action == EObjectHistoryAction::DropTagPerformer) {
                performer = {};
            }
            if (performer) {
                tag->SetPerformer(performer);
            }

            TDBTag dbTag;
            dbTag.SetData(tag);
            dbTag.SetObjectId(session.Car.Id);
            dbTag.SetTagId("fake");

            auto ev = MakeAtomicShared<TTagHistoryEvent>(
                dbTag,
                action,
                tagEvent.HistoryTimestamp,
                tagEvent.HistoryUserId,
                TString{},
                TString{}
            );
            ev->SetHistoryEventId(tagEvent.HistoryEventId);
            result->AddEvent(ev, TBillingSessionSelector::AcceptImpl(*ev));
        }
        return result;
    }
}

struct TContext {
    TString SessionId;
    TVector<std::pair<TInstant, TInstant>> RidingIntervals;

    TDuration GetDuration() {
        TDuration result;
        for (auto&& [start, finish] : RidingIntervals) {
            result += finish - start;
        }
        return result;
    }

    DECLARE_FIELDS(
        Field(SessionId, "session_id"),
        Field(RidingIntervals, "riding_intervals")
    )
};

DECLARE_FIELDS_JSON_SERIALIZER(TContext);

int main_get_insurance_info(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('c', "context", "SessionContext").RequiredArgument("JSON");
    options.AddLongOption('s', "session", "SessionId").RequiredArgument("UUID");
    options.AddLongOption('o', "optimized").Optional().NoArgument();
    options.AddLongOption("drive-host", "Drive host").RequiredArgument("HOST").DefaultValue("admin.carsharing.yandex.net");
    options.AddLongOption("drive-token", "Drive token").RequiredArgument("TOKEN");
    options.AddLongOption("tracks-host", "Drive tracks storage host").RequiredArgument("HOST").DefaultValue("saas-searchproxy-maps.yandex.net");
    options.AddLongOption("tracks-port", "Drive tracks storage port").RequiredArgument("PORT").DefaultValue(17000);
    options.AddLongOption("tracks-service", "Drive tracks storage token").RequiredArgument("SERVICE").DefaultValue("drive_graph");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    TMaybe<TContext> context;
    if (res.Has("context")) {
        context = NJson::FromJson<TContext>(NJson::ReadJsonFastTree(res.Get("context")));
    }
    TString sessionId;
    if (context) {
        sessionId = context->SessionId;
    } else {
        sessionId = res.Get("session");
    }

    const bool optimized = res.Has("optimized");

    const TString& driveHost = res.Get("drive-host");
    const TString& driveExtra = TString();
    const TString& driveToken = res.Has("drive-token") ? res.Get("drive-token") : TString();

    const TString& tracksHost = res.Get("tracks-host");
    const auto tracksPort = FromString<ui16>(res.Get("tracks-port"));
    const TString& tracksService = res.Get("tracks-service");

    NDrive::TSessionRequester requester(driveHost, driveExtra, driveToken);
    NDrive::TTracksClient tracksClient(tracksService, tracksHost, tracksPort);
    NDrive::TTrackPartitionOptions trackPartitionOptions;
    trackPartitionOptions.Threshold = TDuration::Minutes(5);
    trackPartitionOptions.Fault = TDuration::Seconds(30);
    if (optimized) {
        trackPartitionOptions.AlwaysCutEdges = true;
    }

    auto session = requester.GetSession(sessionId);
    auto billingSession = MakeBillingSession(session);

    TInsuranceTasksCompilation compilation;
    Y_ENSURE(billingSession->FillCompilation(compilation));

    TDuration onlineDuration;
    for (auto&& [startEvent, finishEvent] : compilation.GetRideIntervals()) {
        auto start = Yensured(startEvent)->GetHistoryTimestamp();
        auto finish = Yensured(finishEvent)->GetHistoryTimestamp();
        auto duration = finish - start;
        if (!duration) {
            Cerr << "skip empty interval: " << start << Endl;
            continue;
        }
        Cerr << "raw: " << start << '-' << finish << Endl;

        NDrive::TTrackQuery trackQuery;
        trackQuery.RideId = TStringBuilder() << start << '-' << startEvent->GetHistoryAction() << '-' << billingSession->GetSessionId() << "-new";
        auto asyncTracks = tracksClient.GetTracks(trackQuery, TDuration::Seconds(10));
        const NDrive::TTracks& tracks = asyncTracks.GetValueSync();
        if (tracks.empty()) {
            Cerr << "no tracks: " << trackQuery.RideId << Endl;
        }

        const NDrive::TTrack& track = tracks.front();
        const auto partition = NDrive::BuildTrackPartition({start, finish}, track, trackPartitionOptions);
        for (auto&& interval : partition.Intervals) {
            Cerr << "filtered: " << interval.GetMin().Seconds() << '-' << interval.GetMax().Seconds() << Endl;
            onlineDuration += interval.GetMax() - interval.GetMin();
        }
    }

    if (context) {
        auto offlineDuration = context->GetDuration();

        NJson::TJsonValue info;
        info["session_id"] = sessionId;
        info["offline_duration"] = offlineDuration.Seconds();
        info["online_duration"] = onlineDuration.Seconds();
        info["delta"] = onlineDuration.SecondsFloat() - offlineDuration.SecondsFloat();
        Cout << info.GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}
