#include "telematics.h"

#include <drive/library/cpp/telematics/data.h>
#include <drive/library/cpp/yt/node/cast.h>
#include <drive/telematics/protocol/vega.h>

#include <library/cpp/getopt/last_getopt.h>

#include <mapreduce/yt/client/client.h>

namespace {
    class TBlackboxRecordsFilter: public NYT::IMapper<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TBlackboxRecordsFilter() = default;

        void Do(TReader* reader, TWriter* writer) override {
            for (; reader && reader->IsValid(); reader->Next()) {
                auto node = reader->MoveRow();
                try {
                    const auto& type = node["type"].AsString();
                    if (type == "BLACKBOX_RECORDS") {
                        node["blackbox_timestamp"] = NDrive::GetTimestamp(node["data"]).Seconds();
                        writer->AddRow(std::move(node));
                    }
                } catch (const std::exception& e) {
                    node["error"] = FormatExc(e);
                    writer->AddRow(std::move(node), 1);
                }
            }
        }
    };

    class TIterateTelematicsState: public NYT::IReducer<NYT::TTableReader<NYT::TNode>, NYT::TTableWriter<NYT::TNode>> {
    public:
        TIterateTelematicsState() = default;

        void Do(TReader* reader, TWriter* writer) override {
            TMaybe<double> latitude;
            TMaybe<double> longitude;
            TMaybe<double> canSpeed;
            TMaybe<double> server1Connected;
            TMaybe<double> server2Connected;
            TMaybe<double> server3Connected;
            TMaybe<double> server4Connected;
            TMaybe<double> gpsActive;
            TMaybe<double> gpsJammed;
            TMaybe<double> gpsSpoofed;
            TMaybe<double> gsmJammed;
            TMaybe<TInstant> timestamp;

            TString imei;
            TInstant outageStart;
            TVector<double> outageSpeeds;
            for (; reader && reader->IsValid(); reader->Next()) try {
                const auto& node = reader->GetRow();
                const auto& currentImei = node["imei"].ConvertTo<TString>();
                if (!imei) {
                    imei = currentImei;
                } else {
                    Y_ENSURE(imei == currentImei, "imei mismatch: " << imei << ' ' << currentImei);
                }

                const auto& data = node["data"];
                timestamp = NDrive::GetTimestamp(data);
                latitude = NDrive::GetLatitude(data);
                longitude = NDrive::GetLongitude(data);
                if (auto value = NDrive::GetCustomParameter(data, ToString(CAN_SPEED))) {
                    canSpeed = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_TCP_STATE1))) {
                    server1Connected = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_TCP_STATE2))) {
                    server2Connected = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_TCP_STATE3))) {
                    server3Connected = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_TCP_STATE4))) {
                    server4Connected = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_GPS_IS_ACTIVE))) {
                    gpsActive = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_GPS_JAMMED))) {
                    gpsJammed = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_GPS_SPOOF_SENSOR))) {
                    gpsSpoofed = *value;
                }
                if (auto value = NDrive::GetCustomParameter(data, ToString(VEGA_GSM_JAMMED))) {
                    gsmJammed = *value;
                }
                {
                    NYT::TNode result;
                    result["imei"] = imei;
                    result["latitude"] = NYT::ToNode(latitude);
                    result["longitude"] = NYT::ToNode(longitude);
                    result["can_speed"] = NYT::ToNode(canSpeed);
                    result["server_1_connected"] = NYT::ToNode(server1Connected);
                    result["server_2_connected"] = NYT::ToNode(server2Connected);
                    result["server_3_connected"] = NYT::ToNode(server3Connected);
                    result["server_4_connected"] = NYT::ToNode(server4Connected);
                    result["gps_active"] = NYT::ToNode(gpsActive);
                    result["gps_jammed"] = NYT::ToNode(gpsJammed);
                    result["gps_spoofed"] = NYT::ToNode(gpsSpoofed);
                    result["gsm_jammed"] = NYT::ToNode(gsmJammed);
                    result["timestamp"] = NYT::ToNode(timestamp);
                    writer->AddRow(std::move(result));
                }
                auto notNull = [](const TMaybe<double>& value, double fallback = 0) {
                    return value.GetOrElse(fallback) > 0.001;
                };
                bool hasGps = notNull(latitude, 55) && notNull(longitude, 37);
                bool hasGsm = notNull(server1Connected, 1) || notNull(server2Connected, 1) || notNull(server3Connected, 1) || notNull(server4Connected);
                if (!hasGps && !hasGsm) {
                    if (!outageStart) {
                        outageStart = *timestamp;
                    }
                    if (canSpeed) {
                        outageSpeeds.push_back(*canSpeed);
                    }
                } else {
                    if (outageStart) {
                        NYT::TNode result;
                        result["imei"] = imei;
                        result["start"] = NYT::ToNode(outageStart);
                        result["finish"] = NYT::ToNode(timestamp);
                        result["duration"] = (*timestamp - outageStart).Seconds();
                        result["speeds"] = NYT::ToNode(outageSpeeds);
                        std::sort(outageSpeeds.begin(), outageSpeeds.end());
                        result["max_speed"] = !outageSpeeds.empty() ? outageSpeeds.back() : NYT::TNode::CreateEntity();
                        result["min_speed"] = !outageSpeeds.empty() ? outageSpeeds.front() : NYT::TNode::CreateEntity();
                        writer->AddRow(std::move(result), 1);
                        outageStart = TInstant::Zero();
                        outageSpeeds.clear();
                    }
                }
            } catch (const std::exception& e) {
                auto node = reader->MoveRow();
                node["error"] = FormatExc(e);
                writer->AddRow(std::move(node), 2);
            }
        }
    };
}

REGISTER_MAPPER(TBlackboxRecordsFilter);
REGISTER_REDUCER(TIterateTelematicsState);

int main_DRIVEANALYTICS_676(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("yt-destination", "destination YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-server", "YT server").RequiredArgument("SERVER").DefaultValue("hahn.yt.yandex.net");
    options.AddLongOption("yt-source", "source YT table").RequiredArgument("TABLE").Required();
    options.AddLongOption("yt-token", "YT token").RequiredArgument("TOKEN").Optional();
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& server = res.Get("yt-server");
    const TString& source = res.Get("yt-source");
    const TString& destination = res.Get("yt-destination");
    TMaybe<TString> token;
    if (res.Has("yt-token")) {
        token = res.Get("yt-token");
    }

    NYT::TCreateClientOptions cco;
    if (token) {
        cco.Token(*token);
    }
    auto client = NYT::CreateClient(server, cco);
    auto tx = client;
    {
        NYT::TMapOperationSpec spec;
        spec.AddInput<NYT::TNode>(source);
        spec.AddOutput<NYT::TNode>(destination + ".filtered");
        spec.AddOutput<NYT::TNode>(destination + ".filtered.errors");
        tx->Map(spec, MakeIntrusive<TBlackboxRecordsFilter>());
    }
    {
        NYT::TSortOperationSpec spec;
        spec.AddInput(destination + ".filtered");
        spec.Output(destination + ".filtered");
        spec.SortBy({"imei", "blackbox_timestamp"});
        tx->Sort(spec);
    }
    {
        NYT::TReduceOperationSpec spec;
        spec.AddInput<NYT::TNode>(destination + ".filtered");
        spec.AddOutput<NYT::TNode>(destination);
        spec.AddOutput<NYT::TNode>(destination + ".stat");
        spec.AddOutput<NYT::TNode>(destination + ".errors");
        spec.ReduceBy({"imei"});
        spec.SortBy({"imei", "blackbox_timestamp"});
        tx->Reduce(spec, MakeIntrusive<TIterateTelematicsState>());
    }

    return EXIT_SUCCESS;
}
