#include "schemes.h"

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/logger/global/global.h>

#include <util/stream/file.h>

int main_create(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption("state", "state").RequiredArgument("STATE");
    options.AddLongOption('o', "output", "output file").OptionalArgument("FILE");
    options.SetFreeArgsNum(0);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    ESessionState state;
    if (!TryFromString(res.Get("state"), state)) {
        Cout << "Incorrect state" << Endl;
        return EXIT_FAILURE;
    }

    auto scheme = NDrive::NTest::TCurrentSessionSchemes::GetScheme(state);

    if (res.Has("output")) {
        const TString& file = res.Get("output");
        TOFStream output(file);
        TStringStream ss;
        auto json = scheme.SerializeToJson(NDrive::ISchemeElement::ReportValidation);
        NJson::WriteJson(&ss, &json, true, true);
        output.Write(ss.Str());
    } else {
        Cout << scheme.SerializeToJson().GetStringRobust() << Endl;
    }

    return EXIT_SUCCESS;
}

int main_check(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddHelpOption();
    options.AddLongOption('i', "input", "input file with json").RequiredArgument("FILE");
    options.SetFreeArgsNum(0);
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& file = res.Get("input");
    TIFStream input(file);
    TString contentJson = input.ReadAll();
    NJson::TJsonValue json;
    if (!NJson::ReadJsonTree(contentJson, &json)) {
        Cout << "Incorrect json" << Endl;
        return EXIT_FAILURE;
    }
    NDrive::TScheme scheme;
    if (!scheme.DeserializeFromJson(json)) {
        Cout << "Incorrect json" << Endl;
        return EXIT_FAILURE;
    }
    Cout << "Ok" << Endl;
    return EXIT_SUCCESS;
}

int main(int argc, const char** argv) {
    DoInitGlobalLog("cerr", TLOG_INFO, false, false);

    TModChooser modChooser;
    modChooser.AddMode("check", main_check, "validate json");
    modChooser.AddMode("create", main_create, "create json");
    try {
        return modChooser.Run(argc, argv);
    } catch (const std::exception& e) {
        Cerr << "An exception has occurred: " << FormatExc(e) << Endl;
        return EXIT_FAILURE;
    }
    auto scheme = NDrive::NTest::TCurrentSessionSchemes::GetScheme(ESessionState::Undefined);
    Cout << scheme.SerializeToJson() << Endl;
    return 0;
}
