#include "schemes.h"

#include <drive/tools/schemes/schemes.h_serialized.h>

#include <drive/backend/user_devices/manager.h>

struct TModelsInfo {
    TModelsInfo() {
        Scheme.Add<TFSNumeric>("cars_count", "Количество машин").SetMin(0);
        Scheme.Add<TFSVariants>("code", "Идентификатор модели").SetVariants(Models);
        Scheme.Add<TFSVariants>("fuel_cap_side", "С какой стороны лючек бензобака").SetVariants({ "right", "left" });
        Scheme.Add<TFSString>("fuel_type", "Тип топлива");
        Scheme.Add<TFSString>("fuel_icon_url", "Иконка заправки");
        Scheme.Add<TFSString>("image_angle_url");
        Scheme.Add<TFSString>("image_large_url");
        Scheme.Add<TFSString>("image_map_url_2x");
        Scheme.Add<TFSString>("image_map_url_3x");
        Scheme.Add<TFSString>("image_pin_url_2x");
        Scheme.Add<TFSString>("image_pin_url_3x");
        Scheme.Add<TFSString>("image_small_url");
        Scheme.Add<TFSString>("manufacturer", "Производитель");
        Scheme.Add<TFSString>("name", "Полное название");
        Scheme.Add<TFSString>("registry_manufacturer", "Производитель(для загрузки из файлов)");
        Scheme.Add<TFSString>("registry_model", "Модель(для загрузки из файла)");
        Scheme.Add<TFSString>("short_name", "Короткое название модели");

        SpecificationScheme.Add<TFSString>("id", "Идентификатор").SetVisual(TFSString::EVisualType::UUID);
        SpecificationScheme.Add<TFSString>("name", "Название").SetRequired(true);
        SpecificationScheme.Add<TFSNumeric>("position", "Позиция в выдаче").SetMin(0).SetRequired(true);
        SpecificationScheme.Add<TFSString>("value", "Значение").SetRequired(true);
        SpecificationScheme.Add<TFSString>("icon", "Иконка").SetRequired(false);
        Scheme.Add<TFSArray>("specifications", "Особенности").SetElement(SpecificationScheme);

        FullScheme.AddPattern<TFSStructure>("(" + JoinSeq("|", TModelsInfo::Models) + ")").SetStructure(Scheme);
    }

    NDrive::TScheme Scheme;
    NDrive::TScheme FullScheme;
    NDrive::TScheme SpecificationScheme;
    static TSet<TString> Models;

    static NDrive::TScheme GetScheme() {
        return Default<TModelsInfo>().FullScheme;
    }
};

TSet<TString> TModelsInfo::Models = { "porsche_carrera", "kia_rio" };

struct TCarCommonInfo {
    TCarCommonInfo() {
        CommonCarScheme.Add<TFSString>("id", "Идентификатор").SetVisual(TFSString::EVisualType::UUID).SetRequired(true);
        CommonCarScheme.Add<TFSVariants>("model_id", "Идентификатор модели").SetVariants(TModelsInfo::Models).SetRequired(true);
        CommonCarScheme.Add<TFSString>("number", "Гос. номер").SetRequired(true);

        TelematicScheme.Add<TFSNumeric>("fuel_distance", "Расстояние, которое автомобиль может проехать до следующей заправки (км)").SetMin(0).SetRequired(true);
        TelematicScheme.Add<TFSNumeric>("fuel_level", "Уровень топлива (%)").SetMin(0).SetMax(100).SetRequired(true);
        TelematicScheme.Add<TFSBoolean>("is_engine_on", "Флаг включеного двигателя");
        TelematicScheme.Add<TFSBoolean>("is_front_left_door_open", "Флаг открытия передней левой двери");
        TelematicScheme.Add<TFSBoolean>("is_front_right_door_open", "Флаг открытия передней правой двери");
        TelematicScheme.Add<TFSBoolean>("is_rear_left_door_open", "Флаг открытия задней левой двери");
        TelematicScheme.Add<TFSBoolean>("is_rear_right_door_open", "Флаг открытия задней правой двери");
        TelematicScheme.Add<TFSBoolean>("is_hood_open", "Флаг открытия капота");
        TelematicScheme.Add<TFSBoolean>("is_trunk_open", "Флаг открытия багажника");
        TelematicScheme.Add<TFSBoolean>("ble_mac");
        TelematicScheme.Add<TFSBoolean>("ble_passkey");

        LocationScheme.Add<TFSNumeric>("course").SetRequired(true);
        LocationScheme.Add<TFSNumeric>("lat", "Широта").SetRequired(true);
        LocationScheme.Add<TFSNumeric>("lon", "Долгота").SetRequired(true);

        FeatureScheme.Add<TFSString>("tag").SetRequired(true);
        FeatureScheme.Add<TFSString>("performer");
        FeatureScheme.Add<TFSNumeric>("priority").SetMin(0);
        FeatureScheme.SetAdditionalProperties(true);
    }

    NDrive::TScheme CommonCarScheme;
    NDrive::TScheme LocationScheme;
    NDrive::TScheme TelematicScheme;
    NDrive::TScheme FeatureScheme;

    static NDrive::TScheme GetScheme(ESessionState state) {
        NDrive::TScheme result = Default<TCarCommonInfo>().CommonCarScheme;
        if (state == ESessionState::NoRides) {
            return result;
        }
        result.Add<TFSArray>("sf", "Идентификаторы спецификаций авто").SetElement<TFSNumeric>();
        if (state == ESessionState::NoReservation) {
            return result;
        }
        result.Add<TFSArray>("session_features", "Спецификации авто").SetElement(Default<TCarCommonInfo>().FeatureScheme);
        auto& locationValue = result.Add<TFSStructure>("location", "Местоположение");
        locationValue.SetStructure(Default<TCarCommonInfo>().LocationScheme);

        auto& telematicValue = result.Add<TFSStructure>("telematics", "Показания телематики");
        telematicValue.SetStructure(Default<TCarCommonInfo>().TelematicScheme);

        if (state != ESessionState::Undefined) {
            locationValue.SetRequired(true);
            telematicValue.SetRequired(true);
        }
        return result;
    }
};

struct TFeedbackInfo {
    TFeedbackInfo() {
        ButtonScheme.Add<TFSString>("icon", "Иконка").SetRequired(true);
        ButtonScheme.Add<TFSString>("link", "Ссылка").SetRequired(true);
        ButtonScheme.Add<TFSString>("title", "Заголовок").SetRequired(true);
        Scheme.Add<TFSVariants>("set_id").SetVariants({
            "feedback_post",
            "feedback_acceptance",
            "feedback_accepting",
            "feedback_reservation",
            "feedback_riding",
        });
    }

    NDrive::TScheme Scheme;
    NDrive::TScheme ButtonScheme;

    static NDrive::TScheme GetScheme(ESessionState /*state*/) {
        NDrive::TScheme result = Default<TFeedbackInfo>().Scheme;
        auto& buttonValue = result.Add<TFSArray>("buttons");
        buttonValue.SetElement(Default<TFeedbackInfo>().ButtonScheme);
        return result;
    }
};

struct TPOIInfo {
    TPOIInfo() {
        Scheme.Add<TFSString>("url", "Ссылка").SetRequired(true);
        Scheme.Add<TFSString>("message", "Сообщение").SetRequired(true);
        Scheme.Add<TFSString>("title", "Заголовок").SetRequired(true);
        Scheme.Add<TFSString>("image", "Изображение");
        Scheme.Add<TFSString>("small_image", "Уменьшенное изображение");
        Scheme.Add<TFSString>("top_color", "Цвет");
        Scheme.Add<TFSString>("bottom_color", "Цвет");
    }

    NDrive::TScheme Scheme;

    static NDrive::TScheme GetScheme() {
        return Default<TPOIInfo>().Scheme;
    }
};

struct TSupportInfo {
    TSupportInfo() {
        Scheme.Add<TFSString>("support_feedback_form", "Ссылка на форму обратной связи").SetRequired(true);
        Scheme.Add<TFSString>("support_phone", "Телефон поддержки").SetRequired(true);
        Scheme.Add<TFSString>("support_telegram", "Логин поддержки в телеграмм").SetRequired(true);
        Scheme.Add<TFSString>("support_whatsapp", "Телефон поддержки в whatsapp").SetRequired(true);
        Scheme.SetAdditionalProperties(true);
    }

    NDrive::TScheme Scheme;

    static NDrive::TScheme GetScheme(ESessionState state) {
        NDrive::TScheme result = Default<TSupportInfo>().Scheme;
        auto& phoneValue = result.Add<TFSString>("phone", "Контактный телефон пользователя");
        if (state != ESessionState::NoRides && state != ESessionState::Undefined) {
            phoneValue.SetRequired(true);
        }
        return result;
    }
};

struct TTransportationInfo {
    TTransportationInfo() {
        FirstMileScheme.Add<TFSArray>("methods").SetElement<NDrive::TScheme>().SetAdditionalProperties(true);
        Scheme.Add<TFSStructure>("first_mile").SetStructure(FirstMileScheme);
    }

    NDrive::TScheme Scheme;
    NDrive::TScheme FirstMileScheme;

    static NDrive::TScheme GetScheme() {
        return Default<TTransportationInfo>().Scheme;
    }
};

struct TSegmentInfo {
    TSegmentInfo() {
        Scheme.Add<TFSIgnore>("meta").SetRequired(true);
        Scheme.Add<TFSIgnore>("session").SetRequired(true);

        BillRecordScheme.Add<TFSNumeric>("cost").SetRequired(true);
        BillRecordScheme.Add<TFSString>("type").SetRequired(true);
        BillRecordScheme.Add<TFSString>("title").SetRequired(true);
        BillRecordScheme.Add<TFSString>("icon");
        BillRecordScheme.Add<TFSString>("id");
        BillRecordScheme.Add<TFSNumeric>("duration").SetVisual(TFSNumeric::EVisualType::DateTime);
        BillRecordScheme.Add<TFSString>("details");
        BillRecordScheme.Add<TFSNumeric>("distance").SetMin(0);

        auto& multiBillValue = Scheme.Add<TFSStructure>("multi_bill").SetStructure<NDrive::TScheme>();
        multiBillValue.Add<TFSArray>("bills").SetElement<TFSArray>().SetElement(BillRecordScheme);
        multiBillValue.Add<TFSArray>("records").SetElement(BillRecordScheme);
    }

    NDrive::TScheme Scheme;
    NDrive::TScheme BillRecordScheme;

    static NDrive::TScheme GetScheme(ESessionState state) {
        NDrive::TScheme result = Default<TSegmentInfo>().Scheme;
        if (state == ESessionState::NoRides || state == ESessionState::Reservation) {
            return result;
        }
        auto& acceptanceStart = result.Add<TFSNumeric>("session_acceptance_start", "Время начала приемки").SetVisual(TFSNumeric::EVisualType::DateTime);
        if (state != ESessionState::Undefined && state != ESessionState::NoReservation) {
            acceptanceStart.SetRequired(true);
        }
        if (state == ESessionState::Acceptance) {
            return result;
        }
        auto& acceptanceFinish = result.Add<TFSNumeric>("session_acceptance_finish", "Время окончания приемки").SetVisual(TFSNumeric::EVisualType::DateTime);
        if (state != ESessionState::Undefined  && state != ESessionState::NoReservation) {
            acceptanceFinish.SetRequired(true);
        }
        return result;
    }
};

NDrive::NTest::TCurrentSessionSchemes::TCurrentSessionSchemes() {
    Schemes.resize(::GetEnumItemsCount<ESessionState>());
    for (const auto& state : GetEnumAllValues<ESessionState>()) {
        Schemes[(ui32)state] = BuildScheme(state);
    }
}

NDrive::TScheme NDrive::NTest::TCurrentSessionSchemes::BuildScheme(ESessionState state) {
    NDrive::TScheme commonScheme;

    // area_common_user_info
    {
        auto& userCommonInfoBase = commonScheme.Add<TFSStructure>("area_common_user_info", "Алерт поддержки");
        userCommonInfoBase.SetStructure(TSupportInfo::GetScheme(state));
        userCommonInfoBase.SetRequired(true);
    }

    // transportation
    switch (state) {
    case ESessionState::Reservation: {
        auto& transportationValue = commonScheme.Add<TFSStructure>("transportation", "Способы добраться до машины");
        transportationValue.SetStructure(TTransportationInfo::GetScheme());
        transportationValue.SetRequired(true);
        break;
    }
    case ESessionState::Acceptance:
    case ESessionState::Undefined:
        commonScheme.Add<TFSStructure>("transportation", "Способы добраться до машины").SetStructure(TTransportationInfo::GetScheme());
        break;
    default:
        break;
    }

    // feedback
    {
        auto& feedbackValue = commonScheme.Add<TFSStructure>("feedback", "Ссылки на соцсети");
        feedbackValue.SetStructure(TFeedbackInfo::GetScheme(state));
        switch (state) {
        case ESessionState::NoReservation:
            feedbackValue.SetRequired(true);
            break;
        default:
            break;
        }
    }

    // notification
    switch (state) {
    case ESessionState::NoRides:
    case ESessionState::NoReservation:
    case ESessionState::Reservation:
    case ESessionState::Acceptance:
        break;
    case ESessionState::Undefined:
        commonScheme.Add<TFSString>("notification");
        break;
    default:
        commonScheme.Add<TFSString>("notification").SetRequired(true);
        break;
    }

    // poi
    switch (state) {
    case ESessionState::NoRides:
    case ESessionState::NoReservation:
        break;
    case ESessionState::Undefined:
        commonScheme.Add<TFSArray>("poi", "Интересные места в городе").SetElement(TPOIInfo::GetScheme());
        break;
    default:
        auto& poiValue = commonScheme.Add<TFSArray>("poi", "Интересные места в городе");
        poiValue.SetElement(TPOIInfo::GetScheme());
        poiValue.SetRequired(true);
        break;
    }

    commonScheme.Add<TFSIgnore>("sf", "Спецификация авто");

    // fueling_ability
    {
        auto& fuelingValue = commonScheme.Add<TFSBoolean>("fueling_ability");
        switch (state) {
        case ESessionState::Riding:
        case ESessionState::Parking:
            fuelingValue.SetRequired(true);
            break;
        default:
            break;
        }
    }

    // flags
    {
        auto& flagsValue = commonScheme.Add<TFSStructure>("flags");
        flagsValue.SetStructure<NDrive::TScheme>().SetAdditionalProperties(true);
        flagsValue.SetRequired(true);
    }

    // device_diff
    {
        auto& deviceDiffValue = commonScheme.Add<TFSStructure>("device_diff");
        deviceDiffValue.SetStructure<NDrive::TScheme>().SetAdditionalProperties(true);
        switch (state) {
        case ESessionState::NoRides:
        case ESessionState::Undefined:
            break;
        default:
            deviceDiffValue.SetRequired(true);
            break;
        }
    }

    commonScheme.Add<TFSIgnore>("property_patches");

    // models
    {
        auto& modelValue = commonScheme.Add<TFSStructure>("models", "Описание моделей");
        modelValue.SetStructure(TModelsInfo::GetScheme());
        switch (state) {
        case ESessionState::NoRides:
        case ESessionState::Undefined:
            break;
        default:
            modelValue.SetRequired(true);
            break;
        }
    }

    // car
    {
        auto& carValue = commonScheme.Add<TFSStructure>("car", "Информация об авто");
        carValue.SetStructure(TCarCommonInfo::GetScheme(state));
        switch (state) {
        case ESessionState::NoRides:
        case ESessionState::Undefined:
            break;
        default:
            carValue.SetRequired(true);
            break;
        }
    }

    // server_time
    {
        auto& serverTimeValue = commonScheme.Add<TFSNumeric>("server_time", "Серверное время").SetVisual(TFSNumeric::EVisualType::DateTime);
        switch (state) {
        case ESessionState::NoRides:
        case ESessionState::NoReservation:
            serverTimeValue.SetRequired(true);
            break;
        default:
            break;
        }
    }

    commonScheme.Add<TFSIgnore>("user", "Информация о пользователе").SetRequired(true);

    // segment
    {
        auto& segmentValue = commonScheme.Add<TFSStructure>("segment");
        segmentValue.SetStructure(TSegmentInfo::GetScheme(state));
        switch (state) {
        case ESessionState::NoRides:
        case ESessionState::Undefined:
            break;
        default:
            segmentValue.SetRequired(true);
            break;
        }
    }

    commonScheme.Add<TFSVariants>("device_status").InitVariants<ENewDeviceStatus>().SetRequired(true);
    return commonScheme;
}

const NDrive::TScheme& NDrive::NTest::TCurrentSessionSchemes::GetScheme(ESessionState state) {
    CHECK_WITH_LOG((ui32)state < Default<NDrive::NTest::TCurrentSessionSchemes>().Schemes.size());
    return Default<NDrive::NTest::TCurrentSessionSchemes>().Schemes[(ui32)state];
}
