import argparse
import hashlib
import logging
import os
import socket
import time

from nanny_sox_audit.fileutils import parse_checksum_from_dump_json
from nanny_sox_audit.jugglerutils import push_juggler

def calc_checksum(filename):
    h = hashlib.md5()
    with open(filename, 'rb') as f:
        while True:
            chunk = f.read(1024 * 1024)
            if not chunk:
                break
            h.update(chunk)
    return h.hexdigest()

def main():
    logging.basicConfig(format='[%(asctime)s] %(levelname)s: %(message)s', level=logging.INFO)

    parser = argparse.ArgumentParser()
    parser.add_argument('-r', '--root', required=True, help='instance working directory')
    parser.add_argument('-f', '--file', required=True, help='file to monitor', action='append')
    parser.add_argument('-c', '--checksum-file', default='dump.json')
    parser.add_argument('-d', '--delay', type=int, default=300)
    parser.add_argument('-i', '--juggler-instance', default='')
    parser.add_argument('-u', '--juggler-url', default='http://[::1]:31579/events')
    parser.add_argument('--host', default=socket.gethostname())
    parser.add_argument('-s', '--juggler-service', default='drive_sox_audit')
    args = parser.parse_args()

    checksum_file = os.path.join(args.root, args.checksum_file)
    checksums = {}
    for i in args.file:
        filename = os.path.join(args.root, i)
        checksum = parse_checksum_from_dump_json(checksum_file, i)
        checksums[filename] = checksum

    for filename, checksum in checksums.items():
        logging.info("monitoring {} checksum {}".format(filename, checksum))

    while True:
        logging.info("starting periodical check")
        status = 'OK'
        description = 'checksum check ok'
        try:
            for filename, checksum in checksums.items():
                calculated = calc_checksum(filename)
                if calculated != checksum:
                    status = 'CRIT'
                    description = 'checksum check fail: {}'.format(filename)
                    break
        except Exception as e:
            status = 'CRIT'
            description = str(e)

        logging.info('status is {}, {}'.format(status, description))
        r = push_juggler(
            url=args.juggler_url,
            host=args.host,
            service=args.juggler_service,
            status=status,
            instance=args.juggler_instance,
            desc=description
        )
        logging.info('pushed to juggler [{}], sleep for {} seconds...'.format(r.status_code, args.delay))
        time.sleep(args.delay)

if __name__ == '__main__':
    main()
