/**
  ******************************************************************************
  * @file    usbd_cdc_vcp.c
  * @author  MCD Application Team
  * @version V1.2.0
  * @date    09-November-2015
  * @brief   Generic media access Layer.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT 2015 STMicroelectronics</center></h2>
  *
  * Licensed under MCD-ST Liberty SW License Agreement V2, (the "License");
  * You may not use this file except in compliance with the License.
  * You may obtain a copy of the License at:
  *
  *        http://www.st.com/software_license_agreement_liberty_v2
  *
  * Unless required by applicable law or agreed to in writing, software 
  * distributed under the License is distributed on an "AS IS" BASIS, 
  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  ******************************************************************************
  */ 

#ifdef USB_OTG_HS_INTERNAL_DMA_ENABLED 
#pragma     data_alignment = 4 
#endif /* USB_OTG_HS_INTERNAL_DMA_ENABLED */

/* Includes ------------------------------------------------------------------*/
#include "usbd_cdc_vcp.h"

#if defined(USE_FREERTOS)
#include "FreeRTOS.h"
#include "queue.h"
#include "task.h"
#include "semphr.h" 
#endif //USE_FREERTOS

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/                                     
__IO uint16_t usbd_cdc_SOF_timer=0;

static usb_cdc_linecoding_t linecoding=
{
  .bitrate=115200,  /* baud rate*/
  .format=0x00,     /* stop bits-1*/
  .paritytype=0x00, /* parity - none*/
  .datatype=0x08,   /* nb. of bits 8*/
};

static __IO struct VCP_struct
{
  uint8_t  IsDataSent;
  uint8_t  RxBuff[CDC_DATA_OUT_PACKET_SIZE];
  uint16_t AwlLenInRxBuf;
#if defined(USE_FREERTOS)
  SemaphoreHandle_t rx_sync_obj;
#endif //defined(USE_FREERTOS)
}VCP=
{
  .IsDataSent=1,
  .AwlLenInRxBuf=0,
#if defined(USE_FREERTOS)
  .rx_sync_obj=NULL,
#endif //defined(USE_FREERTOS)
};

/* Private function prototypes -----------------------------------------------*/
static uint16_t VCP_DataTx(void);
static uint16_t VCP_DataRx(uint32_t len);
#if defined(USB_CDC_VCP_LINECODING_SUPPORT)
static uint16_t VCP_Ctrl (uint32_t Cmd, uint8_t* Buf, uint32_t Len);
#endif //defined(USB_CDC_VCP_LINECODING_SUPPORT)

CDC_IF_Prop_TypeDef VCP_fops = 
{
  .pIf_DataTx=VCP_DataTx,
  .pIf_DataRx=VCP_DataRx,
#if defined(USB_CDC_VCP_LINECODING_SUPPORT)
  .pIf_Ctrl=VCP_Ctrl,
#endif //defined(USB_CDC_VCP_LINECODING_SUPPORT)
};

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  VCP_DataTx
  *         CDC received data to be send over USB IN endpoint are managed in 
  *         this function.
  * @retval Result of the operation: USBD_OK if all operations are OK else VCP_FAIL
  */
static uint16_t VCP_DataTx(void)
{ 
  VCP.IsDataSent=1;
  return USBD_OK;
}

/**
  * @brief  VCP_DataRx
  *         Data received over USB OUT endpoint are sent over CDC interface 
  *         through this function.
  *           
  *         @note
  *         This function will block any OUT packet reception on USB endpoint 
  *         until exiting this function. If you exit this function before transfer
  *         is complete on CDC interface (ie. using DMA controller) it will result 
  *         in receiving more data while previous ones are still not sent.
  *                 
  * @param  Len: Number of data received (in bytes)
  * @retval Result of the operation: USBD_OK if all operations are OK else VCP_FAIL
  */
static uint16_t VCP_DataRx(uint32_t len)
{ 
  VCP.AwlLenInRxBuf=len;
  
#if defined(USE_FREERTOS)
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  xSemaphoreGiveFromISR(VCP.rx_sync_obj, &xHigherPriorityTaskWoken);
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
#endif //defined(USE_FREERTOS)
  
  return USBD_OK;
}

#if defined(USB_CDC_VCP_LINECODING_SUPPORT)
/**
  * @brief  VCP_Ctrl
  *         Manage the CDC class requests
  * @param  Cmd: Command code            
  * @param  Buf: Buffer containing command data (request parameters)
  * @param  Len: Number of data to be sent (in bytes)
  * @retval Result of the operation (USBD_OK in all cases)
  */
static uint16_t VCP_Ctrl (uint32_t Cmd, uint8_t* Buf, uint32_t Len)
{ 
  switch (Cmd)
  {
  case SEND_ENCAPSULATED_COMMAND:
    /* Not  needed for this driver */
    break;

  case GET_ENCAPSULATED_RESPONSE:
    /* Not  needed for this driver */
    break;

  case SET_COMM_FEATURE:
    /* Not  needed for this driver */
    break;

  case GET_COMM_FEATURE:
    /* Not  needed for this driver */
    break;

  case CLEAR_COMM_FEATURE:
    /* Not  needed for this driver */
    break;

  case SET_LINE_CODING:
    linecoding.bitrate = (uint32_t)(Buf[0] | (Buf[1] << 8) | (Buf[2] << 16) | (Buf[3] << 24));
    linecoding.format = Buf[4];
    linecoding.paritytype = Buf[5];
    linecoding.datatype = Buf[6];
    /* Set the new configuration */
    //VCP_COMConfig(OTHER_CONFIG);
    break;

  case GET_LINE_CODING:
    Buf[0] = (uint8_t)(linecoding.bitrate);
    Buf[1] = (uint8_t)(linecoding.bitrate >> 8);
    Buf[2] = (uint8_t)(linecoding.bitrate >> 16);
    Buf[3] = (uint8_t)(linecoding.bitrate >> 24);
    Buf[4] = linecoding.format;
    Buf[5] = linecoding.paritytype;
    Buf[6] = linecoding.datatype; 
    break;

  case SET_CONTROL_LINE_STATE:
    /* Not  needed for this driver */
    break;

  case SEND_BREAK:
    /* Not  needed for this driver */
    break;    
    
  default:
    break;
  }

  return USBD_OK;
}
#endif //defined(USB_CDC_VCP_LINECODING_SUPPORT)

extern USB_OTG_CORE_HANDLE USB_OTG_dev;

void usbd_cdc_SOF_timer_callback(void)
{
  if(usbd_cdc_SOF_timer) usbd_cdc_SOF_timer--;
}

void USB_CDC_Init(void)
{
#if defined(USE_FREERTOS)
  if(VCP.rx_sync_obj==NULL) vSemaphoreCreateBinary(VCP.rx_sync_obj);
  xSemaphoreTake(VCP.rx_sync_obj, 0);
#endif //defined(USE_FREERTOS)
  
  VCP.IsDataSent=0;
  VCP.AwlLenInRxBuf=0;
  DCD_EP_PrepareRx(&USB_OTG_dev, CDC_OUT_EP, (uint8_t*)VCP.RxBuff, sizeof(VCP.RxBuff));
}

usb_cdc_linecoding_t* USB_CDC_GetVcpLineCodingPtr(void)
{
  return &linecoding;
}

void USB_CDC_Deinit(void)
{
  // 
  DCD_EP_Flush(&USB_OTG_dev, CDC_IN_EP);
  DCD_EP_Flush(&USB_OTG_dev, CDC_OUT_EP);

  //portENTER_CRITICAL();
  //DCD_EP_Close(&USB_OTG_dev,     CDC_IN_EP);
  //DCD_EP_Close(&USB_OTG_dev,     CDC_OUT_EP);
  //DCD_EP_Close(&USB_OTG_dev,     CDC_CMD_EP);
  //portEXIT_CRITICAL();
}

uint8_t USB_CDC_State(void)
{
  if(!(USB_OTG_dev.dev.device_status==USB_OTG_CONFIGURED)) 
  {
    return 0;
  }
  else
  {
    if(!usbd_cdc_SOF_timer)
    {
      return 0;
    }
    return 1;
  }
}

int16_t USB_CDC_Send(uint8_t* data, uint16_t size, uint32_t xTicksToWait)
{
  uint16_t chunks_count;
  
  if(!(USB_OTG_dev.dev.device_status==USB_OTG_CONFIGURED)) return -1;
  
  if(!usbd_cdc_SOF_timer)
  {
    USB_CDC_Deinit();
    return -1;
  }
  
  uint32_t start_send_time=xTaskGetTickCount();
    
  chunks_count=size/CDC_DATA_IN_PACKET_SIZE;
  if(size%CDC_DATA_IN_PACKET_SIZE) chunks_count++;
    
  for(uint16_t i=0; i<chunks_count; i++)
  {
    uint16_t len=CDC_DATA_IN_PACKET_SIZE;
    if(i==(chunks_count-1) && (size%CDC_DATA_IN_PACKET_SIZE)) len=size%CDC_DATA_IN_PACKET_SIZE;
    
    VCP.IsDataSent=0;
    //Prepare the available data buffer to be sent on IN endpoint
    DCD_EP_Tx(&USB_OTG_dev, CDC_IN_EP, data, len);
    while(!VCP.IsDataSent)
    {
      if((xTaskGetTickCount()-start_send_time)>=xTicksToWait) return -1;
    }
    if(i==(chunks_count-1) && len==CDC_DATA_IN_PACKET_SIZE) 
    {
      //Send ZLP to indicate the end of the current transfer
      VCP.IsDataSent=0; 
      DCD_EP_Tx(&USB_OTG_dev, CDC_IN_EP, NULL, 0);
      while(!VCP.IsDataSent)
      {
        if((xTaskGetTickCount()-start_send_time)>=xTicksToWait) return -1;
      }
    }
    data+=len;
  }
  
  return size;
}

//  CDC_DATA_OUT_PACKET_SIZE 
int16_t USB_CDC_Receive_nobloking(uint8_t* data, uint16_t size)
{
  if(!(USB_OTG_dev.dev.device_status==USB_OTG_CONFIGURED)) return -1;
  
  if(!usbd_cdc_SOF_timer)
  {
    USB_CDC_Deinit();
    return -1;
  }

  if(!size) return 0;
  
  uint16_t copy_len, alw_len;
  
  alw_len=VCP.AwlLenInRxBuf;
  copy_len=alw_len;
  if(copy_len>size) copy_len=size;
  
  if(copy_len)
  {
    memcpy(data, (uint8_t*)VCP.RxBuff, copy_len);
    alw_len-=copy_len;
    VCP.AwlLenInRxBuf=alw_len;
    
    if(!alw_len)
    {
      DCD_EP_PrepareRx(&USB_OTG_dev, CDC_OUT_EP, (uint8_t*)VCP.RxBuff, CDC_DATA_OUT_PACKET_SIZE);
    }
    else
    {
      for(uint16_t i=0; i<alw_len; i++) VCP.RxBuff[i]=VCP.RxBuff[i+copy_len];
    }
  }

  return copy_len;
}

int16_t USB_CDC_Receive(uint8_t* data, uint16_t size, uint32_t xTicksToWait)
{
  if(!size) return 0;
  if(size>INT16_MAX) size=INT16_MAX;
  
  uint16_t tot_recv_len=0;
  
  uint32_t start_receive_time=xTaskGetTickCount();
  
  do{
#if defined(USE_FREERTOS)
    xSemaphoreTake(VCP.rx_sync_obj, 1);
#endif //defined(USE_FREERTOS)
    
    int16_t recv_len=USB_CDC_Receive_nobloking(&data[tot_recv_len], size-tot_recv_len);
    
    if(recv_len>0)       {tot_recv_len+=recv_len;}
    else if(recv_len==0) {}//vTaskDelay(1);
    else                 {return recv_len;}// 
  }while(((xTaskGetTickCount()-start_receive_time)<xTicksToWait) && tot_recv_len<size);
  
  return tot_recv_len;
}

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
