/*
** ============================================================================
** FILE			Accel_tilt.c
** DESCRIPTION		Contains all subroutine for LIS3DH chip 
**
** TERGET		STM32L151
** CREATED		Solovyev A.F. Vega-Absolute Ltd.
** DATA			14.12.2012
** ============================================================================
*/

/* Includes ------------------------------------------------------------------*/
#include <stdbool.h>

#include "Accel_sensors_config.h"
#include "Accel\Accel_sense\Accel_tilt.h"
#include "Accel\Accel_sense\Accel_filtr.h"

/* Defines -------------------------------------------------------------------*/
//#define TS_REAL_THRESHOLD                     0
//#define TS_CALIBRATION_THRESHOLD              1
#define TS_CUT_HF_THRESHOLD                     36

// ! 
// TS_COUNT_OUTSIDE_THRESHOLD >  2   TS_BUF_SIZE_SMPL + TS_BUF_SIZE_MA1 + TS_BUF_SIZE_MA2 + TS_BUF_SIZE_MA3

#define TS_BUF_SIZE_SMPL                 50       // 750 ms 
#define TS_BUF_SIZE_MA1                  70       // 1040 ms 
#define TS_BUF_SIZE_MA2                  90       // 1350 ms
#define TS_BUF_SIZE_MA3                  110      // 1650 ms 
#define TS_BUF_SIZE_AVERAGE              50       // 5000 ms -   
#define TS_COUNT_OUTSIDE_THRESHOLD       500      // 4  -    
#define TS_PERIOD_GET_SAMPLES            6        // 15 ms
#define TS_PERIOD_SEARCH_AVERAGE         7        // 105 ms        

//#define TILT_THR_STEP                  1        //   
#define TILT_THR_LOW                     5        //  
#define TS_CALIBRATION_PERIOD            600      //  30  

//ACCEL_TILT_DELAY_TO_STOP
//#define TS_DELAY_TO_STOP		5*400 // 30*400 = 30s * 400kHz

////   //////
const uint32_t TiltThreshold[8] = {
  0,    // 0               
  18,   // 1   
  15,   // 2    
  13,   // 3   
  11,   // 4   
  9,    // 5   
  7,    // 6   
  TILT_THR_LOW,    // 7                        
};


/* Global variables ----------------------------------------------------------*/
static uint8_t AccelTiltSenseLevel = 3; 	//    . 3 - 1
static uint8_t AccelTiltState = 0;		//   

extern ACCEL_POINT_16BIT AcceluSmpl; //  uint16 

/* Private variables ---------------------------------------------------------*/
typedef enum {RESET=0, SET=1} FlagStatus;
typedef enum {
  tss_SetThreshold,
  tss_CheckThreshold,
  tss_CheckImmobility,
  tss_SendAlarm,
} TILT_SENSOR_STATE;
TILT_SENSOR_STATE TiltSensorState = tss_SetThreshold; //    


//-  
ACCEL_POINT_16BIT CmpSmpl; //  
ACCEL_POINT_16BIT TS_BufSmpl[TS_BUF_SIZE_SMPL]; //   
ACCEL_POINT_CTRL TS_CtrlSmpl; //    
//-   1
ACCEL_POINT_16BIT TS_ItemMa1;       
ACCEL_POINT_16BIT TS_BufMa1[TS_BUF_SIZE_MA1];
ACCEL_POINT_CTRL TS_CtrlMa1;
//-   2
ACCEL_POINT_16BIT TS_ItemMa2;      
ACCEL_POINT_16BIT TS_BufMa2[TS_BUF_SIZE_MA2];
ACCEL_POINT_CTRL TS_CtrlMa2;
//-   3
ACCEL_POINT_16BIT TS_ItemMa3;      
ACCEL_POINT_16BIT TS_BufMa3[TS_BUF_SIZE_MA3];
ACCEL_POINT_CTRL TS_CtrlMa3;
//-   4
ACCEL_POINT_16BIT TS_ItemMa4;  
//-  AVERAGE
ACCEL_POINT_16BIT TS_StartAverage;
ACCEL_POINT_16BIT TS_BufAvrg[TS_BUF_SIZE_AVERAGE];
ACCEL_POINT_CTRL TS_CtrlAvrg;


ACCEL_POINT_16BIT TS_CalibrationThresholdLevelUpper; //    
ACCEL_POINT_16BIT TS_CalibrationThresholdLevelLower; //    
ACCEL_POINT_16BIT TS_ThresholdLevelUpper;   		//      
ACCEL_POINT_16BIT TS_ThresholdLevelLower;  		//      

static uint8_t Count_TsSamples = 0x00;				//      
static uint8_t Count_TsSearchAverage = 0x00; 			//         
static FlagStatus F_TS_FAST_NEXT_STATE = RESET; 		//       
static uint16_t Count_OutsideThreshold = 0x0000; 		//       

static FlagStatus SA_TS_CALIBRATION_TRESHOLD = RESET;
static FlagStatus SA_TS_CHECK_TRESHOLD = RESET;
static FlagStatus SA_AVERAGE_LEVEL_IS_READY = RESET;
static uint16_t Count_TiltSensorCalibrationPeriod = 0;
static uint16_t Count_TsDelayCheckStop = 0; // ACCEL_TILT_DELAY_TO_STOP; 

/* Function prototypes -------------------------------------------------------*/
void TiltSensor_CutHighFrequencySurge(ACCEL_POINT_16BIT* _in_val);
void ts_CutAxis(uint16_t* _in_dot, uint16_t* _in_average);

void TiltSensor_SetTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta);
void TiltSensor_SetCalibrationTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta);
bool TiltSensor_CheckTheshold(ACCEL_POINT_16BIT* _in_val);
bool TiltSensor_CheckCalibrationTheshold(ACCEL_POINT_16BIT* _in_val);
void CalibrationTiltSensor_Count(void);
void TiltSensor_CalibrationLaunch(void);
void TiltSensor_ReLaunch(void);
void TiltSensor_Reset(void);

/* Functions -----------------------------------------------------------------*/
//- Init
void AccelTilt_Init(void)
{
  AccelTiltState=0;
  TiltSensor_Reset();
}



//-    , ,  
//         
//      "math_ma"     (),
//             , 
//      . 
//     ,       
//     "math_ma"    .
void TiltSensor_CutHighFrequencySurge(ACCEL_POINT_16BIT* _in_val)
{
  ts_CutAxis(&_in_val->X, &TS_ItemMa4.X);
  ts_CutAxis(&_in_val->Y, &TS_ItemMa4.Y);
  ts_CutAxis(&_in_val->Z, &TS_ItemMa4.Z);
}

//==============================================================================
void ts_CutAxis(uint16_t* _in_dot, uint16_t* _in_average)
{
  uint16_t TS_CHFS_Upper;
  uint16_t TS_CHFS_Lower;

  TS_CHFS_Upper = *_in_average + TS_CUT_HF_THRESHOLD;
  TS_CHFS_Lower = *_in_average - TS_CUT_HF_THRESHOLD;
  
  
  if(*_in_dot > TS_CHFS_Upper)
  {
    *_in_dot = TS_CHFS_Upper;
  }
  else if(*_in_dot < TS_CHFS_Lower)
  {
    *_in_dot = TS_CHFS_Lower;
  }
}


//-   
        //      
        //    .    
        //     ( ).  
        //           . 
        //    .       
        //   .    .
        //           , 
        //   (  ).   - 
        //    ,    
        //       
        // .   -      
        //     
void AccelTilt_Do(void)
{
 // Check to stop
  if(Count_TsDelayCheckStop>0)  {Count_TsDelayCheckStop--;}
  else AccelTiltState=0;

    Count_TsSamples++;
    if(Count_TsSamples>= TS_PERIOD_GET_SAMPLES)    // 15 ms
    {
      Count_TsSamples = 0x00;
      CmpSmpl = AcceluSmpl;
     //           ,   
     //  (  )
      if(SA_AVERAGE_LEVEL_IS_READY == SET) { TiltSensor_CutHighFrequencySurge(&CmpSmpl); }
     //   
      if(Accel_math_ma(CmpSmpl, &TS_ItemMa1, &TS_CtrlSmpl, TS_BufSmpl, TS_BUF_SIZE_SMPL) == true)
      {
        if(Accel_math_ma(TS_ItemMa1, &TS_ItemMa2, &TS_CtrlMa1, TS_BufMa1, TS_BUF_SIZE_MA1) == true)
        {
          if(Accel_math_ma(TS_ItemMa2, &TS_ItemMa3, &TS_CtrlMa2, TS_BufMa2, TS_BUF_SIZE_MA2) == true)
          {       
            if(Accel_math_ma(TS_ItemMa3, &TS_ItemMa4, &TS_CtrlMa3, TS_BufMa3, TS_BUF_SIZE_MA3) == true)
            { 
              SA_AVERAGE_LEVEL_IS_READY = SET;  //   ,       
             //     
              if(SA_TS_CHECK_TRESHOLD == SET)
              {        
                if(TiltSensor_CheckTheshold(&TS_ItemMa4) == true)
                {
                  Count_OutsideThreshold++;
                  if(Count_OutsideThreshold>TS_COUNT_OUTSIDE_THRESHOLD)
                  {
		    AccelTiltState=1; Count_TsDelayCheckStop = DEF_ACCEL_TILT_DELAY_TO_STOP; // MS_DELAY_TO_STOP;
		    Count_OutsideThreshold = 0x0000;  	//  !!!
                    TiltSensor_Reset();
                  }
                } 
                else { if(Count_OutsideThreshold != 0)
		          Count_OutsideThreshold = 0x0000; }  //   
              }
       
              //   
              if(SA_TS_CALIBRATION_TRESHOLD == SET)
              { 
                do
                { F_TS_FAST_NEXT_STATE = RESET;
                  switch(TiltSensorState)
                  {
                   // (2)    
                    case tss_CheckImmobility:
                     // AVERAGE   
                      if(TiltSensor_CheckCalibrationTheshold(&TS_ItemMa4) == false)
                      {
                        Count_TsSearchAverage++;
                        if(Count_TsSearchAverage >= TS_PERIOD_SEARCH_AVERAGE)    // 15 ms
                        {
                          Count_TsSearchAverage = 0x00;
                          if(Accel_math_ma(TS_ItemMa4, &TS_StartAverage, &TS_CtrlAvrg, TS_BufAvrg, TS_BUF_SIZE_AVERAGE) == true)
                          {
                            TiltSensor_SetTheshold(&TS_StartAverage, TiltThreshold[AccelTiltSenseLevel]); //   
                            SA_TS_CALIBRATION_TRESHOLD = RESET;    //  
                            SA_TS_CHECK_TRESHOLD = SET;            //    
                            TiltSensorState = tss_SetThreshold;    //     
                          } 
                        }
                      }
                      else //   
                      {
                        F_TS_FAST_NEXT_STATE = SET;              
                        TiltSensorState = tss_SetThreshold;    //    
                      }
                    break;
                    
                   // (1)     
                    case tss_SetThreshold:
                      TiltSensor_SetCalibrationTheshold(&TS_ItemMa4, TiltThreshold[7]); //    
                      Accel_math_reset(&TS_CtrlAvrg);  		//   
                      Count_TsSearchAverage = 0x00;
                      TiltSensorState = tss_CheckImmobility;
                      break;
                    
                    default: break;
                  }
                }while(F_TS_FAST_NEXT_STATE == SET);

              }    
         
            }
          }
        }
      }
      
    }   // Count_Samples
}


//-   
void CalibrationTiltSensor_Count(void)
{
  Count_TiltSensorCalibrationPeriod++;
  if(Count_TiltSensorCalibrationPeriod == TS_CALIBRATION_PERIOD)
  {
    Count_TiltSensorCalibrationPeriod = 0;
    TiltSensor_CalibrationLaunch();
  }   
}


//-   
void TiltSensor_SetTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta)
{
  TS_ThresholdLevelUpper.X = _in_avr->X+_in_delta;
  TS_ThresholdLevelLower.X = _in_avr->X-_in_delta;
  TS_ThresholdLevelUpper.Y = _in_avr->Y+_in_delta;
  TS_ThresholdLevelLower.Y = _in_avr->Y-_in_delta;
  TS_ThresholdLevelUpper.Z = _in_avr->Z+_in_delta;
  TS_ThresholdLevelLower.Z = _in_avr->Z-_in_delta;
}

//-   
void TiltSensor_SetCalibrationTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta)
{
  TS_CalibrationThresholdLevelUpper.X = _in_avr->X+_in_delta;
  TS_CalibrationThresholdLevelLower.X = _in_avr->X-_in_delta;
  TS_CalibrationThresholdLevelUpper.Y = _in_avr->Y+_in_delta;
  TS_CalibrationThresholdLevelLower.Y = _in_avr->Y-_in_delta;
  TS_CalibrationThresholdLevelUpper.Z = _in_avr->Z+_in_delta;
  TS_CalibrationThresholdLevelLower.Z = _in_avr->Z-_in_delta; 
}


//-   
bool TiltSensor_CheckTheshold(ACCEL_POINT_16BIT* _in_val)
{
  if((_in_val->X > TS_ThresholdLevelUpper.X) || (_in_val->X < TS_ThresholdLevelLower.X))
  {
    return(true);
  }
  if((_in_val->Y > TS_ThresholdLevelUpper.Y) || (_in_val->Y < TS_ThresholdLevelLower.Y))
  {
    return(true);
  }
  if((_in_val->Z > TS_ThresholdLevelUpper.Z) || (_in_val->Z < TS_ThresholdLevelLower.Z))
  {
    return(true);
  } 
  return(false);
}


//-   
bool TiltSensor_CheckCalibrationTheshold(ACCEL_POINT_16BIT* _in_val)
{
  if((_in_val->X > TS_CalibrationThresholdLevelUpper.X) || (_in_val->X < TS_CalibrationThresholdLevelLower.X))
  {
    return(true);
  }
  if((_in_val->Y > TS_CalibrationThresholdLevelUpper.Y) || (_in_val->Y < TS_CalibrationThresholdLevelLower.Y))
  {
    return(true);
  }
  if((_in_val->Z > TS_CalibrationThresholdLevelUpper.Z) || (_in_val->Z < TS_CalibrationThresholdLevelLower.Z))
  {
    return(true);
  } 
  return(false);
}


//   ,   
void TiltSensor_CalibrationLaunch(void)
{
  SA_TS_CALIBRATION_TRESHOLD = SET; 	//  ,      
  TiltSensorState = tss_SetThreshold;   //        
  Accel_math_reset(&TS_CtrlAvrg);
  Count_TsSearchAverage = 0;
}


//-   ,   
void TiltSensor_ReLaunch(void)
{
  SA_TS_CHECK_TRESHOLD = RESET; 	//   
  SA_TS_CALIBRATION_TRESHOLD = SET; 	//  ,      
  TiltSensorState = tss_SetThreshold;   //        
  Accel_math_reset(&TS_CtrlAvrg);
  Count_OutsideThreshold = 0x0000;
  Count_TsSearchAverage = 0;
  Count_TiltSensorCalibrationPeriod = 0; //    
}


//-    
void TiltSensor_Reset(void)
{
  Accel_math_reset(&TS_CtrlSmpl);
  Accel_math_reset(&TS_CtrlMa1);
  Accel_math_reset(&TS_CtrlMa2); 
  Accel_math_reset(&TS_CtrlMa3);  
  SA_AVERAGE_LEVEL_IS_READY = RESET;
  TiltSensor_ReLaunch();
}

//-    
uint8_t AccelTilt_GetState(void)
{
  return AccelTiltState;
}
