/**
  ******************************************************************************
  * File Name          : Accel.c
  * Description        : Accelerometer
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include <stdio.h>

//- Check LIS MEMS choice
#include "Accel_sensors_config.h"

#include "lis3dh_registers.h"
#include "lis2hh12_registers.h"

#include "Accel/Accel_sensors.h"
#include "Accel/Accel_sense/Accel_filtr.h"
#include "Accel/Accel_sense/Accel_move.h"
#include "Accel/Accel_sense/Accel_tilt.h"
#include "lis_spi_driver.h"
#include "nvic_exti_driver.h"

#if (LIS3DH_WHO_AM_I == LIS2HH12_WHO_AM_I)
#define LIS_WHO_AM_I LIS2HH12_WHO_AM_I
#else
#error LIS3DH_WHO_AM_I != LIS2HH12_WHO_AM_I
#endif //(LIS3DH_WHO_AM_I == LIS2HH12_WHO_AM_I)

/* Global variables ---------------------------------------------------------*/
AxesRaw_t AccelAxes_Curr;
ACCEL_POINT_16BIT AcceluSmpl; // входной uint16 семпл
AccelSensorState_t AccelSensor_MoveState, AccelSensor_TiltState;

AxesRaw_t AccelRaw={0};

/* Private variables ---------------------------------------------------------*/
static uint8_t accel_type = 0;

static __IO struct {
  uint8_t icnt;
  uint8_t ilst;
}DataReady = {0,0}; // flag to ready

static __IO struct {
  uint8_t is_sleep_mode;
  uint8_t wakeup_event;
}SleepReg = {0, 0};

/* Private function prototypes -----------------------------------------------*/
//- Low level for LIS MEMS
#define LIS_CMD_RWmsk 	0x80	// mask Read/Write bit in SPI command
#define LIS_CMD_MSmsk 	0x40	// mask Multiple/Single bit in SPI command

#if ACCEL_FRAME_RATE_kHZ == 400
#define LIS3DH_ODR_MASK    (LIS3DH_CTRL1_ODR_400_HZ)
#define LIS2HH12_ODR_MASK  (LIS2HH12_CTRL1_ODR_400_HZ)
#else
#error wrong ACCEL_FRAME_RATE_kHZ
#endif

//Для LIS2HH12 не использовать LIS_ReadRegs!!!
static void LIS_WriteReg(uint8_t reg, uint8_t val);
static void LIS_ReadReg(uint8_t reg, uint8_t* val);
static void LIS_ReadRegs(uint8_t reg, uint8_t* buf, uint8_t len);
static uint8_t LIS_WriteRegAndVerifyExt(uint8_t reg, uint8_t val, uint8_t timeout);
static uint8_t LIS_WriteRegAndVerify(uint8_t reg, uint8_t val);

//- Init
uint8_t AccelSensors_Init(uint8_t is_sleep_mode)
{
  LIS_SPI_Init();
  LIS_CS_Up();

  LIS_ReadReg(LIS_WHO_AM_I, &accel_type);

  if(accel_type != LIS3DH_WHO_AM_I_ID && accel_type != LIS2HH12_WHO_AM_I_ID)
  {
    LIS_SPI_DeInit();
    return 0;
  }

  if(accel_type == LIS3DH_WHO_AM_I_ID)
  {
    LIS_WriteReg(LIS3DH_CTRL_REG5, LIS3DH_BOOT);//soft reset
    LIS_Delay(8);
  }
  else //(accel_type == LIS2HH12_WHO_AM_I_ID)
  {
    LIS_WriteReg(LIS2HH12_CTRL5, LIS2HH12_CTRL5_SOFT_RESET); //soft reset
    vTaskDelay(1);
  }

  if(is_sleep_mode)
  {
    SleepReg.is_sleep_mode=1;
    SleepReg.wakeup_event=0;

    if(accel_type == LIS3DH_WHO_AM_I_ID)
    {
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG1, LIS3DH_XEN|LIS3DH_YEN|LIS3DH_ZEN|LIS3DH_ODR2))//ODR 50Hz
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG2, LIS3DH_HPM1|LIS3DH_HPM0|LIS3DH_HPIS1))//HP filter enabled for Interrupt Generator 1,  Autoreset on interrupt event
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG3, LIS3DH_I1_AOI1))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG4, LIS3DH_HR))//2g
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG5, 0))//Interrupt request not latched
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_INT1_THS, LIS3DH_INT_TH(52, 2000)))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_INT1_DURATION, LIS3DH_DURATION_TIME(200, 50)))//enable duration
        return 0;

      LIS_Delay(1);
      LIS_ReadRegs(LIS3DH_OUT_X_L, (uint8_t*)&AcceluSmpl, 6);// Dummy read to force immediately the HP filter output

      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG6, LIS3DH_H_LACTIVE))//Interrupt active low
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_INT1_CFG,  LIS3DH_ZHIE|LIS3DH_YHIE|LIS3DH_XHIE))
        return 0;

      LIS_Delay(1);
      LIS_ReadRegs(LIS3DH_OUT_X_L, (uint8_t*)&AcceluSmpl, 6);	// Dummy read
    }
    else //(accel_type == LIS2HH12_WHO_AM_I_ID)
    {
      //Фильтр высоких частот в рерывании включен
      //прерывание генерируется по порогу, заданному в ACCEL_mG_TH_DEFAULT при длительности воздействия DURATION_TIME_mS (накопление выключено)

      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL4, 0x42))//Anti-aliasing 200Hz, ±2g, spi 4wire mode
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL5, 0x02))//PP, Interrupt active low
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL1, (LIS2HH12_CTRL1_XEN|LIS2HH12_CTRL1_YEN|LIS2HH12_CTRL1_ZEN|LIS2HH12_CTRL1_BDU|LIS2HH12_CTRL1_ODR_10_HZ)))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL2, (LIS2HH12_CTRL2_HPIS2|LIS2HH12_CTRL2_HPIS1))) // HP filter enabled for Interrupt Generator XX
      //if(!LIS_WriteRegAndVerify(CTRL7, LIS2HH12_CTRL7_DCRM2|LIS2HH12_CTRL7_DCRM1))//duration counter is decremented enable
      //  return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_IG_DUR1, LIS2HH12_DURATION_REG_VAL(500, 10)))//enable duration
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL3, (LIS2HH12_CTRL3_INT1_IG1)))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_IG_THS_X1, LIS2HH12_ACCEL_TH_DEFAULT_VAL(200, 2000)))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_IG_THS_Y1, LIS2HH12_ACCEL_TH_DEFAULT_VAL(200, 2000)))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_IG_THS_Z1, LIS2HH12_ACCEL_TH_DEFAULT_VAL(200, 2000)))
        return 0;

      uint8_t dummy;
      LIS_ReadReg(LIS2HH12_XL_REFERENCE, &dummy);//Dummy read to force immediately the HP filter output
      LIS_ReadReg(LIS2HH12_YL_REFERENCE, &dummy);//Dummy read to force immediately the HP filter output
      LIS_ReadReg(LIS2HH12_ZL_REFERENCE, &dummy);//Dummy read to force immediately the HP filter output

      LIS_Delay(3);
      if(!LIS_WriteRegAndVerify(LIS2HH12_IG_CFG1,  (LIS2HH12_IG_CFG1_XHIE|LIS2HH12_IG_CFG1_XHIE|LIS2HH12_IG_CFG1_ZHIE)))
        return 0;
    }

    __LIS_IRQ_EXTI_ENABLE();
    // __LIS_IRQ_NVIC_ENABLE(); // Разрешение групповых прерываний уровнем выше!

    return 1;
  }
  else
  {
    SleepReg.is_sleep_mode=0;

    if(accel_type == LIS3DH_WHO_AM_I_ID)
    {
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG1, LIS3DH_XEN|LIS3DH_YEN|LIS3DH_ZEN|LIS3DH_ODR_MASK))//ODR 400Hz
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG2, 0))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG3, LIS3DH_I1_DRDY1))//On INT1: DRDY1
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG4, LIS3DH_BDU|LIS3DH_HR))//2g, BlockDataUpdate
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG5, LIS3DH_FIFO_EN))//Interrupt request not latched
        return 0;
      if(!LIS_WriteRegAndVerify(LIS3DH_CTRL_REG6, LIS3DH_H_LACTIVE))//Interrupt active low
        return 0;
    }
    else //(accel_type == LIS2HH12_WHO_AM_I_ID)
    {
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL4, 0x42))//Anti-aliasing 200Hz, ±2g, spi 4wire mode
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL1, (LIS2HH12_CTRL1_XEN|LIS2HH12_CTRL1_YEN|LIS2HH12_CTRL1_ZEN|LIS2HH12_CTRL1_BDU|LIS2HH12_ODR_MASK)))//ODR 400Hz, BDU
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL2, 0))
        return 0;
      if(!LIS_WriteRegAndVerify(LIS2HH12_CTRL5, 0x02))//PP, Interrupt active low
        return 0;
      if(!LIS_WriteRegAndVerifyExt(LIS2HH12_CTRL3, 0x01, 4))//On INT1: DRDY1
        return 0;
    }

    AccelMove_Init();
    AccelTilt_Init();

    //LIS_Delay(1);
    //LIS_ReadRegs(LIS3DH_OUT_X_L, (uint8_t*)&AcceluSmpl, 6);	// Dummy read

    __LIS_IRQ_EXTI_ENABLE();
    // __LIS_IRQ_NVIC_ENABLE(); // Разрешение групповых прерываний уровнем выше!

    //генерируем прерывание
    EXTI_GenerateSWInterrupt(LIS_IRQ_EXTI_LINEx);

    return 1;
  }
}

void AccelSensors_DeInit(void)
{
  LIS_SPI_DeInit();
}

// - Check
uint8_t AccelSensors_Check(void)
{
  if(SleepReg.is_sleep_mode)
  {
    return SleepReg.wakeup_event;
  }
  else
  {
    // Check DataReady
    uint8_t icnt=DataReady.icnt;
    if(icnt==DataReady.ilst) return 0;

    if(MovingSensor_GetState()) {AccelSensor_MoveState=SET;}  else {AccelSensor_MoveState=RESET;}
    if(AccelTilt_GetState())    {AccelSensor_TiltState=SET;}  else {AccelSensor_TiltState=RESET;}

    DataReady.ilst=DataReady.icnt;
    return 1;
  }
}

/*--- Low level for LIS MEMS -----------------*/
static void LIS_WriteReg(uint8_t reg, uint8_t val)
{
  LIS_CS_Dwn();
  SPI_TxRxPoll( LIS_SPI, &reg, 1, NULL, 0 );
  SPI_TxRxPoll( LIS_SPI, &val, 1, NULL, 0 );
  LIS_CS_Up();
}

static uint8_t LIS_WriteRegAndVerifyExt(uint8_t reg, uint8_t val, uint8_t timeout)
{
  uint8_t rval;
  LIS_Delay(1);
  LIS_WriteReg(reg, val);
  LIS_Delay(timeout);
  LIS_ReadReg(reg, &rval);
  if(rval!=val) {return 0;}
  else          {return 1;}
}

#pragma inline=forced_no_body
static uint8_t LIS_WriteRegAndVerify(uint8_t reg, uint8_t val)
{
  return LIS_WriteRegAndVerifyExt(reg, val, 1);
}

static void LIS_ReadReg(uint8_t reg, uint8_t* val)
{
  reg|=LIS_CMD_RWmsk;
  LIS_CS_Dwn();
  SPI_TxRxPoll( LIS_SPI, &reg, 1, val, 1 );
  LIS_CS_Up();
}

static void LIS_ReadRegs(uint8_t reg, uint8_t* buf, uint8_t len)
{
  reg|=LIS_CMD_RWmsk + LIS_CMD_MSmsk;
  LIS_CS_Dwn();
  SPI_TxRx(LIS_SPI, &reg, 1, buf, len );
  LIS_CS_Up();
}

uint8_t LIS_GetAccelType(void)
{
  return accel_type;
}

//- Interrupt hendler
void LIS_IRQ_Handler(void)
{
  if(SleepReg.is_sleep_mode)
  {
    SleepReg.wakeup_event=1;
  }
  else
  {
    if(accel_type == LIS3DH_WHO_AM_I_ID)
    {
      LIS_ReadRegs(LIS3DH_OUT_X_L, (uint8_t*)&AcceluSmpl, 6); // Read for clear LIS_IRQ on pin
    }
    else //(accel_type == LIS2HH12_WHO_AM_I_ID)
    {
      LIS_ReadReg(LIS2HH12_OUT_X_L, (uint8_t*)&AcceluSmpl.X);
      LIS_ReadReg(LIS2HH12_OUT_X_H, (uint8_t*)&AcceluSmpl.X+1);
      LIS_ReadReg(LIS2HH12_OUT_Y_L, (uint8_t*)&AcceluSmpl.Y);
      LIS_ReadReg(LIS2HH12_OUT_Y_H, (uint8_t*)&AcceluSmpl.Y+1);
      LIS_ReadReg(LIS2HH12_OUT_Z_L, (uint8_t*)&AcceluSmpl.Z);
      LIS_ReadReg(LIS2HH12_OUT_Z_H, (uint8_t*)&AcceluSmpl.Z+1);
    }

    AccelRaw.AXIS_X=AcceluSmpl.X;
    AccelRaw.AXIS_Y=AcceluSmpl.Y;
    AccelRaw.AXIS_Z=AcceluSmpl.Z;

    uint8_t icnt=DataReady.icnt;
    if(icnt==DataReady.ilst)
    {
      AccelAxes_Curr.AXIS_X=AcceluSmpl.X;
      AccelAxes_Curr.AXIS_Y=AcceluSmpl.Y;
      AccelAxes_Curr.AXIS_Z=AcceluSmpl.Z;
      DataReady.icnt++;
    }
    Accel_ConvertTwoComplement(&AcceluSmpl); // в прямой код и >>4

#if !defined(DISABLE_ACCEL_MOVE)
    AccelMove_Do();
#endif //!defined(DISABLE_ACCEL_MOVE)

#if !defined(DISABLE_ACCEL_TILT)
    AccelTilt_Do();
#endif //!defined(DISABLE_ACCEL_TILT)
  }
}
