/**
  ******************************************************************************
  * File Name          : Black_box.cpp
  * Description        :   ( , FIFO,  )
  *                      
  *                      
  ******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "Black_box/Black_box.h"
#include <string.h>
#include "crc_lib/crc_lib.h"


#if defined(STM32L1XX_MD) || defined(STM32L1XX_HD) || defined(STM32L1XX_MD_PLUS)
//todo: check_voltage
#include "stm32l1xx.h"
#endif

#if defined(STM32F10X_MD) || defined(STM32F10X_HD) || defined(STM32F10X_XL)
#include "stm32f10x.h"
#endif

#if defined(STM32F2XX)
#include "stm32f2xx.h"
#endif

#if defined(STM32F413_423xx)
#include "stm32f4xx.h"
#endif

#if defined(STM32L433xx)
//todo: check_voltage
#include "stm32l433xx.h"
#endif

//todo:   mem_read  mem_write    

/* C++ code-------------------------------------------------------------------*/
CBlack_box::CBlack_box(mem_driver_init_t init, mem_driver_deinit_t deinit,\
             mem_driver_format_t mem_format,\
             mem_driver_write_t write, mem_driver_read_t read,\
             mem_driver_direct_write_t dir_write, mem_driver_direct_read_t dir_read,\
             mem_driver_direct_erase_sector_t dir_erase, uint16_t dir_sector_count,\
             uint16_t dir_sector_size, uint32_t weared_memory_size)
{
  this->mem_init = init;
  this->mem_deinit = deinit;
  this->mem_format = mem_format;
  this->mem_write = write;
  this->mem_read = read;
  this->mem_dir_read = dir_read;
  this->mem_dir_write = dir_write;
  this->mem_dir_erase_sector = dir_erase;
  this->dir_sector_count = dir_sector_count;
  this->dir_sector_size = dir_sector_size;
  this->weared_memory_size = weared_memory_size;
  #ifdef USE_FREERTOS
  //        
  bb_busy = xSemaphoreCreateMutex();
  #ifdef DEBUG
  vQueueAddToRegistry(this->bb_busy, "black_box");
  #endif // DEBUG  
  #endif
  
  //       
  priority_message_1.sector = 0xFFFF;
  priority_message_1.byte = 0xFFFF;
  priority_message_2.sector = 0xFFFF;
  priority_message_2.byte = 0xFFFF;
  priority_message_3.sector = 0xFFFF;
  priority_message_3.byte = 0xFFFF;
  priority_message_4.sector = 0xFFFF;
  priority_message_4.byte = 0xFFFF;
  priority_message_1_readed = 0;
  priority_message_2_readed = 0;
  priority_message_3_readed = 0;
  priority_message_4_readed = 0;
  
  mem_read_err_counter = 0;
  mem_write_err_counter = 0;
}

// 
BBresult CBlack_box::Init(void)
{
  //    
  use_priority_message_setting = true;
    
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  if(mem_init())
  {
    //      
    this->Load_base_table();
    //      
    this->Register_in_base("file_table", sizeof(this->File_table), 100, (uint8_t*)this->File_table);
    
    for(uint8_t i=0; i<BB_MAX_FILES; i++)
    {
      File_table[i].opened=false;
    }
    
    //      
    #ifndef BOOTLOADER
    //   
    if(this->find_head()) 
    {
      this->find_tails();
    }
    //   ,   
    else 
    {
      this->Clean();
    }
    #endif    
    return BB_OPERATION_OK;
  }
  else return BB_OPERATION_FAILED;
}    
// 
void CBlack_box::Deinit(void)
{
  mem_deinit();
}

BBresult CBlack_box::Register_in_base(const char* base_name, uint16_t base_size, uint16_t reserved_place_size, void* base_ptr)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  //         (      )
  for(uint8_t index = 0; index < BASE_COUNT; index++)
  {
    //      
    if((strncmp(Base_table[index].base_name, base_name, strlen(Base_table[index].base_name)) == 0) &&
       (Base_table[index].base_name[0] != 0))
    {
      //!     ,      !      !
      if(Base_table[index].full_base_size < base_size)
      {
        mem_format();
        NVIC_SystemReset();
      }
      //   
      if(Base_table[index].base_size != base_size) Base_table[index].base_size = base_size;
      //      ,         
      Base_table[index].base_ptr = base_ptr;
      Load_base(base_name);
      return BB_OPERATION_OK;
    }
  }
  //     
  for(uint8_t index=0; index<BASE_COUNT; index++)
  {
    if(Base_table[index].base_name[0] == 0) 
    {
      //    
      if(strlen((const char*)base_name) > BASE_NAME_SIZE) return BB_OPERATION_FAILED;
      //     
      memcpy(&Base_table[index].base_name[0], base_name, strlen((const char*)base_name));
      //    !!!
      if(Base_table[index].base_name[0] == 0) while(1);
      //       
      Base_table[index].base_size = base_size;
      Base_table[index].full_base_size = base_size + reserved_place_size;
      Base_table[index].base_ptr = base_ptr;
      if(index == 0) 
      {
        //              
        Base_table[index].base_start_address = BASE_START_ADDRESS + BASE_TABLE_SIZE;
      }
      else
      {
        //      ,      
        Base_table[index].base_start_address = Base_table[index-1].base_start_address + Base_table[index-1].full_base_size;
        //        
        if(Base_table[index].base_start_address + Base_table[index].full_base_size > BASE_END_ADDRESS)
        {
          mem_format();
          NVIC_SystemReset();
        }
      }
      Load_base(base_name);
      return BB_OPERATION_OK;
    }
  }
  //     
  return BB_OPERATION_FAILED;
}

//    
BBresult CBlack_box::Load_base(const char* base_name)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  for(uint8_t x = 0; x < BASE_COUNT; x++)
  {
    //      
    if((strncmp(Base_table[x].base_name, base_name, strlen(Base_table[x].base_name)) == 0) &&
       (Base_table[x].base_ptr != 0))
    {
      //     
      mem_read(Base_table[x].base_start_address, (uint8_t*)Base_table[x].base_ptr, Base_table[x].base_size);
      return BB_OPERATION_OK;
    }
  }
  return BB_OPERATION_FAILED;
}

//     data c  address
bool CBlack_box::Is_data_same(uint32_t address, const uint8_t* data, uint16_t size)
{
  //return false;
  
  for(uint16_t i = 0; i < size;)
  {
    uint16_t rlen;
    if(sizeof(LOG_BUF) + i > size) {rlen = size - i;}
    else                           {rlen = sizeof(LOG_BUF);}
    
    mem_read(address + i, LOG_BUF, rlen);
    
    if(memcmp(data + i, LOG_BUF, rlen) != 0)
    {
      return false;
    }
    
    i += rlen;
  }
  
  return true;
}

//    
BBresult CBlack_box::Save_base(const char* base_name)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  
  //         
  if(!Is_data_same(BASE_START_ADDRESS, (uint8_t*)&Base_table, BASE_TABLE_SIZE))
  {
    mem_write(BASE_START_ADDRESS, (uint8_t*)&Base_table, BASE_TABLE_SIZE);
  }
  
  if(strncmp(base_name, "all", strlen("all")) == 0)
  {
    //         
    for(uint8_t x = 0; x < BASE_COUNT; x++)
    {
      if(Base_table[x].base_ptr != 0)
      {
        //     
        if(!Is_data_same(Base_table[x].base_start_address, (uint8_t*)Base_table[x].base_ptr, Base_table[x].base_size))
        {
          mem_write(Base_table[x].base_start_address, (uint8_t*)Base_table[x].base_ptr, Base_table[x].base_size);
        }
      }
    }  
    return BB_OPERATION_OK;
  }
  else
  {
    //         
    for(uint8_t x = 0; x < BASE_COUNT; x++)
    {
      //      
      if((strncmp(Base_table[x].base_name, base_name, strlen(Base_table[x].base_name)) == 0) &&
         (Base_table[x].base_ptr != 0))
      {
        //     
        if(!Is_data_same(Base_table[x].base_start_address, (uint8_t*)Base_table[x].base_ptr, Base_table[x].base_size))
        {
          mem_write(Base_table[x].base_start_address, (uint8_t*)Base_table[x].base_ptr, Base_table[x].base_size);
        }
        return BB_OPERATION_OK;
      }
    }
  }
  return BB_OPERATION_FAILED;
}

//   (filetype -  ,   )
BBresult CBlack_box::fcreate(const char* filename, uint32_t size_in_bytes, BB_filetype filetype)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  //      
  uint8_t index = 0;
  for(index = 0; index < BB_MAX_FILES; index++)
  {
    if(File_table[index].file_total_size == 0xFFFFFFFF) break;
  }
  if(strlen(filename) > BB_FILENAME_SIZE) return BB_OPERATION_FAILED;
  memcpy(File_table[index].filename, filename, strlen(filename));
  //           
  if(index == 0) File_table[index].start_address = BASE_END_ADDRESS + 1;
  //      
  else File_table[index].start_address = File_table[index-1].start_address + File_table[index - 1].file_total_size;
  File_table[index].end_address = File_table[index].start_address + size_in_bytes - 1;
  //      
  if(File_table[index].end_address > weared_memory_size) 
  {
    while(1);
  }
  File_table[index].filetype = filetype;
  File_table[index].file_total_size = size_in_bytes;
  File_table[index].opened = false;
  File_table[index].read_ptr = File_table[index].start_address;
  File_table[index].write_ptr = File_table[index].start_address;
  File_table[index].updated = false;
  //   
  Save_base("file_table");
  return BB_OPERATION_OK;
}

//   
BBresult CBlack_box::fopen(const char* filename)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  File_table[index].opened = true;
  return BB_OPERATION_OK;
}

//  
BBresult CBlack_box::fclose(const char* filename)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  File_table[index].opened = false;
  return BB_OPERATION_OK;
}

// ,   
BBresult CBlack_box::fopened(const char* filename, uint8_t* opened)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  *opened = File_table[index].opened;
  return BB_OPERATION_OK;
}

//    (      ,       )
BBresult CBlack_box::fread(const char* filename, uint32_t addr_in_file, uint8_t* buff, uint16_t size)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;  

  uint32_t filesize = 0;
  fsize(filename, &filesize);
  //       
  if((addr_in_file + size) > filesize) return BB_OPERATION_FAILED;  

  if(File_table[index].filetype == SIMPLE_FILE)
  {
    uint32_t memaddr = File_table[index].start_address + addr_in_file;
    mem_read(memaddr, buff, size);
    return BB_OPERATION_OK;
  }  
  else if(File_table[index].filetype == FIFO_FILE)
  {
    //    
    uint32_t memaddr = File_table[index].read_ptr + addr_in_file;
    //      
    if(memaddr > File_table[index].end_address) 
    {  
      memaddr -= File_table[index].file_total_size;
    }
    //        
    if( (memaddr + size) < File_table[index].end_address )
    {
      mem_read(memaddr, buff, size);
      return BB_OPERATION_OK;
    }
    //   
    else
    {
      uint16_t read1_size = File_table[index].end_address - memaddr;
      uint16_t read2_size = size - read1_size;
      mem_read(memaddr, buff, read1_size);
      mem_read(File_table[index].start_address, buff + read1_size, read2_size);
      return BB_OPERATION_OK;
    }
  }
  return BB_OPERATION_FAILED;
}

//  ,     LAT
BBresult CBlack_box::file_self_test(const char* filename)
{
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  
  uint32_t filesize = 0;
  fsize(filename, &filesize);
  //filesize = get_file_total_size(filename);
  
  if(File_table[index].filetype == SIMPLE_FILE)
  {
    for(uint32_t offset = 0; offset < filesize;)
    {
      uint16_t rsize = 1024-1;
      
      if(rsize > (filesize-offset)) rsize=filesize-offset;
      
      if(!mem_read(File_table[index].start_address+offset, NULL, rsize))
      {
        return BB_OPERATION_FAILED;
      }
      
      offset+=rsize;
    }
    
    return BB_OPERATION_OK;
  }
  else if(File_table[index].filetype == FIFO_FILE)
  {
    for(uint32_t offset = 0; offset < filesize;)
    {
      uint16_t rsize = 1024-1;
      
      if(rsize > (filesize-offset)) rsize=filesize-offset;
      
      //    
      uint32_t memaddr = File_table[index].read_ptr + rsize;
      //      
      if(memaddr > File_table[index].end_address) 
      {  
        memaddr -= File_table[index].file_total_size;
      }
      
      //        
      if( (memaddr + rsize) < File_table[index].end_address )
      {
        if(!mem_read(memaddr, NULL, rsize))
        {
          return BB_OPERATION_FAILED;
        }
      }
      //   
      else
      {
        uint16_t read1_size = File_table[index].end_address - memaddr;
        
        if(mem_read(memaddr, NULL, read1_size))
        {
          return BB_OPERATION_FAILED;
        }
        
        uint16_t read2_size = rsize - read1_size;
        
        if(mem_read(File_table[index].start_address, NULL, read2_size))
        {
          return BB_OPERATION_FAILED;
        }
        
      }
      
      offset+=rsize;
    }
    
    return BB_OPERATION_OK;
  }
  
  return BB_OPERATION_FAILED;
}

//    (      ,     )
BBresult CBlack_box::fwrite(const char* filename, uint8_t* buff, uint16_t size)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  //       !
  if(size > 2048) return BB_OPERATION_FAILED;
  
  if(File_table[index].filetype == SIMPLE_FILE)
  {
    //        
    if((File_table[index].write_ptr + size) > File_table[index].end_address) return BB_OPERATION_FAILED;  
    mem_write(File_table[index].write_ptr, buff, size);
    File_table[index].write_ptr += size;
    return BB_OPERATION_OK;
  }
  else if(File_table[index].filetype == FIFO_FILE)
  {
    //    
    if((File_table[index].write_ptr + size) < File_table[index].end_address)
    {
      mem_write(File_table[index].write_ptr, buff, size);
    }
    //    
    if((File_table[index].write_ptr + size) > File_table[index].end_address)
    {
      uint16_t write1_size  = File_table[index].end_address - File_table[index].write_ptr;
      uint16_t write2_size = size - write1_size;
      mem_write(File_table[index].write_ptr, buff, write1_size);
      mem_write(File_table[index].start_address, buff + write1_size, write2_size);
    }
    //      
    File_table[index].write_ptr += size;
    if(File_table[index].write_ptr > File_table[index].end_address)
    {
      //__PRINTF("%s(1): write_ptr: %u, end_address: %u, file_total_size: %u\n", __func__, File_table[index].write_ptr, File_table[index].end_address, File_table[index].file_total_size);
      File_table[index].write_ptr -= File_table[index].file_total_size;
      //__PRINTF("%s(1): write_ptr: %u\n", __func__, File_table[index].write_ptr);
    }
    uint32_t ptr_distance = 0;
    if(File_table[index].write_ptr > File_table[index].read_ptr) 
    {
      ptr_distance = File_table[index].file_total_size - (File_table[index].write_ptr - File_table[index].read_ptr);
    }
    if(File_table[index].write_ptr < File_table[index].read_ptr)
    {
      ptr_distance = File_table[index].read_ptr - File_table[index].write_ptr;
    }
    if(ptr_distance < 3000)
    {
      File_table[index].read_ptr += size;
      if(File_table[index].read_ptr > File_table[index].end_address) File_table[index].read_ptr -= File_table[index].file_total_size;
    }
    return BB_OPERATION_OK;
  }
  return BB_OPERATION_FAILED;
}

//      (     )
BBresult CBlack_box::fsize(const char* filename, uint32_t* size)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  if(File_table[index].filetype == SIMPLE_FILE) 
  {  
    *size = (File_table[index].write_ptr - File_table[index].start_address);
    return BB_OPERATION_OK;
  }
  else if(File_table[index].filetype == FIFO_FILE)
  {
    if(File_table[index].read_ptr < File_table[index].write_ptr)
    {
      *size = File_table[index].write_ptr - File_table[index].read_ptr;
      return BB_OPERATION_OK;
    }
    if(File_table[index].read_ptr > File_table[index].write_ptr)
    {
      *size = File_table[index].file_total_size - (File_table[index].read_ptr - File_table[index].write_ptr);
      return BB_OPERATION_OK;
    }
  }
   return BB_OPERATION_FAILED;
}

//  
BBresult CBlack_box::fflush(const char* filename)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;
  File_table[index].read_ptr = File_table[index].start_address;
  File_table[index].write_ptr = File_table[index].start_address;
  return BB_OPERATION_OK;
}

//    
BBresult CBlack_box::fsave(void)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  Save_base("file_table");
  return BB_OPERATION_OK;
}

//      
BB_mem_ptr CBlack_box::sector_forward(BB_mem_ptr ptr)
{
  ptr.sector++;
  if(ptr.sector >= this->dir_sector_count) ptr.sector = 0;
  ptr.byte = 0;
  return ptr;
}

//      
BB_mem_ptr CBlack_box::sector_back(BB_mem_ptr ptr)
{
  ptr.sector--;
  if(ptr.sector == 0xFFFF) ptr.sector = (this->dir_sector_count - 1);
  ptr.byte = 0;
  return ptr;
}

//    -  
BB_mem_ptr CBlack_box::find_free_sector(void)
{
  BB_message message;
  BB_mem_ptr retval;
  retval.sector = 0xFFFF;
  retval.byte = 0xFFFF;
  for(uint16_t i = 0; i < this->dir_sector_count; i++)
  {
    //      (       )
    retval.sector = i;
    retval = this->sector_forward(retval);
    //    
    if(!this->mem_dir_read(retval.sector, retval.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      mem_read_err_counter++; 
    }        
    if(message.len == 0xFFFF && message.receiver_1 == 0xFF && message.receiver_2 == 0xFF && message.receiver_3 == 0xFF && message.receiver_4 == 0xFF)
    {
      break;
    }
  }
  return retval;
}

//    -  
bool CBlack_box::find_head(void)
{
  //    (  F103)
  if(!check_voltage()) return false;
  BB_message message;
  BB_mem_ptr mem_ptr;
  //   
  mem_ptr = this->find_free_sector();
  //     ,     
  mem_ptr = sector_back(mem_ptr);
  
  //  
  while(1)
  {
    //    
    if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      mem_read_err_counter++; 
      return BB_OPERATION_FAILED;
    }        
    //    
    if(message.len == 0xFFFF) 
    {  
      FIFO.write_ptr = mem_ptr;
      break;
    }
    //    
    mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
    //   ,    ,  
    if(mem_ptr.byte >= this->dir_sector_size || message.len == 0) 
    {
      return false;
    } 
  }
  return true;
}

//  ,   ,   
void CBlack_box::find_tails(void)
{
  //    (  F103)
  if(!check_voltage()) return;
  BB_message message;
  BB_mem_ptr mem_ptr = this->sector_forward(FIFO.write_ptr);
  //   
  uint8_t tail_1_finded = 0, tail_2_finded = 0, tail_3_finded = 0, tail_4_finded = 0;
  //       
  mem_ptr = this->sector_forward(mem_ptr);
  uint8_t we_in_head_sector = 0;
  //  
  while(1)
  {
    //    
    if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      mem_read_err_counter++; 
    }        
    // ,        
    if(message.len && message.len != 0xFFFF && message.receiver_1)
    {
      if(!tail_1_finded) { FIFO.read_ptr_1 = mem_ptr; tail_1_finded = 1; }
      FIFO.message_count_1++;
    }
    if(message.len && message.len != 0xFFFF && message.receiver_2)
    {
      if(!tail_2_finded) { FIFO.read_ptr_2 = mem_ptr; tail_2_finded = 1; }
      FIFO.message_count_2++;
    }
    if(message.len && message.len != 0xFFFF && message.receiver_3)
    {
      if(!tail_3_finded) { FIFO.read_ptr_3 = mem_ptr; tail_3_finded = 1; }
      FIFO.message_count_3++;
    }
    if(message.len && message.len != 0xFFFF && message.receiver_4)
    {
      if(!tail_4_finded) { FIFO.read_ptr_4 = mem_ptr; tail_4_finded = 1; }
      FIFO.message_count_4++;
    }
    //    
    mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
    //      ,    
    if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || message.len == 0 || (mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size) 
    {
      mem_ptr = this->sector_forward(mem_ptr);
      if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
    } 
    //       ,  
    if(we_in_head_sector && (mem_ptr.sector > FIFO.write_ptr.sector || mem_ptr.byte >= FIFO.write_ptr.byte))
    {
      //       ,    
      if(!tail_1_finded) {FIFO.read_ptr_1 = FIFO.write_ptr; FIFO.message_count_1 = 0;}
      if(!tail_2_finded) {FIFO.read_ptr_2 = FIFO.write_ptr; FIFO.message_count_2 = 0;}
      if(!tail_3_finded) {FIFO.read_ptr_3 = FIFO.write_ptr; FIFO.message_count_3 = 0;}
      if(!tail_4_finded) {FIFO.read_ptr_4 = FIFO.write_ptr; FIFO.message_count_4 = 0;}
      break;
    }
  }
}

//    
BBresult CBlack_box::add_message(BB_message* message)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  if(message->len > BB_MESSAGE_MAXSIZE || message->len == 0) return BB_OPERATION_FAILED;
  //      ,     
  if((FIFO.write_ptr.byte + message->len + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)
  {
    //      ( )
    FIFO.write_ptr = this->sector_forward(FIFO.write_ptr);
    //        ,  ,       
    if(FIFO.message_count_1 == 0) FIFO.read_ptr_1 = FIFO.write_ptr;
    if(FIFO.message_count_2 == 0) FIFO.read_ptr_2 = FIFO.write_ptr;
    if(FIFO.message_count_3 == 0) FIFO.read_ptr_3 = FIFO.write_ptr;
    if(FIFO.message_count_4 == 0) FIFO.read_ptr_4 = FIFO.write_ptr;
    //    
    BB_mem_ptr free = this->sector_forward(FIFO.write_ptr);
    // ,          
    uint8_t bkp1 = priority_message_1_readed; priority_message_1_readed = 0;
    uint8_t bkp2 = priority_message_2_readed; priority_message_2_readed = 0;
    uint8_t bkp3 = priority_message_3_readed; priority_message_3_readed = 0;
    uint8_t bkp4 = priority_message_4_readed; priority_message_4_readed = 0;
    if(free.sector == FIFO.read_ptr_1.sector) while(free.sector == FIFO.read_ptr_1.sector && FIFO.message_count_1) {delete_messages(1, 1);}
    if(free.sector == FIFO.read_ptr_2.sector) while(free.sector == FIFO.read_ptr_2.sector && FIFO.message_count_2) {delete_messages(2, 1);}
    if(free.sector == FIFO.read_ptr_3.sector) while(free.sector == FIFO.read_ptr_3.sector && FIFO.message_count_3) {delete_messages(3, 1);}
    if(free.sector == FIFO.read_ptr_4.sector) while(free.sector == FIFO.read_ptr_4.sector && FIFO.message_count_4) {delete_messages(4, 1);}
    priority_message_1_readed = bkp1;
    priority_message_2_readed = bkp2;
    priority_message_3_readed = bkp3;
    priority_message_4_readed = bkp4;
    
    //   
    this->mem_dir_erase_sector(free.sector);
  }
  if(message->receiver_1) message->receiver_1 = 0xFF;
  if(message->receiver_2) message->receiver_2 = 0xFF;
  if(message->receiver_3) message->receiver_3 = 0xFF;  
  if(message->receiver_4) message->receiver_4 = 0xFF;
  uint16_t tmp_sector = FIFO.write_ptr.sector;
  uint16_t tmp_byte = FIFO.write_ptr.byte;
  //         
  FIFO.write_ptr.byte += (BB_MESSAGE_HEADER_SIZE + message->len);
  if(message->receiver_1) FIFO.message_count_1 ++;
  if(message->receiver_2) FIFO.message_count_2 ++;
  if(message->receiver_3) FIFO.message_count_3 ++;  
  if(message->receiver_4) FIFO.message_count_4 ++;
  
  if(use_priority_message_setting)
  {
    //     ,    
    if(message->receiver_1 && priority_message_1.sector == 0xFFFF) {priority_message_1.sector = tmp_sector; priority_message_1.byte = tmp_byte; priority_message_1_readed = 0;}
    if(message->receiver_2 && priority_message_2.sector == 0xFFFF) {priority_message_2.sector = tmp_sector; priority_message_2.byte = tmp_byte; priority_message_2_readed = 0;}
    if(message->receiver_3 && priority_message_3.sector == 0xFFFF) {priority_message_3.sector = tmp_sector; priority_message_3.byte = tmp_byte; priority_message_3_readed = 0;}
    if(message->receiver_4 && priority_message_4.sector == 0xFFFF) {priority_message_4.sector = tmp_sector; priority_message_4.byte = tmp_byte; priority_message_4_readed = 0;}  
  }
  
  //     
  if(!this->mem_dir_write(tmp_sector, tmp_byte, (uint8_t*)message, BB_MESSAGE_HEADER_SIZE)) 
  {
    LOG("Black_box: write messahe header error to sector %d, byte %d, len %d\n", tmp_sector, tmp_byte, BB_MESSAGE_HEADER_SIZE);
    mem_write_err_counter++; 
    return BB_OPERATION_FAILED;
  }
  tmp_byte += BB_MESSAGE_HEADER_SIZE;
  //    
  if(!this->mem_dir_write(tmp_sector, tmp_byte, message->payload, message->len))
  {
    LOG("Black_box: write message error to sector %d, byte %d, len %d\n", tmp_sector, tmp_byte, message->len);
    mem_write_err_counter++; 
    return BB_OPERATION_FAILED;
  }
  return BB_OPERATION_OK;
}

//     (receiver -    , message_address -  ,       )
BBresult CBlack_box::read_message(uint8_t* buff, uint8_t receiver, uint16_t message_address)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  BB_message message;
  BB_mem_ptr mem_ptr;
  //             
  switch(receiver)
  {
    case 1: 
    {
      if(message_address >= FIFO.message_count_1) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_1.sector; mem_ptr.byte = FIFO.read_ptr_1.byte;
      //            ,    
      if(priority_message_1.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_1.sector, priority_message_1.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_1.sector, priority_message_1.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        //    
        if(!this->mem_dir_read(priority_message_1.sector, priority_message_1.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_1.sector, (priority_message_1.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_1_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 2: 
    {
      if(message_address >= FIFO.message_count_2) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_2.sector; mem_ptr.byte = FIFO.read_ptr_2.byte;
      //            ,    
      if(priority_message_2.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_2.sector, priority_message_2.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_2.sector, priority_message_2.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_2.sector, priority_message_2.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_2.sector, (priority_message_2.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_2_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 3: 
    {
      if(message_address >= FIFO.message_count_3) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_3.sector; mem_ptr.byte = FIFO.read_ptr_3.byte;
      //            ,    
      if(priority_message_3.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_3.sector, priority_message_3.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_3.sector, priority_message_3.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_3.sector, priority_message_3.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_3.sector, (priority_message_3.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_3_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 4: 
    {
      if(message_address >= FIFO.message_count_4) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_4.sector; mem_ptr.byte = FIFO.read_ptr_4.byte;
      //            ,    
      if(priority_message_4.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_4.sector, priority_message_4.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_4.sector, priority_message_4.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_4.sector, priority_message_4.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_4.sector, (priority_message_4.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_4_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    default: return BB_OPERATION_FAILED;
  }
  //            ,     
  if(priority_message_1.sector != 0xFFFF && receiver == 1 && priority_message_1_readed && message_address > 0) message_address--;
  if(priority_message_2.sector != 0xFFFF && receiver == 2 && priority_message_2_readed && message_address > 0) message_address--;
  if(priority_message_3.sector != 0xFFFF && receiver == 3 && priority_message_3_readed && message_address > 0) message_address--;
  if(priority_message_4.sector != 0xFFFF && receiver == 4 && priority_message_4_readed && message_address > 0) message_address--;
  uint8_t we_in_head_sector = 0;
  if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
  uint16_t message_num = 0;
  while(1)
  {
    //    
    if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      LOG("Black_box: read message header error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte, BB_MESSAGE_HEADER_SIZE);
      mem_read_err_counter++; 
      return BB_OPERATION_FAILED;
    }      
    //   
    if(message.len && message.len != 0xFFFF && message.receiver_1 && receiver == 1) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_2 && receiver == 2) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_3 && receiver == 3) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_4 && receiver == 4) message_num++;
    //     ,   (  ,   )
    if(message.len && message.len != 0xFFFF && message.receiver_1 && receiver == 1 && mem_ptr.sector == priority_message_1.sector && mem_ptr.byte == priority_message_1.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_2 && receiver == 2 && mem_ptr.sector == priority_message_2.sector && mem_ptr.byte == priority_message_2.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_3 && receiver == 3 && mem_ptr.sector == priority_message_3.sector && mem_ptr.byte == priority_message_3.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_4 && receiver == 4 && mem_ptr.sector == priority_message_4.sector && mem_ptr.byte == priority_message_4.byte) message_num--;
    
    //  , 
    if(message_num == message_address+1)
    {
      if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
      {
        LOG("Black_box: read message error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_HEADER_SIZE, message.len);
        mem_read_err_counter++; 
        return BB_OPERATION_FAILED;
      }        
      break;
    }
    //    
    mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
    //   ,    
    if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
    {
      mem_ptr = this->sector_forward(mem_ptr);
      if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
    } 
    //       ,     ,  
    if( we_in_head_sector && (mem_ptr.sector > FIFO.write_ptr.sector || mem_ptr.byte >= FIFO.write_ptr.byte))
    {
      if(receiver == 1) { FIFO.read_ptr_1.sector = FIFO.write_ptr.sector; FIFO.read_ptr_1.byte = FIFO.write_ptr.byte; FIFO.message_count_1 = 0; }
      if(receiver == 2) { FIFO.read_ptr_2.sector = FIFO.write_ptr.sector; FIFO.read_ptr_2.byte = FIFO.write_ptr.byte; FIFO.message_count_2 = 0; }    
      if(receiver == 3) { FIFO.read_ptr_3.sector = FIFO.write_ptr.sector; FIFO.read_ptr_3.byte = FIFO.write_ptr.byte; FIFO.message_count_3 = 0; }   
      if(receiver == 4) { FIFO.read_ptr_4.sector = FIFO.write_ptr.sector; FIFO.read_ptr_4.byte = FIFO.write_ptr.byte; FIFO.message_count_4 = 0; }   
      return BB_OPERATION_FAILED;
    }
  } 
  return BB_OPERATION_OK;
  
}

//     (receiver -    , message_address -  ,       )
//  ,     ( ) 
BBresult CBlack_box::ext_read_message(uint8_t* buff, uint8_t receiver, uint32_t message_address)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  BB_message message;
  BB_mem_ptr mem_ptr;
  //             
  switch(receiver)
  {
    case 1: 
    {
      if(message_address >= FIFO.message_count_1) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_1.sector; mem_ptr.byte = FIFO.read_ptr_1.byte;
      //            ,    
      if(priority_message_1.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_1.sector, priority_message_1.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_1.sector, priority_message_1.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        //    
        if(!this->mem_dir_read(priority_message_1.sector, priority_message_1.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_1.sector, (priority_message_1.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_1_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 2: 
    {
      if(message_address >= FIFO.message_count_2) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_2.sector; mem_ptr.byte = FIFO.read_ptr_2.byte;
      //            ,    
      if(priority_message_2.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_2.sector, priority_message_2.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_2.sector, priority_message_2.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_2.sector, priority_message_2.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_2.sector, (priority_message_2.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_2_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 3: 
    {
      if(message_address >= FIFO.message_count_3) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_3.sector; mem_ptr.byte = FIFO.read_ptr_3.byte;
      //            ,    
      if(priority_message_3.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_3.sector, priority_message_3.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_3.sector, priority_message_3.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_3.sector, priority_message_3.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_3.sector, (priority_message_3.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_3_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    case 4: 
    {
      if(message_address >= FIFO.message_count_4) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_4.sector; mem_ptr.byte = FIFO.read_ptr_4.byte;
      //            ,    
      if(priority_message_4.sector != 0xFFFF && message_address == 0)
      {  
        //    
        if(!this->mem_dir_read(priority_message_4.sector, priority_message_4.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_4.sector, priority_message_4.byte, BB_MESSAGE_HEADER_SIZE);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        //    
        if(!this->mem_dir_read(priority_message_4.sector, priority_message_4.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
        {
          LOG("Black_box: read priority_message error from sector %d, byte %d, len %d\n", priority_message_4.sector, (priority_message_4.byte + BB_MESSAGE_HEADER_SIZE), message.len);
          mem_read_err_counter++; 
          return BB_OPERATION_FAILED;
        }
        priority_message_4_readed = 1;
        return BB_OPERATION_OK;
      }
      break;
    }
    default: return BB_OPERATION_FAILED;
  }
  //            ,     
  if(priority_message_1.sector != 0xFFFF && receiver == 1 && priority_message_1_readed && message_address > 0) message_address--;
  if(priority_message_2.sector != 0xFFFF && receiver == 2 && priority_message_2_readed && message_address > 0) message_address--;
  if(priority_message_3.sector != 0xFFFF && receiver == 3 && priority_message_3_readed && message_address > 0) message_address--;
  if(priority_message_4.sector != 0xFFFF && receiver == 4 && priority_message_4_readed && message_address > 0) message_address--;
  uint8_t we_in_head_sector = 0;
  if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
  uint32_t message_num = 0;
  while(1)
  {
    //    
    if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      LOG("Black_box: read message header error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte, BB_MESSAGE_HEADER_SIZE);
      mem_read_err_counter++; 
      return BB_OPERATION_FAILED;
    }      
    //   
    if(message.len && message.len != 0xFFFF && message.receiver_1 && receiver == 1) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_2 && receiver == 2) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_3 && receiver == 3) message_num++;
    if(message.len && message.len != 0xFFFF && message.receiver_4 && receiver == 4) message_num++;
    //     ,   (  ,   )
    if(message.len && message.len != 0xFFFF && message.receiver_1 && receiver == 1 && mem_ptr.sector == priority_message_1.sector && mem_ptr.byte == priority_message_1.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_2 && receiver == 2 && mem_ptr.sector == priority_message_2.sector && mem_ptr.byte == priority_message_2.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_3 && receiver == 3 && mem_ptr.sector == priority_message_3.sector && mem_ptr.byte == priority_message_3.byte) message_num--;
    if(message.len && message.len != 0xFFFF && message.receiver_4 && receiver == 4 && mem_ptr.sector == priority_message_4.sector && mem_ptr.byte == priority_message_4.byte) message_num--;
    
    //  , 
    if(message_num == message_address+1)
    {
      if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_HEADER_SIZE, buff, message.len))
      {
        LOG("Black_box: read message error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_HEADER_SIZE, message.len);
        mem_read_err_counter++; 
        return BB_OPERATION_FAILED;
      }        
      break;
    }
    //    
    mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
    //   ,    
    if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
    {
      mem_ptr = this->sector_forward(mem_ptr);
      if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
    } 
    //       ,     ,  
    if( we_in_head_sector && (mem_ptr.sector > FIFO.write_ptr.sector || mem_ptr.byte >= FIFO.write_ptr.byte))
    {
      if(receiver == 1) { FIFO.read_ptr_1.sector = FIFO.write_ptr.sector; FIFO.read_ptr_1.byte = FIFO.write_ptr.byte; FIFO.message_count_1 = 0; }
      if(receiver == 2) { FIFO.read_ptr_2.sector = FIFO.write_ptr.sector; FIFO.read_ptr_2.byte = FIFO.write_ptr.byte; FIFO.message_count_2 = 0; }    
      if(receiver == 3) { FIFO.read_ptr_3.sector = FIFO.write_ptr.sector; FIFO.read_ptr_3.byte = FIFO.write_ptr.byte; FIFO.message_count_3 = 0; }   
      if(receiver == 4) { FIFO.read_ptr_4.sector = FIFO.write_ptr.sector; FIFO.read_ptr_4.byte = FIFO.write_ptr.byte; FIFO.message_count_4 = 0; }   
      return BB_OPERATION_FAILED;
    }
  } 
  return BB_OPERATION_OK;
  
}

//  
BBresult CBlack_box::delete_messages(uint8_t receiver, uint8_t message_count)
{
  //    (  F103)
  if(!check_voltage()) return BB_OPERATION_FAILED;
  BB_message message;
  BB_mem_ptr mem_ptr;
  uint8_t free = 0;
  //             
  switch(receiver)
  {
    case 1: 
    {
      if(message_count > FIFO.message_count_1) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_1.sector; mem_ptr.byte = FIFO.read_ptr_1.byte;
      //     ,   
      if(priority_message_1.sector != 0xFFFF && priority_message_1_readed)
      {
        if(!this->mem_dir_write(priority_message_1.sector, priority_message_1.byte + BB_MESSAGE_RECV_1_OFFSET, &free, 1))
        {
          LOG("Black_box: delete priority message error from sector %d, byte %d, len %d\n", priority_message_1.sector, priority_message_1.byte + BB_MESSAGE_RECV_1_OFFSET, 1);
          mem_write_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        message_count--;
        FIFO.message_count_1 --;
        priority_message_1.sector = 0xFFFF;
        priority_message_1.byte = 0xFFFF;
      }
      break;
    }
    case 2: 
    {
      if(message_count > FIFO.message_count_2) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_2.sector; mem_ptr.byte = FIFO.read_ptr_2.byte;
      //     ,   
      if(priority_message_2.sector != 0xFFFF && priority_message_2_readed)
      {
        if(!this->mem_dir_write(priority_message_2.sector, priority_message_2.byte + BB_MESSAGE_RECV_2_OFFSET, &free, 1))
        {
          LOG("Black_box: delete priority message error from sector %d, byte %d, len %d\n", priority_message_2.sector, priority_message_2.byte + BB_MESSAGE_RECV_2_OFFSET, 1);
          mem_write_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        message_count--;
        FIFO.message_count_2 --;
        priority_message_2.sector = 0xFFFF;
        priority_message_2.byte = 0xFFFF;
      }
      break;
    }
    case 3: 
    {
      if(message_count > FIFO.message_count_3) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_3.sector; mem_ptr.byte = FIFO.read_ptr_3.byte;
      //     ,   
      if(priority_message_3.sector != 0xFFFF && priority_message_3_readed)
      {
        if(!this->mem_dir_write(priority_message_3.sector, priority_message_3.byte + BB_MESSAGE_RECV_3_OFFSET, &free, 1))
        {
          LOG("Black_box: delete priority message error from sector %d, byte %d, len %d\n", priority_message_3.sector, priority_message_3.byte + BB_MESSAGE_RECV_3_OFFSET, 1);
          mem_write_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        message_count--;
        FIFO.message_count_3 --;
        priority_message_3.sector = 0xFFFF;
        priority_message_3.byte = 0xFFFF;
      }
      break;
    }
    case 4: 
    {
      if(message_count > FIFO.message_count_4) return BB_OPERATION_FAILED;
      mem_ptr.sector = FIFO.read_ptr_4.sector; mem_ptr.byte = FIFO.read_ptr_4.byte;
      //     ,   
      if(priority_message_4.sector != 0xFFFF && priority_message_4_readed)
      {
        if(!this->mem_dir_write(priority_message_4.sector, priority_message_4.byte + BB_MESSAGE_RECV_4_OFFSET, &free, 1))
        {
          LOG("Black_box: delete priority message error from sector %d, byte %d, len %d\n", priority_message_4.sector, priority_message_4.byte + BB_MESSAGE_RECV_4_OFFSET, 1);          
          mem_write_err_counter++; 
          return BB_OPERATION_FAILED;
        }        
        message_count--;
        FIFO.message_count_4 --;
        priority_message_4.sector = 0xFFFF;
        priority_message_4.byte = 0xFFFF;
      }
      break;
    }
    default: return BB_OPERATION_FAILED;
  }
  //       ,      
  if(FIFO.message_count_1 == 0) {FIFO.read_ptr_1.sector = FIFO.write_ptr.sector; FIFO.read_ptr_1.byte = FIFO.write_ptr.byte;}
  if(FIFO.message_count_2 == 0) {FIFO.read_ptr_2.sector = FIFO.write_ptr.sector; FIFO.read_ptr_2.byte = FIFO.write_ptr.byte;}
  if(FIFO.message_count_3 == 0) {FIFO.read_ptr_3.sector = FIFO.write_ptr.sector; FIFO.read_ptr_3.byte = FIFO.write_ptr.byte;}
  if(FIFO.message_count_4 == 0) {FIFO.read_ptr_4.sector = FIFO.write_ptr.sector; FIFO.read_ptr_4.byte = FIFO.write_ptr.byte;}
  //           ,    .
  if(message_count == 0) return BB_OPERATION_OK;

  uint8_t we_in_head_sector = 0;
  if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
  uint8_t message_num = message_count;
  while(1)
  {
    //    
    if(!this->mem_dir_read(mem_ptr.sector, mem_ptr.byte, (uint8_t*)&message, BB_MESSAGE_HEADER_SIZE))
    {
      LOG("Black_box: read message header while delete error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte, BB_MESSAGE_HEADER_SIZE);
      mem_read_err_counter++; 
      return BB_OPERATION_FAILED;
    }        
    //           
    if(message.len && message.len != 0xFFFF && message.receiver_1 && receiver == 1) 
    {  
      if(!this->mem_dir_write(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_1_OFFSET, &free, 1))
      {
        LOG("Black_box: delete message s1 error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_1_OFFSET, 1);          
        mem_write_err_counter++; 
        return BB_OPERATION_FAILED;
      }        

      message_num--;
      FIFO.message_count_1 --;
    }
    if(message.len && message.len != 0xFFFF && message.receiver_2 && receiver == 2) 
    {  
      if(!this->mem_dir_write(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_2_OFFSET, &free, 1))
      {
        LOG("Black_box: delete message s2 error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_2_OFFSET, 1);          
        mem_write_err_counter++; 
        return BB_OPERATION_FAILED;
      }        
      message_num--;
      FIFO.message_count_2 --;      
    }
    if(message.len && message.len != 0xFFFF && message.receiver_3 && receiver == 3) 
    {  
      if(!this->mem_dir_write(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_3_OFFSET, &free, 1))
      {
        LOG("Black_box: delete message s3 error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_3_OFFSET, 1);          
        mem_write_err_counter++; 
        return BB_OPERATION_FAILED;
      }        
      message_num--;
      FIFO.message_count_3 --;
    }  
    if(message.len && message.len != 0xFFFF && message.receiver_4 && receiver == 4) 
    {  
      if(!this->mem_dir_write(mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_4_OFFSET, &free, 1))
      {
        LOG("Black_box: delete message s4 error from sector %d, byte %d, len %d\n", mem_ptr.sector, mem_ptr.byte + BB_MESSAGE_RECV_4_OFFSET, 1);          
        mem_write_err_counter++; 
        return BB_OPERATION_FAILED;
      }        
      message_num--;
      FIFO.message_count_4 --;
    }
    //    ,       
    if(message_num == 0)
    {
      //      
      if(receiver == 1) 
      {
        mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
        //   ,    
        if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
        {
          mem_ptr = this->sector_forward(mem_ptr);
        } 
        FIFO.read_ptr_1 = mem_ptr; 
      }
      if(receiver == 2) 
      {
        mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
        //   ,    
        if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
        {
          mem_ptr = this->sector_forward(mem_ptr);
        } 
        FIFO.read_ptr_2 = mem_ptr; 
      }
      if(receiver == 3) 
      {
        mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
        //   ,    
        if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
        {
          mem_ptr = this->sector_forward(mem_ptr);
        } 
        FIFO.read_ptr_3 = mem_ptr; 
      }
      if(receiver == 4) 
      {
        mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
        //   ,    
        if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
        {
          mem_ptr = this->sector_forward(mem_ptr);
        } 
        FIFO.read_ptr_4 = mem_ptr; 
      }
      //    ,      
      if(FIFO.message_count_1 == 0) {FIFO.read_ptr_1.sector = FIFO.write_ptr.sector; FIFO.read_ptr_1.byte = FIFO.write_ptr.byte;}
      if(FIFO.message_count_2 == 0) {FIFO.read_ptr_2.sector = FIFO.write_ptr.sector; FIFO.read_ptr_2.byte = FIFO.write_ptr.byte;}
      if(FIFO.message_count_3 == 0) {FIFO.read_ptr_3.sector = FIFO.write_ptr.sector; FIFO.read_ptr_3.byte = FIFO.write_ptr.byte;}
      if(FIFO.message_count_4 == 0) {FIFO.read_ptr_4.sector = FIFO.write_ptr.sector; FIFO.read_ptr_4.byte = FIFO.write_ptr.byte;}
      break;
    }
    //    
    mem_ptr.byte = mem_ptr.byte + message.len + BB_MESSAGE_HEADER_SIZE;
    //   ,    
    if(mem_ptr.byte >= this->dir_sector_size || message.len > this->dir_sector_size || ((mem_ptr.byte + BB_MESSAGE_HEADER_SIZE) >= this->dir_sector_size)) 
    {
      mem_ptr = this->sector_forward(mem_ptr);
      if(mem_ptr.sector == FIFO.write_ptr.sector) we_in_head_sector = 1;
    } 
    //       ,     ,  
    if( we_in_head_sector && (mem_ptr.sector > FIFO.write_ptr.sector || mem_ptr.byte >= FIFO.write_ptr.byte))
    {
      if(receiver == 1) { FIFO.read_ptr_1.sector = FIFO.write_ptr.sector; FIFO.read_ptr_1.byte = FIFO.write_ptr.byte; FIFO.message_count_1 = 0; }
      if(receiver == 2) { FIFO.read_ptr_2.sector = FIFO.write_ptr.sector; FIFO.read_ptr_2.byte = FIFO.write_ptr.byte; FIFO.message_count_2 = 0; }    
      if(receiver == 3) { FIFO.read_ptr_3.sector = FIFO.write_ptr.sector; FIFO.read_ptr_3.byte = FIFO.write_ptr.byte; FIFO.message_count_3 = 0; }   
      if(receiver == 4) { FIFO.read_ptr_4.sector = FIFO.write_ptr.sector; FIFO.read_ptr_4.byte = FIFO.write_ptr.byte; FIFO.message_count_4 = 0; }   
      return BB_OPERATION_FAILED;
    }
  } 
  return BB_OPERATION_OK;    
}

//      
uint32_t CBlack_box::get_message_count(uint8_t receiver)
{
  switch(receiver)
  {
    case 1: return FIFO.message_count_1;
    case 2: return FIFO.message_count_2;
    case 3: return FIFO.message_count_3;    
    case 4: return FIFO.message_count_4;
    default: return 0;
  }
}

//   
BBresult CBlack_box::Clean(void)
{
  //   
  uint32_t dummy = 0;
  for(uint16_t i = 0; i < this->dir_sector_count; i++)
  {  
    if(!this->mem_dir_write(i, 0, (uint8_t*)&dummy, sizeof(dummy)))
    {
      mem_write_err_counter++; 
      return BB_OPERATION_FAILED;
    }        
  }
  //    
  this->mem_dir_erase_sector(0);
  this->mem_dir_erase_sector(1);

  //  FIFO
  memset(&this->FIFO, 0x00, sizeof(this->FIFO));
  return BB_OPERATION_OK;
}

uint32_t CBlack_box::get_file_read_ptr(const char* filename)
{
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;  
  return this->File_table[index].read_ptr;
}


uint32_t CBlack_box::get_file_total_size(const char* filename)
{
  uint8_t index = ffind(filename);
  if(index == 0xFF) return BB_OPERATION_FAILED;  
  return this->File_table[index].file_total_size;
}

#ifdef USE_FREERTOS
//  
BaseType_t CBlack_box::Grab(TickType_t xTicksToWait)
{
  return xSemaphoreTake(this->bb_busy, xTicksToWait);  
}

//  
void CBlack_box::Release (void)
{
  xSemaphoreGive(this->bb_busy);  
}
#endif


void CBlack_box::Load_base_table(void)
{
  //    (  F103)
  if(!check_voltage()) return;
  this->mem_read(BASE_START_ADDRESS, (uint8_t*)&Base_table, BASE_TABLE_SIZE);
  //      ,    
  if(*(uint8_t*)&Base_table == 0xFF)
  {
    memset(&Base_table, 0x00, BASE_TABLE_SIZE);
    return;
  }  
  //      
  for(uint8_t index=0; index<BASE_COUNT; index++)
  {
    Base_table[index].base_ptr = 0;
  }
}

//     
uint8_t CBlack_box::ffind(const char* filename)
{
  for(uint8_t index = 0; index < BB_MAX_FILES; index++)
  {
    if(memcmp(filename,File_table[index].filename, strlen(filename)) == 0) 
    {
      return index;
    }
  }
  return 0xFF;
}

//   ,       
bool CBlack_box::check_voltage(void)
{
  #if (defined(STM32F10X_HD) || defined(STM32F2XX) || defined(STM32F413_423xx))
  if(PWR_GetFlagStatus(PWR_FLAG_PVDO) == SET) 
  {  
    EXTI_GenerateSWInterrupt(EXTI_Line16);
    return false;
  }
  #endif
    
  return true;
}

#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include "time_utilities\time_utilities.h"
#include "rtc_driver.h"
//   
void CBlack_box::LOG(const char* string, ...)
{
  //       
  memset(LOG_BUF, 0x00, sizeof(LOG_BUF));
  uint32_t unixtime = GetUnixTime();
  time_struct_t time; 
  conv_unixtime_to_time(&time, unixtime);
  snprintf((char*)LOG_BUF, sizeof(LOG_BUF), "[%d/%d/%d][%d:%d:%d]", time.day, time.mon, time.year, time.hour, time.min, time.sec);
  uint8_t LOG_BUF_offset = strlen((char*)LOG_BUF);
  //        
  va_list ap;
  va_start(ap, string);
  int retval = vsnprintf((char*)LOG_BUF + LOG_BUF_offset, sizeof(LOG_BUF) - LOG_BUF_offset, string, ap);
  va_end(ap);  
  uint8_t size = strlen((char*)LOG_BUF);
  
  fwrite("LOG", LOG_BUF, size);
  fsave();
}
