/**
  ******************************************************************************
  * File Name          : Black_box.h
  * Description        :   ( , FIFO,  )
  *                      
  *                      
  ******************************************************************************
*/
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef BLACK_BOX_H
#define BLACK_BOX_H

/* Includes ------------------------------------------------------------------*/
#include "Black_box_config.h"
#include "stdint.h"

#ifdef USE_FREERTOS
#include "FreeRTOS.h"
#include "semphr.h"
#endif

/* Defines- ------------------------------------------------------------------*/
//    
#define BASE_TABLE_SIZE         sizeof(Base_table)
//    
#define BB_MESSAGE_HEADER_SIZE   6
//       
#define BB_MESSAGE_RECV_1_OFFSET 2
#define BB_MESSAGE_RECV_2_OFFSET 3
#define BB_MESSAGE_RECV_3_OFFSET 4
#define BB_MESSAGE_RECV_4_OFFSET 5

//     
typedef enum
{
  BB_OPERATION_OK     =  1,
  BB_OPERATION_FAILED =  0,
}BBresult;

//   
typedef struct
{
  //     
  char base_name[BASE_NAME_SIZE];
  //     
  uint32_t base_start_address;
  //      
  void* base_ptr;
  //   
  uint16_t base_size;
  //   -  +  
  uint16_t full_base_size;
} TBase_table[BASE_COUNT];

//  ,      
typedef uint8_t (*mem_driver_init_t)(void);
typedef void (*mem_driver_deinit_t)(void);
typedef uint8_t (*mem_driver_format_t)(void);
typedef uint8_t (*mem_driver_write_t)(uint32_t address, uint8_t* data, uint16_t size);
typedef uint8_t (*mem_driver_read_t)(uint32_t address, uint8_t* data, uint16_t size);
typedef uint8_t (*mem_driver_direct_write_t)(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size);
typedef uint8_t (*mem_driver_direct_read_t)(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size);
typedef uint8_t (*mem_driver_direct_erase_sector_t)(uint16_t sector);

//  ,  
typedef enum
{
  FIFO_FILE = 0,   //  
  SIMPLE_FILE = 1  //  
}BB_filetype;

//    
typedef struct
{
  //  
  char filename[BB_FILENAME_SIZE];
  //  
  BB_filetype filetype;
  //    
  uint32_t start_address;
  //     
  uint32_t end_address;
  //    (      ,      )
  uint32_t read_ptr;
  //    ( ,   )
  uint32_t write_ptr;
  //   (  )
  uint32_t file_total_size;
  //   (  )
  uint8_t updated;
  //    (  )
  bool opened;
}BB_file;

//     FIFO  
typedef struct
{
  //  
  uint16_t sector;
  //   
  uint16_t byte;
}BB_mem_ptr;

// FIFO  
typedef struct
{
  //     FIFO
  BB_mem_ptr write_ptr;
  //      
  BB_mem_ptr read_ptr_1;
  BB_mem_ptr read_ptr_2;
  BB_mem_ptr read_ptr_3;
  BB_mem_ptr read_ptr_4;
  //     
  uint32_t message_count_1;
  uint32_t message_count_2;
  uint32_t message_count_3;
  uint32_t message_count_4;
}BB_fifo;

//   
typedef __packed struct
{
  //  
  uint16_t len;
  //   (    )
  uint8_t receiver_1;
  uint8_t receiver_2; 
  uint8_t receiver_3;
  uint8_t receiver_4;
  //  
  uint8_t* payload;
}BB_message;

/* C++ code-------------------------------------------------------------------*/
class CBlack_box
{
  public:
  // 
  CBlack_box(mem_driver_init_t init, mem_driver_deinit_t deinit,\
             mem_driver_format_t mem_format,\
             mem_driver_write_t write, mem_driver_read_t read,\
             mem_driver_direct_write_t dir_write, mem_driver_direct_read_t dir_read,\
             mem_driver_direct_erase_sector_t dir_erase, uint16_t dir_sector_count,\
             uint16_t dir_sector_size, uint32_t weared_memory_size);
  
  // 
  BBresult Init(void);
  // 
  void Deinit(void);
  //    .       !
  void Disable_priority_messages(void) {use_priority_message_setting = false;}
  
  //    
  //      
  BBresult Register_in_base(const char* base_name, uint16_t base_size, uint16_t reserved_place_size, void* base_ptr);
  //     
  BBresult Load_base(const char* base_name);
  //     
  BBresult Save_base(const char* base_name); 

  ////   
  //   (filetype -  ,   )
  BBresult fcreate(const char* filename, uint32_t size_in_bytes, BB_filetype filetype);
  //   (     )
  BBresult fopen(const char* filename);
  //  
  BBresult fclose(const char* filename);
  // ,   
  BBresult fopened(const char* filename, uint8_t* opened);
  //    (      ,       )
  BBresult fread(const char* filename, uint32_t addr_in_file, uint8_t* buff, uint16_t size);
  //    (      ,     )
  BBresult fwrite(const char* filename, uint8_t* buff, uint16_t size);
  //      (     )
  BBresult fsize(const char* filename, uint32_t* size);
  //  
  BBresult fflush(const char* filename);
  //    
  BBresult fsave(void);
  //      
  inline bool is_file_present(const char* filename) {return (ffind(filename)==0xFF)?(false):(true);}

  //   FIFO  
  //  
  BB_mem_ptr sector_forward(BB_mem_ptr ptr);
  BB_mem_ptr sector_back(BB_mem_ptr ptr);
  BB_mem_ptr find_free_sector(void);
  bool find_head(void);
  void find_tails(void);

  //    
  BBresult add_message(BB_message* message);
  //     (receiver -    , message_address -  ,       )
  BBresult read_message(uint8_t* buff, uint8_t receiver, uint16_t message_address);
  BBresult ext_read_message(uint8_t* buff, uint8_t receiver, uint32_t message_address);
  //  
  BBresult delete_messages(uint8_t receiver, uint8_t message_count);
  //      
  uint32_t get_message_count(uint8_t receiver);
  //   
  BBresult Clean(void);
  
  uint16_t get_mem_read_err_counter(void) {return mem_read_err_counter;}
  uint16_t get_mem_write_err_counter(void) {return mem_write_err_counter;}
  void reset_mem_err_counters(void) {mem_read_err_counter = 0; mem_write_err_counter = 0;}
  
  #ifdef USE_FREERTOS
  //  
  BaseType_t Grab(TickType_t xTicksToWait = portMAX_DELAY);
  //  
  void Release (void);
  //        
  SemaphoreHandle_t bb_busy;
  #endif
  //      
  uint32_t get_file_read_ptr(const char* filename);
  uint32_t get_file_total_size(const char* filename);
  
  //  ,     LAT
  BBresult file_self_test(const char* filename);
  
  //     data c  address
  bool Is_data_same(uint32_t address, const uint8_t* data, uint16_t size);
  
  private:
  uint16_t dir_sector_count;
  uint16_t dir_sector_size;
  uint32_t weared_memory_size;
  BB_file File_table[BB_MAX_FILES];
  BB_fifo FIFO;
  mem_driver_init_t mem_init;
  mem_driver_deinit_t mem_deinit;
  mem_driver_format_t mem_format;
  mem_driver_write_t mem_write;
  mem_driver_read_t mem_read;
  mem_driver_direct_write_t mem_dir_write;
  mem_driver_direct_read_t mem_dir_read;
  mem_driver_direct_erase_sector_t mem_dir_erase_sector;
  
  uint16_t mem_read_err_counter;
  uint16_t mem_write_err_counter;
  
  //     .   .
  bool use_priority_message_setting;
  
  //    
  BB_mem_ptr priority_message_1;
  BB_mem_ptr priority_message_2;
  BB_mem_ptr priority_message_3;
  BB_mem_ptr priority_message_4;
  uint8_t priority_message_1_readed;
  uint8_t priority_message_2_readed;
  uint8_t priority_message_3_readed;
  uint8_t priority_message_4_readed;
  
  
  //   
  TBase_table Base_table;
  //      
  void Load_base_table(void);
  //     
  uint8_t ffind(const char* filename);
  //   ,       
  bool check_voltage(void);
  
  void LOG(const char* string, ...);
  uint8_t LOG_BUF[128];
  
};


#endif //BLACK_BOX_H