/**
  ******************************************************************************
  * File Name          : NOR_FTL.cpp
  * Description        :       
  *                      flash-      
  *                      
  ******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include "NOR_FTL/NOR_FTL.h"
#include "macro_utils.h"

#if !defined(__FTL_WRITE_DEBUG)
#define __FTL_WRITE_DEBUG(...) {}
#define __GET_MS_COUNTER()     (0)
#endif //!defined(__FTL_WRITE_DEBUG)

#if defined(BOOTLOADER)
#if !defined(__GET_MS_COUNTER)
uint32_t __GET_MS_COUNTER(void) {extern volatile uint32_t SysTickReg; return SysTickReg;}
#endif //!defined(__GET_MS_COUNTER)
#endif

//NOR_LAT_element LAT[NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT]; //for debug only

/* C++ code-------------------------------------------------------------------*/
// 
CNOR_ftl::CNOR_ftl(NOR_driver_init_t init, NOR_driver_deinit_t deinit,\
                   NOR_driver_write_t write, NOR_driver_read_t read,\
                   NOR_driver_erase_sector_t erase_sector,\
                   NOR_driver_erase_chip_t erase_chip)
{
  // ,       
  if(!init || !deinit || !write || !read || !erase_sector || !erase_chip) while(1);
  //  FTL    
  this->mem_init = init;
  this->mem_deinit = deinit;
  this->mem_write = write;
  this->mem_read = read;
  this->mem_erase_sector = erase_sector;
  this->mem_erase_chip = erase_chip;
}

//  
NORresult CNOR_ftl::Init(void)
{
  //  
  if(!mem_init()) return NOR_OPERATION_FAILED;
  //Format_memory();
  
  lat_recovery_inf.event_flag = 0;
  LAT_formatted_flag = 0;
  
  if(Is_memory_formatted()) 
  {
    bool ret;
    uint8_t range_ok;
    uint16_t addr_dublication, not_found_addr;
    
    //    
    ret = (NOR_OPERATION_OK == Load_LAT());
    
    __PRINTF("LAT load: %s\n", (ret)?("ok"):("fail"));
    
    //  
    ret = Check_LAT(&addr_dublication, &not_found_addr, &range_ok);
    
    __PRINTF("LAT check: %s, dublication_addr: %u, not_found_addr: %u, range: %s\n", (ret)?("pass"):("fail"), addr_dublication, not_found_addr, (range_ok)?("ok"):("err"));
    
    //    ,    
    if(!ret && addr_dublication == 0 && not_found_addr == 1)
    {
      ret = RecoverySingleLostSector();
    }
    
    if(!ret)
    {
      __PRINTF("format memory\n");
      //for(;;);
      Format_memory();
      LAT_formatted_flag = 1;
    }
    
    return NOR_OPERATION_OK;
  }
  else
  {
    //    NOR FTL
    __PRINTF("format memory\n");
    //for(;;);
    Format_memory();
    LAT_formatted_flag = 1;
    
    return NOR_OPERATION_OK;
  }
}

//  
void CNOR_ftl::Deinit(void)
{
  mem_deinit();
}

//  
NORresult CNOR_ftl::Write_data(uint32_t logical_address, uint8_t* buff, uint16_t size)
{
  NOR_address fizical_address_1;
  NOR_address fizical_address_2;
  uint16_t sector1_write_size = 0;
  uint16_t sector2_write_size = 0;
  
  //     
  if(logical_address + size > Get_memory_size())
  {
    __FTL_WRITE_DEBUG("%s: bad args, addr: %u, size: %u\n", __func__, logical_address, size);
    
    return NOR_OPERATION_FAILED;
  }
  
  //        
  if(size > NOR_FTL_SECTOR_SIZE)
  {
    __FTL_WRITE_DEBUG("%s: bad args, addr: %u, size: %u\n", __func__, logical_address, size);
    
    return NOR_OPERATION_FAILED;
  }
  
  //    
  if(Find_fizical_address(logical_address, &fizical_address_1) != NOR_OPERATION_OK) return NOR_OPERATION_FAILED;
  
  //   
  if(fizical_address_1.byte + size > NOR_FTL_SECTOR_SIZE)
  {
    //     
    if(Find_fizical_address(logical_address + (NOR_FTL_SECTOR_SIZE - fizical_address_1.byte), &fizical_address_2) != NOR_OPERATION_OK) return NOR_OPERATION_FAILED;
    
    sector1_write_size = NOR_FTL_SECTOR_SIZE - fizical_address_1.byte;
    sector2_write_size = size - sector1_write_size;
  }
  else
  {
    //      
    sector1_write_size = size;
  }
  //      
  if(sector1_write_size) Write_sector(fizical_address_1.sector, fizical_address_1.byte, buff, sector1_write_size);
  //    
  if(sector2_write_size) Write_sector(fizical_address_2.sector, fizical_address_2.byte, buff + sector1_write_size, sector2_write_size);
  //    
  return NOR_OPERATION_OK;
}

//  
NORresult CNOR_ftl::Read_data(uint32_t logical_address, uint8_t* buff, uint16_t size) 
{
  NOR_address fizical_addr;
  
  //     
  if(logical_address + size > Get_memory_size()) 
  {
    __FTL_WRITE_DEBUG("%s: bad args, addr: %u, size: %u\n", __func__, logical_address, size);
    
    return NOR_OPERATION_FAILED;
  }
  //       
  if(size > NOR_FTL_SECTOR_SIZE)
  {
    __FTL_WRITE_DEBUG("%s: bad args, addr: %u, size: %u\n", __func__, logical_address, size);
    
    return NOR_OPERATION_FAILED;
  }
  
  //   
  if(Find_fizical_address(logical_address, &fizical_addr) != NOR_OPERATION_OK) return NOR_OPERATION_FAILED;
  //__PRINTF("%s: Find_fizical_address, logical: %u, fizical_sector: %u, fizical_byte: %u\n", __func__, logical_address, fizical_addr.sector, fizical_addr.byte);
  
  //      
  if(NOR_FTL_SECTOR_SIZE - fizical_addr.byte > size)
  {
    if(buff!=NULL)
    {
      mem_read(fizical_addr.sector, fizical_addr.byte, buff, size);
    }
  }
  else
  {
    uint16_t read_first_size = NOR_FTL_SECTOR_SIZE - fizical_addr.byte;
    uint16_t read_second_size = size - read_first_size;
    
    //     
    if(buff!=NULL)
    {
      mem_read(fizical_addr.sector, fizical_addr.byte, buff, read_first_size);
    }
    
    //   
    if(Find_fizical_address(logical_address + read_first_size, &fizical_addr) != NOR_OPERATION_OK) return NOR_OPERATION_FAILED;
    //__PRINTF("%s: Find_fizical_address, logical: %u, fizical_sector: %u, fizical_byte: %u\n", __func__, logical_address, fizical_addr.sector, fizical_addr.byte);
    
    //    
    if(buff!=NULL)
    {
      mem_read(fizical_addr.sector, fizical_addr.byte, buff + read_first_size, read_second_size);
    }
  }
  return NOR_OPERATION_OK;
}

//    
uint32_t CNOR_ftl::Get_memory_size(void)
{
  return ((NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT - NOL_FTL_FREE_SECTOR_COUNT) * NOR_FTL_SECTOR_SIZE - sizeof(NOR_spare_header)*NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT);
}

// fixme:
//   , - ,   Format_memory    NOL_FTL_FREE_SECTOR_COUNT.
//      Format_memory  NOL_FTL_FREE_SECTOR_COUNT  <=   Get_memory_size  (NOL_FTL_FREE_SECTOR_COUNT+1)
//   , ..      .

//  
NORresult CNOR_ftl::Format_memory(void)
{
  NOT_USED(uint32_t debug_timer;)
  
  NOR_spare_header header;
  //   
  this->mem_erase_chip();
  //      
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    LAT[i].sector_address = i;
    if(i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT - NOL_FTL_FREE_SECTOR_COUNT) LAT[i].sector_busy = 1;
    else
    {
      LAT[i].sector_busy = 0;
      LAT[i].sector_address = 0;
    }
  }
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    header.wear_level = 0;
    header.sector_address = LAT[i].sector_address;
    header.sector_busy = 0xFF;
    if(LAT[i].sector_busy == 0) header.sector_busy = 0x00;
    header.sector_out_move_start = 0xFF;
    header.sector_in_move_end = 0x00;
    
    __FTL_WRITE_DEBUG("%s: -> mem_write(%u, %u, %p, %u)\n", __func__, i, 0, (uint8_t*)&header, sizeof(header)); debug_timer=__GET_MS_COUNTER();
    mem_write(i, 0, (uint8_t*)&header, sizeof(header));
    __FTL_WRITE_DEBUG("%s: <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  }
  return NOR_OPERATION_OK;
}

//    (    IDLE)
void CNOR_ftl::Static_wear_leveller(void)
{
  NOT_USED(uint32_t debug_timer;)
  
  uint16_t minimum_weared_sector;
  uint16_t maximum_weared_sector;
  uint32_t minimum_wear = 0xFFFFFFFF;
  uint32_t maximum_wear = 0;
  //      
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    NOR_spare_header sector_header;
    
    mem_read(i, 0, (uint8_t*)&sector_header, sizeof(sector_header));
    //     
    if(sector_header.sector_busy && sector_header.wear_level < minimum_wear)
    {
      minimum_wear = sector_header.wear_level;
      minimum_weared_sector = i;
    }
    //     
    if(sector_header.sector_busy == 0x00 && sector_header.wear_level > maximum_wear)  
    {
      maximum_wear = sector_header.wear_level;
      maximum_weared_sector = i;
    }
  }
  //    ,    
  if( maximum_wear > minimum_wear && (maximum_wear - minimum_wear) > NOR_FTL_MAXIMUM_WEAR_DISBALANCE ) 
  {
    //         
    __FTL_WRITE_DEBUG("%s: -> Move_sector(%u, %u, %u, %u)\n", __func__, minimum_weared_sector, maximum_weared_sector, 0, 0); debug_timer=__GET_MS_COUNTER();
    Move_sector(minimum_weared_sector, maximum_weared_sector);
    __FTL_WRITE_DEBUG("%s: <- Move_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  }
}

//     
NORresult CNOR_ftl::Direct_write_data(uint16_t sector, uint16_t address, uint8_t* buff, uint16_t size)
{
  NOT_USED(uint32_t debug_timer;)
  
  sector = sector + NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT;
  if(sector >=  NOR_FTL_SECTOR_COUNT) return NOR_OPERATION_FAILED;
  if(address + size > NOR_FTL_SECTOR_SIZE) return NOR_OPERATION_FAILED;
  
  __FTL_WRITE_DEBUG("%s: -> mem_write(%u, %u, %p, %u)\n", __func__, sector, address, buff, size); debug_timer=__GET_MS_COUNTER();
  mem_write(sector, address, buff, size);
  __FTL_WRITE_DEBUG("%s: <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  
  return NOR_OPERATION_OK;
}

//     
NORresult CNOR_ftl::Direct_read_data(uint16_t sector,uint16_t address, uint8_t* buff, uint16_t size)
{
  sector = sector + NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT;
  if(sector >=  NOR_FTL_SECTOR_COUNT) return NOR_OPERATION_FAILED;
  if(address + size > NOR_FTL_SECTOR_SIZE) return NOR_OPERATION_FAILED;
  
  mem_read(sector, address, buff, size);
  
  return NOR_OPERATION_OK;
}

//    (     )
NORresult CNOR_ftl::Direct_erase_sector(uint16_t sector)
{
  NOT_USED(uint32_t debug_timer;)
  
  sector = sector + NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT;
  if(sector >=  NOR_FTL_SECTOR_COUNT) return NOR_OPERATION_FAILED;
  
  __FTL_WRITE_DEBUG("%s: -> mem_erase_sector(%u)\n", __func__, sector); debug_timer=__GET_MS_COUNTER();
  mem_erase_sector(sector);
  __FTL_WRITE_DEBUG("%s: <- mem_erase_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  return NOR_OPERATION_OK;
}

//      
NORresult CNOR_ftl::Load_LAT(void)
{
  NOT_USED(uint32_t debug_timer;)
  
  NOR_spare_header tmp_header;
  NOR_spare_header error_in_header;
  uint16_t error_out_sector = 0;
  uint16_t error_in_sector = 0;
  uint32_t maximum_wear = 0;
    
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    mem_read(i, 0, (uint8_t*)&tmp_header, sizeof(tmp_header));
    LAT[i].sector_address = tmp_header.sector_address;
    LAT[i].sector_busy = tmp_header.sector_busy;
    //     
    if(tmp_header.wear_level > maximum_wear && tmp_header.wear_level != 0xFFFFFFFF) maximum_wear = tmp_header.wear_level;

    //         
    if(tmp_header.sector_busy && tmp_header.sector_in_move_end != 0x00) 
    {
      //    
      error_in_sector = (i+1);
      error_in_header.sector_address = tmp_header.sector_address;
      error_in_header.sector_busy = tmp_header.sector_busy;
      error_in_header.wear_level = tmp_header.wear_level;
      error_in_header.sector_in_move_end = tmp_header.sector_in_move_end;
      
      __FTL_WRITE_DEBUG("%s: found incomplete dst sector move: %u\n", __func__, i);
    }
    //  ,      
    if(tmp_header.sector_out_move_start == 0x00 && tmp_header.sector_busy)
    {
      error_out_sector = (i+1);
      
      __FTL_WRITE_DEBUG("%s: found incomplete src sector move: %u\n", __func__, i);
    }
  }

  //    
  //   1 (   (  ),   )
  if(error_in_sector && !error_out_sector && error_in_header.wear_level == 0xFFFFFFFF)
  {
    //        ,   
    error_in_sector--;
    
    __FTL_WRITE_DEBUG("%s(1): -> mem_erase_sector(%u)\n", __func__, error_in_sector); debug_timer=__GET_MS_COUNTER();
    mem_erase_sector(error_in_sector);
    __FTL_WRITE_DEBUG("%s(1): mem_erase_sector (%u ms) <-\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    tmp_header.sector_address = 0;
    tmp_header.sector_busy = 0;
    tmp_header.sector_out_move_start = 0xFF;
    tmp_header.sector_in_move_end = 0xFF;
    tmp_header.wear_level = maximum_wear;
    
    __FTL_WRITE_DEBUG("%s(1): -> mem_write(%u, %u, %p, %u)\n", __func__, error_in_sector, 0, (uint8_t*)&tmp_header, sizeof(tmp_header)); debug_timer=__GET_MS_COUNTER();
    mem_write(error_in_sector, 0, (uint8_t*)&tmp_header, sizeof(tmp_header));
    __FTL_WRITE_DEBUG("%s(1): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    LAT[error_in_sector].sector_busy = 0;
    LAT[error_in_sector].sector_address = 0;
  }
  //   2 (   ,     ,   )
  else if(error_in_sector && !error_out_sector && error_in_header.wear_level != 0xFFFFFFFF)
  {
    //      
    error_in_sector--;
    LAT[error_in_sector].sector_busy = 0;
    LAT[error_in_sector].sector_address = 0;
    const uint8_t busy_and_address[3] = {0, 0, 0}; //   ?
    
    __FTL_WRITE_DEBUG("%s(2.2): -> mem_write(%u, %u, %p, %u)\n", __func__, error_in_sector, NOR_FTL_SECTOR_BUSY_OFFSET, busy_and_address, sizeof(busy_and_address)); debug_timer=__GET_MS_COUNTER();
    mem_write(error_in_sector, NOR_FTL_SECTOR_BUSY_OFFSET, busy_and_address, sizeof(busy_and_address));
    __FTL_WRITE_DEBUG("%s(2.2): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    /*
    const uint8_t busy = 0;
    const uint16_t address = 0;
    
    __FTL_WRITE_DEBUG("%s(2.1): -> mem_write(%u, %u, %p, %u)\n", __func__, error_in_sector, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, sizeof(error_in_header.sector_busy)); debug_timer=__GET_MS_COUNTER();
    mem_write(error_in_sector, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, sizeof(error_in_header.sector_busy));
    __FTL_WRITE_DEBUG("%s(2.1): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    __FTL_WRITE_DEBUG("%s(2.2): -> mem_write(%u, %u, %p, %u)\n", __func__, error_in_sector, NOR_FTL_SECTOR_ADDRESS_OFFSET, (uint8_t*)&address, sizeof(address)); debug_timer=__GET_MS_COUNTER();
    mem_write(error_in_sector, NOR_FTL_SECTOR_ADDRESS_OFFSET, (uint8_t*)&address, sizeof(address));
    __FTL_WRITE_DEBUG("%s(2.2): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    */
  }  
  //   3 (  ,   ,        )
  else if(error_in_sector && error_out_sector && error_in_header.sector_in_move_end != 0) 
  {
    //         
    error_in_sector--;
    error_out_sector--;
    __FTL_WRITE_DEBUG("%s(3): -> Move_sector(%u, %u, %u, %u)\n", __func__, error_out_sector, error_in_sector, 0, 0); debug_timer=__GET_MS_COUNTER();
    Move_sector(error_out_sector, error_in_sector);
    __FTL_WRITE_DEBUG("%s(3): <- Move_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  }
  //   4 ( ,       ,      )
  else if(!error_in_sector && error_out_sector)
  {
    //     .
    const uint8_t busy = 0;
    error_out_sector--;
    __FTL_WRITE_DEBUG("%s(4): -> mem_write(%u, %u, %p, %u)\n", __func__, error_out_sector, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, 1); debug_timer=__GET_MS_COUNTER();
    mem_write(error_out_sector, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, 1); 
    __FTL_WRITE_DEBUG("%s(4): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    LAT[error_out_sector].sector_busy = 0;
  }
  
  return NOR_OPERATION_OK;
}

//     
//   LAT ,  sector     
NOR_address CNOR_ftl::Find_fizical_address_(uint32_t logical_address)
{
  NOR_address retaddr;
  //   
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    if( (LAT[i].sector_address == (logical_address/(NOR_FTL_SECTOR_SIZE - sizeof(NOR_spare_header)))) && (LAT[i].sector_busy)) 
    {  
      retaddr.sector = i;
      break;
    }
  }
  //      
  retaddr.byte = logical_address%(NOR_FTL_SECTOR_SIZE - sizeof(NOR_spare_header)) + sizeof(NOR_spare_header);
  
  return retaddr;
}

//     
NORresult CNOR_ftl::Find_fizical_address(uint32_t logical_addr, NOR_address* fizical_addr)
{  
  //   
  const uint32_t sector_address = logical_addr/(NOR_FTL_SECTOR_SIZE - sizeof(NOR_spare_header));
  
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    if( LAT[i].sector_busy && LAT[i].sector_address == sector_address ) 
    {
      //__PRINTF("%s: logical_addr: %u, sector_address: %u\n", __func__, logical_addr, sector_address);
      fizical_addr->sector = i;
      
      //      
      fizical_addr->byte = logical_addr%(NOR_FTL_SECTOR_SIZE - sizeof(NOR_spare_header)) + sizeof(NOR_spare_header);
      
      return NOR_OPERATION_OK;
    }
  }
  
  __FTL_WRITE_DEBUG("%s: addr %u not found\n", __func__, logical_addr);
  
  return NOR_OPERATION_FAILED;
}

//   
uint16_t CNOR_ftl::Find_free_sector(uint16_t current_sector)
{
  for(uint16_t i = current_sector; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    if(!LAT[i].sector_busy) return i;
  }
  for(uint16_t i = 0; i < current_sector; i++)
  {
    if(!LAT[i].sector_busy) return i;
  }
  //     (   )
  __FTL_WRITE_DEBUG("%s: not found, current_sector: %u\n", __func__, current_sector);
  while(1);
}

bool CNOR_ftl::Find_free_sector_(uint16_t current_sector, uint16_t* free_sector)
{
  for(uint16_t i = current_sector; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    if(!LAT[i].sector_busy)
    {
      free_sector[0] = i;
      return true;
    }
  }
  for(uint16_t i = 0; i < current_sector; i++)
  {
    if(!LAT[i].sector_busy)
    {
      free_sector[0] = i;
      return true;
    }
  }
  //     (   )
  __FTL_WRITE_DEBUG("%s: not found, current_sector: %u\n", __func__, current_sector);
  
  return false;
}

//        (c    )
void CNOR_ftl::Move_sector(uint16_t source, uint16_t dest, const uint8_t* buff, uint16_t size, uint16_t addr_in_sector)
{
  NOT_USED(uint32_t debug_timer;)
  
  //static volatile uint8_t dgb=0; if(dest==0) {dgb++;}
  //     3  dest==0,        ,
  //    busy.          .
  
  // ,    
  NOR_spare_header source_header;
  NOR_spare_header dest_header;
  //   
  mem_read(dest, 0, (uint8_t*)&dest_header, sizeof(source_header)); 
  mem_read(source, 0, (uint8_t*)&source_header, sizeof(source_header)); 
  //       
  dest_header.sector_address = source_header.sector_address;
  //   ! //     ,    

  // 1.   
  __FTL_WRITE_DEBUG("%s(1): -> mem_erase_sector(%u)\n", __func__, dest); debug_timer=__GET_MS_COUNTER();
  mem_erase_sector(dest); // 74 ms
  __FTL_WRITE_DEBUG("%s(1): <- mem_erase_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  
  //   ! //    (  ),        ,   

  // 2.       
  dest_header.wear_level++;
  __FTL_WRITE_DEBUG("%s(2): -> mem_write(%u, %u, %p, %u)\n", __func__, dest, NOR_FTL_WEAR_LEVEL_OFFSET, (uint8_t*)&dest_header.wear_level, sizeof(dest_header.wear_level)); debug_timer=__GET_MS_COUNTER();
  mem_write(dest, NOR_FTL_WEAR_LEVEL_OFFSET, (uint8_t*)&dest_header.wear_level, sizeof(dest_header.wear_level)); //1 ms
  __FTL_WRITE_DEBUG("%s(2): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  //   ! //      .     .     

  // 3.          
  uint8_t move_start = 0;
  __FTL_WRITE_DEBUG("%s(3.1): -> mem_write(%u, %u, %p, %u)\n", __func__, source, NOR_FTL_SECTOR_OUT_MOVE_START_OFFSET, &move_start, 1); debug_timer=__GET_MS_COUNTER();
  mem_write(source, NOR_FTL_SECTOR_OUT_MOVE_START_OFFSET, &move_start, 1); //1 ms
  __FTL_WRITE_DEBUG("%s(3.1): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  //   ! //         (   )
  //    
  uint16_t buff_index = 0;
  for(uint16_t i = sizeof(NOR_spare_header); i < NOR_FTL_SECTOR_SIZE;) // 31 ms
  {
    // ,      
    uint16_t rw_size = sizeof(move_buffer);
    if(i+sizeof(move_buffer) >= NOR_FTL_SECTOR_SIZE) rw_size = NOR_FTL_SECTOR_SIZE - i;
    //       
    mem_read(source, i, move_buffer, rw_size);
    if(size && buff != NULL)
    {
      //   
      for(uint16_t x = i; x < i + rw_size; x++)
      {
        if(x >= addr_in_sector && x < addr_in_sector + size) 
        {
          move_buffer[x-i] = buff[buff_index];
          buff_index++;  
        }
      }
    }
    //     
    __FTL_WRITE_DEBUG("%s(3.2): -> mem_write(%u, %u, %p, %u)\n", __func__, dest, i, move_buffer, rw_size); debug_timer=__GET_MS_COUNTER();
    mem_write(dest, i, move_buffer, rw_size);
    __FTL_WRITE_DEBUG("%s(3.2): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    i += rw_size;
  }
  
  //      
  __FTL_WRITE_DEBUG("%s(3.3): -> mem_write(%u, %u, %p, %u)\n", __func__, dest, NOR_FTL_SECTOR_ADDRESS_OFFSET, (uint8_t*)&dest_header.sector_address, sizeof(dest_header.sector_address)); debug_timer=__GET_MS_COUNTER();
  mem_write(dest, NOR_FTL_SECTOR_ADDRESS_OFFSET, (uint8_t*)&dest_header.sector_address, sizeof(dest_header.sector_address));
  __FTL_WRITE_DEBUG("%s(3.3): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  //   ! //         (   )

  //    
  dest_header.sector_in_move_end = 0x00;
  __FTL_WRITE_DEBUG("%s(3.4): -> mem_write(%u, %u, %p, %u)\n", __func__, dest, NOR_FTL_SECTOR_IN_MOVE_END_OFFSET, (uint8_t*)&dest_header.sector_in_move_end, sizeof(dest_header.sector_in_move_end)); debug_timer=__GET_MS_COUNTER();
  mem_write(dest, NOR_FTL_SECTOR_IN_MOVE_END_OFFSET, (uint8_t*)&dest_header.sector_in_move_end, sizeof(dest_header.sector_in_move_end)); 
  __FTL_WRITE_DEBUG("%s(3.4): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  //   ! //      .     .
  // 4.   
  const uint8_t busy = 0;
  __FTL_WRITE_DEBUG("%s(4): -> mem_write(%u, %u, % p, %u)\n", __func__, source, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, 1); debug_timer=__GET_MS_COUNTER();
  mem_write(source, NOR_FTL_SECTOR_BUSY_OFFSET, &busy, 1);
  __FTL_WRITE_DEBUG("%s(4): <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  //   ! //     ,   
  
  // 6.     LAT
  LAT[source].sector_busy = 0;
  LAT[source].sector_address = 0;
  LAT[dest].sector_busy = 1;
  LAT[dest].sector_address = dest_header.sector_address;
}

//      (    )
void CNOR_ftl::Write_sector(uint16_t sector, uint16_t address, uint8_t* buff, uint16_t size)
{
  NOT_USED(uint32_t debug_timer;)
  
  bool need_to_move_sector = 0;
  
  //   
  if(address + size > NOR_FTL_SECTOR_SIZE) while(1);
  //      (        )
  for(uint16_t i = address; i < address + size; i++)
  {
    uint8_t tmp_byte;
    
    mem_read(sector, i, &tmp_byte, 1);
    if(tmp_byte != 0xFF)
    {
      need_to_move_sector = 1;
      break;
    }
  }
  
  if(need_to_move_sector)
  {
    uint16_t dest = Find_free_sector(sector);
    //         
    __FTL_WRITE_DEBUG("%s: -> Move_sector(%u, %u, %u, %u)\n", __func__, sector, dest, size, address); debug_timer=__GET_MS_COUNTER();
    Move_sector(sector, dest, buff, size, address);
    __FTL_WRITE_DEBUG("%s: <- Move_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  }
  else
  {
    //    ,     
    __FTL_WRITE_DEBUG("%s: -> mem_write(%u, %u, %p, %u)\n", __func__, sector, address, buff, size); debug_timer=__GET_MS_COUNTER();
    mem_write(sector, address, buff, size);
    __FTL_WRITE_DEBUG("%s: <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
  }
}

// ,     FTL
bool CNOR_ftl::Is_memory_formatted(void)
{
  NOR_spare_header header;
  uint16_t not_weared_sectors = 0;
  
  __FTL_WRITE_DEBUG("NOR spare:\nindex\tbusy\taddr\twlevel\tmove_start\tmove_end\n")
  
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    mem_read(i, 0, (uint8_t*)&header, sizeof(header));
    if(header.wear_level == 0xFFFFFFFF) not_weared_sectors++;
    
    __FTL_WRITE_DEBUG("%3.3u\t%2.2X\t%u\t%8.8X\t%2.2X\t%2.2X\n", i, header.sector_busy, header.sector_address, header.wear_level, header.sector_out_move_start, header.sector_in_move_end);
  }
  
  if(not_weared_sectors > 1) return false; //       ,   ,       
  return true;
}

#include "debug_port.h"
//      
bool CNOR_ftl::Check_LAT(uint16_t* addr_dublication, uint16_t* not_found_addr, uint8_t* range_ok)
{
  /*
  static const uint8_t bad_lat[768*2] =
  {
    0x07, 0x03, 0x4B, 0x02, 0x85, 0x00, 0xAB, 0x04, 0x1D, 0x01, 0xA9, 0x02, 0x43, 0x05, 0xA9, 0x01, 0x19, 0x01, 0x5D, 0x01, 0x4B, 0x00, 0x6B, 0x01, 0x89, 0x04, 0xF7, 0x00, 0x3B, 0x03, 0xAB, 0x01, 0xB9, 0x05, 0x79, 0x02, 0xA1, 0x03, 0xC1, 0x04, 0xA3, 0x03, 0x0F, 0x00, 0xE1, 0x04, 0x43, 0x00, 0x65, 0x01, 0x43, 0x02, 0xBF, 0x00, 0x1B, 0x04, 0x67, 0x02, 0x97, 0x05, 0xD1, 0x00, 0x31, 0x00, 0xE7, 0x03, 0xB9, 0x02, 0x29, 0x01, 0xD3, 0x03, 0x33, 0x02, 0x3B, 0x04, 0x81, 0x04, 0x87, 0x03, 0x35, 0x02, 0x81, 0x03, 0x07, 0x04, 0x1B, 0x05, 0xDB, 0x03, 0x73, 0x02, 0xDB, 0x02, 0x61, 0x03, 0x3B, 0x01, 0x6F, 0x02, 0xA3, 0x04, 0xA1, 0x05, 0x47, 0x03, 0xB3, 0x03, 0x83, 0x03, 0x49, 0x03, 0x35, 0x05, 0x53, 0x05, 0x6D, 0x01, 0x1D, 0x03, 0x37, 0x00, 0x89, 0x01, 0x2B, 0x00, 0x93, 0x01, 0xFF, 0x04, 0x2B, 0x05, 0xB5, 0x02, 0x93, 0x04, 0x07, 0x02, 0xFD, 0x04, 0xD9, 0x05, 0x1B, 0x03, 0xAB, 0x05, 0x55, 0x00, 0x55, 0x03, 0x33, 0x03, 0x5F, 0x01, 0x57, 0x04, 0xEF, 0x03, 0xC1, 0x01, 0x4F, 0x03, 0x03, 0x03, 0xBD, 0x01, 0xEF, 0x00, 0x99, 0x01, 0x77, 0x04, 0x3F, 0x05, 0x99, 0x05, 0x37, 0x03, 0xCF, 0x04, 0x39, 0x01, 0xBD, 0x00, 0xA7, 0x03, 0x05, 0x02, 0x37, 0x04, 0x8F, 0x00,
    0xE5, 0x03, 0x2F, 0x02, 0x81, 0x05, 0xBF, 0x03, 0xD7, 0x04, 0xA9, 0x04, 0x7F, 0x02, 0xFF, 0x02, 0xE3, 0x01, 0x99, 0x02, 0xC9, 0x00, 0x53, 0x01, 0xD3, 0x05, 0x0D, 0x04, 0x01, 0x02, 0xF7, 0x01, 0x29, 0x05, 0xBD, 0x04, 0x75, 0x03, 0x53, 0x03, 0x15, 0x04, 0x8B, 0x05, 0x4D, 0x05, 0x5B, 0x01, 0xF1, 0x03, 0xD9, 0x01, 0x79, 0x03, 0xF9, 0x04, 0xAB, 0x02, 0x4F, 0x01, 0xBD, 0x03, 0x05, 0x00, 0x7B, 0x05, 0x6B, 0x04, 0xE5, 0x04, 0x13, 0x00, 0xDF, 0x02, 0xA3, 0x02, 0x53, 0x02, 0x47, 0x04, 0x0B, 0x00, 0x3F, 0x00, 0x45, 0x01, 0xE5, 0x00, 0xC7, 0x03, 0x0F, 0x04, 0x41, 0x02, 0xBF, 0x01, 0xC7, 0x01, 0x63, 0x04, 0x4B, 0x04, 0x1D, 0x05, 0xAB, 0x03, 0x51, 0x03, 0xC5, 0x02, 0xFB, 0x02, 0x29, 0x03, 0x2B, 0x03, 0xFF, 0x00, 0x21, 0x04, 0x09, 0x01, 0x09, 0x04, 0x2D, 0x04, 0xC7, 0x05, 0x71, 0x01, 0xF9, 0x02, 0xBD, 0x02, 0x4F, 0x04, 0x4F, 0x02, 0x65, 0x05, 0x5D, 0x00, 0x9B, 0x03, 0x39, 0x05, 0x8F, 0x01, 0x87, 0x01, 0x9D, 0x00, 0x49, 0x04, 0x59, 0x05, 0xB1, 0x01, 0x49, 0x01, 0xF5, 0x00, 0x7D, 0x00, 0x5F, 0x03, 0x57, 0x02, 0xAD, 0x02, 0x7F, 0x05, 0x6F, 0x01, 0xEB, 0x04, 0xC3, 0x04, 0xA7, 0x05, 0x23, 0x05, 0x39, 0x04, 0xE5, 0x01, 0x2D, 0x03, 0xB5, 0x05, 0xC9, 0x02,
    0x87, 0x02, 0xD3, 0x04, 0x7B, 0x04, 0x13, 0x03, 0x45, 0x05, 0xC3, 0x01, 0x91, 0x04, 0x27, 0x00, 0x27, 0x02, 0x97, 0x00, 0x59, 0x00, 0x95, 0x05, 0x05, 0x04, 0xDB, 0x05, 0xB5, 0x00, 0x41, 0x04, 0x4F, 0x00, 0x73, 0x05, 0xA3, 0x00, 0x8D, 0x05, 0x25, 0x03, 0x2B, 0x04, 0xC3, 0x02, 0xD7, 0x01, 0x91, 0x05, 0x83, 0x02, 0x71, 0x03, 0x9F, 0x00, 0x99, 0x00, 0x1D, 0x00, 0x21, 0x00, 0xA1, 0x00, 0xAD, 0x00, 0x9B, 0x00, 0xBB, 0x00, 0x23, 0x00, 0x8D, 0x00, 0x3D, 0x04, 0xE3, 0x03, 0x1B, 0x04, 0xA5, 0x00, 0xFD, 0x00, 0xA7, 0x00, 0xA1, 0x02, 0xBB, 0x04, 0x31, 0x03, 0xD3, 0x00, 0x2B, 0x02, 0xBF, 0x02, 0x87, 0x04, 0xB7, 0x00, 0xD5, 0x02, 0x1F, 0x05, 0x61, 0x02, 0xF1, 0x02, 0x99, 0x03, 0x1B, 0x01, 0xF3, 0x02, 0xCB, 0x00, 0xC1, 0x03, 0x77, 0x02, 0x6B, 0x05, 0x1F, 0x00, 0x0F, 0x03, 0x53, 0x04, 0x3F, 0x03, 0x63, 0x03, 0x81, 0x02, 0x1B, 0x00, 0x65, 0x04, 0xE1, 0x01, 0x6B, 0x02, 0x8F, 0x05, 0xA9, 0x00, 0xC7, 0x02, 0xA5, 0x05, 0x23, 0x04, 0x6F, 0x05, 0x81, 0x01, 0xA1, 0x04, 0xBB, 0x03, 0x51, 0x04, 0x33, 0x00, 0x21, 0x02, 0x5B, 0x05, 0x19, 0x05, 0xA9, 0x03, 0x3D, 0x03, 0x25, 0x00, 0x8B, 0x01, 0x39, 0x00, 0x4B, 0x03, 0xEF, 0x04, 0xCD, 0x02, 0xAD, 0x05, 0x01, 0x04,
    0x03, 0x04, 0xCB, 0x00, 0x87, 0x00, 0x8D, 0x03, 0x79, 0x04, 0x91, 0x00, 0xDB, 0x00, 0x09, 0x05, 0xFB, 0x01, 0x47, 0x01, 0x45, 0x03, 0xD7, 0x03, 0xD9, 0x00, 0xF9, 0x00, 0xBB, 0x02, 0xED, 0x01, 0xB3, 0x04, 0xF5, 0x01, 0x35, 0x00, 0xC3, 0x03, 0xFB, 0x00, 0xB1, 0x04, 0x5B, 0x03, 0x83, 0x04, 0x73, 0x01, 0xA7, 0x04, 0x39, 0x03, 0xC5, 0x04, 0x45, 0x02, 0x67, 0x03, 0x0B, 0x04, 0x07, 0x01, 0x3D, 0x01, 0x00, 0x00, 0x25, 0x05, 0xB7, 0x05, 0xA5, 0x01, 0x29, 0x04, 0x9F, 0x04, 0xEB, 0x03, 0x3B, 0x00, 0x49, 0x05, 0x39, 0x02, 0x00, 0x00, 0xB3, 0x02, 0xAD, 0x04, 0x9B, 0x04, 0x29, 0x00, 0x4F, 0x05, 0x55, 0x05, 0x57, 0x00, 0xA9, 0x05, 0x5B, 0x00, 0x21, 0x01, 0x89, 0x03, 0x7F, 0x03, 0xDB, 0x04, 0x8B, 0x00, 0x23, 0x02, 0x89, 0x00, 0x25, 0x02, 0xED, 0x00, 0xE9, 0x00, 0xAB, 0x00, 0x2F, 0x03, 0x3D, 0x05, 0x7B, 0x02, 0x6F, 0x00, 0xC5, 0x05, 0xF9, 0x03, 0xB3, 0x01, 0x71, 0x02, 0x2F, 0x01, 0x51, 0x01, 0x77, 0x01, 0xC3, 0x05, 0xAF, 0x00, 0xD5, 0x05, 0x2D, 0x01, 0x0B, 0x02, 0x00, 0x00, 0xED, 0x03, 0x00, 0x00, 0x5F, 0x02, 0x57, 0x03, 0x67, 0x04, 0x5D, 0x03, 0xB1, 0x00, 0xFB, 0x03, 0xF3, 0x03, 0x43, 0x01, 0x00, 0x00, 0x57, 0x01, 0xD1, 0x02, 0x00, 0x00, 0x00, 0x00,
    0xC1, 0x05, 0x73, 0x03, 0x41, 0x05, 0x55, 0x04, 0xFD, 0x02, 0x5D, 0x02, 0x59, 0x02, 0xB5, 0x04, 0x99, 0x04, 0x3D, 0x00, 0x00, 0x00, 0x93, 0x03, 0x5F, 0x04, 0x49, 0x02, 0x75, 0x00, 0x03, 0x01, 0xD9, 0x03, 0x73, 0x00, 0x49, 0x00, 0x55, 0x01, 0x07, 0x05, 0x0B, 0x01, 0x71, 0x00, 0x2B, 0x01, 0x91, 0x03, 0xCD, 0x01, 0xE3, 0x02, 0x00, 0x00, 0x31, 0x01, 0x13, 0x01, 0xE7, 0x02, 0x5B, 0x02, 0xD7, 0x05, 0x15, 0x01, 0xB7, 0x02, 0xF3, 0x04, 0x00, 0x00, 0xF7, 0x02, 0xEF, 0x02, 0x81, 0x00, 0x00, 0x00, 0x7F, 0x00, 0x00, 0x00, 0xCB, 0x04, 0x0F, 0x02, 0x91, 0x02, 0xB3, 0x00, 0x51, 0x02, 0xE5, 0x02, 0x85, 0x01, 0x01, 0x01, 0x17, 0x02, 0x6F, 0x03, 0x97, 0x04, 0x69, 0x04, 0x23, 0x03, 0xF9, 0x01, 0x59, 0x04, 0x13, 0x04, 0xF5, 0x02, 0x77, 0x05, 0x37, 0x02, 0xC7, 0x04, 0x93, 0x00, 0x45, 0x00, 0x03, 0x05, 0xA7, 0x01, 0x69, 0x03, 0x4D, 0x01, 0x6D, 0x02, 0x8D, 0x04, 0x01, 0x03, 0x23, 0x01, 0x17, 0x03, 0x97, 0x01, 0x9D, 0x03, 0x15, 0x00, 0x17, 0x04, 0xF7, 0x03, 0xA7, 0x02, 0x35, 0x03, 0x9B, 0x02, 0x97, 0x03, 0x61, 0x05, 0x95, 0x02, 0xCF, 0x01, 0xA5, 0x03, 0x8D, 0x01, 0xD5, 0x01, 0xBB, 0x05, 0xCB, 0x03, 0xCD, 0x00, 0xCD, 0x04, 0x43, 0x04, 0x8F, 0x04, 0x51, 0x00,
    0x8D, 0x02, 0x61, 0x01, 0x11, 0x03, 0x75, 0x01, 0xFB, 0x04, 0xD1, 0x03, 0xC1, 0x00, 0x05, 0x03, 0x47, 0x02, 0x47, 0x00, 0x11, 0x02, 0x19, 0x04, 0x13, 0x05, 0xE9, 0x03, 0x77, 0x00, 0x85, 0x04, 0xAF, 0x02, 0x31, 0x04, 0xC5, 0x03, 0x89, 0x02, 0x1F, 0x03, 0xEB, 0x01, 0x5D, 0x04, 0x1F, 0x04, 0xD1, 0x05, 0x19, 0x00, 0x2F, 0x05, 0xC5, 0x01, 0x69, 0x00, 0x61, 0x00, 0x4D, 0x04, 0x95, 0x04, 0x87, 0x05, 0x03, 0x02, 0x37, 0x05, 0xFD, 0x03, 0x65, 0x03, 0xD5, 0x04, 0x63, 0x02, 0x45, 0x04, 0x73, 0x04, 0xAF, 0x03, 0xCB, 0x05, 0xCD, 0x03, 0xB1, 0x05, 0x8F, 0x03, 0x3B, 0x02, 0x47, 0x05, 0xCF, 0x03, 0x6D, 0x04, 0x7B, 0x01, 0xCB, 0x02, 0x4D, 0x02, 0xDF, 0x03, 0x2F, 0x00, 0x3F, 0x04, 0x15, 0x03, 0x15, 0x05, 0xDD, 0x05, 0x6B, 0x03, 0x0F, 0x05, 0xF1, 0x04, 0x9F, 0x05, 0x3B, 0x05, 0x59, 0x03, 0xED, 0x04, 0xE1, 0x03, 0xAD, 0x03, 0xF1, 0x00, 0x0B, 0x03, 0xF1, 0x01, 0x33, 0x04, 0xB5, 0x01, 0x01, 0x05, 0xED, 0x02, 0x43, 0x03, 0xC9, 0x05, 0xB9, 0x03, 0x67, 0x00, 0xE3, 0x00, 0xC7, 0x00, 0x93, 0x05, 0x1D, 0x02, 0x5B, 0x04, 0x4D, 0x03, 0xD3, 0x01, 0x4B, 0x01, 0x7D, 0x05, 0xC9, 0x01, 0x75, 0x02, 0xD3, 0x02, 0x95, 0x03, 0x1B, 0x04, 0xEB, 0x02, 0x35, 0x04, 0xC1, 0x02,
    0xE7, 0x00, 0x2D, 0x02, 0x89, 0x05, 0x65, 0x02, 0xF5, 0x03, 0xE3, 0x04, 0x13, 0x02, 0xA3, 0x05, 0x95, 0x01, 0xE7, 0x01, 0xD5, 0x00, 0x5F, 0x05, 0x0D, 0x01, 0xA1, 0x01, 0x0B, 0x05, 0xEB, 0x00, 0x6F, 0x04, 0xFF, 0x03, 0xA5, 0x04, 0xB9, 0x00, 0xE1, 0x00, 0x6D, 0x05, 0xCD, 0x05, 0x5D, 0x05, 0x33, 0x05, 0xF3, 0x00, 0xE1, 0x02, 0x15, 0x02, 0x9F, 0x02, 0x2D, 0x05, 0x83, 0x00, 0xFD, 0x01, 0x75, 0x05, 0x53, 0x00, 0x77, 0x03, 0x11, 0x01, 0x8B, 0x04, 0xBF, 0x04, 0x75, 0x04, 0xC9, 0x04, 0x95, 0x00, 0x8B, 0x02, 0x7F, 0x04, 0x2F, 0x04, 0xD7, 0x00, 0x9F, 0x01, 0x31, 0x05, 0x97, 0x02, 0x63, 0x01, 0xB3, 0x05, 0x79, 0x01, 0xCF, 0x02, 0x59, 0x04, 0xF7, 0x04, 0x9D, 0x02, 0xAD, 0x01, 0x59, 0x01, 0x2D, 0x00, 0xDD, 0x04, 0x57, 0x05, 0x9D, 0x04, 0x3D, 0x02, 0x69, 0x01, 0xA3, 0x01, 0xDD, 0x01, 0xC9, 0x03, 0xDF, 0x04, 0xEF, 0x01, 0x7D, 0x02, 0x19, 0x03, 0xD9, 0x04, 0x6D, 0x00, 0x69, 0x05, 0xE7, 0x04, 0x41, 0x03, 0xD1, 0x04, 0x1F, 0x01, 0x63, 0x00, 0xD7, 0x02, 0x1F, 0x02, 0xC3, 0x00, 0x41, 0x00, 0x61, 0x04, 0xFF, 0x01, 0x8F, 0x02, 0x7D, 0x03, 0x83, 0x01, 0x17, 0x05, 0x25, 0x04, 0xB9, 0x04, 0x93, 0x02, 0x27, 0x03, 0x35, 0x01, 0x1B, 0x02, 0x11, 0x00, 0x69, 0x02,
    0x63, 0x05, 0x83, 0x05, 0x65, 0x00, 0x51, 0x05, 0x21, 0x05, 0xB5, 0x03, 0x7B, 0x03, 0xBB, 0x01, 0x09, 0x00, 0x9D, 0x01, 0x09, 0x02, 0xD1, 0x01, 0x09, 0x03, 0xB1, 0x02, 0xE9, 0x02, 0x7D, 0x04, 0x4B, 0x05, 0x25, 0x01, 0x79, 0x00, 0x8B, 0x03, 0x71, 0x04, 0x29, 0x02, 0x3F, 0x01, 0x85, 0x05, 0xDD, 0x03, 0xF3, 0x01, 0xCB, 0x01, 0x7F, 0x01, 0x6D, 0x03, 0xD5, 0x03, 0x27, 0x04, 0xDB, 0x01, 0xDF, 0x05, 0x0D, 0x03, 0xE9, 0x01, 0x9B, 0x05, 0x07, 0x00, 0xB7, 0x04, 0xCF, 0x00, 0x9D, 0x05, 0x7D, 0x01, 0xA5, 0x02, 0xAF, 0x01, 0x41, 0x01, 0x1D, 0x04, 0x33, 0x01, 0xCB, 0x04, 0x55, 0x02, 0x0D, 0x05, 0xB7, 0x01, 0x4D, 0x00, 0xC5, 0x00, 0x0D, 0x02, 0xB9, 0x01, 0x11, 0x05, 0x0F, 0x01, 0xAF, 0x05, 0xDD, 0x02, 0xB1, 0x03, 0x9F, 0x03, 0xDF, 0x01, 0x5F, 0x00, 0x31, 0x02, 0xE9, 0x04, 0x19, 0x02, 0xD9, 0x02, 0xB7, 0x03, 0x27, 0x01, 0xDF, 0x00, 0x79, 0x05, 0xBF, 0x05, 0x91, 0x01, 0x7B, 0x00, 0xF5, 0x04, 0x9B, 0x01, 0x85, 0x03, 0x17, 0x01, 0x67, 0x01, 0x17, 0x00, 0x05, 0x01, 0xCF, 0x05, 0x21, 0x03, 0x03, 0x00, 0x6B, 0x00, 0x27, 0x05, 0x05, 0x05, 0x71, 0x05, 0x85, 0x02, 0x3F, 0x02, 0xDD, 0x00, 0xBD, 0x05, 0x11, 0x04, 0x37, 0x01, 0x0D, 0x00, 0xAF, 0x04, 0x01, 0x00,
  };
  
  const NOR_LAT_element* const LAT = (const NOR_LAT_element* const)bad_lat;
  */
  
  addr_dublication[0] = 0;
  not_found_addr[0] = NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT;
  range_ok[0] = 1;
  
  uint16_t min = 0xFFFF;
  uint16_t max = 0; //  (NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT - NOL_FTL_FREE_SECTOR_COUNT - 1)?
  
  for(uint16_t i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
  {
    if(LAT[i].sector_busy)
    {
      if(min > LAT[i].sector_address)
      {
        min = LAT[i].sector_address;
      }
      
      if(max < LAT[i].sector_address)
      {
        max = LAT[i].sector_address;
      }
    }
    
    for(uint16_t j = 0; j < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; j++)
    {
      if(i != j && 
         LAT[i].sector_address == LAT[j].sector_address && 
         LAT[i].sector_busy && LAT[j].sector_busy)
      {
        addr_dublication[0]++;
        
        __PRINTF("found sector dublication: %u, index: %u %u\n", LAT[i].sector_address, i, j);
      }
    }
    //__PRINTF("i:%u\t addr:%u\t busy:%u\n", i, LAT[i].sector_address, LAT[i].sector_busy);
  }
  
  if(addr_dublication[0]%2) addr_dublication[0]++;
  
  addr_dublication[0]=addr_dublication[0]/2;
  
  if(min != 0 || max != (NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT - NOL_FTL_FREE_SECTOR_COUNT - 1))
  {
    range_ok[0] = 0;
    __PRINTF("wrong sector addr range, min: %u, max: %u\n", min, max);
  }
  
  if(max <= NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT && min < max)
  {
    not_found_addr[0] = 0;
    
    for(uint16_t sector_address = min; sector_address <= max; sector_address++)
    {
      uint16_t i;
      
      for(i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
      {
        if(LAT[i].sector_busy)
        {
          if(sector_address == LAT[i].sector_address) break;
        }
      }
      
      if(i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT && sector_address == LAT[i].sector_address) continue;
            
      not_found_addr[0]++;
     
     __PRINTF("not found sector: %u\n", sector_address);
     
     if(not_found_addr[0]>=NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT) break;
    }
  }
  
  if(!not_found_addr[0] && !addr_dublication[0] && range_ok[0])
  {
    return true;
  }
  else
  {
    return false;
  }
}

//      Load_LAT  bootloader   ,     
//  1     
// index busy    addr    wlevel          move_start      move_end
// XXX   00      YYY     00000F80        FF              00
bool CNOR_ftl::RecoverySingleLostSector(void)
{
  NOT_USED(uint32_t debug_timer;)
  
  static const uint16_t min = 0;
  static const uint16_t max = (NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT - NOL_FTL_FREE_SECTOR_COUNT - 1);
  
  lat_recovery_inf.event_flag = 1;
  
  //    
  for(uint16_t lost_sector = min; lost_sector <= max; lost_sector++)
  {
    uint16_t i;
    
    for(i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
    {
      if(LAT[i].sector_busy)
      {
        if(lost_sector == LAT[i].sector_address) break;
      }
    }
    
    if(i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT && lost_sector == LAT[i].sector_address) continue;
    
    lat_recovery_inf.sector = lost_sector;
    
    //   ,          (   ?    ?)
    for(i = 0; i < NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT; i++)
    {
      if(LAT[i].sector_busy == 0x00 && LAT[i].sector_address == lost_sector)
      {
        NOR_spare_header header;
        
        mem_read(i, 0, (uint8_t*)&header, sizeof(header));
        
        if(header.sector_busy == 0x00 && 
           header.sector_address == lost_sector && 
           header.wear_level != 0xFFFFFFFF && 
           header.sector_out_move_start == 0xFF && 
           header.sector_in_move_end == 0x00)
        {
          uint16_t free_sector;
          
          if(!Find_free_sector_(i+1, &free_sector))
          {
            break;
          }
          
          __FTL_WRITE_DEBUG("%s: -> Move_sector(%u, %u, %u, %u)\n", __func__, i, free_sector, 0, 0); debug_timer=__GET_MS_COUNTER();
          Move_sector(i, free_sector);
          __FTL_WRITE_DEBUG("%s: <- Move_sector %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
          
          __PRINTF("Recovery sector %u ok\n", lost_sector);
          
          lat_recovery_inf.with_corrupt = 0;
          lat_recovery_inf.status = 1;
          
          return true;
        }
      }
    }
       
    __PRINTF("Recovery sector %u failed\n", lost_sector);
    
    //   FTL   
    //    ,   
    if(lost_sector < 3) break;
    
    uint16_t free_sector;
    
    if(!Find_free_sector_(0, &free_sector))
    {
      break;
    }
    
    NOR_spare_header header;
    
    mem_read(free_sector, 0, (uint8_t*)&header, sizeof(header));
    
    __FTL_WRITE_DEBUG("%s: -> mem_erase_sector(%u)\n", __func__, free_sector); debug_timer=__GET_MS_COUNTER();
    mem_erase_sector(free_sector);
    __FTL_WRITE_DEBUG("%s: mem_erase_sector (%u ms) <-\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    LAT[free_sector].sector_address = lost_sector;
    LAT[free_sector].sector_busy = 1;
    
    header.wear_level++; if(header.wear_level == 0xFFFFFFFF) {header.wear_level = 0xFFFFFFFF/2;}
    header.sector_address = LAT[free_sector].sector_address;
    header.sector_busy = 0xFF;
    header.sector_out_move_start = 0xFF;
    header.sector_in_move_end = 0x00;
    
    __FTL_WRITE_DEBUG("%s: -> mem_write(%u, %u, %p, %u)\n", __func__, free_sector, 0, (uint8_t*)&header, sizeof(header)); debug_timer=__GET_MS_COUNTER();
    mem_write(free_sector, 0, (uint8_t*)&header, sizeof(header));
    __FTL_WRITE_DEBUG("%s: <- mem_write %u ms\n", __func__, __GET_MS_COUNTER()-debug_timer);
    
    __PRINTF("Recovery sector %u ok (data corrupted)\n", lost_sector);
    
    lat_recovery_inf.with_corrupt = 1;
    lat_recovery_inf.status = 1;
    
    return true;
  }
  
  lat_recovery_inf.status = 0;
  
  __PRINTF("Recovery failed\n");
  
  return false;
}