/**
  ******************************************************************************
  * File Name          : NOR_FTL.h
  * Description        :       
  *                      flash-      
  *                      
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef NOR_FTL_H
#define NOR_FTL_H

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include <intrinsics.h>
#include "NOR_FTL_config.h"

#ifdef USE_FREERTOS
#include "FreeRTOS.h"
#include "semphr.h" 
#endif

/* Defines- ------------------------------------------------------------------*/
//         
//         
//  (static wear leveling)
#define NOR_FTL_MAXIMUM_WEAR_DISBALANCE 10
#define NOL_FTL_FREE_SECTOR_COUNT       16

//      
#define NOR_FTL_SECTOR_BUSY_OFFSET            0
#define NOR_FTL_SECTOR_ADDRESS_OFFSET         1
#define NOR_FTL_WEAR_LEVEL_OFFSET             3
#define NOR_FTL_SECTOR_OUT_MOVE_START_OFFSET  7
#define NOR_FTL_SECTOR_IN_MOVE_END_OFFSET     8

//     NOR
typedef enum
{
  NOR_OPERATION_OK     =  1,
  NOR_OPERATION_FAILED =  0,
}NORresult;

//    NOR
typedef struct
{
  uint16_t sector;
  uint16_t byte;
}NOR_address;

//       (LAT = logical address table)
__packed typedef struct
{
  uint8_t  sector_busy    : 1;
#if   (NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT > 8191)
#error 
#elif (NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT > 4095)
  uint16_t sector_address : 13;
  uint8_t  reserved       : 2 ;
#else //<=4095
  uint16_t sector_address : 12;
  uint8_t  reserved       : 3 ;
#endif
}NOR_LAT_element; 

__packed typedef struct
{
  uint8_t sector_busy;
  uint16_t sector_address;
  uint32_t wear_level;
  uint8_t sector_out_move_start;
  uint8_t sector_in_move_end;
}NOR_spare_header; 

__packed typedef struct
{
  uint16_t event_flag:1;
  uint16_t with_corrupt:1;
  uint16_t status:1;
  uint16_t sector:13;
}lat_recovery_inf_t;

//  ,      
typedef uint8_t (*NOR_driver_init_t)(void);
typedef void (*NOR_driver_deinit_t)(void);
typedef void (*NOR_driver_write_t)(uint16_t sector, uint16_t address, const uint8_t* data, uint16_t size);
typedef void (*NOR_driver_read_t)(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size);
typedef void (*NOR_driver_erase_sector_t)(uint16_t sector);
typedef void (*NOR_driver_erase_chip_t)(void);


/* C++ code-------------------------------------------------------------------*/
class CNOR_ftl
{
  public:
  // 
  CNOR_ftl(NOR_driver_init_t init, NOR_driver_deinit_t deinit,\
           NOR_driver_write_t write, NOR_driver_read_t read,\
           NOR_driver_erase_sector_t erase_sector,\
           NOR_driver_erase_chip_t erase_chip);
  //  
  NORresult Init(void);
  //  
  void Deinit(void);
  //  
  NORresult Write_data(uint32_t logical_address, uint8_t* buff, uint16_t size);
  //  
  NORresult Read_data(uint32_t logical_address, uint8_t* buff, uint16_t size);  
  //    ( )
  void Static_wear_leveller(void);
  //    
  uint32_t Get_memory_size(void);
  
  //     
  NORresult Direct_write_data(uint16_t sector,uint16_t address, uint8_t* buff, uint16_t size);
  //     
  NORresult Direct_read_data(uint16_t sector,uint16_t address, uint8_t* buff, uint16_t size);  
  //    (     )
  NORresult Direct_erase_sector(uint16_t sector);  
  //  
  NORresult Format_memory(void);
  
  //   ,   
  bool Check_LAT(uint16_t* addr_dublication, uint16_t* not_found_addr, uint8_t* range_ok);
  
  //    
  inline const lat_recovery_inf_t* Get_Recovery_Inf(void) {return &lat_recovery_inf;}
  //    LAT
  inline bool Get_memory_LAT_formated_flag(void) {return LAT_formatted_flag;}
private:
  //   
  NOR_LAT_element LAT[NOR_FTL_WEAR_LEVELLED_SECTOR_COUNT];
  // ,     
  uint8_t move_buffer[256];
  //     
  NOR_driver_init_t mem_init;
  //     
  NOR_driver_deinit_t mem_deinit;
  //       
  NOR_driver_read_t mem_read;
  //       
  NOR_driver_write_t mem_write;
  //      
  NOR_driver_erase_sector_t mem_erase_sector;
  //     
  NOR_driver_erase_chip_t mem_erase_chip;
  //      
  NORresult Load_LAT(void);
  //     
  NOR_address Find_fizical_address_(uint32_t logical_address);
  NORresult Find_fizical_address(uint32_t logical_addr, NOR_address* fizical_addr);
  //   
  uint16_t Find_free_sector(uint16_t current_sector);
  bool Find_free_sector_(uint16_t current_sector, uint16_t* free_sector);
  //     
  void Move_sector(uint16_t source, uint16_t dest, const uint8_t* buff = NULL, uint16_t size = 0, uint16_t addr_in_sector = 0); 
  //      (    )
  void Write_sector(uint16_t sector, uint16_t address, uint8_t* buff, uint16_t size);
  // ,     FTL
  bool Is_memory_formatted(void);
  //   
  bool RecoverySingleLostSector(void);
  
  //   
  uint8_t LAT_formatted_flag;
  //    
  lat_recovery_inf_t lat_recovery_inf;
};



#endif //NOR_FTL_H