/**
  ******************************************************************************
  * File Name          : ONFI_NAND.h
  * Description        :      NAND 
  *                      
  *                      
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef ONFI_NAND_H
#define ONFI_NAND_H

/* Includes ------------------------------------------------------------------*/
#include "stm32f10x.h"

/* Defines- ------------------------------------------------------------------*/
//  ,    .
//            ,   
//    
#define BLOCKS_PER_UNIT    1024 
//    
#define PAGES_PER_BLOCK    64
//   
#define BYTES_PER_PAGE     2048
//       
#define SPARE_AREA_SIZE    64
//   
#define NUM_OF_ADDR_CYCLES 34
//   
#define NUM_OF_LUNS         1 
//  ,    bad 
#define RESERVED_BLOCKS     32

//   ,     NAND
#define NAND_MINIMUM_WRITE_SIZE  512
//   ,      
#define NAND_MINIMUM_ERASE_SIZE  BYTES_PER_PAGE*PAGES_PER_BLOCK

//    
#define FIRST_RESERVED_BLOCK (BLOCKS_PER_UNIT-RESERVED_BLOCKS)
//    ,    (  bad )
#define RESERVED_BLOCK_FREE     0xFFFA
#define RESERVED_BLOCK_NOT_USED 0xFFFB

//      FSMC    .  tCS, tCLS, tALS, tCLR 
//      
//
// tHCLK = 1/HCLK = 1/72000000 = 0.01us = 13ns
// (FSMC_SetupTime + 1) x tHCLK                                                            >= max (tCS, tCLS, tALS, tCLR, tAR) - tWP
// (FSMC_WaitSetupTime + 1) x tHCLK                                                        >= max (tWP,tRP)
// (FSMC_HiZSetupTime + 1) x tHCLK                                                         >= max(tCS, tALS, tCLS) + (tWP - tDS)
// (FSMC_HoldSetupTime + 1) x tHCLK                                                        >= max (tCH, tCLH, tALH)
// ((FSMC_WaitSetupTime + 1) + ( FSMC_HoldSetupTime + 1) + (FSMC_SetupTime + 1)) x tHCLK   >= max (tWC/RC)
//
// Considering the different timings in the FSMC and the memory, the equations become:
// WAIT must verify:
//
// (FSMC_WaitSetupTime + 1)* tHCLK                                                         >= (tREA + tsu(D-NOE))
// WAIT                                                                                   >= (tREA + tsu(D-NOE)) /tHCLK - 1

//   FSMC
#define FSMC_SETUP_TIME          0
#define FSMC_WAIT_SETUP_TIME     2
#define FSMC_HOLD_SETUP_TIME     1
#define FSMC_HIZ_SETUP_TIME      0

//  FSMC    NAND
#define NAND_CMD_AREA                   ((uint32_t)(1<<16))  /* A16 = CLE high */
#define NAND_ADDR_AREA                  ((uint32_t)(1<<17))  /* A17 = ALE high */
#define NAND_DATA_AREA                  ((uint32_t)0x00000000) 
#define NAND_BANK_ADDR                  ((uint32_t)0x70000000)
#define FSMC_BANK_NAND                  FSMC_Bank2_NAND
#define NAND_COMMAND_REGISTER  *(__IO uint8_t *)(NAND_BANK_ADDR | NAND_CMD_AREA)
#define NAND_ADDR_REGISTER     *(__IO uint8_t *)(NAND_BANK_ADDR | NAND_ADDR_AREA)
#define NAND_DATA_REGISTER     *(__IO uint8_t *)(NAND_BANK_ADDR | NAND_DATA_AREA)

//  NAND 
#define NAND_CMD_AREA_A            ((uint8_t)0x00)
#define NAND_CMD_AREA_B            ((uint8_t)0x01)
#define NAND_CMD_AREA_C            ((uint8_t)0x50)
#define NAND_CMD_AREA_TRUE1        ((uint8_t)0x30)
#define NAND_CMD_CACHE_READ1        ((uint8_t)0x00)
#define NAND_CMD_CACHE_READ2        ((uint8_t)0x31)
#define NAND_CMD_RANDOM_DATA_OUT1   ((uint8_t)0x05)
#define NAND_CMD_RANDOM_DATA_OUT2   ((uint8_t)0xE0)

#define NAND_CMD_WRITE0            ((uint8_t)0x80)
#define NAND_CMD_WRITE_TRUE1       ((uint8_t)0x10)  
#define NAND_CMD_ERASE0            ((uint8_t)0x60)
#define NAND_CMD_ERASE1            ((uint8_t)0xD0)  
#define NAND_CMD_READID            ((uint8_t)0x90)  
#define NAND_CMD_STATUS            ((uint8_t)0x70)
#define NAND_CMD_LOCK_STATUS       ((uint8_t)0x7A)
#define NAND_CMD_RESET             ((uint8_t)0xFF)
#define NAND_CMD_CACHE_PROGRAM0    ((uint8_t)0x80)
#define NAND_CMD_CACHE_PROGRAM1    ((uint8_t)0x15)
#define NAND_CMD_RANDOM_DATA_IN    ((uint8_t)0x85)

#define NAND_ONFI_SIGNATURE_ADDR   ((uint8_t)0x20)  
#define NAND_ONFI_PAGE_PARAM_READ  ((uint8_t)0xEC) 

//     NAND
typedef enum
{
  NAND_STATE_BUSY   =  0,
  NAND_OPERATION_OK =  1,
  NAND_WRITE_FAILED =  2,
  NAND_ERASE_FAILED =  3,
  NAND_READ_FAILED  =  4,
  NAND_NOT_DETECTED =  5,
  NAND_NOT_SUPPORTED = 6,
  NAND_LUT_FAILED    = 7, 
  NAND_SIZE_ERROR    = 8,  
  NAND_ADDRESS_ERROR = 9,  
  NAND_BLOCK_ERASED  = 10  
}NANDresult;

//   
typedef struct
{
  uint8_t  PageSignature[4];
  uint8_t  DeviceManufacturer[12];
  uint8_t  DeviceModel[20];
  uint32_t DataBytePerPage;
  uint16_t SpareBytePerPage;
  uint32_t NumberOfPagePerBlock;
  uint32_t NumberOfBlockPerUnit;
  uint8_t  NumberOfLogicalUnit;
  uint8_t  NumberOfAddrCycle;
  uint8_t  NumberOfBitsPerCell;
}NAND_Info;

//      
typedef enum
{
  S_NAND_PASSED=0, S_NAND_FAILED = !S_NAND_PASSED
} NAND_passed;

typedef enum
{
  S_NAND_BUSY=0, S_NAND_READY = !S_NAND_BUSY
} NAND_ready;

typedef enum
{
  S_NAND_PROTECTED=0, S_NAND_UNPROTECTED = !S_NAND_PROTECTED
} NAND_protected;

typedef enum
{
  BIT_01_CORRECTED = 0,
  BIT_2_CORRECTED = 2,
  BIT_3_CORRECTED = 1,
  BIT_4_CORRECTED = 3
} NAND_int_ecc_status;

//   
typedef union
{
  struct
  {
    NAND_passed         chip_status:1;
    NAND_passed         cache_program_result:1;
    uint8_t             not_used:1;
    NAND_int_ecc_status ecc_status:2;
    NAND_ready          RB1:1;
    NAND_ready          NAND_ready:1;
    NAND_protected      Protected:1;
  }bits;
  uint8_t byte;
}NAND_Status;

typedef struct
{
  uint16_t block;
  uint8_t  page;
  uint16_t byte;
}NAND_address;

typedef struct
{
  //   bad-
  uint16_t bad_blocks_count;
  //  BAD        
  uint16_t LUT[RESERVED_BLOCKS];
  uint8_t reserved[10];
}NAND_settings;


/* C++ code-------------------------------------------------------------------*/
class CNAND
{
  public:
  // 
  CNAND(void);
  //  
  void Power_on(void);
  //  
  void Power_off(void);  
  //  FSMC     
  NANDresult Init(void);
  //  
  void Deinit(void);

  //  
  NANDresult Write_data(uint32_t address, uint8_t* buff, uint16_t size);
  //  
  NANDresult Read_data(uint32_t address, uint8_t* buff, uint16_t size);  
  
  //    
  uint32_t Get_logical_size(void);
  
  //  
  NANDresult Read_page(NAND_address address, uint8_t* buff, uint16_t size);
  //  
  NANDresult Write_page(NAND_address address, uint8_t* buff, uint16_t size);
  //  
  NANDresult Erase_block(NAND_address address);

private:
  
  //  NAND,    
  NAND_settings settings;  
  //    
  NAND_Info info;
  //    
  NAND_Status status;
  //     ,    
  uint8_t Page_buffer[2112];
  
  //     
  NANDresult Read_ONFI(void);
  //   bad 
  NANDresult BuildLUT(void);
  //   
  void Get_status(void);
  //  bad   
  NANDresult Add_bad_block(uint16_t block_num);
  //     
  NAND_address Convert_address_to_logical(uint32_t address);
  //     
  NAND_address Convert_address_to_fizical(NAND_address address);
  //       
  void Copy_block(NAND_address source, NAND_address dest);
  //     
  void Send_address(NAND_address address, uint8_t start_cycle = 1, uint8_t stop_cycle = 4);

  
};

/* Global variables-----------------------------------------------------------*/
extern CNAND NAND;

#endif //NAND_BLACK_BOX_H