/**
  ******************************************************************************
  * File Name          : System_settings.h
  * Description        : Настройки системы
  *
  *
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef SYSTEM_H
#define SYSTEM_H

/* Includes  -----------------------------------------------------------------*/
#ifdef USE_FREERTOS
#include "FreeRTOS.h"
#include "semphr.h"
#endif
#include "time_utilities/time_utilities.h"
#include "serv_protocols_lib/vega/vega_protocol.h"
#include "system_config.h"
#include "system_typedefs.h"

#ifndef __ICCARM__
#include "getsystem.h"
#undef interface
#else

#if defined(NAVITERM_PRESENT)
#include "alarm_manager/alarm_manager.h"
#endif //defined(NAVITERM_PRESENT)
#endif

#include "can_signals/vega_can_typedefs.h"

#include "canpro/system_canlog_v2.h"

#if defined(CANLOG_V4_PRESENT)
#include "canpro/canlog_v4.h"
#endif //defined(CANLOG_V4_PRESENT)

#if defined(DM5002X_MANOMETR_PRESENT)
#include "manometer_dm5002x_lib/dm5002x.h"
#endif //defined(DM5002X_MANOMETR_PRESENT)

#if defined(BASE_STATION_BT_PRESENT)
#include "base_station_bt_lib/base_station_bt_lib/base_station_bt_protocol.h"
#endif //defined(BASE_STATION_BT_PRESENT)

#if defined(BLE_SENSORS_PRESENT)
#include "ble_sensors_lib/ble_sensors_lib.h"
#endif //defined(BLE_SENSORS_PRESENT)

#if defined(RS_TACHO_PRESENT)
#include "tacho_lib/common_tacho.h"
#endif //defined(RS_TACHO_PRESENT)

#if defined(SAVER2021_PRESENT)
#include "saver_lib/saver2021_lib.h"
#endif //defined(SAVER2021_PRESENT)

#define RUSSIAN_LANG           0
#define ENGLISH_LANG           1

#if defined(VEGA_HARD_PASS_PRESENT)
#define VEGA_PIN_SIZE   17
#else //
#define VEGA_PIN_SIZE   5
#endif //VEGA_HARD_PASS_PRESENT

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
#ifndef __ICCARM__
#include "../../lib/ble_nfc_ext_board_lib/ble_nfc_ext_board_protocol.h"
#else
#include "ble_nfc_ext_board_lib/ble_nfc_ext_board_protocol.h"
#endif
#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)

#define MAX_SENSOR_R_SIZE  (0xFF-sizeof(TVega_get_param_ans_args)) // Максимальный размер запрашиваемого параметра

/////////Параметры конфигурации системы ////////////////////////////////////////
#pragma pack(1)
// Настройки соединения
typedef struct
{
  char APN[30];
  char user[16];
  char password[16];
} TGSM_APN;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  SERVER_OFF = 0,
  EGTS = 1,
  WIALON = 2,
  VEGA = 3,
  NDTP = 4,
  WIALON_COMBINE = 5,
  TACHO_ARM = 6,
  NAVIS_PROTOCOL = 7,
  EGTS_LIGHT_1 = 8,
  EGTS_LIGHT_2 = 9,
  SMART_FLEET_V2 = 10,
}ESERVER_proto;

typedef struct
{
  char address[GSM_MAX_SERVER_ADDR_LEN];
  uint16_t connection_period;
  ESERVER_proto server_protocol;
} TGSM_server;

typedef struct
{
  uint8_t translate_with_track;
  uint16_t translate_period;
  uint8_t translate_with_change;
} TGSM_sensor_translation;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  PIN_UNUSED   = 0,
  PIN_DISABLED = 1,
  PIN_ENABLED  = 2,
}EConnection_pin_cfg;

typedef struct
{
  EConnection_pin_cfg   pin_cfg;
  char                  pin[VEGA_PIN_SIZE];
}TConnection_security;

typedef struct
{
  union
  {
    uint8_t _period;

    struct
    {
      uint8_t period:6;  //часы, 0 - выкл
      uint8_t reserve:2; //резерв
    };
  };

  char requst[10];   //запрос
}TBalance_request_settings;

typedef struct
{
  uint8_t disable_priority_messages;  //Запрет приоритетной отправки новых сообщений впреред сообщений из черного ящика.
  uint8_t vega_protocol_bblox_enable; //Разрешить запись ЧЯ для протокола VEGA, битовое поле
  uint8_t add_lbs_param;              //Добавлять набор параметров LBS в каждую запись ЧЯ
} TBlackBox_settings;

typedef struct
{
  TGSM_server server[MAX_SERVERS_COUNT];                         //Настройки сервера
  TGSM_APN APN[MAX_SIM_COUNT];                                   //APN
//#if defined(BALANCE_REQUEST_PRESENT)
  TBalance_request_settings balance_request_settings[MAX_SIM_COUNT]; //Настройки запроса баланса
//#endif //defined(BALANCE_REQUEST_PRESENT)
  TGSM_sensor_translation regular_translation[MAX_SENSOR_COUNT]; //Настройки регулярной передачи информации с датчиков на сервер
  uint32_t ndtp_address[MAX_SERVERS_COUNT];                      //Адрес терминала(для NDTP и WCOMBINE)
  TConnection_security security[MAX_SERVERS_COUNT];              //Настройки безопасности соединений
  uint8_t main_yandex_server_id;
} TConnection_settings;

typedef struct
{
  uint8_t spk_volume; //0%-100%
  uint8_t mic_gain;  //0-15
  uint8_t allow_all_incoming; //разрешать входящие со всех номеров
}TVoicecall_settings;

// Настройки трека
typedef struct
{
  uint16_t time_step;        // Шаг времени, в секундах. Если 0, данный фильтр не применяется
  uint16_t distance_step;    // Шаг пройденного пути, в метрах. Если 0, данный фильтр не применяется
  uint8_t course_step;       // Шаг курса, в градусах. Если 0, данный фильтр не применяется
} TTrack_param;

typedef struct
{
 bool gps;
 bool glonass;
 bool galileo;
 bool beidou;
 bool qzss;
 bool sbass;
}TNav_system_settings;

typedef struct
{
  uint8_t cno_thresh;            // C/N0 threshold for deciding whether to attempt a fix, dBHz
  uint8_t cno_thresh_num_svs;    // Number of satellites required to have C/N0 above cnoThresh for a fix to be attempted
  uint8_t min_elev;              // Minimum Elevation for a GNSS satellite to be used in NAV, deg
  uint8_t static_hold_thresh;    // Static hold threshold, cm/s
  uint16_t static_hold_max_dist; // Static hold distance threshold (before quitting static hold), m
  uint16_t p_acc;                // Position Accuracy Mask, m
}TNav_extended_filters;

typedef struct
{
  uint8_t intreface:3;           // 0 - выкл, 1 - RS232, 2 - RS485, 3 - UART
  uint8_t rmc_only:1;            // Транслировать только строку xRMC
  uint8_t reserve_1:4;           // Резерв
  uint32_t baud_rate:17;         // Скорость 9600 (только совместно с rmc_only) или 115200.
  uint32_t reserve_2:15;         // Резерв
}TNav_nmea_to_serial_port_translate;

typedef struct
{
  TTrack_param track_param;               // Параметры записи трека
  uint8_t translate_with_change_option;   // Режим работы regular_translation.translate_with_change: 0 - генерируется сообщение только с измененной величиной, 1 - генерируется сообщение со всем набором translate_with_track
  uint16_t time_step_with_ign_off;        // Шаг времени при отключенном зажигании, в секундах. Если 0, данный фильтр не применяется
  uint8_t filter_track_on_stop;           // Фильтрация от выбросов трека на стоянках
  uint8_t use_inertial_navigation;        // Использовать инерциальную навигацию
  uint8_t filter_track_hdop;              // Фильтрация трека по HDOP
  uint8_t reset_odometer_on_stop;         // Сбрасывать одометр на стоянках
  uint8_t always_locate;                  // Передавать последнюю валидную координату
  uint8_t odometer_by_ign;                // Считать одометр только при включенном зажигании
  TNav_system_settings nav_systems;       // Используемые навигационные системы
  TNav_extended_filters extended_filters; // Расширенные настройки GNSS приемника
#if defined(NMEA_TO_SERIAL_PORT_TRANSLATE_PRESENT)
  TNav_nmea_to_serial_port_translate nmea_to_serial_port_translate; //Настройки трасляции NMEA в последоавательный порт
#endif //defined(NMEA_TO_SERIAL_PORT_TRANSLATE_PRESENT)
} TTrack_settings;

// Настройки энергосбережения
typedef struct
{
  uint8_t sleep_by_ignition;            // Переходить в спящий режим через Х минут после выключения зажигания
  uint8_t sleep_by_stop;                // Переходить в спящий режим через Х минут после остановки
  uint16_t sleep_by_wakeup;             // Переходить в спящий режим через Х минут после пробуждения
  uint8_t slep_and_case;                // Переходить в спящий режим при выполнении всех выбранных условий
  uint8_t wakeup_by_ignition;           // Просыпаться при включении зажигания
  uint8_t wakeup_by_move;               // Просыпаться при наличии движения
  uint16_t wakeup_by_time;              // Просыпаться через Х минут после перехода в спящий режим
}TPower_settings;

// Настройки энергосбережения 2
typedef struct
{
  uint16_t wake_by_dig_in;               // Просыпаться по активному уровню на цифровом входе, битовое поле
}TPower_settings_2;

// Настройки энергосбережения gnss
typedef struct
{
  // Если стоит одна из опций use, то энергосбережения для приемника включен
  uint8_t use_gsens:1;                  // Использовать датчик движения акселерометра для перехода/выхода из энергосберегающего режима
  uint8_t use_can:1;                    // Использовать активность can для перехода/выхода из энергосберегающего режима
  uint8_t use_ign:1;                    // Использовать ign для перехода/выхода из энергосберегающего режима
  uint8_t reserve:5;

  uint8_t position_fixed_active_timeM;  // Время активности приемника, после захвата позиции, 1 - 255 минут
  uint8_t end_wait_fix_timeM;           // Время ожидания захвата позиции. Если позици не найдена, то уходим в psave, 1 - 255 минут

  uint16_t pos_fixed_sleep_timeM;       // Время нахождения в psave в случае, если позиция найдена
  uint16_t pos_not_fixed_sleep_timeM;   // Время нахождения в psave в случае, если позиция не найдена
}TGnss_psave_settings;

// Настройки энергосбережения gsm
typedef struct
{
  uint8_t use_psave:1;
  uint8_t reserve:7;
}TGsm_psave_settings;

// Настройки безопасности
typedef struct
{
  char pin[VEGA_PIN_SIZE];                            //Пин-код  доступа к устройству
  uint8_t use_pin;                                    //Использовать парольный доступ
  char autorized_phones[MAX_AUTORIZED_PHONES][14];    //Авторизованные телефоны
  uint8_t autorized_keys[MAX_AUTORIZED_KEYS][8];      //Авторизованные ключи
}TSecurity_settings;

// Настройки геозон
typedef struct
{
  float lat;
  float lon;
  uint32_t radius;
}TGeozone;

typedef struct
{
  TGeozone geozone[MAX_GEOZONES]; //Настройки геозоны
}TGeozone_settings;

// Настройки датчиков
typedef struct
{
  uint8_t din_num; //Номера цифрового входа для подсчета импульсов (0 – не включен)
}TPulse_input_settings;

typedef struct
{
  uint8_t din_num;//Номер цифрового входа для измерения частоты импульсов (0 – не включен)
}TFreq_input_settings;

typedef struct
{
  uint8_t dout_num;    //Номер цифрового выхода, который используется для вывода частоты (0 – не включен)
}TFreq_output_settings;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  DUT_OFF = 0,
  DUT_RS485 = 1,
  DUT_RS232 = 2,
}EDut_type;

typedef struct
{
  EDut_type type;      //Тип датчика (0 – выключен, 1- RS-485)
  uint8_t bus_address; //Адрес нашине 485 (для варианта 485)
  uint8_t protocol;    //Проткол обмена (для варианта 485, 0 – Omnicom, 1 - Epsilon)
}TDut_settings;

typedef struct
{
 uint8_t sensor_id[8]; //ID датчика (указан на самом датчике)
}TExt_temp_sensor_settings;

typedef struct
{
 uint8_t din_num; //номер цифрового входа для датчика зажигания (0 – работать  по напряжению бортовой сети)
}TIgnition_settings;


typedef struct
{
  uint16_t stopfix_time; //Время, через которое фиксируется остановка
}TGsensor_movesensor_settings;

typedef struct
{
  uint16_t stopfix_time; //Время, через которое фиксируется остановка
}TGPS_movesensor_settings;

typedef struct
{
  uint8_t din_num;  //Номер цифрового входа, к которому подключена
  uint8_t alarm_button_timeout;     //Длительность нажатия тревожной кнопки
}TAlarm_button_settings;

typedef struct
{
  uint8_t reset_after_stop;  //Сбрасывать авторизацию через х минут после остановки (0 – не сбрасывать по остановке)
  uint8_t reset_by_ignition; //Сбрасывать авторизацию при выключении зажигания
}TAutorization_sensor_settings;

typedef struct
{
  uint8_t reset_by_digin:7;         //Сбрасывать авторизацию при активном цировом входе
  uint8_t reset_by_digin_inv:1;     //Инверсия

  uint8_t reset_by_digout:7;        //Сбрасывать авторизацию при активном цировом выходе
  uint8_t reset_by_digout_inv:1;    //Инверсия

  uint8_t reset_by_ext_digin:7;     //Сбрасывать авторизацию при активном цировом входе БР
  uint8_t reset_by_ext_digin_inv:1; //Инверсия

  uint8_t reset_by_ext_digout:7;     //Сбрасывать авторизацию при активном цировом выходе БР
  uint8_t reset_by_ext_digout_inv:1; //Инверсия
}TAutorization_sensor_settings_2;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  IGNITION = 0,
  VOLTAGE = 1
}EMotohour_source;

typedef struct
{
  EMotohour_source source;//0 - Работать по зажиганию 1 - Работать по напряжению бортовой сети
}TMotohours_settings;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  CAM_OV528  = 0,
  CAM_QQZM   = 1,
  CAM_QQZM_2 = 2,
  CAM_VC0706 = 3,
}ECamera_type;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  CAM_OFF = 0,
  CAM_RS232 = 1,
  CAM_RS485 = 2,
  CAM_RSUART = 3,
}ECamera_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  CAM_RES320 = 0,
  CAM_RES640,
  CAM_RES1024,
  CAM_RES1280
}ECamera_resolution;

typedef struct
{
  uint16_t interval;              // Делать снимки каждые Х минут, если 0, не делать по времени
  ECamera_interface interface;    //Интерфейс камеры
  ECamera_resolution resolution;  //Разрешение (1 – 320х240, 2 – 640х480, 3 – 1024х768, 4 - 1280х1024)
}TCam_settings;

typedef struct
{
  uint8_t polarity;     //Полярность входа
}TDigital_input_settings;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  BR_OFF   = 0,
  BR_RS232 = 1,
  BR_RS485 = 2
}EBr_interface;

typedef struct
{
  EBr_interface interface;
}TBr_settings;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  DIG_IN = 0,
  FREQ_IN = 1,
  PULSE_IN = 2,
  ANALOG_IN = 3,
}EMfi_type;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  ACTIVE_0 = 0,
  ACTIVE_1 = 1,
}EMfi_active_polarity;

typedef struct
{
  EMfi_type input_type; //Тип многофункционального входа
  EMfi_active_polarity polarity; //Полярность активного сигнала
}TMfi_settings;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  FRIDGE_OFF   = 0,
  FRIDGE_RS232 = 1,
  FRIDGE_RS485 = 2,
}EFridge_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  ZANOTTI_OFF   = 0,
  ZANOTTI_RS232 = 1,
}EZanotti_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  SR3_OFF   = 0,
  SR3_ON    = 1,
}ESR3_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  PARTNER_PROTOCOL   = 0,
}EFridge_protocol;

#ifdef __ICCARM__
typedef enum
#else
enum class EFlowmeter_interface: uint8_t
#endif
{
  FM_OFF   = 0,
  FM_RS232 = 1,
  FM_RS485 = 2,
#ifdef __ICCARM__
} EFlowmeter_interface;
#else
};
#endif


#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  CARRIER_OFF   = 0,
  CARRIER_RS232 = 1,
}ECarrier_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  CARRIER_STANDARD   = 0,
  CARRIER_ADVANCE = 1,
  CARRIER_APX = 2,
}ECarrier_type;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  MDAS9_OFF   = 0,
  MDAS9_RS232 = 1,
}EMDAS9_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  MDSM7_OFF   = 0,
  MDSM7_RS232 = 1,
}MDSM7_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  OMNICOMM_OFF   = 0,
  OMNICOMM_RS485 = 1,
}OMNICOMM_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  RS232_2_OFF   = 0,
  RS232_2_TELEM = 1,
  RS232_2_IRIDIUM = 2,
  RS232_2_DISPLAY = 3,
  RS232_2_EMDC600 = 4,
}RS232_2_assign;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  HUM_IAKM_OFF   = 0,
  HUM_IAKM_RS485 = 1,
}EHum_iakm_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  SCD30_OFF   = 0,
  SCD30_RS485 = 1,
}EScd30_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  THERMALMASTER_OFF   = 0,
  THERMALMASTER_RS232 = 1,
}Thermalmaster_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  EXCERA_OFF   = 0,
  EXCERA_RS232 = 1,
}EExcera_interface;

typedef struct
{
  EExcera_interface interface;    //Интерфейс РСТ Excera
}
TExcera_settings;

typedef struct
{
  EFridge_interface interface;    //Интерфейс холодильника
  EFridge_protocol protocol;
}
TFridge_settings;

typedef struct
{
  EFridge_interface interface;    //Интерфейс Touchprint
}
TTouchprint_settings;

typedef struct
{
  EZanotti_interface interface;    //Интерфейс Zanotti
}
TZanotti_settings;

typedef struct
{
  ESR3_interface interface;    //Интерфейс SR3 ХОУ Thermo_King_SLX_T_UT
}
TSR3_settings;

typedef struct
{
  EFlowmeter_interface interface;    //Интерфейс Flowmeter
  uint8_t addr; //1...247
}
TFlowmeter_settings;

typedef struct
{
  EHum_iakm_interface interface;    //Интерфейс MODBUS Датчика влажности и температуры  IAKM.00.F148.517.S53
  uint8_t addr; //1...247
}
THum_iakm_settings;

typedef struct
{
  EScd30_interface interface;    //Интерфейс MODBUS Датчика CO2, влажности и температуры  SCD30
}
TScd30_settings;

typedef struct
{
  Thermalmaster_interface interface;    //Интерфейс THERMAL MASTER
}
THERMALMASTER_settings;

typedef struct
{
  uint16_t voltage; //Порог напряжения питания на засыпание
  uint16_t timer;   //Время на переход в режим сна, в сек
}
TAdj_sleep_settings;

typedef struct
{
  ECarrier_interface interface;    //Интерфейс холодильника Carrier
  ECarrier_type type;
}
TCarrier_settings;

typedef struct
{
  EMDAS9_interface interface;    //Интерфейс MDAS9
}
TMDAS9_settings;

typedef struct
{
  MDSM7_interface interface;    //Интерфейс MDSM7
}
MDSM7_settings;

typedef struct
{
  OMNICOMM_interface interface;    //Интерфейс OMNICOMM ICON
}
OMNICOMM_settings;

typedef struct
{
 RS232_2_assign assign; //Назначение RS232-2
 uint8_t read_param;
 uint8_t set_param;
 uint8_t   set_ref;
}
RS232_2_settings;

typedef struct
{
  EFridge_interface interface;    //Интерфейс i-BOX
}
TIbox_settings;

#if defined(LORA_SENSORS_PRESENT)
typedef struct
{
  uint8_t addr[5];              // 5-байтовый адрес в HEXе
  uint8_t period;               // В минутах от 1 до 255
  uint8_t power;                // В dBm от 2 до 20
  uint8_t sense_threshold;      // Чувствительность датчика отрыва 1-5
}TLoraSens_subsettings;

typedef struct
{
  float temperature;
  uint8_t events;
  uint8_t charge_level;
  uint8_t rssi;
  uint8_t online_status;
  uint8_t version;
#if (defined(EXTENDED_ADVANTUM_LORA_SENSORS_PRESENT) || defined(IS_BEACON_LORA_BUILD))
  uint8_t axis;
#endif //EXTENDED_ADVANTUM_LORA_SENSORS_PRESENT
}TLoraSens_state;

typedef struct
{
  uint8_t events_0;// bit 0 - цепь
  uint8_t events_1;// bit 1 - датчик Холла 1
  uint8_t events_2;// bit 2 - датчик Холла 2
  uint8_t events_3;// bit 3 - датчик вскрытия корпуса
  uint8_t events_4;// bit 4 - датчик отрыва
}TLoraSens_state_bits;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  LORA_SENS_OFF   = 0,
  LORA_SENS_RS232 = 1,
  LORA_SENS_RS485 = 2
}ELoraSens_interface;

typedef struct
{
  ELoraSens_interface interface;
  TLoraSens_subsettings sensors[MAX_LORA_SENSORS_COUNT];
}TLoraSens_settings;
#endif //LORA_SENSORS_PRESENT

#if defined(NRF_BEACONS_SCANNER_PRESENT)
typedef struct
{
  uint32_t visible_beacons_bitfield[MAX_NRF_AUTORIZED_BEACONS_COUNT/32];
  uint32_t battlow_beacons_bitfield[MAX_NRF_AUTORIZED_BEACONS_COUNT/32];
  char visible_beacons_string[80+1];
  uint8_t  is_autorized; // Хотя бы одна авторизованная метка в поле видимости
}TNrfBeaconsScanner_state;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  NRF_BEACONS_SCANNER_OFF   = 0,
  NRF_BEACONS_SCANNER_RS232 = 1,
  NRF_BEACONS_SCANNER_RS485 = 2
}ENrfBeaconsScanner_interface;

typedef struct
{
  ENrfBeaconsScanner_interface interface;
  uint32_t autorized_beacons[MAX_NRF_AUTORIZED_BEACONS_COUNT];
}TNrfBeaconsScanner_settings;
#endif //NRF_BEACONS_SCANNER_PRESENT

#if defined(INTERNAL_NRF_PRESENT)
typedef struct
{
  uint8_t  ready_for_command;
  uint32_t relay_id;
  uint8_t  desired_relay_state;
  uint8_t  actual_relay_state;//0xff - неизвестно

  uint32_t marks_id[MAX_INTERNAL_NRF_MARKS_COUNT];
  uint32_t visible_marks_bf;
  uint32_t battlow_marks_bf;
#if defined(NAVITERM_PRESENT)
  uint16_t mark_btns_pressed;
#endif //(NAVITERM_PRESENT)
}TInternalNrf_state;
#endif //INTERNAL_NRF_PRESENT

#if defined(TENZO_M_WEIGHER_PRESENT)
typedef struct
{
  uint8_t net_raw_data[4];
  uint8_t gross_raw_data[4];
}TTenzoMWegher_state;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  TENZO_M_WEIGHER_OFF   = 0,
  TENZO_M_WEIGHER_RS232 = 1,
  TENZO_M_WEIGHER_RS485 = 2
}ETenzoMWegher_interface;

typedef struct
{
  ETenzoMWegher_interface interface;
  uint8_t addr; //0x01…0x9F
}TTenzoMWegher_settings;
#endif //TENZO_M_WEIGHER_PRESENT

#if defined(PROCESSOR_WEIGHT_PV310_PRESENT)
typedef struct
{
  uint8_t net_raw_data[4];
  //uint8_t gross_raw_data[4];
}TPV310Wegher_state;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  PV310_WEIGHER_OFF   = 0,
  PV310_WEIGHER_RS232 = 1,
  PV310_WEIGHER_RS485 = 2
}EPV310Wegher_interface;

typedef struct
{
  EPV310Wegher_interface interface;
  uint8_t addr; //1...247
}TPV310Wegher_settings;
#endif //PROCESSOR_WEIGHT_PV310_PRESENT

#if (defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))
typedef struct
{
  float mt;
  float t2;
  float t3;
  float sp;
  float sp2;
  float sp3;
  float ambt;
  float afzt;
  uint16_t rpm;
  uint8_t conf;
  uint8_t conf2;
  uint8_t conf3;
  uint8_t state;
  uint8_t state2;
  uint8_t state3;
  uint8_t dr_bf;
  float batv;
  float bata;
  float hm;
  float hme;
  float hmt;
  char refsn[21];
  char trsn[14];
  uint8_t alcount;
  uint8_t al1;
  uint8_t al2;
  uint8_t al3;
  uint8_t al4;
  uint8_t al5;
  uint8_t al6;
  char sn[16];
  char ver[11];
  uint8_t reftype;
  uint8_t no_connect;
  uint8_t in_bf;
  uint32_t uptime;
  float ad1;
  float ad2;
  float ad3;
  float ad4;
  float ad5;
  float ad6;
  float hs;
  uint16_t err_bf;
  uint8_t lic;
  uint8_t dtp;
  //char time[20];//Время по UTC (только в опорном кадре)
  char vrn[16];//Гос. номер iQFreeze
  //char regtime[20];//Актуальное время регистрации последней записи (UTC)
}TIqfreeze_state;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  IQFREEZE_OFF   = 0,
  IQFREEZE_RS232 = 1,
  IQFREEZE_RS485 = 2
}EIqfreeze_interface;

typedef struct
{
  EIqfreeze_interface interface;
  uint8_t poll_time_s;
}TIqfreeze_settings;
#endif //(defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  DM5002X_OFF   = 0,
  DM5002X_RS232 = 1,
  DM5002X_RS485 = 2
}EDm5002x_interface;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  BS_BT_OFF   = 0,
  BS_BT_RS485 = 2

}Bs_bt_interface;

#if defined(TPMS_PRESENT) //tire pressure monitoring system
typedef struct
{
 // uint8_t func[MAX_TPMS_DAT_COUNT];//Функционал
 // uint8_t vinf[MAX_TPMS_DAT_COUNT];//Объем информации датчика
 // uint8_t indexinf;//Индекс информации
 // uint8_t trailerID1;//Прицеп ID1
 // uint8_t trailerID2;//Прицеп ID2
 // uint8_t trailerID3;//Прицеп ID3
 // uint8_t poswh;//Определение позиции колес (от 1 до 42)
  uint32_t id[MAX_TPMS_DAT_COUNT];//ID
  float pressure[MAX_TPMS_DAT_COUNT];//Давление Диапазон: 100-1400КРА (абсолютное)  Расчет: Относительное Давление=Р*5.5-101.3 КРА  если информация о давлении не поступает, Бит Данных будет 0хFF
  float temper[MAX_TPMS_DAT_COUNT];//Температура Диапазон: -50° - +205°C  Расчет: Температура=Т-50 °С    Если информация о температуре не поступает, Бит Данных будет 0хFF
  uint8_t flags[MAX_TPMS_DAT_COUNT];//флаги
  //Предупреждение о высоком/низком давлении (3 бита)
  //Предупреждение о высокой температуре (1 бит)
  //Предупреждение о быстрой утечке воздуха (1 бит)
  //Предупреждение о проблеме с датчиком (1 бит)
}TTPMS_state;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  TPMS_OFF   = 0,
  TPMS_RS232 = 1
}ETPMS_interface;

typedef struct
{
  ETPMS_interface interface;
  uint8_t poll_time_s;
}TTPMS_settings;
#endif //TPMS_PRESENT

typedef struct
{
#if (MAX_DIGITAL_INPUTS > 0)
  TDigital_input_settings digital_in[MAX_DIGITAL_INPUTS];
#endif //(MAX_DIGITAL_INPUTS > 0)

#if (MAX_PULSE_INPUTS > 0)
  TPulse_input_settings pulse_in[MAX_PULSE_INPUTS]; //Импульсный вход Х
#endif //(MAX_PULSE_INPUTS > 0)

#if (MAX_FREQ_INPUTS > 0)
  TFreq_input_settings freq_in[MAX_FREQ_INPUTS]; //Частотный вход X
#endif //(MAX_FREQ_INPUTS > 0)

#if (MAX_FREQ_OUTPUTS > 0)
  TFreq_output_settings freq_out[MAX_FREQ_OUTPUTS];//Частотный выход Х
#endif //(MAX_FREQ_OUTPUTS > 0)

#if (MAX_MF_INPUTS > 0)
  TMfi_settings mfi[MAX_MF_INPUTS]; //Многофункциональные входы
#endif //(MAX_MF_INPUTS > 0)

#if defined(BR_PRESENT)
  TBr_settings br_settings;
  TMfi_settings ext_mfi[MAX_EXT_MF_INPUTS]; //Расширенные многофункциональные входы
#endif //BR_PRESENT

#if defined(FRIDGE_PRESENT)
  TFridge_settings fridge;
#endif //FRIDGE_PRESENT

#if defined(LORA_SENSORS_PRESENT)
  TLoraSens_settings lora_sens;
#endif //LORA_SENSORS_PRESENT

#if defined(NRF_BEACONS_SCANNER_PRESENT)
  TNrfBeaconsScanner_settings nrf_beacons_scanner;
#endif //NRF_BEACONS_SCANNER_PRESENT

#if defined(TENZO_M_WEIGHER_PRESENT)
  TTenzoMWegher_settings tenzo_m_weigher;
#endif //TENZO_M_WEIGHER_PRESENT

#if defined(PROCESSOR_WEIGHT_PV310_PRESENT)
  TPV310Wegher_settings pv310_weigher;
#endif //PROCESSOR_WEIGHT_PV310_PRESENT

#if (defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))
  TIqfreeze_settings iqfreeze;
#endif //(defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))

#if defined(TOUCHPRINT_PRESENT)
  TTouchprint_settings touchprint;
#endif // TOUCHPRINT_PRESENT

#if defined(ZANOTTI_MODBUS_PRESENT)
  TZanotti_settings zanotti;
#endif // ZANOTTI_MODBUS_PRESENT

#if defined(SR3_PRESENT)
  TSR3_settings sr3;
#endif // SR3_PRESENT

#if defined(EXCERA_PRESENT)
  TExcera_settings excera;
#endif // EXCERA_PRESENT

#if defined(IBOX_PRESENT)
  TIbox_settings ibox;
#endif // IBOX_PRESENT

#if defined(FLOWMETER_PRESENT)
  TFlowmeter_settings flowmeter;
#endif// FLOWMETER_PRESENT

#if defined(TPMS_PRESENT)
  TTPMS_settings tpms;
#endif //TPMS_PRESENT

#if defined(CARRIER_PRESENT)
  TCarrier_settings carrier;
#endif //CARRIER_PRESENT

#if defined(MDAS9_PRESENT)
  TMDAS9_settings mdas9;
#endif //MDAS9_PRESENT

#if defined(MDSM7_PRESENT)
  MDSM7_settings mdsm7;
#endif //MDSM7_PRESENT

#if defined(OMNICOMM_ICON_PRESENT)
  OMNICOMM_settings omnicomm;
#endif //OMNICOMM_ICON_PRESENT

#if defined(RS232_2_PRESENT)
  RS232_2_settings rs232_2;
#endif //RS232_2_PRESENT

#if defined(HUM_SENSOR_IAKM_PRESENT)
  THum_iakm_settings hum_iakm;
#endif //HUM_SENSOR_IAKM_PRESENT

#if defined(SCD30_PRESENT)
  TScd30_settings scd30;
#endif //SCD30_PRESENT

#if defined(ADJ_SLEEP_MODE_PRESENT)
  TAdj_sleep_settings adj_sleep;
#endif //ADJ_SLEEP_MODE_PRESENT

#if defined(THERMALMASTER_PRESENT)
  THERMALMASTER_settings thermalmaster;
#endif //defined(THERMALMASTER_PRESENT)

#if defined(REPORT_GEN_PRESENT)
  uint32_t Type_dat;//тип темературных датчиков 1-LORA, 2-DALLAS, 3-аналоговый (терморезистор)
  uint32_t MaxNumCompartment;//макс.кол-во отсеков
  uint32_t SensCompartment[8];//SensCompartment[0,2,4,6]:  digital_input -> 0,1,2,3,4 Tamper->5,6,7 lora_sens[0].events i->8,9,10,11 lora_sens[1].events i->12,13,14,15  lora_sens[2].events i->16,17,18,19
  //    lora_sens[3].events i->20,21,22,23lora_sens[4].events i->24,25,26,27  lora_sens[5].events i->28,29,30,31
  //SensCompartment[1,3,5,7]: lora_sens[6].events i->0,1,2,3 lora_sens[7].events i->4,5,6,7  lora_sens[8].events i->8,9,10,11  Адреса подключенных к отсеку датчиков № ВЕГА-ТД 1...10 ->16...25 analog_input -> 26,27,28,29,30,31
  uint32_t Period_BB_TR_TIME;   //периодичность считывания темп.датчиков 1...60мин
  uint8_t VIN_auto[16];//VIN автомобиля
  uint8_t TR_header[32];//Текст заголовка
  uint8_t ID_owner[16];//ID владельца
  uint8_t Lora_ddat[10];//Настройки цифровых датчиков LORA bit0-цепь, bit1-, bit2-датчик отрыва, bit3-датчик вскрытия, bit4-датчик температуры
#endif // REPORT_GEN_PRESENT

  TDut_settings dut[MAX_DUTS]; //Датчик уровня топлива Х
#if defined(MAX_DALLAS_TEMP_SENSORS) && (MAX_DALLAS_TEMP_SENSORS > 0)
  TExt_temp_sensor_settings ext_temp[MAX_DALLAS_TEMP_SENSORS]; //Датчик температуры внешний Х (шина dallas)
#endif
  TIgnition_settings ignition; //Датчик зажигания
  TGsensor_movesensor_settings gsensor_movesensor; //Датчик движения акселерометра
  uint8_t gsensor_senslevel;
  TGPS_movesensor_settings gps_movesensor;//Датчик движения GPS
  TAlarm_button_settings alarm_button;//Тревожная кнопка
  TAutorization_sensor_settings autorization;//Датчик авторизации
  TAutorization_sensor_settings_2 autorization_2;//Датчик авторизации
  TMotohours_settings motohours;//Датчик моточасов
#if defined(MAX_CAMERAS) && (MAX_CAMERAS > 0)
  TCam_settings camera[MAX_CAMERAS];//Фотокамера
  ECamera_type  camera_type[MAX_CAMERAS];//Тип фотокамеры
#endif
  uint8_t temp_add_mode;            // Режим добавления всех новых видимых датчиков температуры
  uint8_t keys_add_mode;            // Режим добавления всех новых видимых ключей
  uint8_t use_can_ignition;         // Использовать CAN для определения зажигания
}TSensor_settings;

#if defined(ATOM_PRESENT)
typedef struct
{
  uint8_t is_ok;
  uint8_t last_error;
  uint32_t camera_disconnected_bitfield;
  uint32_t camera_image_problem_bitfield;
  uint16_t door_pass_in[3];
  uint16_t door_pass_out[3];
  uint8_t door_number;
  uint16_t door_in;
  uint16_t door_out;
  uint32_t video_timestamp;
  uint32_t time_for_fix;
  uint32_t video_id;
}TAtom_state;
extern TAtom_state atom_state, atom_state_res;
#endif

#if defined(EXT_GSENSOR_PRESENT)
typedef struct
{
  uint8_t communication_state;
  uint16_t flags;
  char maneuver[64];
}TExt_gsens_state;
extern TExt_gsens_state ext_gsens_state, ext_gsens_state_res;
#endif //EXT_GSENSOR_PRESENT

#if defined(STRIZH_BS_PRESENT)
typedef struct
{
  uint8_t communication_state;
  char last_data[22*2+1];
}TStrizh_bs_state;
extern TStrizh_bs_state strizh_bs_state, strizh_bs_state_res;
#endif //STRIZH_BS_PRESENT


// Настройки сценариев
#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  VALUE_LAY_DOWN = 0, // Значение стало меньше на sensor_value/ and_sensor_value
  VALUE_RISE_UP  = 1, // Значение стало больше на sensor_value/ and_sensor_value
  VALUE_CHANGED  = 2, // Значение изменилось на sensor_value/ and_sensor_value
  VALUE_EQUAL    = 3  // Значение стало равным sensor_value/ and_sensor_value
}ESensor_event;

typedef struct
{
  uint8_t call_to_number;	     //позвонить на номер
  uint8_t dout_on_num;	             //включить цифровой выход
  uint8_t dout_on_time;	             //включить цифровой выход на время
  uint8_t dout_off_num; 	     //выключить цифровой выход
  uint8_t dout_off_time;	     //выключить цифровой выход на время
  uint8_t send_sms_to_num;	     //отправить СМС на номер
  uint8_t sms_text[20];	             //текст СМС
  uint8_t take_photo;                //сделать фото
  uint8_t ext_dout_on_num; 	     //включить внешний цифровой выход
  uint8_t ext_dout_on_time; 	     //включить внешний цифровой выход на время
  uint8_t ext_dout_off_num; 	     //выключить цифровой выход
  uint8_t ext_dout_off_time; 	     //выключить внешний цифровой выход на время

  uint8_t can_blinker_flash:1;       //вспышка поворотников
  uint8_t can_close_doors:1;         //закрыть двери
  uint8_t can_open_doors:1;          //открыть двери
  uint8_t can_open_driver_door:1;    //открыть дверь водителя
  uint8_t can_horn:1;                //подать сигнал клаксона
  uint8_t can_trunk:1;		     //открыть багажник
  uint8_t can_stop_engine:1;	     //остановить двигатель
  uint8_t can_start_engine:1;        //запустить двигатель
  uint8_t can_start_webasto:1;       //запустить webasto
  uint8_t can_stop_webasto:1;        //остановить webasto
  uint8_t can_door_emulation:1;      //эмуляция открытия двери водителя
  uint8_t can_horn_and_brink:1;      //вспышка поворотников и клаксон
  uint8_t can_windows_closing_3s:1;  //закрытие окон 3 секунды
  uint8_t can_windows_closing_7s:1;  //закрытие окон 7 секунд
  uint8_t can_windows_closing_11s:1; //закрытие окон 11 секунд
  uint8_t can_windows_closing_15s:1; //закрытие окон 15 секунд
  uint8_t can_windows_closing_19s:1; //закрытие окон 19 секунд
  uint8_t can_windows_closing_23s:1; //закрытие окон 23 секунд
  uint8_t can_windows_closing_29s:1; //закрытие окон 29 секунд
  uint8_t can_windows_opening_3s:1;  //открытие окон 3 секунд
  uint8_t can_windows_opening_7s:1;  //открытие окон 7 секунд
  uint8_t can_windows_opening_11s:1; //открытие окон 11 секунд
  uint8_t can_windows_opening_15s:1; //открытие окон 15 секунд
  uint8_t can_windows_opening_19s:1; //открытие окон 19 секунд
  uint8_t can_windows_opening_23s:1; //открытие окон 23 секунд
  uint8_t can_windows_opening_29s:1; //открытие окон 29 секунд
  uint8_t user_can_script:6;         //пользовательский can скрипт
}TScenario_action;

typedef struct
{
  uint16_t sensor_source_id;      // id первого датчика, значение которого участвует в сценарии.
  float    sensor_value;          // Значение 1-го датчика
  ESensor_event sensor_event;     // Событие первого датчика

  uint8_t use_and_condition:1;    // В условии сценария участвуют два датчика
  uint8_t reserve:1;              // Резерв
  uint8_t offset:3;               // Смещение значения 1-го битового датчика
  uint8_t and_offset:3;           // Смещение значения 2-го битового датчика

  uint16_t and_sensor_source_id;  // id второго датчика
  float    and_sensor_value;      // Значение 2-го датчика
  ESensor_event and_sensor_event; // Событие второго датчика
  TScenario_action action;        // Действие, при возникновении события
}TScenario;

typedef struct
{
  TScenario scenario[MAX_SCENARIES];
}TScenario_settings;

typedef struct
{
  uint16_t program_no;                   //Номер программы
  uint8_t frame_interval;                //Интервал времени между кадрами, секунды
}TCan_settings;

typedef struct
{
  uint16_t sensor_source_id;      // id первого датчика, значение которого участвует в сценарии.
  float    sensor_value;          // Значение 1-го датчика
  ESensor_event sensor_event;     // Событие первого датчика
  uint8_t  use_and_condition;     // В условии сценария участвуют два датчика
  uint16_t and_sensor_source_id;  // id второго датчика
  float    and_sensor_value;      // Значение 2-го датчика
  ESensor_event and_sensor_event; // Событие второго датчика
  uint8_t  code;                  // Настройка передача кодов состояния через Iridium, при возникновении события
}TScenario_i;
#if defined(IRIDIUM_PRESENT)
typedef struct
{
  TScenario_i scenario_i[MAX_SCENARIES_I];
}TScenario_settings_i;

#endif // defined(IRIDIUM_PRESENT)
/////////Текущее состояние системы /////////////////////////////////////////////

// Текущее состояние GPS
#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  INACTIVE = 0,
  NOT_AVALIABLE  = 1,
  FIX2D = 2,
  FIX3D = 3,
}EFix_type;

#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  INVALID = 0,
  VALID  = 1
} ECoord_status;

typedef struct
{
    uint8_t     id;         // PRN спутника
    uint8_t     elv;        // Возвышение над горизонтом в градусах (от 0 до 90)
    uint16_t    azimuth;    // Азимут от истинного севера, в градусах (от 0 до 359)
    uint8_t     sig;        // Уровень сигнала dB (от 0 до 99)
} TGNSS_satellite;

typedef struct
{
    uint8_t         inview;                    // Количество видимых спутников для данной ГНСС
    TGNSS_satellite sat[GPS_MAX_SAT_VISIBLE];  // Информация о видимых спутниках для данной ГНСС
} TGNSS_satinfo;

typedef struct
{
    uint8_t is_on;              // Включен ли приемник
    uint8_t is_sleep;           // Включен ли менеджер
    uint8_t receiver_is_ok;     // Работоспособен ли приемник
    uint8_t in_move;            // Устройство находится в движении
    uint32_t trip_counter;      // Номер поездки
    uint8_t jammed;             // Датчик глушения сигнала
    uint8_t spoof;               // Датчик подмены сигнала (0 = неизвестно или выключен, 1 = не обнаружено, 2 = подмена, 3 = множественная подмена)
    uint8_t receiver_type;       // Тип ГНСС приемника (модель)
    uint8_t receiver_version[GPS_RECV_VER_SIZE]; // Версия ПО ГНСС приемника
    time_struct_t utc;           // Текущие время и дата
    ECoord_status coord_status; // Статус координат (валидные/невалидные)
    float    lat;                // Широта (знаковая. градусы, доли градуса. )
    float    lon;                // Долгота (знаковая. градусы, доли градуса. )
    float    alt;                // Высота над средним уровнем моря (в метрах)
    float    diff;               // Разница между эллипсоидом WGS-84 и средним уровнем моря (в метрах)
    float    speed;              // Скорость
    float    direction;          // Курс
    uint8_t  dgps_mode;          // Режим DGPS (A = автономный режим, D = с использованием дифференциальной коррекции)
    uint32_t dgps_age;           // Возраст данных DGPS в секундах
    uint16_t dgps_sid;          // Номер станции DGPS
    int     sig;                // Индикатор качества (0 = Invalid; 1 = Fix; 2 = Differential, 3 = Sensitive)
    EFix_type fix_type;         // Тип фиксации (1 = Fix not available; 2 = 2D; 3 = 3D)
    float   PDOP;               // Position Dilution Of Precision
    float   HDOP;               // Horizontal Dilution Of Precision
    float   VDOP;               // Vertical Dilution Of Precision
    uint8_t sat_inuse;          // Количество спутников, используемых в навигационном решении
    uint8_t sat_prn[GPS_MAX_SAT_USED]; // PRN спутников, используемых в навигационном решении
    TGNSS_satinfo gps_satinfo;         // Информация о спутниках GPS
    TGNSS_satinfo glonass_satinfo;     // информация о спутниках GLONASS
    TGNSS_satinfo galileo_satinfo;     // Информация о спутниках Galileo
    TGNSS_satinfo beidou_satinfo;      // информация о спутниках BeiDou
    uint8_t total_sat_inview;
    float odometer;
    uint8_t geozones[MAX_GEOZONES];
} TGNSS_state;

typedef struct
{
    uint32_t      unix_time;          // Время и дата
    ECoord_status coord_status;       // Статус координат (валидные/невалидные)
    float         lat;                // Широта (знаковая. градусы, доли градуса. )
    float         lon;                // Долгота (знаковая. градусы, доли градуса. )
    int16_t       alt;                // Высота над средним уровнем моря (в метрах) * 10
    uint32_t      diff;               // Разница между эллипсоидом WGS-84 и средним уровнем моря (в метрах) * 10
    uint16_t      speed;              // Скорость * 10
    uint16_t      direction;          // Курс * 10
    uint8_t       dgps_mode;          // Режим DGPS (A = автономный режим, D = с использованием дифференциальной коррекции)
    uint32_t      dgps_age;           // Возраст данных DGPS в секундах
    uint16_t      dgps_sid;           // Номер станции DGPS
    uint8_t       sig;                // Индикатор качества (0 = Invalid; 1 = Fix; 2 = Differential, 3 = Sensitive)
    EFix_type     fix_type;           // Тип фиксации (1 = Fix not available; 2 = 2D; 3 = 3D)
    uint8_t       sat_inuse;          // Количество спутников, используемых в навигационном решении
    uint16_t      PDOP;               // Position Dilution Of Precision * 100
    uint16_t      HDOP;               // Horizontal Dilution Of Precision * 100
    uint16_t      VDOP;               // Vertical Dilution Of Precision * 100
}TGNSS_raw_state;

// Текущее состояние сервер менеджера
typedef struct
{
  uint16_t MCC;
  uint16_t MNC;
  uint32_t LAC;
  uint32_t CELLID;
  uint8_t RegStat; //Not registered, ME is currently not searching for new operator: 0, Registered to home network: 1, Not registered, but ME is currently searching for a new operator: 2, Registration denied: 3, Unknown (reset state): 4, ME is registered at a foreign network (national or international network): 5
  uint8_t AcT;     //GSM: 0, Unknown (reset state): 1, UTRAN: 2, GSM w/EGPRS: 3, UTRAN w/HSDPA: 4, UTRAN w/HSUPA: 5, UTRAN w/HSDPA and w/HSUPA: 6, E-UTRAN: 7
} TGSM_cellinfo;

typedef struct
{
  uint32_t rx;
  uint32_t tx;
} TGSM_traffic_counter;

typedef struct
{
  uint8_t is_on;                 // Включен ли модем
  uint8_t is_sleep;              // Включен ли менеджер
  uint8_t modem_is_ok;           // Работоспособен ли модем
  uint8_t jammed;                // Датчик глушения сигнала
  uint8_t modem_type;            // Тип модема
  char modem_version[MODEM_VER_SIZE]; // Версия ПО модема
  char IMEI[16];                 // IMEI модема
#if defined(NAVITERM_PRESENT)
  char IMSI[16];                 // IMSI модема
#endif //(NAVITERM_PRESENT)
  char ICCID[MAX_SIM_COUNT][20]; // ICCID SIM
  TGSM_cellinfo serving_cell;    // Данные обслуживающей базовой станции
  uint8_t used_sim;              // Используемая симкарта, 0 - неизвестно, 1 - основная, 2 - резевная
  #if defined(YANDEX_EXTENDED_SENSORS_PRESENT)
  char ext_serving_cells_info[MAX_SENSOR_R_SIZE];
  uint8_t scan_mode;
  #endif //YANDEX_EXTENDED_SENSORS_PRESENT
  uint8_t tcp_connect_state[MAX_SERVERS_COUNT]; // Признак соединения с сервером по TCP
  uint8_t csd_connect_state[MAX_SERVERS_COUNT]; // Признак соединения с сервером по CSD
  uint8_t signal_rssi;        // Уровень сигнала, RSSI
  int8_t signal_dbm;          // Уровень сигнала, dBm

#if defined(BALANCE_REQUEST_PRESENT)
  float balance[MAX_SIM_COUNT];// Баланс SIM-карты
#endif //defined(BALANCE_REQUEST_PRESENT)

//#ifndef __ICCARM__
  uint16_t input_buff_size;   //Размер входящего буфера
  uint16_t output_buff_size;  //Размер исходящего буфера
//#endif
  uint8_t usb_connected;      // Признак подключенного USB
  TGSM_traffic_counter traffic_counter[MAX_SERVERS_COUNT]; //счётчик трафика в килобайтах
} TServer_state;

// Текущее состояние сигнального менеджера
#ifdef __ICCARM__
typedef enum
#else
typedef enum : uint8_t
#endif
{
  UNKNOWN_POWER =  0,
  EXTERNAL_POWER = 1,
  ACC_POWER      = 2
}Power_source;

#if defined(FRIDGE_PRESENT)
typedef struct
{
  uint8_t flags; //Bit 0: on/off state, Bit 1: valid state, Bit 2: alarm state, Bit 3: error state
  float   value;
}TFridge_AnSensor_data;

typedef struct
{
  uint8_t on_off : 1;
  uint8_t valid  : 1;
  uint8_t alarm  : 1;
  uint8_t error  : 1;
  uint8_t no_matters : 4;
}TFridge_Sensor_flags;

typedef struct
{
  uint8_t avaliable  : 1;
  uint8_t on         : 1;
  uint8_t defrosting : 1;
  uint8_t pre_trip   : 1;
  uint8_t no_matters : 4;
}TCompartment_state_flags;

typedef struct
{
  uint8_t avaliable  : 1;
  uint8_t error      : 1;
  uint8_t alarm      : 1;
  uint8_t digital_sensor : 1;
  uint8_t no_matters : 4;
}TBat_fuel_state_flags;

typedef struct
{
  uint8_t flags; //Bit 0: on/off state, Bit 1: valid state, Bit 2: alarm state, Bit 3: error state
  uint8_t value;
}TFridge_DigSensor_data;

typedef struct
{
  uint8_t flags; //Bit 0: Available, Bit 1: Error, Bit 2: Alarm
  uint32_t value;
}TFridge_Hours_data;

typedef struct
{
  uint8_t compartment_state; //Bit 0: Available, Bit 1: Comparment on, Bit 2: Defrosting, Bit 3: In pre-trip
  uint8_t compartment_mode; //0 Off, 1 Heating, 2 Cooling, 3 Idle, 4 Defrost, 5 Pretrip, 15 Unknown
  TFridge_AnSensor_data setpoint;
  TFridge_AnSensor_data supply_air_sensor_1;
  TFridge_AnSensor_data supply_air_sensor_2;
  TFridge_AnSensor_data return_air_sensor_1;
  TFridge_AnSensor_data return_air_sensor_2;
  TFridge_AnSensor_data evaporator_temperature;
}TFridge_compartment_data;

typedef struct
{
  uint8_t alarm_level;
  uint8_t alarm_codes[MAX_FRIDGE_ALARM_CODES];
}TFridge_alarm_data;

typedef struct
{
  uint32_t serial_number;
  TFridge_AnSensor_data an_data[MAX_FRIDGE_AIN];
  TFridge_DigSensor_data dig_data[MAX_FRIDGE_DIN];
  uint8_t run_mode; //0: Start/Stop (SS), 1: Continous, 2: Cycle sentry (CS), 7: Unknown / Not available
  uint8_t power_mode; //0: Engine, 1: Electric (Standby)
  uint8_t speed_mode; //0: Normal speed, 1: High speed
  uint8_t battery_state_flags; //Bit 0: Available, Bit 1: Error, Bit 2: Alarm, Bit 3: Digital sensor
  float battery_voltage;
  uint8_t fuel_state_flags; //Bit 0: Available, Bit 1: Error, Bit 2: Alarm, Bit 3: Digital sensor
  uint8_t fuel_level; //In %
  TFridge_AnSensor_data ambient_temperature;
  TFridge_Hours_data electric_hours;
  TFridge_Hours_data standby_hours;
  TFridge_Hours_data diesel_hours;
  TFridge_Hours_data maintenance_hours[MAX_FRIDGE_MAINTENANCE];
  TFridge_compartment_data compartment_data[MAX_FRIDGE_COMPARTMENT];
  TFridge_alarm_data alarm_data;
}TFridge_state;
#endif //FRIDGE_PRESENT
#if defined(TOUCHPRINT_PRESENT)
typedef struct
{
  uint16_t dsr_var_ok;//Data in other Register variable are valid only if variable DSR_Var_OK assume value 2
//--Data logger inputs (analog ones) value range and unit of measure are inputs configuration dependent,
//   please see E2_I1_CFG to E2_I6_CFG parameters value range
  float temp_sensor1;
  float temp_sensor2;
  float temp_sensor3;
  float temp_sensor4;
  float temp_sensor5;
  float temp_sensor6;

  uint8_t dig_sensor;
  float hum_sensor1;
  float hum_sensor2;
  float hum_sensor3;
  float hum_sensor4;
  float hum_sensor5;
  float hum_sensor6;
 //--E2_I1_CFG to E2_I6_CFG parameters value range--
  uint16_t config_sensor1;
  uint16_t config_sensor2;
  uint16_t config_sensor3;
  uint16_t config_sensor4;
  uint16_t config_sensor5;
  uint16_t config_sensor6;
//--parameters of VP truck Controller (DSR)--
  float s1_probe;
  float s2_probe;
  uint16_t maintenance_hours;
  uint16_t total_hours;
  uint16_t road_hours;
  uint16_t stby_hours;
  uint16_t alarm_status;
  float set_point1;
  float set_point2;
  uint16_t  dig_input;
  uint8_t  status;
  uint8_t  op_mode_compartment1;
  uint8_t  op_mode_compartment2;
  uint32_t  state_dig_output;
  uint16_t firmware_version;
  uint16_t firmware_release;
}TTouchprint_state;
#endif //TOUCHPRINT_PRESENT

#if defined(IBOX_PRESENT)
typedef struct
{
  float  fuel_level;
  float battery_potential;
  char     soft_ident[32];
  float total_electric_hours;
  char     component_ident[32];
  float total_vehicle_hours;
  float total_engine_hours;
  //Trailer1 Zone1
  uint8_t  t1_z1_a_zones_alrm_type;
  uint8_t  t1_z1_alrm_code;
  uint8_t  t1_z1_temp_sensor_assign;
  float    t1_z1_return_air1_temp;
  float    t1_z1_supply1_temp;
  float    t1_z1_temp_setpoint;
  float    t1_z1_evaporator_coil_temp;
  float    t1_z1_return_air2_temp;
  float    t1_z1_supply2_temp;
  uint8_t  t1_z1_operating_mode;
  //Trailer1 Zone2
  uint8_t  t1_z2_alrm_type;
  uint8_t  t1_z2_alrm_code;
  uint8_t  t1_z2_temp_sensor_assign;
  float    t1_z2_return_air1_temp;
  float    t1_z2_supply1_temp;
  float    t1_z2_temp_setpoint;
  float    t1_z2_evaporator_coil_temp;
  float    t1_z2_return_air2_temp;
  float    t1_z2_supply2_temp;
  uint8_t  t1_z2_operating_mode;
  //Trailer1 Zone3
  uint8_t  t1_z3_alrm_type;
  uint8_t  t1_z3_alrm_code;
  uint8_t  t1_z3_temp_sensor_assign;
  float    t1_z3_return_air1_temp;
  float    t1_z3_supply1_temp;
  float    t1_z3_temp_setpoint;
  float    t1_z3_evaporator_coil_temp;
  float    t1_z3_return_air2_temp;
  float    t1_z3_supply2_temp;
  uint8_t  t1_z3_operating_mode;
  //DAS or Cargo Watch Sensors Information
  uint8_t  t1_din0_3;
  uint8_t  t1_sens_assign;
  float    t1_temp_sensor1;
  float    t1_temp_sensor2;
  float    t1_temp_sensor3;
  float    t1_temp_sensor4;
  float    t1_temp_sensor5;
  float    t1_temp_sensor6;
  float  t1_humidity_sensor;
  float  t1_fuel_level_sensor;
  uint8_t  mid_t;
}TIbox_state;
#endif //IBOX_PRESENT
#if defined(FLOWMETER_PRESENT)
typedef struct
{
  float     fuel_volume;     //Объем топлива от момента включения датчика (в 0.01 л.)
  float     flow_rate;       //Текущая скорость потока (в 0.1 л/ч.)
  uint16_t  status;          //Статус*
  float     fuel_volume_fc;  //Объем топлива камеры подачи от момента включения датчика (в 0.01 л.) //Fuel volume of the feed chamber
  float     flow_rate_fc;    //Текущая скорость потока камеры подачи (в 0.1 л/ч.)
  float     temp_fc;         //Температура камеры подачи в (в 0.1 °С)
  float     fuel_volume_rc;  //Объем топлива камеры обратки от момента включения датчика (в 0.01 л/ч.)//Fuel volume of the return chamber
  float     flow_rate_rc;    //Текущая скорость потока камеры обратки (в 0.1 л/ч.)
  float     temp_rc;         //Температура камеры обратки в (в 0.1 °С)
  float     acc_fuel_idl;    //Накопленный объем топлива в режиме "Холостой ход" (в 0.01 л)// Accumulated fuel in mode "Idling"
  float     acc_fuel_nom;    //Накопленный объем топлива в режиме "Номинальный" (в 0.01 л.)
  float     acc_fuel_over;   //Накопленный объем топлива в режиме "Перегрузка" (в 0.01 л.)
  float     acc_fuel_ch;     //Накопленный объем топлива в режиме "Накрутка" (в 0.01 л.)//cheat
  float     acc_fuel_neg;    //Накопленный объем топлива в режиме "Отрицательный" (в 0.01 л.)
  float     acc_fuel_fc_idl; //Накопленный объем топлива камеры подачи в режиме "Холостой ход" (в 0.01 л)
  float     acc_fuel_fc_nom; //Накопленный объем топлива камеры подачи в режиме "Номинальный" (в 0.01 л)
  float     acc_fuel_fc_over;//Накопленный объем топлива камеры подачи в режиме "Перегрузка" (в 0.01 л.)
  float     acc_fuel_fc_ch;  //Накопленный объем топлива камеры подачи в режиме "Накрутка" (в 0.01 л.)
  float     acc_fuel_rc_idl; //Накопленный объем топлива камеры обратки в режиме "Холостой ход" (в 0.01 л)
  float     acc_fuel_rc_nom; //Накопленный объем топлива камеры обратки в режиме "Номинальный" (в 0.01 л)
  float     acc_fuel_rc_over;//Накопленный объем топлива камеры обратки в режиме "Перегрузка" (в 0.01 л.)
  float     acc_fuel_rc_ch;  //Накопленный объем топлива камеры обратки в режиме "Накрутка" (в 0.01 л.)
  uint32_t  time_idl;        //Время нахождения в режиме "Холостой ход" (в сек.)
  uint32_t  time_nom;        //Время нахождения в режиме "Номинальный" (в сек.)
  uint32_t  time_over;       //Время нахождения в режиме "Перегрузка" (в сек.)
  uint32_t  time_ch;         //Время нахождения в режиме "Накрутка" (в сек.)
  uint32_t  time_neg;        //Время нахождения в режиме "Отрицательный" (в сек.)
  uint32_t  time_fc_idl;     //Время нахождения камеры подачи в режиме "Холостой ход" (в сек.)
  uint32_t  time_fc_nom;     //Время нахождения камеры подачи в режиме "Номинальный" (в сек.)
  uint32_t  time_fc_over;    //Время нахождения камеры подачи в режиме "Перегрузка" (в сек.)
  uint32_t  time_fc_ch;      //Время нахождения камеры подачи в режиме "Накрутка" (в сек.)
  uint32_t  time_rc_idl;     //Время нахождения камеры обратки в режиме "Холостой ход" (в сек.)
  uint32_t  time_rc_nom;     //Время нахождения камеры обратки в режиме "Номинальный" (в сек.)
  uint32_t  time_rc_over;    //Время нахождения камеры обратки в режиме "Перегрузка" (в сек.)
  uint32_t  time_rc_ch;      //Время нахождения камеры обратки в режиме "Накрутка" (в сек.)
  uint32_t  time_interv;     //Время нахождения в режиме "Вмешательство" (в сек.)//Time spent in the "Intervention" mode
  uint32_t  time_sens;       //Время работы датчика (в сек.)
  uint32_t  serial_num;      //Серийный номер прибора
  uint16_t  device_type;      //Тип прибора
}TFlowmeter_state;
#endif //FLOWMETER_PRESENT

#if defined(HUM_SENSOR_IAKM_PRESENT)
typedef struct
{
  float    temp_sensor;      //Температура, °C
  uint8_t  temp_error_sensor;//Коды ошибок канала температуры
  float    hum_sensor;       //Относительная влажность, %
  uint8_t  hum_error_sensor; //Коды ошибок канала относительной влажности
}THum_iakm_state;
#endif //HUM_SENSOR_IAKM_PRESENT

#if defined(SCD30_PRESENT)
typedef struct
{
  float    temp_sensor;      //Температура, °C
  float    hum_sensor;       //Относительная влажность, %
  float    co2_sensor;       //CO2
}TScd30_state;
#endif //SCD30_PRESENT


#if defined(THERMALMASTER_PRESENT)
typedef struct
{
  uint16_t  instemp;   //1.Current inside temperature
  uint16_t  outind;    //2.Output indication value
  uint16_t  voltage;   //3.Voltage
  uint16_t  reftemp;   //4.Current refrigerated temperature
  uint32_t  ophoursw;  //5.Current operation hours
  uint16_t  defrtemp;  //6.Current defrost temperature
  uint16_t  settemp;   //7.Set temperature
  uint8_t   defrcycle; //8.Defrost cycle
  uint8_t   defrtime;  //9.Defrost time min*10
  uint8_t   onoff;     //10.ON/OFF value
  uint16_t  tempcorr;  //11.Temperature correction value
  uint8_t   status;    //12.Refrigeration unit status
}Thermalmaster_state;
#endif //defined(THERMALMASTER_PRESENT)

#if defined(CARRIER_PRESENT)
typedef struct
{
  uint8_t    source;           //Адрес источника команды (22H)
  uint8_t    command_id;       //ID команды ХОУ Carrier
  uint8_t    data[6];          //содержимое команды
}command_t;

typedef struct
{
  uint8_t    source;           //Адрес источника статуса команды ( ХОУ AAH)
  uint8_t    command_id;       //ID команды ХОУ Carrier
  uint8_t    status[3];        //статуса выполнения конфигурационной команды
}command_ans_t;

typedef struct
{
  uint8_t   over_stat;
  uint8_t   mode_op_comp1;
  uint8_t   mode_op_comp2;
  uint8_t   mode_op_comp3;
  uint8_t   engine_status;
  uint8_t   alarm_status;
  uint8_t   num_act_alarms;
  uint8_t   alarm_byte1;
  uint8_t   alarm_byte2;
  uint8_t   alarm_byte3;
  uint8_t   alarm_byte4;
  uint8_t   alarm_byte5;
  uint8_t   alarm_byte6;
  uint8_t   alarm_byte7;
  uint8_t   alarm_byte8;
  uint8_t   alarm_byte9;
  uint8_t   alarm_byte10;
  uint8_t   alarm_byte11;
  uint8_t   alarm_byte12;
  uint8_t   alarm_byte13;
  uint8_t   alarm_byte14;
  uint8_t   alarm_byte15;
  uint8_t   alarm_byte16;
  uint8_t   num_inact_alarms;
  uint8_t   ina_alarm_byte1;
  uint8_t   ina_alarm_byte2;
  uint8_t   ina_alarm_byte3;
  uint8_t   ina_alarm_byte4;
  uint8_t   ina_alarm_byte5;
  uint8_t   ina_alarm_byte6;
  uint8_t   ina_alarm_byte7;
  uint8_t   ina_alarm_byte8;
  uint8_t   ina_alarm_byte9;
  uint8_t   ina_alarm_byte10;
  uint8_t   ina_alarm_byte11;
  uint8_t   ina_alarm_byte12;
  uint8_t   ina_alarm_byte13;
  uint8_t   ina_alarm_byte14;
  uint8_t   ina_alarm_byte15;
  uint8_t   ina_alarm_byte16;
  char      trailer_id[12];
  char      unit_sn[12];
  char      soft_rev[9];
  float     comp1_sp;
  float     comp2_sp;
  float     comp3_sp;
  float     ambient;
  float     comp1_ret;
  float     comp1_sup;
  float     comp2_ret;
  float     comp2_sup;
  float     comp3_ret;
  float     comp3_sup;
  float     rem_sens1;
  float     rem_sens2;
  float     rem_sens3;
  float     bat_vol;
  uint8_t   fuel_level;
  uint8_t   door_sw;
  uint8_t   remote_sw1;
  uint8_t   remote_sw2;
  uint8_t   remote_sw3;
  uint8_t   remote_sw4;
  float     eng_coolant;
  float     rpm;
  uint32_t  eng_hours;
  uint32_t  stby_hours;
  uint32_t  sw_on_hours;
  uint32_t  prev_hour_m1;
  uint32_t  prev_hour_m2;
  uint32_t  prev_hour_m3;
  uint32_t  prev_hour_m4;
  uint32_t  prev_hour_m5;
  char      act_int_set[21];
  char      act_int_fn[26];
  uint8_t   act_nm_int;
  char      int1_name[21];
  char      int2_name[21];
  char      int3_name[21];
  char      int4_name[21];
  char      int5_name[21];
  char      int6_name[21];
  char      int7_name[21];
  char      int8_name[21];
  char      int9_name[21];
  char      int10_name[21];
  char      int11_name[21];
  char      int12_name[21];
  char      int13_name[21];
  char      int14_name[21];
  char      int15_name[21];
  char      int16_name[21];
  char      int17_name[21];
  char      int18_name[21];
  char      int19_name[21];
  char      int20_name[21];
  char      int21_name[21];
  char      int22_name[21];
  char      int23_name[21];
  char      int24_name[21];
  char      int25_name[21];
  char      int26_name[21];
  char      int27_name[21];
  char      int28_name[21];
  char      int29_name[21];
  char      int30_name[21];
  char      int31_name[21];
  uint8_t   c1_contr_probe;
  uint8_t   temp_units;
  uint8_t   press_units;
  uint8_t   air_flow;
  uint8_t   defrost_int;
  uint8_t   fresh_protect;
  uint8_t   c2_contr_probe;
  uint8_t   afax_status;
  uint8_t   afax_setting;
  uint8_t   afax_cfm_contr;
  uint8_t   pret_test_num;
  uint8_t   pret_subtest_num;
  uint16_t  pret_time_disp;
  uint8_t   pret_type;
  uint8_t   over_pret_stat;
  uint8_t   pret_test[25];//Test 1 – Test 25
  char      model_num[13];
  uint8_t   oper_mode;
  uint8_t   oper_mode_lock;
  uint8_t   c_piek_ov_stat;
  uint8_t   c_piek_m_stat;
  float     suct_press;
  float     disch_press;
  float     disch_temp;
  uint8_t   smv_pos;
  uint8_t   exp_valve_pos;
  float     outl_temp;
  float     outl_press;
  float     dtt1;
  float     dtt2;
  float     comp2_dtt;
  float     comp3_dtt;
  float     bat_curr;
  float     ac_curr1;
  float     ac_curr2;
  uint16_t   map_sens;
  uint8_t   len_act_alarms; //Length of Active Alarm Queue/Index of Inactive Alarm Queue
  uint16_t  a_alarm_byte1; //Alarm Number 1
  uint16_t  a_alarm_byte2; //Alarm Number 2
  uint16_t  a_alarm_byte3; //Alarm Number 3
  uint16_t  a_alarm_byte4; //Alarm Number 4
  uint16_t  a_alarm_byte5; //Alarm Number 5
  uint16_t  a_alarm_byte6; //Alarm Number 6
  uint16_t  a_alarm_byte7; //Alarm Number 7
  uint16_t  a_alarm_byte8; //Alarm Number 8
  uint16_t  a_alarm_byte9; //Alarm Number 9
  uint16_t  a_alarm_byte10; //Alarm Number 10
  uint16_t  a_alarm_byte11; //Alarm Number 11
  uint16_t  a_alarm_byte12; //Alarm Number 12
  uint16_t  a_alarm_byte13; //Alarm Number 13
  uint16_t  a_alarm_byte14; //Alarm Number 14
  uint16_t  a_alarm_byte15; //Alarm Number 15
  uint16_t  a_alarm_byte16; //Alarm Number 16
  char      kohler_pcod1[6];
  char      kohler_pcod2[6];
  char      kohler_pcod3[6];
  char      kohler_pcod4[6];
  char      kohler_pcod5[6];
  char      kohler_pcod6[6];
  char      kohler_pcod7[6];
  char      kohler_pcod8[6];
  char      kohler_pcod9[6];
  char      kohler_pcod10[6];
  char      kohler_pcod11[6];
  char      kohler_pcod12[6];
  char      kohler_pcod13[6];
  char      kohler_pcod14[6];
  char      kohler_pcod15[6];
  char      kohler_pcod16[6];
  uint8_t   type_mp;
  uint8_t   fuel_level_a;
  command_t command;
  command_ans_t command_ans;
}TCarrier_state;
#endif //CARRIER_PRESENT

#if defined(ZANOTTI_MODBUS_PRESENT)
typedef struct
{
  uint16_t regmode;
  float    realtemp;
  float    targtemp;
  uint32_t timeupdate;
  uint32_t timecurrent;
  uint16_t commerrorcnt;
  uint16_t regerror1;
  uint16_t regerror2;
  uint16_t regerror3;
  uint16_t regerror4;
  uint16_t regerror5;
  uint16_t regerror6;
  uint16_t regerror7;
  uint16_t regerror8;
  uint32_t flags;
}TZanotti_state;
#endif //ZANOTTI_MODBUS_PRESENT

#if defined(SR3_PRESENT)
typedef struct
{
  uint8_t  mode_op_comp1; //Mode of Operation – Compartment 1 Режим  отсека №1
  uint8_t  mode_op_comp2; //Mode of Operation – Compartment 2 Режим  отсека №2
  uint8_t  engine_status; //Engine / Standby Status Режим питания ХОУ Дизельный или электрический
  uint8_t  run_mode;      // Режим поддержания Т  0 = Старт/Стоп; 1 = Непрерывный;
  uint8_t  speed_mode;    // Скорость двигателя 0 = Низкая; 1 = Высокая;
  float    eng_coolant;   //SR3_ENG_COOLANT        Температура  двигателя
  uint16_t rpm;           //SR3_ENG_RPM            Обороты двигателя
  uint8_t  alarm_status;  //SR3_ALRM_STAT          Статус сигналов ХОУ 0 - нет активных сигналов, 1 - информационные сигналы, 2 - аварийное откл
  uint8_t  num_act_alarms;//SR3_NUM_ALRMS          Количество активных сигналов
  uint8_t  signal1;       //SR3_SIGNAL1            Сигнал №1
  uint8_t  signal2;       //SR3_SIGNAL2            Сигнал №2
  uint8_t  signal3;       //SR3_SIGNAL3            Сигнал №3
  uint8_t  signal4;       //SR3_SIGNAL4            Сигнал №4
  uint8_t  signal5;       //SR3_SIGNAL5            Сигнал №5
  float    comp1_sp;      //SR3_COMP_SP1           Заданная температура отсек №1
  float    comp2_sp;      //SR3_COMP_SP1           Заданная температура отсек №2
  float    ambient;       //SR3_AMBIENT_TEMP       Наружная температура
  float    comp1_sup;     //SR3_TEMP_COMP1_SUPL    Температура подачи отсек №1
  float    comp1_ret;     //SR3_TEMP_COMP1_RET     Температура возврата отсек №1
  float    comp2_sup;     //SR3_TEMP_COMP2_SUPL    Температура подачи отсек №2
  float    comp2_ret;     //SR3_TEMP_COMP2_RET     Температура возврата отсек №2
  uint8_t  door_sw1;      // SR3_DOOR_SW1           Door Switch Status Состояние двери отсек №1
  uint8_t  door_sw2;      // SR3_DOOR_SW2           Состояние двери отсек №2
  uint32_t sw_on_hours;   //SR3_SW_ON_HOURS        Total Switch On Hours Общие часы ХОУ
  uint32_t eng_hours;     //SR3_ENG_HOURS          Total Engine Hours Дизель-часы ХОУ
  uint32_t stby_hours;    //SR3_STBY_HOURS         Total Standby Hours Электро-часы ХОУ
  uint16_t prev_hour_m1;  //SR3_PREV_HOUR_M_1      ТО по счётчику №1
  uint16_t prev_hour_m2;  //SR3_PREV_HOUR_M_2      ТО по счётчику №2
  uint16_t lim_hour_m1;   //SR3_LIM_HOUR_M_1       Выставленный предел в М\Ч до ТО счётчик №1
  uint16_t lim_hour_m2;   //SR3_LIM_HOUR_M_2       Выставленный предел в М\Ч до ТО счётчик №2
  uint16_t suct_press;    //SR3_SUCT_PRESS         Suction Pressure Давление всасывания компрессора
  uint16_t disch_press;   //SR3_DISCH_PRESS        Discharge Pressure Давление нагнетания компрессора
  uint8_t  lev_coolant;   //SR3_LEVEL_COOLANT      Уровень ОЖ ДВС
  uint8_t  lev_oil;       //SR3_LEVEL_OIL          Уровень масла
  uint8_t  oil_press;     //SR3_OIL_PRESS          Давление масла
  uint8_t  on_el;         //SR3_ON_EL              Электросеть подключена
  uint8_t  max_defrost;   //SR3_MAX_DEFROST        Максимальное время оттайки
  float    bat_curr;      //SR3_BATT_CURRENT       Battery DC Current Ток АКБ ХОУ
  uint16_t bat_vol;       //SR3_BATTERY_VOL        Напряжение АКБ ХОУ
  float    outlet_temp1;  //SR3_OUTL_TEMP1         Температура испарителя 1
  float    outlet_temp2;  //SR3_OUTL_TEMP2         Температура испарителя 2
  uint16_t soft_rev;      //SR3_SOFT_REV           Версия ПО ХОУ
  uint16_t soft_rel;      //SR3_SOFT_REALISE       Релиз ПО ХОУ
  uint8_t  mp;            //SR3_TYPE_MP            Тип МП ХОУ
}TSR3_state;
#endif //SR3_PRESENT

#if defined(MDAS9_PRESENT)
typedef struct
{
  uint8_t   speed;
  uint8_t   left_turn;
  uint8_t   right_turn;
  uint8_t   brake;
  uint16_t  rpm;
  uint8_t   ldw_left;
  uint8_t   ldw_right;
  uint16_t  left_dist;
  uint16_t  right_dist;
  uint8_t   ttc;
  uint8_t   sda;
  uint8_t   fvsa;
  uint8_t   fpw;
  uint8_t   fcw;
  uint8_t   pcw;
  uint8_t   record;
  uint8_t   error_code;
  uint16_t  ahead_dist;
  uint16_t  ahead_speed;
  uint8_t   slr_state;
  uint8_t   slr_recog;
  uint8_t   slr_sensit;
}TMDAS9_state;
#endif //MDAS9_PRESENT

#if defined(MDSM7_PRESENT)
typedef struct
{
  uint8_t  speed;             //Speed (0 ~ 255 /0x00 – 0xFF)
  //uint16_t dummy_carinfo;     //For Extension
  char     car_info[3];       //Dummy_CARINFO For Extension
  uint8_t  event;             //DSM Algorithm Status
                              //0bit – Drowsiness (0: none, 1: active)
                              //1bit – Distraction (0: none, 1: active)
                              //2bit – Yawning (0: none, 1: active)
                              //3bit – Phone (0: none, 1: active)
                              //4bit – Smoking (0: none, 1: active)
                              //5bit – Driver Empty (0: none, 1: active)
                              //6~7bit: dummy
  char driver_name[10];       //Driver Name String ASCII, None Data Byte mustbe 0x00.
  uint8_t  record_status;     //Record Status 0x00- 0x02
                              //0x00 – None
                              //0x01 – Connected & Data None
                              //0x02 – Connected & Data Valuable
  uint8_t gps_status;        //GPS Status 0x00- 0x02
                              //0x00 – None
                              //0x01 – Connected & Data None
                              //0x02 – Connected & Data Valuable
  uint32_t latitude;          //GPS Latitude Data/10000 (Decimal Point 4)
  uint32_t longitude;         //GPS Longitude Data/10000 (Decimal Point 4)
  uint8_t error_code;         //Error Code
                              //0bit – Camera Fail (0: none, 1: active)
                              //1bit – GPS Fail (0: none, 1: active)
                              //2bit – CAN Fail (0: none, 1: active)
                              //3bit – Low visibility (0: none, 1: active)
                              //4bit – Camera Blocked (0: none, 1: active)
                              //5~7bit: Dummy
  uint16_t year;              //Year
  uint8_t mon;                //Month
  uint8_t day;                //Day
  uint8_t hour;               //Hour
  uint8_t min;                //Min
  uint8_t sec;                //Sec
  uint8_t g_sensor;           //G-Sensor Event 0x00- 0x01
}MDSM7_state;
#endif //MDSM7_PRESENT

#if defined(YANDEX_EXTENDED_SENSORS_PRESENT)
typedef struct
{
  //0-закрыт, 1-открыт
  uint8_t driver_door:1;
  uint8_t driver_door_prepared:1;

  uint8_t pass_door:1;
  uint8_t pass_door_prepared:1;

  uint8_t l_rear_door:1;
  uint8_t l_rear_door_prepared:1;

  uint8_t r_rear_door:1;
  uint8_t r_rear_door_prepared:1;
}doors_bf_t;

typedef struct
{
  //0-закрыт, 1-открыт
  uint8_t sunroof:1;
  uint8_t sunroof_prepared:1;

  uint8_t reserve:6;
}ext_doors_bf_t;
#endif //defined(YANDEX_EXTENDED_SENSORS_PRESENT)

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
typedef struct
{
  uint8_t inteface_state; //Cвязь с платой BLE: 0 - плата не отвечает, 1 - Ok, 2 - отключена в настройках
  uint8_t firmware_download_pct; //Процент загрузки файла прошивки
  char boot_vers[24]; //Версия загрузчика
  char app_vers[24]; //Версия прошивки
  ext_board_ble_mac_t board_mac; //MAC адресс платы
  ext_board_ble_info_t ble_conn_status; //Состояние соединения BLE
#if (MAX_EXTERNAL_BOARD_BLE_BEACONS_INFO_COUNT > 0)
  ext_board_ble_beacon_info_t beacons_info[MAX_EXTERNAL_BOARD_BLE_BEACONS_INFO_COUNT][6]; //должно быть меньше MAX_SENSOR_R_SIZE
#endif //(MAX_EXTERNAL_BOARD_BLE_BEACONS_INFO_COUNT > 0)
  uint8_t session_key[32];
}TExt_board_ble_state;

typedef struct
{
  uint8_t inteface_state;
  uint8_t firmware_download_pct;
  char boot_vers[16];
  char app_vers[16];
  ext_board_ble_mac_t board_mac;
}TExt_board_ble_state_res;

typedef struct
{
  char ble_name[32]; //null terminated
  char ble_passkey[7]; //6 digit fixed, 0..9 digit only, null terminated
  uint8_t ble_min_security_conn_level; //0 - No protection, open link, 1 - Encryption, but no MITM protection, 2 - Encryption and MITM protectionб 3 - LESC encryption and MITM protection
}TExt_board_ble_settings;
#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)

typedef struct
{
  uint8_t is_on;              // Включен ли модем
  uint8_t is_sleep;           // Включен ли менеджер
  uint8_t gsensor_is_ok;      // Работоспособен ли акселерометр

  Power_source power_source;   // Текущий источник питания

#if (MAX_ANALOG_INPUTS > 0)
  float analog_input[MAX_ANALOG_INPUTS]; //Аналоговый вход Х
#endif //(MAX_ANALOG_INPUTS > 0)

#if (MAX_ANALOG_OUTPUTS > 0)
  float analog_output[MAX_ANALOG_OUTPUTS]; //Аналоговый выход Х
#endif //(MAX_ANALOG_OUTPUTS > 0)

#if (MAX_DIGITAL_INPUTS > 0)
  uint8_t digital_input[MAX_DIGITAL_INPUTS];  //Цифровой вход Х
#endif //(MAX_DIGITAL_INPUTS > 0)

#if (MAX_DIGITAL_OUTPUTS > 0)
  uint8_t digital_output[MAX_DIGITAL_OUTPUTS]; //Цифровой выход Х
#endif //(MAX_DIGITAL_OUTPUTS > 0)

#if (MAX_PULSE_INPUTS > 0)
  uint32_t pulse_input[MAX_PULSE_INPUTS]; //Импульсный вход Х
#endif //(MAX_PULSE_INPUTS > 0)

#if (MAX_FREQ_INPUTS > 0)
  float freq_input[MAX_FREQ_INPUTS];   //Частотный вход X
#endif //(MAX_FREQ_INPUTS > 0)

#if (MAX_FREQ_OUTPUTS > 0)
  uint16_t freq_output[MAX_FREQ_OUTPUTS]; //Частотный выход Х
#endif //(MAX_FREQ_OUTPUTS > 0)

  #if defined(BR_PRESENT)
  uint8_t is_br_ok; //0 - не найден, 1 - блок расширения отвечает, 2 - блок расширения отключен  в настройках
  uint8_t ext_digital_input[MAX_EXT_DIGITAL_INPUTS];  //Расширенный цифровой вход Х
  float ext_analog_input[MAX_EXT_ANALOG_INPUTS]; //Расширенный аналоговый вход Х
  float ext_freq_input[MAX_EXT_FREQ_INPUTS];   //Расширенный частотный вход X
  uint32_t ext_pulse_input[MAX_EXT_PULSE_INPUTS]; //Расширенный импульсный вход Х
  uint8_t ext_digital_output[MAX_EXT_DIGITAL_OUTPUTS]; //Расширенный цифровой выход Х
  #endif //BR_PRESENT

  #if defined(FRIDGE_PRESENT)
  uint8_t is_fridge_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TFridge_state fridge_state;
  #endif //FRIDGE_PRESENT

  #if defined(LORA_SENSORS_PRESENT)
  uint8_t is_lora_sens_ok; //0 - не найден, 1 - шлюз отвечает, 2 - шлюз отключен  в настройках
  TLoraSens_state lora_sens[MAX_LORA_SENSORS_COUNT];
  uint8_t lora_coproc_firmware_version[32];
  float lora_sens_hum[MAX_LORA_SENSORS_COUNT];    // Влажность
  #endif //LORA_SENSORS_PRESENT

  #if defined(REPORT_GEN_PRESENT)
  TLoraSens_state_bits lora_sens_bits[MAX_LORA_SENSORS_COUNT];
  #endif //REPORT_GEN_PRESENT

  #if defined(NRF_BEACONS_SCANNER_PRESENT)
  uint8_t is_nrf_beacons_scanner_ok; //0 - не найден, 1 - шлюз отвечает, 2 - шлюз отключен в настройках
  TNrfBeaconsScanner_state  nrf_beacons_scanner;
  #endif //NRF_BEACONS_SCANNER_PRESENT

  #if defined(TENZO_M_WEIGHER_PRESENT)
  uint8_t is_tenzo_m_weigher_ok; //0 - не найдены, 1 - отвечают, 2 - отключены в настройках
  TTenzoMWegher_state  tenzo_m_weigher;
  #endif //TENZO_M_WEIGHER_PRESENT

  #if defined(PROCESSOR_WEIGHT_PV310_PRESENT)
  uint8_t is_pv310_weigher_ok; //0 - не найдены, 1 - отвечают, 2 - отключены в настройках
  TPV310Wegher_state  pv310_weigher;
  #endif //PROCESSOR_WEIGHT_PV310_PRESENT

  #if (defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))
  uint8_t is_iqfreeze_ok; //0 - iqfreeze не найден, 1 - iqfreeze отвечает, 2 - отключен  в настройках
  TIqfreeze_state iqfreeze;
  #endif //(defined(IQFREEZE_PRESENT)||(defined(IQFREEZE_MODBUS_PRESENT)))

  #if defined(TPMS_PRESENT)
  uint8_t is_tpms_ok; //0 - tpms не найден, 1 - tpms отвечает, 2 - отключен  в настройках
  TTPMS_state tpms;
  #endif //TPMS_PRESENT

  #if defined(TOUCHPRINT_PRESENT)
  uint8_t is_touchprint_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TTouchprint_state touchprint_state;
  #endif //TOUCHPRINT_PRESENT

  #if defined(IBOX_PRESENT)
  uint8_t is_ibox_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TIbox_state ibox_state;
  #endif //IBOX_PRESENT

  #if defined(INTERNAL_NRF_PRESENT)
  TInternalNrf_state nrf_state;
  #endif //INTERNAL_NRF_PRESENT

  uint8_t dut_is_ok[MAX_DUTS];         //Работоспособен ли ДУТ X
  float dut[MAX_DUTS];                 //Датчик уровня топлива Х
  int8_t dut_temp[MAX_DUTS];           //Температура датчик уровня топлива Х

  #if defined(MAX_DALLAS_TEMP_SENSORS) && (MAX_DALLAS_TEMP_SENSORS > 0)
  float ext_temp_sensor[MAX_DALLAS_TEMP_SENSORS]; //Датчик температуры внешний Х
  #endif

  #if MAX_TAMPERS > 0
  uint8_t tamper[MAX_TAMPERS];         //Датчик вскрытия Х
  #endif
  uint8_t ignition;     //Зажигание
  #if defined(NAVITERM_PRESENT)
  uint8_t door_opened;                 // Двери открыты
  uint8_t in_btn;                      // Кнопка подтверждения
  uint8_t vsense;                      // Датчик объема
  uint8_t in_auto;                     // Контроль автозапуска
  uint8_t s_mode;                      // Программный режим
  uint8_t h_mode;                      // Аппаратный режим
  #endif //(NAVITERM_PRESENT)

  float gsensor_axis_x; //Датчик оси акселерометра X
  float gsensor_axis_y; //Датчик оси акселерометра Y
  float gsensor_axis_z; //Датчик оси акселерометра Z
  //float green_driving;  //Датчик качества вождения
  //uint8_t crash_sensor; //Датчик аварии
  uint8_t evacuation_sensor; //Датчик эвакуации
  uint8_t gsensor_move_sensor;   //Датчик движения акселерометра
  float internal_temp; //Датчик температуры внутренний
  uint8_t alarm_button; //Тревожная кнопка
  float external_voltage; //Датчик напряжения бортовой сети
  float internal_acc_voltage; //Датчик напряжения встроенного аккумулятора
  uint8_t internal_acc_charge_level; //Датчик заряда встроенного аккумулятора
  uint8_t autorization_sensor; //Датчик авторизации
  char current_autorized_key[17]; //Текущий авторизованный ключ \0
  char current_ow_key[17]; //Текущий ключ \0
  uint8_t any_ow_key_present; //Поднесен любой ключ 1-wire
  float motohours; //Датчик моточасов
  uint8_t accel_no_init;

#if defined(YANDEX_EXTENDED_SENSORS_PRESENT)
  uint8_t selector_position; //'U' - Unknown, 'P' - Parking, 'D' - Drive, 'R' - Reverse, 'N' - Neutral, 'M' - Manual , 'S' - Sport
  uint8_t key_position; //'U' - Unknown, 'O' - Off, 'A' - ACC, 'I' - IGN, 'S' - STARTER, пока для Genesis
  doors_bf_t windows_position;
  doors_bf_t cls_state;
  ext_doors_bf_t roof;
  uint8_t launch_sensor;
  uint8_t drift_sensor;
  uint8_t drive_on_handbrake;
  uint8_t eco_mode;
  uint8_t alarm;
  uint8_t unauthorized_start_sensor;
  uint8_t tank_flap:1;
  uint8_t washer_liquid;
  uint8_t emergency_lights;
  uint16_t current_audio_file;
  uint8_t current_audio_script;
  char vin_sensor[18];
  char dtc_list[2][64];
#endif //YANDEX_EXTENDED_SENSORS_PRESENT

#if defined(FLOWMETER_PRESENT)
  uint8_t is_flowmeter_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TFlowmeter_state flowmeter_state;
#endif //FLOWMETER_PRESENT

#if defined(IRIDIUM_PRESENT)
  uint8_t is_iridium_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  uint16_t iridium_mess_state; //статус передачи сообщения через IRIDIUM: Попытка отправки через IRIDIUM успешна/неуспешна + код ошибки
  uint32_t iridium_traffic;    //счетчик трафика через IRIDIUM в байтах
#endif //IRIDIUM_PRESENT

#if defined(CARRIER_PRESENT)
  uint8_t is_carrier_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TCarrier_state carrier_state;
#endif //CARRIER_PRESENT

#if defined(MDAS9_PRESENT)
  uint8_t is_mdas9_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TMDAS9_state mdas9_state;
#endif //MDAS9_PRESENT

#if defined(MDSM7_PRESENT)
  uint8_t is_mdsm7_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  MDSM7_state mdsm7_state;
#endif //MDSM7_PRESENT

#if defined(OMNICOMM_ICON_PRESENT)
  uint8_t is_omnicomm_icon_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
#endif //OMNICOMM_ICON_PRESENT

#if defined(DISPLAY_PRESENT)
  uint8_t is_display_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
#endif //DISPLAY_PRESENT

#if defined(HUM_SENSOR_IAKM_PRESENT)
  uint8_t is_hum_iakm_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  THum_iakm_state hum_iakm_state;
#endif //HUM_SENSOR_IAKM_PRESENT

#if defined(SCD30_PRESENT)
  uint8_t is_scd30_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TScd30_state scd30_state;
#endif //SCD30_PRESENT

#if defined(THERMALMASTER_PRESENT)
  uint8_t is_thermalmaster_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  Thermalmaster_state thermalmaster_state;
#endif //defined(THERMALMASTER_PRESENT)

#if defined(ZANOTTI_MODBUS_PRESENT)
  uint8_t is_zanotti_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TZanotti_state zanotti_state;
#endif //defined(ZANOTTI_MODBUS_PRESENT)

#if defined(SR3_PRESENT)
  uint8_t is_sr3_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках
  TSR3_state sr3_state;
#endif //defined(SR3_PRESENT)

#if defined(EXCERA_PRESENT)
  uint8_t is_excera_ok; //0 - не найден, 1 - отвечает, 2 - отключен  в настройках, 3 - не используется
#endif //EXCERA_PRESENT
} TSignal_state;

// Текущее состояние центральной задачи
typedef struct
{
  uint8_t is_sleep;                 // Включен ли менеджер
  uint8_t mcu_firmware_version[50]; //Версия прошивки MCU
  uint32_t serial_num;              //Серийный номер
  uint32_t uptime;                  //Время с последней перезагрузки
  uint32_t current_time;            //Текущее время
  uint8_t operation_mode;           //Режим работы устройства
  uint32_t message_count_in_bb[MAX_SERVERS_COUNT];      //Количество записей в черном ящике для каждого из серверов
  uint32_t can1_last_activity;
  uint32_t can2_last_activity;
  uint32_t can3_last_activity;
  uint32_t flash_se_max_time;
  //uint32_t battery_capacity; //TODO:
  float    wear_condition;
} TSyscom_state;

typedef struct
{
  uint8_t    stat_transm;
  uint16_t   outp_spd_kp;
  uint8_t    clutch_slip;
  uint16_t   input_spd_kp;
  uint8_t    sel_gear;
  uint16_t   gear_ratio;
  uint8_t    current_gear;
  uint8_t    shift_finger;
  uint8_t    neutral_ind;
  uint8_t    nmv_state;
  uint8_t    pto1_state;
  uint8_t    pto2_state;
  uint8_t    park_brake;
  uint8_t    clutch_sw;
  uint8_t    brake_sw;
  uint8_t    cc_active;
  uint8_t    pto_state;
  uint8_t    abs_active;
  uint8_t    bp_position;
  uint8_t    shift_rear;
  uint8_t    fuel_low;
  uint8_t    emergency_sh;
  uint8_t    act_eng_torque;
  uint8_t    ap_kick_sw;
  uint8_t    ap_low_idle;
  uint8_t    load_curr;
  uint8_t    eng_oil_press;
  float      fuel_temp;
  float      oil_temp;
  uint8_t    supply_press;
  uint8_t    park_press;
  uint8_t    s_brake_press1;
  uint8_t    s_brake_press2;
  uint8_t    aux_press;
  uint8_t    air_s_press;
  uint8_t    bar_press;
  float      amb_air_temp;
  uint8_t    sec;
  uint8_t    min;
  uint8_t    hrs;
  uint8_t    mon;
  uint8_t    day;
  uint8_t    year;
  uint8_t    sh_min_gmt;
  uint8_t    sh_hrs_gmt;
  uint32_t   trip_dist;
  uint8_t    axle_loc_ebs;
  uint16_t   axle_weight_ebs;
  uint8_t    axle_loc_eac;
  uint16_t   axle_weight_eac;
  uint8_t    motion_detect;
  uint8_t    driver2_status;
  uint8_t    driver1_status;
  uint8_t    over_speed;
  uint8_t    driver1_card;
  uint8_t    driver1_data;
  uint8_t    driver2_card;
  uint8_t    driver2_data;
  uint8_t    system_oper;
  uint8_t    service_mess;
  uint8_t    system_events;
  uint16_t   veh_speed;
  uint8_t    act_rx_torque;
  uint8_t    query_support;
  uint8_t    diag_support;
  char       ver_soft[6];
  char       vin[18];
  uint8_t    ret_torq_re;
  uint8_t    ret_torq_rd;
  uint16_t   fuel_rate;
  uint16_t   ins_fuel_eco;
  uint8_t    low_oil_press;
  uint8_t    high_coolant_t;
}TCan_man_data;

typedef struct
{
  TCan_security_flags    sec_flags;     //Флаги состояния безопасности
  TCan_controls_flags    cont_flags;    //Флаги состояния контроля
  TCan_agriculture_flags agr_flags;     //Флаги состояния сельскохозяйственного оборудования

  /*
  uint8_t                agr_flags_asis[12];   //Флаги состояния сельскохозяйственного оборудования, как есть
  uint8_t                spec_falgs_asis[19]; //Специальные Флаги, как есть
  uint8_t                sec_flags_asis[7];   //Флаги состояния безопасности, как есть
  uint8_t                cont_flags_asis[16]; //Флаги состояния контроля, как есть
  */

  uint64_t               agr_navis_flags;
  uint32_t               cont_navis_flags;
  uint32_t               sec_navis_flags;
  uint32_t               spec_flags;

#if defined(EXTENDED_ADVANTUM_CAN_PRESENT)
  TCan_man_data          man_data;
#endif //EXTENDED_ADVANTUM_CAN_PRESENT

  TCan_numeric_data      numeric_data;  //Числовые данные
  TCan_sys_inf           sys_inf;       //Информация о сопроцессоре
  float fw_update_pct;                  //проценты зазрузки ПО при обновлении
}TCan_state;


#if defined(CANLOG_V4_PRESENT)
typedef struct
{
  canlog_system_info_type_t               system_info;

#if defined(CANLOG_V4_CONNECTIONS_FLAGS_PRESENT)
  canlog_can_connection_flags_t           can_connection_flags;
#endif //defined(CANLOG_V4_CONNECTIONS_FLAGS_PRESENT)

#if defined(CANLOG_V4_SECURITY_FLAGS_PRESENT)
  canlog_security_flags_t                 security_flags;
#endif //defined(CANLOG_V4_SECURITY_FLAGS_PRESENT)

#if defined(CANLOG_V4_STATE_FLAGS_PRESENT)
  canlog_state_flags_t                    state_flags;
#endif //defined(CANLOG_V4_STATE_FLAGS_PRESENT)

#if defined(CANLOG_V4_INDICATORS_FLAGS_PRESENT)
  canlog_state_indicators_flags_t         state_indicators_flags;
#endif //defined(CANLOG_V4_INDICATORS_FLAGS_PRESENT)

#if defined(CANLOG_V4_AGRICULTURAL_VEHICLES_FLAGS_PRESENT)
  canlog_agricultural_vehicles_flags_t    agricultural_vehicles_flags;
#endif //defined(CANLOG_V4_AGRICULTURAL_VEHICLES_FLAGS_PRESENT)

#if defined(CANLOG_V4_UTILITY_VEHICLES_FLAGS_PRESENT)
  canlog_utility_vehicles_flags_t         utility_vehicles_flags;
#endif //defined(CANLOG_V4_UTILITY_VEHICLES_FLAGS_PRESENT)

#if defined(CANLOG_V4_CISTERN_FLAGS_PRESENT)
  canlog_cisterns_flags_t                 cisterns_flags;
#endif //defined(CANLOG_V4_CISTERN_FLAGS_PRESENT)

#if defined(CANLOG_V4_NUMERIC_DATA_PRESENT)
  canlog_numeric_data_t                   numeric_data;
#endif //defined(CANLOG_V4_NUMERIC_DATA_PRESENT)

#if defined(CANLOG_V4_AGR_NUMERIC_DATA_PRESENT)
  canlog_agricultural_numeric_data_t      agr_numeric_data;
#endif //defined(CANLOG_V4_AGR_NUMERIC_DATA_PRESENT)

#if defined(CANLOG_V4_TACHO_PRESENT)
  canlog_tacho_data_t                     tacho_data;
#endif //defined(CANLOG_V4_TACHO_PRESENT)
}TCanLog_v4_state;
#endif //CANLOG_V4_PRESENT

typedef struct
{
  uint32_t id_type:3;
  uint32_t value:29;
  uint32_t mask:29;
  uint32_t is_enable:1;
}can_fwd_settings_t;

typedef struct
{
  uint8_t bt_enable:1;       //разрешить работу BT модуля
  uint8_t spp_enable:1;      //разрешить подключение по SPP по BT Classic
  uint8_t visible:1;         //видимость BT+BLE
  uint8_t reserve:1;         //резерв
  uint8_t le_scan_period:4;  //период сканирования BLE, 0 - не сканировать, шаг 5 секунд
}modem_bt_settings_t;

typedef union
{
  struct
  {
    uint8_t immo_type;              // тип: 1 - Nissan/Renault
    uint8_t can_id:3;               // ID CAN-шины 0-2
    uint8_t reserve:5;              // Резерв
    uint8_t flanker_br_output:4;    // привязка обходчика к выходу БР, 0 - не используется
    uint8_t engine_on_br_output:4;  // привязка запуска двигателя к выходу БР, 0 - не используется
    uint8_t ign_output:7;           // номер выхода зажигания, 0 - не используется
    uint8_t is_ign_output_br:1;     // использовать выход БР
    uint8_t starter_output:7;       // номер выхода стартера, 0 - не используется
    uint8_t is_starter_output_br:1; // использовать выход БР

    union
    {
      struct
      {
        uint8_t nonce[12];
        uint8_t data[23];
        uint8_t tag[16];
      }ext_key; //шифрованный ключ

      uint8_t ext_key_w_nonce_tag[12+23+16];
    };

  }nissan;
}immo_settings_t;

#if defined(NAVITERM_PRESENT)
#ifdef __ICCARM__
// Текущее состояние ALARM Manager задачи
typedef struct
{
  uint8_t is_sleep;                     // Включен ли менеджер
  alarm_state_t state;                  // текущее состояние
  alarm_state_t state_old;              // предыдущее состояние
  uint8_t cmd_set;                      // признаки внешних команд на постановку на охрану
  uint8_t cmd_reset;                    // признаки внешних команд на снятие с охраны
  uint8_t cmd_service;                  // признаки внешних команд на переход в режим сервиса и выход из этого режима
  uint8_t cmd_sirena;                   // признаки внешних команд на включение и выключение сирены
  uint16_t alarm_timer;                 // таймер
  uint16_t onguard_timer;               // таймер продолжительности пребывания в режиме охраны без тревоги
  uint16_t gps_timer;                   // таймер перехода в спящий режим
  uint8_t gps_mode;                     // признак работы GPS приемника
  uint16_t autoset_timer;               // таймер перехода в режим охраны по отсутствию меток и операций с дверями
  uint8_t radio_tout;                   // флаг таймаута контроля радиоканала в охране
//  uint8_t nopower_timer;                // таймер перехода в режим энергосбережения
//  uint16_t sirena_timer_ext;            // таймер работы сирены по внешнему сигналу
//  uint16_t sirena_timer_int;            // таймер работы сирены по внутреннему сигналу
//  uint8_t sirena_beeps;                 // необходимое количество бипов сирены
  uint8_t engine_lock_timer;            // таймер блокировки двигателя
  uint8_t engine_lock_flags;            // флаги блокировки двигателя
  uint8_t accel_sensor_flags;           // флаги датчика ускорения
  uint8_t engine_autostart_state;       // состояние автозапуска двигателя
} TAlarm_state;
#endif
#endif //defined(NAVITERM_PRESENT)
// Таблица датчиков для поиска имен и их показаний//////////////////////////////

//#ifdef __ICCARM__
typedef struct
{
  uint16_t id;
#if defined(TINY_FLASH_SYSTEM_PRESENT)
#if (__VER__ >= 7080000)
    char name[1];
#else
  char name[0];
#endif //(__VER__ >= 7080000)
#else
  const char* const name;
#endif //defined(TINY_FLASH_SYSTEM_PRESENT)

  void* ptr;
  void* ptr_bkp;
  uint8_t size;
  TSensor_data_type data_type;

#if defined(TINY_FLASH_SYSTEM_PRESENT)
  uint16_t egts_param_id[0];
#else
  uint16_t egts_param_id;
#endif //defined(TINY_FLASH_SYSTEM_PRESENT)
} TSensor_table;
//#else
#ifndef __ICCARM__
typedef struct
{
  uint16_t id;
  char const rus_name[512];
  char const eng_name[512];
  //void* ptr;
  //void* ptr_bkp;
  //uint8_t size;
  //TSensor_data_type data_type;
  //uint16_t egts_param_id;
} TSensor_table_add;
#endif

typedef struct
{
  uint16_t id;
  uint8_t subid;
  void* ptr;
  uint8_t size;
} TSettings_table;

typedef struct
{
  uint16_t id;
  uint16_t subid;
  uint16_t size;
} TSetting_header;

typedef struct
{
  uint32_t counter;
  uint32_t crc;
}settings_info_t;

#pragma pack()

// Класс системы
class CSystem
{
  public:
  // Конструктор
  CSystem(void);

  #ifdef USE_FREERTOS
  // Захватить состояние системы
  bool Grab(TickType_t xTicksToWait = portMAX_DELAY);
  // Отпустить состояние системы
  void Release(void);
  #endif

  /// Параметры конфигурации системы//////////////////
  uint8_t first_start;
  uint8_t settings_version;
  //Версия платы (нумерация с нуля)
  uint8_t Board_hw_version;

  settings_info_t settings_info;

  TTrack_settings      track_settings;      // Настройки трека
  TConnection_settings connection_settings; // Настройки соединения

  TBlackBox_settings   blackbox_settings;   // Настройки черного ящика

  TPower_settings      power_settings;      // Настройки энергопотребления
  TPower_settings_2    power_settings_2;    // Настройки энергопотребления 2

#if defined(GNSS_PSAVE_PRESENT)
  TGnss_psave_settings gnss_psave_settings; // Настройки энергосбережения GNSS
#endif //defined(GNSS_PSAVE_PRESENT)

#if defined(MODEM_PSAVE_PRESENT)
  TGsm_psave_settings gsm_psave_settings;   // Настройки энергосбережения GSM
#endif //defined(MODEM_PSAVE_PRESENT)

  uint8_t wakeup_by_can_activity;           // Просыпаться при активномти can процессора
  TSecurity_settings   security_settings;   // Настройки безопасности
  TGeozone_settings    geozone_settings;    // Настройки геозон
  TSensor_settings     sensor_settings;     // Настройки датчиков
  TScenario_settings   scenario_settings;   // Настройки сценариев
#if defined(IRIDIUM_PRESENT)
  TScenario_settings_i scenario_settings_i; // Настройки сценариев IRIDIUM
#endif //IRIDIUM_PRESENT

#ifdef __ICCARM__
#if (defined(CANPRO_PRESENT))// || defined(VEGA_CAN_PRESENT))
  TCan_settings        can_settings;        // Настройки CAN
#elif defined(CANLOG_V4_PRESENT)
 canlog_settings_t     can_settings;        // Настройки CAN
#endif //CANLOG_V4_PRESENT
#else // for configurator
 TCan_settings         can_settings;        // Настройки CAN
#endif // __ICCARM__

#if defined(VOICE_CALL_PRESENT)
  TVoicecall_settings voicecall_settings;
#endif //VOICE_CALL_PRESENT

#if defined(CAN_RS485_UNITED_PRESENT)
  uint8_t use_can_as_rs485_setting;
  uint8_t use_can_as_rs485_shadow_setting;
#endif //defined(CAN_RS485_UNITED_PRESENT)

#if defined(CUSTOM_CAN_SENSORS_PRESENT)
  char canCustomSettingsGroupName[48];//пользовательское имя настроек
  can_sensor_setting_v1_t canCustomSensorsSettings[MAX_CUSTOM_CAN_SENSORS_COUNT];
  can_sensor_setting_v1_t canCustomSensorsSettings_shadow[MAX_CUSTOM_CAN_SENSORS_COUNT];
  isotp_can_sensor_setting_t IsoTpCanSensorsSettings[MAX_ISOTP_CAN_SENSORS_SETTINGS_COUNT];
  isotp_can_sensor_setting_t IsoTpCanSensorsSettings_shadow[MAX_ISOTP_CAN_SENSORS_SETTINGS_COUNT];
  stm_can_settings_t stmCanSettings[MAX_CAN_INTERFACES];
  can_tx_script_settings_t canTxScriptSettings[MAX_CAN_TX_SCRIPT_COUNT];
  uint8_t can_sensor_setting_key_hash[16];
  uint8_t canCustomSensors_bad_crc_bf[(MAX_CUSTOM_CAN_SENSORS_COUNT + 8-1)/8];
  uint8_t IsoTpcanCustomSensors_bad_crc_bf[(MAX_ISOTP_CAN_SENSORS_SETTINGS_COUNT + 8-1)/8];
#if defined(CUSTOM_CAN_SENSORS_TRANS_TABLE_PRESENT)
  uint16_t canCustomSensorsTransIdTable[32];
#endif //defined(CUSTOM_CAN_SENSORS_TRANS_TABLE_PRESENT)
#endif //defined(CUSTOM_CAN_SENSORS_PRESENT)

#if defined(IMMO_FLANKER_PRESENT)
  immo_settings_t immoFlankerSettings;
  uint8_t immoLearnFlankerCmd;
  struct
  {
    struct
    {
      uint8_t state; // 0 - inactive, 1 - in progress, 2 - found, 3 - not found, 4 - timeout
      uint16_t counter[2];
    }status[2];

    struct
    {
      uint8_t nonce[12];

      union
      {
        uint8_t data[1+13*2];
        struct
        {
          uint8_t immo_type;
          uint8_t args[13*2];
        };
      };

      uint8_t tag[16];
    }util_ags;

  }immoFlankerLearn;
#endif //defined(IMMO_FLANKER_PRESENT)

#if defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)
  char canlogv4CustomSettingsGroupName[32];//пользовательское имя настроек canlog_v4
  canlog_v4_custom_sensor_setting_t canlogv4CustomSensorsSettings[MAX_CUSTOM_CAN_SENSORS_COUNT];
#endif //defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)

#if defined(FASTDATA_PRESENT)
  uint8_t fastdata_fwd_can_server_id;
  uint8_t fastdata_fwd_step;
#endif //defined(FASTDATA_PRESENT)

#if defined(VEGA_CAN_PRESENT)
  uint8_t fwd_can_server_id;
  can_fwd_settings_t fwd_can_setting[MAX_CAN_INTERFACES];
  can_hw_state_t can_hw_state[MAX_CAN_INTERFACES];
#endif //defined(VEGA_CAN_PRESENT)

  /// Текущее состояние системы////////////////////////
#ifdef __ICCARM__
#if defined(NAVITERM_PRESENT)
  TAlarm_state alarm_state;   // Структура с состоянием alarm менеджера
#endif //defined(NAVITERM_PRESENT)
#endif
  TGNSS_state   gnss_state;   // Структура с состоянием GPS менеджера
  TGNSS_raw_state gnss_raw_state; //
  TServer_state server_state; // Структура с состоянием server менеджера
  TSignal_state signal_state; // Структура с состоянием signal менеджера
  TSyscom_state syscom_state; // Структура с состоянием syscom менеджера

  TVega_lbs_data server_lbs_state;

#if defined(TELTONIKA_CMD_ANS_MODE_PRESENT)
  char cmd_ans_buff[33];
#endif //defined(TELTONIKA_CMD_ANS_MODE_PRESENT)

#if defined(CANPRO_PRESENT)
  TCan_state    can_state;    // Структура с состоянием CAN
  TCan_state    can_state_res;// Структура с состоянием CAN
#endif //CANPRO_PRESENT

#if defined(CANLOG_V4_PRESENT)
  TCanLog_v4_state can_v4_state;    // Структура с состоянием CAN
  TCanLog_v4_state can_v4_state_res;// Структура с состоянием CAN
#endif //CANLOG_V4_PRESENT

#if defined(DM5002X_MANOMETR_PRESENT)
  dm5002x_state_t dm5002x_state;
  dm5002x_state_t dm5002x_state_res;
  EDm5002x_interface dm5002x_interface_setting;
#endif //defined(DM5002X_MANOMETR_PRESENT)

#if defined(MODEM_FTP_PRESENT)
  struct
  {
    uint8_t state; //0 - iddle, 1 - in progress, 2 - done, 3 - download error,
                   //4 - bad ftp link format error, 5 - file signature error, 6 - device file system error,
                   //7 - dst file or file path error, 8 - timeout error
    uint8_t pct;   //0-100%
  }ftp_dwn_state;
#endif //defined(MODEM_FTP_PRESENT)

#if defined(BASE_STATION_BT_PRESENT)
  bt_sensor_state_t  bt_sensor_state[MAX_BT_SENSORS];
  bt_sensor_state_t  bt_sensor_state_res[MAX_BT_SENSORS];
  base_station_bt_state_t base_station_bt_state;
  base_station_bt_state_t base_station_bt_state_res;
  Bs_bt_interface bs_bt_interface_setting;
  uint8_t mac_bt_sensor[MAX_BT_SENSORS][6];
#endif //defined(BASE_STATION_BT_PRESENT)

#if defined(MODEM_BT_PRESENT)
  modem_bt_settings_t   modem_bt_settings;
#endif //defined(MODEM_BT_PRESENT)

#if defined(BLE_SENSORS_PRESENT)
  ble_sensor_settings_t ble_sensor_settings[MAX_BLE_SENSORS_COUNT];
  ble_sensor_data_t     ble_sensor_data[MAX_BLE_SENSORS_COUNT];
  ble_sensor_data_t     ble_sensor_data_res[MAX_BLE_SENSORS_COUNT];
#endif //defined(BLE_SENSORS_PRESENT)

#if defined(RS_TACHO_PRESENT)
  tacho_settings_t tacho_settings;
  tacho_state_t tacho_state;
  tacho_state_t tacho_state_res;
#endif //defined(RS_TACHO_PRESENT)

#if defined(SAVER2021_PRESENT)
  saver_settings_t saver_settings;
  saver2021_state_t saver2021_state;
  saver2021_state_t saver2021_state_res;
#endif //defined(SAVER2021_PRESENT)

#if defined(CUSTOM_CAN_SENSORS_PRESENT)
  united_sensor_t custom_can_sensor[MAX_CUSTOM_CAN_SENSORS_COUNT];
  united_sensor_t custom_can_sensor_res[MAX_CUSTOM_CAN_SENSORS_COUNT];

  char custom_can_str_sensor[MAX_CUSTOM_CAN_STR_SENSORS_COUNT][96];
  char custom_can_str_sensor_res[MAX_CUSTOM_CAN_STR_SENSORS_COUNT][96];
#endif //defined(CUSTOM_CAN_SENSORS_PRESENT)

#if defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)
  l_united_sensor_t custom_can_sensor[MAX_CUSTOM_CAN_SENSORS_COUNT];
  l_united_sensor_t custom_can_sensor_res[MAX_CUSTOM_CAN_SENSORS_COUNT];
#endif //defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)

#ifdef __ICCARM__
#if (DEVICE_TYPE == DEVICE_TYPE_OBU_TACHO)
  char kimax2_string[61];
  char kimax2_string_res[sizeof(kimax2_string)];
  __packed uint64_t fms_tco1;
  __packed uint64_t fms_tco1_res;
  __packed uint64_t cpc_dummy;
  __packed uint64_t cpc_dummy_res;
#endif //DEVICE_YPE == DEVICE_TYPE_OBU_TACHO

#if (DEVICE_TYPE == DEVICE_TYPE_VEGA_MT_LIAZ)
  char can_dm1[7*10+1];
  char can_dm1_res[sizeof(can_dm1)];
  __packed uint64_t liaz_fms[8];
  __packed uint64_t liaz_fms_res[8];
#endif //DEVICE_YPE == DEVICE_TYPE_VEGA_MT_LIAZ
#endif //__ICCARM__

#if defined(PROD_TESTING_PRESENT)
  static const int PROT_TESTING_STATE_SIZE = 48;
  char ptest_state[PROT_TESTING_STATE_SIZE];
  char ptest_state_res[PROT_TESTING_STATE_SIZE];
#endif //PROD_TESTING_PRESENT

  TGNSS_state gnss_state_res;     // Структура с резервным состоянием GPS менеджера
  TServer_state server_state_res; // Структура с резервным состоянием server менеджера
  TSignal_state signal_state_res; // Структура с резервным состоянием signal менеджера
  TSyscom_state syscom_state_res; // Структура с резервным состоянием syscom менеджера

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
  TExt_board_ble_state ext_board_ble_state;
  TExt_board_ble_state_res ext_board_ble_state_res;
  TExt_board_ble_settings ext_board_ble_settings;
#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)

  // Таблица датчиков устройства. Использовать для поиска имени датчика, его данных и их размера по id протокола вега
  static const TSensor_table sensor_table[];
#ifndef __ICCARM__
  static const TSensor_table_add sensor_table_add[];
#endif  // __ICCARM__
  // Таблица настроек устройства. Использовать для поиска указателей на настройки
  static const TSettings_table settings_table[];
#ifdef __ICCARM__
  // Тип устройства
  static const uint8_t Dev_type;
  // Тип устройства в текстовом представлении
  static const uint8_t Dev_type_string[20];
#else
  // Тип устройства
  uint8_t Dev_type;
  // Тип устройства в текстовом представлении
  uint8_t Dev_type_string[20];
#endif
  // Текущая версия протокола VEGA
  static const uint8_t Protocol_version;
  // Общее количество датчиков в системе
  static const uint16_t Total_sensor_count;
#ifndef __ICCARM__
  static const uint16_t add_sensor_table_size;
#endif
  // Общее количество настроек в системе
  static const uint16_t Total_settings_count;

  // Имя датчика в протоколе WIPS для сборки с малым flash
  // Функция не реентерабельная!
  const char* Get_tiny_flash_wips_name(const uint16_t* const sensor_id) const;

  // Поиск датчика
  uint8_t Find_sensor(uint16_t sensor_id,
                       const char** sensor_name,
                       void** sensor_data_address,
#ifndef __ICCARM__
                       void** sensor_bckp_address,
#endif
                       TSensor_data_type* data_type,
                       uint16_t* egts_param_id) const;

  // Проверка на сортировку ID в таблице датчиков
  bool Is_sensor_table_sorted(const bool is_verbose);

  // Тест быстрого поиска датчика в таблице
  bool Find_sensor_fast_self_test(const bool is_verbose);

  // Найти датчик. Быстрый поиск в отсортированной таблице
  uint8_t Find_sensor_fast(const uint16_t* const,
                       const char** sensor_name,
                       void** sensor_data_address,
#ifndef __ICCARM__
                       void** sensor_bckp_address,
#endif
                       TSensor_data_type* data_type,
                       uint16_t* egts_param_id) const;

  // Размер датчика в зависимости от типа, в байтах
  uint8_t GetSensorTypeSize(const sensor_type_t sensor_type);
  // Быстрый поиск в пользовательских CAN-датчиках
  uint8_t FindCustomCanSensor(const uint16_t sensor_id, void** sensor_data_address, TSensor_data_type* data_type);
  // Поиск в пользовательских CAN-датчиках с коррекцией типа и размера в зависимости от настройки
  uint8_t FindCustomCanSensorWithCorrectTypeAndSize(const uint16_t sensor_id, TSensor_data_type* data_type);
  // Отключить пердачу для всех ненастроенных пользовательских CAN-датчиков
  void DisableTranslateSettingForEmtyCanSensors(void);
  // Возврат настройки по индексу CAN-датчика
  const can_sensor_setting_v1_t* Get_can_sensor_settings_from_index(const uint16_t index);
  // Возврат настройки по индексу ISO_TP CAN-датчика
  const isotp_can_sensor_setting_t* Get_isotp_can_sensor_settings_from_index(const uint16_t index);
  const canlog_v4_custom_sensor_setting_t* Get_canlog_v4_custom_sensor_settings_from_index(const uint16_t index);
  uint16_t Get_custom_can_sensors_trans_id(const uint16_t id);
#ifndef __ICCARM__
  // Поиск датчика по английски
  uint8_t Find_eng_sensor(uint16_t sensor_id, const char** sensor_name, void** sensor_data_address, void **sensor_bckp_address, TSensor_data_type* data_type, uint16_t* egts_param_id) const ;
  #endif
  // Поиск настройки
  uint16_t Find_setting(uint16_t setting_id, uint16_t setting_subid, uint8_t** setting_data_address);
  // Получение настройки используя ее порядковый номер
  uint16_t Get_setting(uint16_t setting_num, uint16_t *setting_id, uint16_t *setting_subid, uint8_t** setting_data_address) const;
  // Id из диапазона настроек?
  bool Is_setting_id(const uint16_t id);
  // Сенсоры canlog_v4, которые в сценариях нужно обрабатывать c учетом пакованного значения сенсора
  bool Is_canlog_v4_scenario_bitfild_sensor(const uint16_t sensor_id);

#ifndef __ICCARM__
  bool Is_sensor_present(uint16_t id) const;
#endif

#ifdef __ICCARM__
#define count_type_t static const uint8_t
#else
  typedef uint8_t count_type_t;
#endif

#ifdef __ICCARM__
#define count_u16_type_t static const uint16_t
#else
  typedef uint16_t count_u16_type_t;
#endif

  // Количество различных датчиков в системе
  count_type_t sim_count;
  count_type_t din_count;
  count_type_t dout_count;
  count_type_t ain_count;
  count_type_t aout_count;
  count_type_t fin_count;
  count_type_t fout_count;
  count_type_t pin_count;
  count_type_t dut_count;
  count_type_t one_wire_present;
  count_type_t dallas_tm_count;
  count_type_t dallas_temp_count;
  count_type_t phones_count;
  count_type_t geozones_count;
  count_type_t scenaries_count;
  count_type_t tampers_count;
  count_type_t servers_count;
  count_type_t mfi_count;
  #if defined(BR_PRESENT)
  count_type_t ext_mfi_count;
  count_type_t ext_dout_count;
  #endif //BR_PRESENT
  //#if defined(NRF_BEACONS_SCANNER_PRESENT)
  count_u16_type_t nrf_beacons_count;
  //#endif //NRF_BEACONS_SCANNER_PRESENT
  #if defined(LORA_SENSORS_PRESENT)
  count_type_t lora_sensors_count;
  #endif //LORA_SENSORS_PRESENT
  #if defined(IRIDIUM_PRESENT)
  count_type_t scenaries_count_i;
  #endif //IRIDIUM_PRESENT

  #if defined(CUSTOM_CAN_SENSORS_PRESENT)
  count_type_t custom_can_sensors_count;
  count_type_t custom_can_sensors_settnigs_number;
  count_type_t stm_can_modul_count;
  count_type_t can_custom_sensors_setting_version;
  count_type_t can_isotp_sensors_setting_version;
  count_type_t can_tx_script_count_count;
  count_type_t isotp_can_sensors_count;
  #endif //CUSTOM_CAN_SENSORS_PRESENT

#if defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)
  count_type_t custom_canlogv4_sensors_count;
  count_type_t custom_canlogv4_sensors_settings_number;
#endif //defined(CANLOG_V4_CUSTOM_SENSORS_PRESENT)

  #if defined(BLE_SENSORS_PRESENT)
  count_type_t ble_sensors_count;
  #endif //defined(BLE_SENSORS_PRESENT)

  // Присутствие различных функций в системе
  #ifndef __ICCARM__
  bool autorization_present;
  bool gsensor_present;
  bool motohours_present;
  bool camera_present;
  bool alarm_button_present;
  bool ignition_present;
  bool internal_akb_present;
  bool internal_akb_charge_level_present;
  bool can_present;

  bool canV4_present;
  bool canv4_conections_flags_present;
  bool canv4_security_flags_present;
  bool canv4_state_flags_present;
  bool canv4_indicators_flags_present;
  bool canv4_agricultural_vehicles_present;
  bool canv4_utility_vehicles_present;
  bool canv4_cistern_flags_present;
  bool canv4_numeric_data_present;
  bool canv4_agr_numeric_present;
  bool canv4_tacho_present;

  bool can_agro_present;
  bool can_guard_present;
  bool can_numeric_present;
  bool can_dash_present;
  bool mfi_present;
  bool ain_present;
  bool din_present;
  bool voice_present;
  bool inertial_navigation_present;
  bool ext_br_present;
  bool fridge_present;
  bool nrf_beacons_present;
  bool internal_nrf_present;
  bool tenzo_m_present;
  bool lora_sensors_present;
  bool iqfreeze_present;
  bool touchprint_present;
  bool ibox_present;
  bool yandex_extended_sensors_present;
  bool tregistrator_present;
  bool pv310_present;
  bool flowmeter_present;
  bool tpms_present;
  bool iridium_present;
  bool carrier_present;
  bool mdas9_present;
  bool can_man_present;
  bool ublox_gnns_receiver_present;
  bool mediatek_gnns_receiver_present;
  bool stm_gnns_receiver_present;
  bool custom_cansensor_present;
  bool rs232_2_present;
  bool op320_present;
  bool iakm_present;
  bool scd30_present;
  bool adj_sleep_present;
  bool alarm_button_timeout_present;
  bool nt_present;
  bool can_scanner_present;
  bool can_scripts_present;
  bool ble_external_board_present;
  bool dm5002x_present;
  bool vega_can_wakeup_present;
  bool bs_bt_present;
  bool modem_bt_present;
  bool modem_ble_present;
  bool ble_sensors_present;
  bool mdsm7_present;
  bool icon_present;
  bool thermalmaster_present;
  bool custom_canlogv4_sensors_present;
  bool immo_flanker_present;
  bool gsm_balance_present;
  bool gsm_jamming_sensor_present;
  bool fls_settings_present;
  bool rs_tacho_present;
  bool nmea_to_serial_present;
  bool zanotti_present;
  bool sr3_present;
  bool modem_psave_present;
  #endif

  void set_fuel_level_p(uint8_t value);
  uint32_t get_last_time_set_fuel_level_p() const { return last_time_set_fuel_level_p; }

  private:

  #ifdef USE_FREERTOS
  // Семафор для обеспечения атомарности доступа к настройкам системы
  SemaphoreHandle_t busy;
  TickType_t last_time_set_fuel_level_p = 0;
  #else
  uint32_t last_time_set_fuel_level_p = 0;
  #endif
};

// Класс системы
#ifdef __ICCARM__
extern CSystem System;
#endif

#endif /* SYSTEM_SETTINGS_H */
