/**
  ******************************************************************************
  * File Name          : ubx_protocol.cpp
  * Description        : Протокол UBX для работы с навигационными примниками U-blox
  *                       
  *                      
  ******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "UBX_protocol/ubx_protocol.h"
#include <string.h>

/* C code---------------------------------------------------------------------*/
// Добавление контрольной суммы к пакету
void ubx_add_crc(T_ubx_msg* packet, uint8_t* buf)
{
  // Проверка, является ли пакет пакетом UBX
  if( !((buf[0] == 0xB5) && (buf[1] == 0x62)) ) return;
  uint8_t CK_A = 0, CK_B = 0;
  uint16_t i;
  for(i=2; i< (packet->len + UBX_HEADER_SIZE); i++)
  {
     CK_A = CK_A + buf[i];
     CK_B = CK_B + CK_A;
  }
  buf[i] = CK_A;
  buf[++i] = CK_B;
}

// Проверка контрольной суммы
uint8_t ubx_crc_is_ok(T_ubx_msg* packet, const uint8_t* buf)
{
  // Проверка, является ли пакет пакетом UBX
  if( !((buf[0] == 0xB5) && (buf[1] == 0x62)) ) return 0;
  uint8_t CK_A = 0, CK_B = 0;
  uint16_t i;
  for(i=2; i< (packet->len + UBX_HEADER_SIZE); i++)
  {
     CK_A = CK_A + buf[i];
     CK_B = CK_B + CK_A;
  }
  if(packet->crc[0] == CK_A && packet->crc[1] == CK_B) return 1;
  else return 0;
}

// Вычисление количества пакетов UBX в буфере с данными от примниека
uint8_t ubx_get_packets_count(const uint8_t* data, uint16_t size)
{
  T_ubx_msg packet;
  uint8_t packet_count = 0;
  for(uint16_t i =0; i < size; i++)
  {
    // Если найден маркер начала пакета ub, то посчитать CRC и если сошлась, считать пакет валидным. 
    // Если не сошлась, считать весь приемный буффер невалидным
    if( (data[i] == UBX_HEADER_0) && (data[i+1] == UBX_HEADER_1)) 
    {  
      memcpy(&packet, &data[i], UBX_HEADER_SIZE);
      packet.pPayload = &data[i+UBX_HEADER_SIZE];
      // Проверка длины пакета на адекватность (пакет с учетом длины не должен выходить за границы принятых данных)
      if(&data[i+UBX_HEADER_SIZE+packet.len+UBX_CRC_SIZE-1] > &data[size-1]) return UBX_ERROR;
      memcpy(&packet.crc, &data[i+UBX_HEADER_SIZE+packet.len], UBX_CRC_SIZE);
      // Проверка контрольной суммы пакета
      if(!ubx_crc_is_ok(&packet, &data[i])) return UBX_ERROR;
      packet_count++;
    }
  }
  return packet_count;
}

// Упаковка пакета формата UBX
uint16_t ubx_pack_packet(T_ubx_msg* packet, uint8_t* buf, uint16_t bufsize)
{
  // Защита от переполнения буфера
  if(bufsize < (UBX_HEADER_SIZE + packet->len + UBX_CRC_SIZE)) return UBX_ERROR; 
  // Копирование в буфер пакета и полезной нагрузки
  memcpy(buf, packet, UBX_HEADER_SIZE);
  memcpy(buf+UBX_HEADER_SIZE, packet->pPayload, packet->len);
  // Добавление контрольной суммы
  ubx_add_crc(packet, buf);
  // Возврат длины сгенерированного пакета
  return (UBX_HEADER_SIZE + UBX_CRC_SIZE + packet->len);
}

// Распаковка пакета формата UBX
T_ubx_msg ubx_unpack_packet(const uint8_t* data, uint16_t size, uint8_t packet_num)
{
  T_ubx_msg packet;
  uint8_t packet_count = 0;
  for(uint16_t i = 0; i < size ; i++)
  {
    if( (data[i] == UBX_HEADER_0) && (data[i+1] == UBX_HEADER_1)) packet_count++;
    if(packet_count == (packet_num + 1))
    {
      memcpy(&packet, &data[i], UBX_HEADER_SIZE);
      packet.pPayload = &data[i+UBX_HEADER_SIZE];
      break;
    }
  }
  return packet;
}

