/**
  ******************************************************************************
  * File Name          : ubx_protocol.h
  * Description        : Протокол UBX для работы с навигационными примниками U-blox
  *                       
  *                      
  ******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef UBX_PROTOCOL_H
#define UBX_PROTOCOL_H

/* Includes ------------------------------------------------------------------*/
#include <stdint.h>

/* Defines- ------------------------------------------------------------------*/
#define UBX_HEADER_0 0xB5 // u
#define UBX_HEADER_1 0x62 // b

#define UBX_HEADER 0x62B5 // ub
#define UBX_HEADER_SIZE 6 // Заголовок пакета UBX до payload всегда 6 байт
#define UBX_CRC_SIZE 2    // Контрольная сумма в протоколе UBX занимает два байта
#define UBX_MSG_NAV  0x01 //Navigation Results: Position, Speed, Time, Acceleration, Heading, DOP, SVs used
  #define UBX_MSG_NAV_STATUS 0x03
  #define UBX_MSG_NAV_SVINFO 0x30
  #define UBX_MSG_NAV_SVINFO_POLL 0x00
  #define UBX_MSG_NAV_PVT 0x07
  #define UBX_MSG_NAV_PVT_POLL 0x00
  #define UBX_MSG_NAV_DOP 0x04
  #define UBX_MSG_NAV_DOP_POLL 0x00
#define UBX_MSG_MON  0x0A //
  #define UBX_MSG_MON_HW  0x09 //

#define UBX_MSG_ESF  0x10          //External Sensor Fusion Messages
  #define UBX_MSG_ESF_MEAS  0x02   //External Sensor Fusion Measurements

#define UBX_MSG_RXM  0x02 //Receiver Manager Messages: Satellite Status, RTC Status
#define UBX_MSG_INF  0x04 //Information Messages: Printf-Style Messages, with IDs such as Error, Warning, Notice
#define UBX_MSG_ACK  0x05 //Ack/Nack Messages: as replies to CFG Input Messages
  #define UBX_MSG_ACK_ACK  0x01 //AckMessage
  #define UBX_MSG_ACK_NACK 0x00 //AckMessage
#define UBX_MSG_CFG  0x06 //Configuration Input Messages: Set Dynamic Model, Set DOP Mask, Set Baud Rate, etc.
  #define UBX_MSG_CFG_RATE 0x01 //Set message rate      
  #define UBX_MSG_CFG_RATE_LEN 0x03
    #define UBX_MSG_CFG_RATE_NMEA 0xF0
        #define UBX_MSG_CFG_RATE_NMEA_GGA 0x00
        #define UBX_MSG_CFG_RATE_NMEA_GLL 0x01
        #define UBX_MSG_CFG_RATE_NMEA_GSA 0x02
        #define UBX_MSG_CFG_RATE_NMEA_GSV 0x03
        #define UBX_MSG_CFG_RATE_NMEA_RMC 0x04
        #define UBX_MSG_CFG_RATE_NMEA_VTG 0x05
        #define UBX_MSG_CFG_RATE_NMEA_GRS 0x06
        #define UBX_MSG_CFG_RATE_NMEA_GST 0x07
        #define UBX_MSG_CFG_RATE_NMEA_ZDA 0x08
        #define UBX_MSG_CFG_RATE_NMEA_GBS 0x09
        #define UBX_MSG_CFG_RATE_NMEA_DTM 0x0A
        #define UBX_MSG_CFG_RATE_NMEA_GNS 0x0D
        #define UBX_MSG_CFG_RATE_NMEA_THS 0x0D
        #define UBX_MSG_CFG_RATE_NMEA_VLW 0x0D
    #define UBX_MSG_CFG_RATE_NAV 0x01
        #define UBX_MSG_CFG_RATE_NAV_STATUS 0x03
        #define UBX_MSG_CFG_RATE_NAV_DOP    0x04
        #define UBX_MSG_CFG_RATE_NAV_PVT    0x07
        #define UBX_MSG_CFG_RATE_NAV_SVINFO 0x30
    #define UBX_MSG_CFG_RATE_MON 0x0A
        #define UBX_MSG_CFG_RATE_MON_HW     0x09
    #define UBX_MSG_CFG_RATE_ESF 0x10
        #define UBX_MSG_CFG_RATE_ESF_STATUS  0x10
        #define UBX_MSG_CFG_RATE_ESF_MEAS    0x02
        #define UBX_MSG_CFG_RATE_ESF_INS     0x15

  #define UBX_MSG_CFG_FREQ 0x08

  #define UBX_MSG_CFG_NAV5 0x24
  #define UBX_MSG_CFG_NAV5_LEN 0x24 // Длина сообщения 36 байт, фиксированная
        #define UBX_MSG_CFG_NAV5_DYN_MODEL_PORTABLE    0x00  // Настройки динамической модели
        #define UBX_MSG_CFG_NAV5_DYN_MODEL_STATIONARY  0x02
        #define UBX_MSG_CFG_NAV5_DYN_MODEL_PEDESTRIAN  0x03
        #define UBX_MSG_CFG_NAV5_DYN_MODEL_AUTOMOTIVE  0x04
        #define UBX_MSG_CFG_NAV5_DYN_MODEL_SEA         0x05
        #define UBX_MSG_CFG_NAV5_FIX_MODE_2DONLY       0x01
        #define UBX_MSG_CFG_NAV5_FIX_MODE_3DONLY       0x02
        #define UBX_MSG_CFG_NAV5_FIX_MODE_AUTO         0x03
  #define UBX_MSG_CFG_NAVX5 0x23
  #define UBX_MSG_CFG_NAVX5_LEN 0x28 // Длина сообщения 40 байт, фиксированная

  #define UBX_MSG_CFG_GNSS 0x3E // GNSS system configuration
  #define UBX_MSG_CFG_GNSS_LEN (4+8*7) // 4 + 8*numConfigBlocks

#define UBX_MSG_UPD  0x09 //Firmware Update Messages: Memory/Flash erase/write, Reboot, Flash identification, etc.
#define UBX_MSG_MON  0x0A //Monitoring Messages: Comunication Status, CPU Load, Stack Usage, Task Status
        #define UBX_MSG_MON_HW 0x09
#define UBX_MSG_AID  0x0B //AssistNow Aiding Messages: Ephemeris, Almanac, other A-GPS data input
#define UBX_MSG_TIM  0x0D //Timing Messages: Time Pulse Output, Timemark Results
#define UBX_MSG_MGA  0x13 //Multi-GNSS Assistance: Assistance data for various GNSS
#define UBX_MSG_LOG  0x21 //Logging Messages: Log creation, deletion, info and retrieval

#define UBX_FIX_TYPE_NOFIX      0x00
#define UBX_FIX_TYPE_DEAD_ONLY  0x01
#define UBX_FIX_TYPE_2DFIX      0x02
#define UBX_FIX_TYPE_3DFIX      0x03
#define UBX_FIX_TYPE_GNSS_DEAD  0x04
#define UBX_FIX_TYPE_TIME_ONLY  0x05

// Команда для переключения приемника на работу от внутреннего тактового генератора
const uint8_t ubx_ext_crystal_off_message[] = {0xB5, 0x62, 0x06, 0x41, 0x0C, 0x00, 0x00, 0x00, 0x03, 0x1F, 0x47, 0xF2, 0xD7, 0xAD, 0xFF, 0xFF, 0xFC, 0xFF, 0x2B, 0x3D};
// Команда смены скорости на 115200
const uint8_t ubx_baud_115200_message[] = {0xB5, 0x62, 0x06, 0x00, 0x14, 0x00, 0x01, 0x00, 0x00, 0x00, 0xD0, 0x08, 0x00, 0x00, 0x00, 0xC2, 0x01, 0x00, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBC, 0x5E};

const uint8_t ubx_glonass_only_message[] = {0xB5, 0x62, 0x06, 0x3E, 0x3C, 0x00, 0x00, 0x00, 0x1C, 0x07, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x03, 0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x04, 0x08, 0x00, 0x00, 0x00, 0x01, 0x01, 0x03, 0x08, 0x10, 0x00, 0x00, 0x00, 0x01, 0x01, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x01, 0x01, 0x05, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x01, 0x06, 0x08, 0x10, 0x00, 0x01, 0x00, 0x01, 0x01, 0x2A, 0x41};

const uint8_t ubx_setting_nmea4_1_message[] = {0xB5, 0x62, 0x06, 0x17, 0x14, 0x00, 0x00, 0x41, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x75, 0x57};

// Пакет данных формата UBX
typedef struct 
{
  // Заголовок
  uint16_t header;
  // Класс
  uint8_t clas;
  // id сообщения
  uint8_t message_id; 
  // Длина сообщения
  uint16_t len;
  // Полезная нагрузка
  const void *pPayload;
  // Контрольная сумма
  uint8_t crc[2];
} T_ubx_msg;

// Нагрузка сообщения конфигурации выдачи сообщений
typedef struct 
{
  // Класс сообщения
  uint8_t message_class;
  // Id сообщения
  uint8_t message_id;
  // Частота выдачи
  uint8_t message_rate;
} T_ubx_cfg_rates_payload;

//
typedef struct 
{
  uint8_t msgVer; //Message version (=0 for this version)
  uint8_t numTrkChHw; //Number of tracking channels available in hardware (read only)
  uint8_t numTrkChUse; //Number of tracking channels to use. Must be >0, <= numTrkChHw. If 0xFF, then number of tracking channels to use will be set to numTrkChHw
  uint8_t numConfigBlocks; //Number of configuration blocks following, fixed to 7!
  
  struct
  {
    uint8_t gnssId; //System identifier
    uint8_t resTrkCh; //Number of reserved (minimum) tracking channels for this system.
    uint8_t maxTrkCh; //Maximum number of tracking channels used for this system. Must be > 0, >= resTrkChn, <= numTrkChUse and <= maximum number of tracking channels supported for this system.
    uint8_t reserved1; //Reserved
    
    union
    {
      uint32_t u32;
      struct
      {
        uint32_t enable:1; //Enable this system
        uint32_t reserved2:15; //Reserved
        uint32_t sigCfgMask:8; //Signal configuration mask
        uint32_t reserved3:8; //Reserved
      }bf;
    }flags; //bitfield of flags. At least one signal must be configured in every enabled system. 
    
  }ConfigBlocks[7];
} T_ubx_cfg_gnss_payload;

// Нагрузка сообщения конфигурации навигационного движка
typedef struct 
{
  // Битовая маска, какие настройки применять (см. даташит)
  union
  {
    uint16_t u16;
    struct
    {
      uint16_t dyn:1; //Apply dynamic model settings
      uint16_t minEl:1; //Apply minimum elevation settings
      uint16_t posFixMode:1; //Apply fix mode settings
      uint16_t drLim:1; //Reserved
      uint16_t posMask:1;//Apply position mask settings
      uint16_t timeMask:1; //Apply time mask settings
      uint16_t staticHoldMask:1; //Apply static hold settings
      uint16_t dgpsMask:1; //Apply DGPS settings.
      uint16_t cnoThreshold:1; //Apply CNO threshold settings (cnoThresh, cnoThreshNumSVs)
      uint16_t reserved1:1; //Reserved
      uint16_t utc:1; //Apply UTC settings (not supported in protocol versions less than 16)
      uint16_t reserved2:5; //Reserved
    }bf;
  }bit_mask;
  
  // Dynamic platform model:
  //0: portable
  //2: stationary
  //3: pedestrian
  //4: automotive
  //5: sea
  //6: airborne with <1g Acceleration
  //7: airborne with <2g Acceleration
  //8: airborne with <4g Acceleration
  uint8_t dyn_model;
  // Position Fixing Mode: 1: 2D only 2: 3D only 3: auto 2D/3D
  uint8_t fix_mode;
  // Fixed altitude (mean sea level) for 2D fix mode.
  int32_t fixed_alt;
  //Fixed altitude variance for 2D mode
  uint32_t fixed_alt_var;
  // Minimum Elevation for a GNSS satellite to be used in NAV
  int8_t min_elev;
  // Reserved
  uint8_t dr_limit;
  // Position DOP Mask to use
  uint16_t pdop;
  // Time DOP Mask to use
  uint16_t tdop;
  // Position Accuracy Mask
  uint16_t pacc;
  // Time Accuracy Mask
  uint16_t tacc;
  // Static hold threshold
  uint8_t static_hld_trshld;
  // DGPS timeout
  uint8_t dgps_tim;
  // Number of satellites required to have C/N0 above cnoThresh for a fix to be attempted
  uint8_t cno_thresh_num_svs;
  // C/N0 threshold for deciding whether to attempt a fix
  uint8_t cno_thresh;
  // Reserved
  uint8_t reserved1[2];
  // Static hold distance threshold (before quitting static hold)
  uint16_t static_hold_max_dist;
  // UTC standard to be used:
  //0: not specified; receiver may choose freely
  //3: UTC as operated by the U.S. Naval
  //Observatory (USNO); derived from GPS time
  //6: UTC as operated by the former Soviet Union;
  //derived from GLONASS time
  //7: UTC as operated by the National Time Service
  //Center, China; derived from BeiDou time
  //(not supported in protocol versions less than 16).
  uint8_t utc_standart;
  // Reserved
  uint8_t reserved[5];
} T_ubx_cfg_nav5_payload;

// Нагрузка сообщения расширенной конфигурации (в т.ч. иннерциалка)
typedef struct 
{
  //Message version (2 for this version)
  uint16_t version;
  
  //First parameters bitmask.
  union
  {
    uint16_t u16;
    
    struct
    {
      uint16_t reserved1:2; //Reserved
      uint16_t minMax:1; //1 = apply min/max SVs settings
      uint16_t minCno:1; //1 = apply minimum C/N0 setting
      uint16_t reserved2:2; //Reserved
      uint16_t initial3dfix:1; //1 = apply initial 3D fix settings
      uint16_t reserved3:2; //Reserved
      uint16_t wknRoll:1;//1 = apply GPS weeknumber rollover settings
      uint16_t ackAid:1;//1 = apply assistance acknowledgement settings
      uint16_t reserved4:2; //Reserved
      uint16_t ppp:1; //1 = apply usePPP flag
      uint16_t aop:1; //1 = apply aopCfg (useAOP flag) and aopOrbMaxErr settings (AssistNow Autonomous)
      uint16_t reserved5:1; //Reserved
    }bf;
  }mask1;
  
  // Second parameters bitmask.
  union
  {
    uint32_t u32;
    
    struct
    {
      uint32_t reserved1:6; //Reserved
      uint32_t adr:1; //Apply ADR/UDR sensor fusion on/off setting (useAdr flag)
      uint32_t sigAttenComp:1; //Only supported on certain products. Apply signal attenuation compensation feature settings
      uint32_t reserved2:24; //Reserved
    }bf;
  }mask2;
  
  //Reserved
  uint8_t reserved1[2];
  //Minimum number of satellites for navigation 
  uint8_t minSVs; //#SVs 
  //Maximum number of satellites for navigation
  uint8_t maxSVs; //#SVs 
  //Minimum satellite signal level for navigation
  uint8_t minCNO; //dBHz
  //Reserved
  uint8_t reserved2; 
  //1 = initial fix must be 3D
  uint8_t iniFix3D;
  //Reserved
  uint8_t reserved3[2];
  //1 = issue acknowledgements for assistance message input
  uint8_t ackAiding;
  //GPS week rollover number; GPS week numbers
  //will be set correctly from this week up to 1024
  //weeks after this week. Setting this to 0 reverts
  //to firmware default.
  uint16_t wknRollover;
  // Only supported on certain products Permanently attenuated signal compensation (0 = disabled, 255 = automatic, 1..63 = maximum expected C/N0 value)
  uint8_t sigAttenCompMode; //dBHz
  
  //Reserved
  uint8_t reserved4;
  //Reserved
  uint8_t reserved5[2];
  //Reserved
  uint8_t reserved6[2];
  // 1 = use Precise Point Positioning (only available with the PPP product variant)
  uint8_t usePPP;
  
  // AssistNow Autonomous configuration.
  union
  {
    uint8_t u8;
    
    struct
    {
      uint8_t useAOP:1; //1 = enable AssistNow Autonomous
      uint8_t reserved1:7; //Reserved
    }bf;
  }aopCfg;
  
  //Reserved
  uint8_t reserved7[2];
  //Maximum acceptable (modeled) AssistNow Autonomous orbit error (valid range = 5..1000, or 0 = reset to firmware default)
  uint16_t aopOrbMaxErr; //m
  //Reserved
  uint8_t reserved8[4];
  //Reserved
  uint8_t reserved9[3];
  //Only supported on certain products Enable/disable ADR sensor fusion (if 0: sensor fusion is disabled - if 1: sensor fusion is enabled).
  uint8_t useAdr;
  //Reserved
  uint8_t reserved10[2];
  //Reserved
  uint8_t reserved11[2];
} T_ubx_cfg_navx5_payload;

// Нагрузка сообщения о погрешностях от UBLOX
typedef struct 
{
  // GPS time of week of the navigation epoch.
  uint32_t iTOW;
  // Geometric DOP
  uint16_t gDOP;
  // Position DOP
  uint16_t pDOP;
  // Time DOP
  uint16_t tDOP;
  // Vertical DOP
  uint16_t vDOP;
  // Horisontal DOP
  uint16_t hDOP;
  // Northing DOP
  uint16_t nDOP;
  // Easting DOP
  uint16_t eDOP;
} T_ubx_nav_dop_payload;

// Нагрузка сообщения о навигационном решении от UBLOX
typedef struct 
{
  // GPS time of week of the navigation epoch.
  uint32_t iTOW;
  uint16_t year;
  uint8_t month;
  uint8_t day;
  uint8_t hour;
  uint8_t min;
  uint8_t sec;
  // Вылидность времени задана в виде битового поля
    uint8_t valid_date     : 1;
    uint8_t valid_time     : 1;
    uint8_t fully_resolved : 1;
    uint8_t no_matters_5   : 5;

  // Time accuracy estimate (UTC)
  uint32_t tacc;
  // Fraction of second, range -1e9 .. 1e9 (UTC)
  uint32_t nano;
  // GNSSfix Type, range 0..5  //0x00 = No Fix  //0x01 = Dead Reckoning only  //0x02 = 2D-Fix  //0x03 = 3D-Fix  //0x04 = GNSS + dead reckoning combined  //0x05 = Time only fix  //0x06..0xff: reserved
  uint8_t fix_type;

  //Fix Status Flags - заданы в виде битового поля
    //A valid fix (i.e within DOP & accuracy masks)
    uint8_t flags_gnssFixOK     : 1;
    // 1 if differential corrections were applied
    uint8_t flags_diffSoln      : 1;
    // Power Save Mode state (see Power Management): 0 = n/a (i.e no PSM is active) 1 = ENABLED (an intermediate state before ACQUISITION state 2 = ACQUISITION 3 = TRACKING 4 = POWER OPTIMIZED TRACKING 5 = INACTIVE
    uint8_t flags_psmState      : 3;
    // Heading of vehicle is valid
    uint8_t flags_headVehValid  : 1;
    uint8_t no_matters_2        : 2;

  // Reserved
  uint8_t reserved1;
  //Number of satellites used in Nav Solution
  uint8_t numSV;
  // Longitude
  uint32_t lon;
  // Latitude
  uint32_t lat;
  // Height above ellipsoid
  uint32_t height;
  // Height above mean sea level
  int32_t hMSL;
  // Horizontal accuracy estimate
  uint32_t hACC;
  // Vertical accuracy estimate
  uint32_t vACC;
  // NED north velocity
  uint32_t velN;
  // NED east velocity
  uint32_t velE;
  // NED down velocity
  uint32_t velD;
  // Ground Speed (2-D)
  uint32_t gSpeed;
  // Heading of motion (2-D)  курс
  int32_t headMot;
  // Speed accuracy estimate
  uint32_t sACC;
  // Heading accuracy estimate (both motion and vehicle)
  uint32_t HeadACC;
  // Position DOP
  uint16_t pDOP;
  // Reserved
  uint8_t reserved2[6];
  // Heading of vehicle (2-D)
  uint32_t HeadVeh;
  // Reserved
  uint8_t reserved3[4];
} T_ubx_nav_pvt_payload;

// Сообщение с данными о скорости
typedef struct 
{
  // GPS time of week of the navigation epoch.
  uint32_t timeTag;
  uint16_t flags; //Set all to 0
  uint16_t id;   //Data provider
  int32_t Speed : 24; // Sete speed there
  int32_t dataType  : 6;  // Set 11 for speed 
  int32_t dummy     : 2;  
} T_ubx_esf_meas_speed_payload;

// Структура используется в составе структуры с информацией о спутниках
typedef struct 
{
  uint8_t chn; // Channel number, 255 for SVs not assigned to a channel
  uint8_t svid;   // Satellite ID, see Satellite numbering for assignment
  // Bitfield flags
  uint8_t flags_svUsed      : 1; // SV is used for navigation
  uint8_t flags_diffCorr    : 1; // Differential correction data is available for this SV
  uint8_t flags_orbitAvail  : 1; // Orbit information is available for this SV (Ephemeris or Almanac)
  uint8_t flags_orbitEph    : 1; // Orbit information is Ephemeris
  uint8_t flags_unhealthy   : 1; // SV is unhealthy / shall not be used
  uint8_t flags_orbitAlm    : 1; // Orbit information is Almanac Plus
  uint8_t flags_orbitAop    : 1; // Orbit information is AssistNow Autonomous
  uint8_t flags_smoothed    : 1; // Carrier smoothed pseudorange used
  // Bitfield quality
  uint8_t quality_qualityInd  : 4; // Signal Quality indicator (range 0..7). The following list shows the meaning of the different QI values:
  uint8_t quality_nomatters   : 4; // Зарезервировано
  uint8_t cno;                     // Carrier to Noise Ratio (Signal Strength)
  int8_t  elev;                    // Elevation in integer degrees
  int16_t azim;                    // Azimuth in integer degrees
  float prRes;                     // Pseudo range residual in centimeters
} T_ubx_nav_svinfo;

// Нагрузка сообщения о спутниках от UBLOX
typedef struct 
{
  // GPS time of week of the navigation epoch.
  uint32_t iTOW; // GPS time of week of the navigation epoch. 
  uint8_t numCh; // Number of channels
  uint8_t globalFlags;   // bitmask не используется
  uint8_t reserved1[2];  // Резервное поле
  T_ubx_nav_svinfo svinfo[40]; // Собственно информация о спутниках
} T_ubx_nav_svinfo_payload;

// Нагрузка сообщения ACK/NACK
typedef struct 
{
  // Класс сообщения
  uint8_t clsID;
  // Id сообщения
  uint8_t msgID;
} T_ubx_ack_nack_payload;

typedef enum 
{
  UBX_ERROR            = 0,  // Ошибка
  UBX_SUCCESS          = 1,  // Успешно
} E_ubx_result;



/* C code---------------------------------------------------------------------*/
// Функция упаковки пакета формата UBX
uint16_t ubx_pack_packet(T_ubx_msg* packet, uint8_t* buf, uint16_t bufsize);
// Функция распаковки пакета UBX
T_ubx_msg ubx_unpack_packet(const uint8_t* data, uint16_t size, uint8_t packet_num);
// Функция определения количества пакетов UBX в буфере data размером size
uint8_t ubx_get_packets_count(const uint8_t* data, uint16_t size);
//
void ubx_add_crc(T_ubx_msg* packet, uint8_t* buf);
#endif //UBX_PROTOCOL_H