/**
  ******************************************************************************
  * @file    usb_endp.c
  * @author  MCD Application Team
  * @version V4.0.0
  * @date    21-January-2013
  * @brief   Endpoint routines
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT 2013 STMicroelectronics</center></h2>
  *
  * Licensed under MCD-ST Liberty SW License Agreement V2, (the "License");
  * You may not use this file except in compliance with the License.
  * You may obtain a copy of the License at:
  *
  *        http://www.st.com/software_license_agreement_liberty_v2
  *
  * Unless required by applicable law or agreed to in writing, software 
  * distributed under the License is distributed on an "AS IS" BASIS, 
  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  ******************************************************************************
  */


/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include "USB_CDC_STM32/STM32_USB-FS-Device_Driver/inc/usb_lib.h"
#include "USB_CDC_STM32/usb_desc.h"
#include "USB_CDC_STM32/usb_pwr.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/

/* Interval between sending IN packets in frame number (1 frame = 1ms) */
#define VCOMPORT_IN_FRAME_INTERVAL              5
#define ENABLE_BUFFERED_TX_FUNK                 1
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
//__IO uint8_t RESET_cnt=0;
__IO uint8_t WKUP_cnt=0;
__IO uint8_t SUSP_cnt=0;
__IO uint8_t WKUP_event=0;

#if ENABLE_BUFFERED_TX_FUNK == 1
static uint8_t buffered_chars[64];
static uint16_t buffered_char_pos=0;
#endif

#ifdef USE_FREERTOS
#else
extern uint32_t timeAfter(uint32_t time_now, uint32_t time_alarm);
static uint32_t getSysTick(void)
{
  extern volatile uint32_t SysTickReg;
  return SysTickReg;
}
#endif

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : EP1_IN_Callback
* Description    :
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void EP1_IN_Callback (void)
{
#ifdef USE_FREERTOS
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  xSemaphoreGiveFromISR(UsbTxBinSemaphore, &xHigherPriorityTaskWoken);
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
#else
  UsbTxBinSemaphore=1;
#endif
}

/*******************************************************************************
* Function Name  : EP3_OUT_Callback
* Description    :
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void EP3_OUT_Callback(void)
{
  /*
  //packet_receive = 1;
  Usb_Receive_length = GetEPRxCount(ENDP3);
  PMAToUserBufferCopy((unsigned char*)Usb_Receive_Buffer, ENDP3_RXADDR, Usb_Receive_length);
  usb_packet_receive = 1;
  */
#ifdef USE_FREERTOS
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  xSemaphoreGiveFromISR(UsbRxBinSemaphore, &xHigherPriorityTaskWoken);
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
#else
  UsbRxBinSemaphore=1;
#endif
}

/*******************************************************************************
* Function Name  : SOF_Callback / INTR_SOFINTR_Callback
* Description    :
* Input          : None.
* Output         : None.
* Return         : None.
*******************************************************************************/
void SOF_Callback(void)//not used
{
  static uint16_t FrameCount = 0;
  
  if(bDeviceState == CONFIGURED)
  {
    if (FrameCount++ == VCOMPORT_IN_FRAME_INTERVAL)
    {
    SetEPTxCount(ENDP1, 0);
    SetEPTxValid(ENDP1); 
    FrameCount = 0;
    }
  }  
}
void RESET_Callback(void)
{
  //RESET_cnt++;
}
void WKUP_Callback(void)
{
  WKUP_cnt++;
  WKUP_event=1;
}
void SUSP_Callback(void)
{
  bDeviceState = SUSPENDED;
  SUSP_cnt++;
}

/*  *data_ptr  ,   len ;
return:   , -1  conn;*/
int16_t usb_read(uint8_t* mem, uint16_t len, uint32_t timeout)
{
  uint16_t recv_len;
  uint16_t actual_recv_len=0;
  static uint8_t recv_buf[VIRTUAL_COM_PORT_DATA_SIZE];
  
  static uint8_t prev_recv_fill=0;//      
  static uint8_t prev_recv_offset=0;
  
  
  if(WKUP_event)
  {//flush
    prev_recv_fill=0;
    prev_recv_offset=0;
    WKUP_event=0;
  }
  
  if(mem==NULL)
  {//flush
    prev_recv_fill=0;
    prev_recv_offset=0;
    return 0;
  }
  
  if(prev_recv_fill)
  {//     ...
    if(len<=prev_recv_fill)
    {
      memcpy(mem, &recv_buf[prev_recv_offset], len);     
      prev_recv_fill-=len;
      if(prev_recv_fill==0)
      {
#ifdef USE_FREERTOS
        if(!uxQueueMessagesWaiting((QueueHandle_t)UsbRxBinSemaphore))
#else
        if(!UsbRxBinSemaphore)
#endif          
        {//            ,     (  )
          SetEPRxValid(ENDP3);//Enable the receive of data on EP3
        }

        prev_recv_offset=0;
      }
      prev_recv_offset+=len;
      actual_recv_len=len;
      len=0;
    }
    else
    {//len>prev_recv_fill
      /*       ...*/
      memcpy(mem, &recv_buf[prev_recv_offset], prev_recv_fill);
#ifdef USE_FREERTOS      
      if(!uxQueueMessagesWaiting((QueueHandle_t)UsbRxBinSemaphore))
#else
       if(!UsbRxBinSemaphore)
#endif 
      {//            ,     (  )
        SetEPRxValid(ENDP3);//Enable the receive of data on EP3
      }
      mem+=prev_recv_fill;
      actual_recv_len+=prev_recv_fill;
      len-=prev_recv_fill;
      prev_recv_fill=0;
      prev_recv_offset=0;
    }
  }
  
  while(len)
  {//  
#ifdef USE_FREERTOS 
    if(xSemaphoreTake(UsbRxBinSemaphore, timeout))
    {
#else
    uint32_t end_recv_time = timeout + getSysTick();
    uint8_t alv_data=1;
    while(!UsbRxBinSemaphore)
    {
      if(timeAfter(getSysTick(), end_recv_time)) {alv_data=0; break;} //timeout
    }
    if(alv_data) 
    {
      UsbRxBinSemaphore=0;
#endif
      recv_len = GetEPRxCount(ENDP3);
      PMAToUserBufferCopy(recv_buf, ENDP3_RXADDR, recv_len);
      
      if(recv_len<=len)
      {
        memcpy(mem, recv_buf, recv_len);
        
        SetEPRxValid(ENDP3);//Enable the receive of data on EP3
        
        len-=recv_len;
        mem+=recv_len;
        actual_recv_len+=recv_len;
        prev_recv_offset=0;
      }
      else/* recv_buf     +  */
      {
        memcpy(mem, recv_buf, len);
        prev_recv_fill=recv_len-len;
        prev_recv_offset=len;
        actual_recv_len+=len;
        len=0;
      }
    }
    else
    {
      break;
    }
  }
  
  return (int16_t)actual_recv_len;
}

#if ENABLE_BUFFERED_TX_FUNK == 1
void usb_write_char_buffered_flush(void)
{
  if(buffered_char_pos>0)
  {
    usb_write(buffered_chars, buffered_char_pos, 100);
    buffered_char_pos=0;
  }
}

void usb_write_char_buffered(uint8_t ch)
{
  buffered_chars[buffered_char_pos++]=ch;
  
  if(buffered_char_pos>=sizeof(buffered_chars))
  {
    usb_write_char_buffered_flush();
  }
}
#endif

int16_t usb_write(uint8_t *mem, uint16_t len, uint32_t timeout)
{
  uint16_t chain_count;
  uint16_t chain_modulo;
  
  chain_count=len/VIRTUAL_COM_PORT_DATA_SIZE;
  chain_modulo=len%VIRTUAL_COM_PORT_DATA_SIZE;
     
#ifdef USE_FREERTOS
#else
    UsbTxBinSemaphore=0;
#endif
  
  for(uint16_t i=0; i<chain_count; i++)
  {
    /* send  packet to PMA*/
    UserToPMABufferCopy((unsigned char*)mem, ENDP1_TXADDR, VIRTUAL_COM_PORT_DATA_SIZE);
    mem+=VIRTUAL_COM_PORT_DATA_SIZE;
    SetEPTxCount(ENDP1, VIRTUAL_COM_PORT_DATA_SIZE);
    SetEPTxValid(ENDP1);
    
#ifdef USE_FREERTOS
    if(!xSemaphoreTake(UsbTxBinSemaphore, timeout))
    {
      //       ?
      return-1;
    }
#else
    uint32_t end_recv_time = timeout + getSysTick();
    while(!UsbTxBinSemaphore)
    {
      if(timeAfter(getSysTick(), end_recv_time)) return -1; //timeout
    }
    UsbTxBinSemaphore=0;
#endif
  }
  
  if(chain_modulo)
  {
    /* send  packet to PMA*/
    UserToPMABufferCopy((unsigned char*)mem, ENDP1_TXADDR, chain_modulo);
    SetEPTxCount(ENDP1, chain_modulo);
    SetEPTxValid(ENDP1);
    
#ifdef USE_FREERTOS
    if(!xSemaphoreTake(UsbTxBinSemaphore, timeout))
    {
      //       ?
      return-1;
    }
#else
    uint32_t end_recv_time = timeout + getSysTick();
    while(!UsbTxBinSemaphore)
    {
      if(timeAfter(getSysTick(), end_recv_time)) return -1; //timeout
    }
    UsbTxBinSemaphore=0;
#endif
  }
  else//    VIRTUAL_COM_PORT_DATA_SIZE
  {
    SetEPTxCount(ENDP1, 0);//     
    SetEPTxValid(ENDP1); 
    
#ifdef USE_FREERTOS
    if(!xSemaphoreTake(UsbTxBinSemaphore, timeout))
    {
      //       ?
      return-1;
    }
#else
    uint32_t end_recv_time = timeout + getSysTick();
    while(!UsbTxBinSemaphore)
    {
      if(timeAfter(getSysTick(), end_recv_time)) return -1; //timeout
    }
    UsbTxBinSemaphore=0;
#endif
  }

    return (int16_t)len;
}


/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
