/**
******************************************************************************
* File Name          : can_signals.cpp
* Description        : 
*
*
******************************************************************************
*/

#include <math.h>
#include "can_signals.h"

static_assert(!(sizeof(can_sensor_setting_v0_t)!=sizeof(can_sensor_setting_v1_t)), "wrong can_sensor_setting_v0_t or can_sensor_setting_v1_t size");
static_assert(!(CUSTOM_CAN_SENSORS_SETTINGS_VERSION!=1), "wrong CUSTOM_CAN_SENSORS_SETTINGS_VERSION, this file for v1");
static_assert(!(ISOTP_CAN_SENSORS_SETTINGS_VERSION!=0), "wrong ISOTP_CAN_SENSORS_SETTINGS_VERSION, this file for v0");

bool check_can_sensor_setting(const can_sensor_setting_v1_t* setting)
{
  const uint8_t max_dlen=8;
  
  if(setting->version!=1) return false;
  
  if(setting->can_chanel>=MAX_CAN_INTERFACES)
  {
    return false;
  }
  
  if(setting->start_bit>7)
  {
    return false;
  }
  
  switch (setting->sensor_type)
  {
  case BOOL_SYSTEM_T:
    {
      if(max_dlen<=setting->byte_index)
      {
        return false;
      }
    }
    break;
    
  case UINT8_SYSTEM_T:
  case UINT16_SYSTEM_T:
  case UINT32_SYSTEM_T:
  case UINT64_SYSTEM_T:
  case INT8_SYSTEM_T:
  case INT16_SYSTEM_T:
  case INT32_SYSTEM_T:
  case INT64_SYSTEM_T:
  case FLOAT_SYSTEM_T:
  case DOUBLE_SYSTEM_T:
    {
      if(!setting->is_bf_vega_sensor)
      {
        if(setting->byte_index>7 || setting->lenght==0 || setting->lenght>64)
        {
          return false;
        }
        
        uint8_t copy_len=(setting->start_bit+setting->lenght)/8 + !!((setting->start_bit+setting->lenght)%8);
        
        if(copy_len==0 || copy_len>8)
        {
          return false;
        }
        
        uint8_t byte_index=setting->byte_index;
        
        if(setting->b_order==BIG_ENDIAN_ORDER)
        {
          if(copy_len>(byte_index+1))
          {
            return false;
          }
          
          byte_index-=copy_len-1;
        }
        
        if(max_dlen<=byte_index)
        {
          return false;
        }
        
        if((8*byte_index + setting->start_bit + setting->lenght) > 64)
        {
          return false;
        }
      }
      else
      {
        switch (setting->sensor_type)
        {
        case UINT8_SYSTEM_T:
          if(setting->dst_bit_pos>=8) return false;
          break;
          
        case UINT16_SYSTEM_T:
          if(setting->dst_bit_pos>=16) return false;
          break;
          
        case UINT32_SYSTEM_T:
          if(setting->dst_bit_pos>=32) return false;
          break;
          
        case UINT64_SYSTEM_T:
          if(setting->dst_bit_pos>=64) return false;
          break;
          
        default:
          return false;
        }
      }
    }
    break;
    
  default:
    return false;
  }
  
  return true;
}

bool check_isotp_can_sensor_setting(const isotp_can_sensor_sensor_subsetting_t* sensor_setting, const isotp_can_sensor_signal_subsetting_t* signal_setting)
{
  if(signal_setting->start_bit>7 ||
     signal_setting->byte_index>127)
  {
    return false;
  }
  
  if(sensor_setting->sensor_type==STRING_SYSTEM_T && signal_setting->decode_as_dtc)
  {
    return true; //ok
  }
  
  switch (sensor_setting->sensor_type)
  {
  case BOOL_SYSTEM_T:
    {
      
    }
    break;
    
  case STRING_SYSTEM_T:
    {
      if(signal_setting->start_bit!=0 ||
         signal_setting->lenght==0 || 
         signal_setting->lenght%8)
      {
        return false;
      }
    }
    break;
    
  case UINT8_SYSTEM_T:
  case UINT16_SYSTEM_T:
  case UINT32_SYSTEM_T:
  case UINT64_SYSTEM_T:
  case INT8_SYSTEM_T:
  case INT16_SYSTEM_T:
  case INT32_SYSTEM_T:
  case INT64_SYSTEM_T:
  case FLOAT_SYSTEM_T:
  case DOUBLE_SYSTEM_T:
    {
      if(signal_setting->byte_index>127 || signal_setting->lenght==0 || signal_setting->lenght>64)
      {
        return false;
      }
      
      uint8_t copy_len=(signal_setting->start_bit+signal_setting->lenght)/8 + !!((signal_setting->start_bit+signal_setting->lenght)%8);
      
      if(copy_len==0 || copy_len>8)
      {
        return false;
      }
      
      uint8_t byte_index=signal_setting->byte_index;
      
      if(signal_setting->b_order==BIG_ENDIAN_ORDER)
      {
        if(copy_len>(byte_index+1))
        {
          return false;
        }
        
        byte_index-=copy_len-1;
      }
      
      if((8*byte_index + signal_setting->start_bit + signal_setting->lenght) > 1024)
      {
        return false;
      }
    }
    break;
    
  default:
    return false;
  }
  
  return true;
}

/*
ret:
-1: conv err
0: out of min/max or mask
1: conv ok
*/
#ifdef __ICCARM__
#pragma optimize=speed
#endif
int convert_raw_frame_to_signal(const can_rx_frame_t* frame, const can_sensor_setting_v1_t* setting, united_sensor_t* const sensor)
{
  if(setting->version!=1) return -1;
  
  if(setting->start_bit>7)
  {
    return -1;
  }
  
  if(frame->dlen==0)
  {
    return -1;
  }
  
  switch (setting->sensor_type)
  {
    val_64 val;
    
  case BOOL_SYSTEM_T:
    {
      if(frame->dlen<=setting->byte_index)
      {
        return -1;
      }
      
      //  
      if(setting->use_mask_filter)
      {
        val.uval=0;
        
        for(uint8_t i=0; i<frame->dlen; i++)
        {
          *((uint8_t*)&val.uval+(frame->dlen-1)-i)=frame->data[i];
        }
        
        if((val.uval&setting->mask_filt.mask)!=(setting->mask_filt.value&setting->mask_filt.mask))
        {
          return 0;
        }
      }
      
      if((((frame->data[setting->byte_index])&(1<<setting->start_bit)) && !setting->invert) \
        || (!((frame->data[setting->byte_index])&(1<<setting->start_bit)) && setting->invert))
      {
        sensor->b=true;
      }
      else
      {
        sensor->b=false;
      }
    }
    break;
    
  case UINT8_SYSTEM_T:
  case UINT16_SYSTEM_T:
  case UINT32_SYSTEM_T:
  case UINT64_SYSTEM_T:
  case INT8_SYSTEM_T:
  case INT16_SYSTEM_T:
  case INT32_SYSTEM_T:
  case INT64_SYSTEM_T:
  case FLOAT_SYSTEM_T:
  case DOUBLE_SYSTEM_T:
    {
      if(!setting->is_bf_vega_sensor)
      {
        if(setting->byte_index>7 || setting->lenght==0 || setting->lenght>64)
        {
          return -1;
        }
        
        const uint8_t copy_len=(setting->start_bit+setting->lenght)/8 + !!((setting->start_bit+setting->lenght)%8);
        
        if(copy_len==0 || copy_len>8)
        {
          return -1;
        }
        
        uint8_t byte_index=setting->byte_index;
        
        if(setting->b_order==BIG_ENDIAN_ORDER)
        {
          if(copy_len>(byte_index+1))
          {
            return -1;
          }
          
          byte_index-=copy_len-1;
        }
        
        if(frame->dlen<=byte_index)
        {
          return -1;
        }
        
        if((8*byte_index + setting->start_bit + setting->lenght) > 64)
        {
          return -1;
        }
        
        //  
        if(setting->use_mask_filter)
        {
          val.uval=0;
          
          for(uint8_t i=0; i<frame->dlen; i++)
          {
            *((uint8_t*)&val.uval+(frame->dlen-1)-i)=frame->data[i];
          }
          
          if((val.uval&setting->mask_filt.mask)!=(setting->mask_filt.value&setting->mask_filt.mask))
          {
            return 0;
          }
        }
        
        val.uval=0;
        
        memcpy(&val.uval, &frame->data[byte_index], copy_len);
        
        if(setting->b_order==BIG_ENDIAN_ORDER /*&& setting->lenght>8*/)
        {
          uint64_t tmp=val.uval;
          
          for(uint8_t i=0; i<copy_len; i++)
          {
            *((uint8_t*)&val.uval+(copy_len-1)-i)=*((uint8_t*)&tmp+0+i);
          }
        }
        
        if(setting->start_bit>0)
        {
          val.uval=val.uval>>setting->start_bit;
        }
        
        if(((setting->start_bit+setting->lenght)%8)!=0)
        {
          val.uval=(val.uval<<(64-setting->lenght))>>(64-setting->lenght);
        }
        
        //   
        if(setting->is_signed)
        {
          const uint64_t c=((uint64_t)1)<<(setting->lenght-1);
          val.uval=(c^val.uval)-c;
        }
        
        if((setting->use_mask_filter) || \
           (!setting->is_signed && val.uval>=setting->min_max_filt.min.uval && val.uval<=setting->min_max_filt.max.uval) || \
           (setting->is_signed && val.sval>=setting->min_max_filt.min.sval && val.sval<=setting->min_max_filt.max.sval))
        {
          if(setting->scale==1.0f && setting->offset==0.0f)
          {
            if(!setting->is_signed)
            {
              switch (setting->sensor_type)
              {
              case UINT8_SYSTEM_T:  sensor->u8=(uint8_t)(val.uval); break;
              case INT8_SYSTEM_T:   sensor->i8=(int8_t)(val.uval); break;
              case UINT16_SYSTEM_T: sensor->u16=(uint16_t)(val.uval); break;
              case INT16_SYSTEM_T:  sensor->i16=(int16_t)(val.uval); break;
              case UINT32_SYSTEM_T: sensor->u32=(uint32_t)(val.uval); break;
              case INT32_SYSTEM_T:  sensor->i32=(int32_t)(val.uval); break;
              case UINT64_SYSTEM_T: sensor->u64=(uint64_t)(val.uval); break;
              case INT64_SYSTEM_T:  sensor->i64=(int64_t)(val.uval); break;
              case FLOAT_SYSTEM_T:  sensor->f=(float)(val.uval); break;
              case DOUBLE_SYSTEM_T: sensor->d=(double)(val.uval); break;
              default:              return -1;
              }
            }
            else
            {
              switch (setting->sensor_type)
              {
              case UINT8_SYSTEM_T:  sensor->u8=(uint8_t)(val.sval); break;
              case INT8_SYSTEM_T:   sensor->i8=(int8_t)(val.sval); break;
              case UINT16_SYSTEM_T: sensor->u16=(uint16_t)(val.sval); break;
              case INT16_SYSTEM_T:  sensor->i16=(int16_t)(val.sval); break;
              case UINT32_SYSTEM_T: sensor->u32=(uint32_t)(val.sval); break;
              case INT32_SYSTEM_T:  sensor->i32=(int32_t)(val.sval); break;
              case UINT64_SYSTEM_T: sensor->u64=(uint64_t)(val.sval); break;
              case INT64_SYSTEM_T:  sensor->i64=(int64_t)(val.sval); break;
              case FLOAT_SYSTEM_T:  sensor->f=(float)(val.sval); break;
              case DOUBLE_SYSTEM_T: sensor->d=(double)(val.sval); break;
              default:              return -1;
              }
            }
          }
          else
          {
            union
            {
              double dval;
              float fval;
            }dfval;
            
            switch (setting->sensor_type)
            {
            case UINT8_SYSTEM_T:
            case INT8_SYSTEM_T:
            case UINT16_SYSTEM_T:
            case INT16_SYSTEM_T:
            case FLOAT_SYSTEM_T:
              {
                if(!setting->is_signed)
                {
                  dfval.fval=val.uval*setting->scale+setting->offset;
                }
                else
                {
                  dfval.fval=val.sval*setting->scale+setting->offset;
                }
                
                if(setting->sensor_type!=FLOAT_SYSTEM_T)
                {
                  dfval.fval=roundf(dfval.fval);
                }
                
                break;
              }
              
            case UINT32_SYSTEM_T:
            case INT32_SYSTEM_T: 
            case UINT64_SYSTEM_T:
            case INT64_SYSTEM_T:
            case DOUBLE_SYSTEM_T:
              {
                if(!setting->is_signed)
                {
                  dfval.dval=val.uval*(double)setting->scale+(double)setting->offset;
                }
                else
                {
                  dfval.dval=val.sval*(double)setting->scale+(double)setting->offset;
                }
                
                if(setting->sensor_type!=DOUBLE_SYSTEM_T)
                {
                  dfval.dval=round(dfval.dval);
                }
                
                break;
              }
              
            default:     return -1;
            }
            
            switch (setting->sensor_type)
            {
            case UINT8_SYSTEM_T:  sensor->u8=(uint8_t)(dfval.fval); break;
            case INT8_SYSTEM_T:   sensor->i8=(int8_t)(dfval.fval); break;
            case UINT16_SYSTEM_T: sensor->u16=(uint16_t)(dfval.fval); break;
            case INT16_SYSTEM_T:  sensor->i16=(int16_t)(dfval.fval); break;
            case FLOAT_SYSTEM_T:  sensor->f=(float)(dfval.fval); break;
            
            case UINT32_SYSTEM_T: sensor->u32=(uint32_t)(dfval.dval); break;
            case INT32_SYSTEM_T:  sensor->i32=(int32_t)(dfval.dval); break;
            case UINT64_SYSTEM_T: sensor->u64=(uint64_t)(dfval.dval); break;
            case INT64_SYSTEM_T:  sensor->i64=(int64_t)(dfval.dval); break;
            case DOUBLE_SYSTEM_T: sensor->d=(double)(dfval.dval); break;
            default:              return -1;
            }
          }
        }
        else
        {
          return 0;
        }
      }
      else
      {
        bool is_set_bit=false;
        
        if((((frame->data[setting->byte_index])&(1<<setting->start_bit)) && !setting->invert) \
          || (!((frame->data[setting->byte_index])&(1<<setting->start_bit)) && setting->invert))
        {
          is_set_bit=true;
        }
        
        switch (setting->sensor_type)
        {
        case UINT8_SYSTEM_T:
          if(setting->dst_bit_pos>=8) return -1;
          if(is_set_bit) sensor->u8|=(1<<(setting->dst_bit_pos));
          else           sensor->u8&=~(1<<(setting->dst_bit_pos));
          break;
          
        case UINT16_SYSTEM_T:
          if(setting->dst_bit_pos>=16) return -1;
          if(is_set_bit) sensor->u16|=(1<<(setting->dst_bit_pos));
          else           sensor->u16&=~(1<<(setting->dst_bit_pos));
          break;
          
        case UINT32_SYSTEM_T:
          if(setting->dst_bit_pos>=32) return -1;
          if(is_set_bit) sensor->u32|=(1<<(setting->dst_bit_pos));
          else           sensor->u32&=~(1<<(setting->dst_bit_pos));
          break;
          
        case UINT64_SYSTEM_T:
          if(setting->dst_bit_pos>=64) return -1;
          if(is_set_bit) sensor->u64|=(1<<(setting->dst_bit_pos));
          else           sensor->u64&=~(1<<(setting->dst_bit_pos));
          break;
          
        default:
          return -1;
        }
      }
    }
    break;
    
  default:
    return -1;
  }
  
  return 1;
}
