/**
******************************************************************************
* File Name          : vega_can_typefs.h
* Description        : 
*
*
******************************************************************************
*/

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __VEGA_CAN_TYPEDEFS
#define __VEGA_CAN_TYPEDEFS

#ifdef __cplusplus
 extern "C" {
#endif
   
/* Includes ------------------------------------------------------------------*/
#include <stdint.h>
#include <string.h>
#include <stdbool.h>
#include <math.h>

#include "System/system_typedefs.h"

#define LITTLE_ENDIAN_ORDER               0
#define BIG_ENDIAN_ORDER                  1

#define CAN_MODUL_FILTER_COUNT            14
#define CAN_TX_SCRIPT_STEP_COUNT          4
   
#define ISOTP_CAN_SENSOR_TYPE0_SUBSETTINS_COUNT 4
#define ISOTP_CAN_SENSOR_TYPE1_SUBSETTINS_COUNT 3
   
typedef TSensor_data_type sensor_type_t;

typedef struct
{  
  union
  {
    uint64_t u64;
    int64_t  i64;
    uint32_t u32;
    int32_t  i32;
    uint16_t u16;
    int16_t  i16;
    uint8_t  u8;
    int8_t   i8;
    float    f;
    double   d;
    bool     b;
  };
}united_sensor_t;

typedef struct
{  
  union
  {
    uint64_t u64;
    int64_t  i64;
    uint32_t u32;
    int32_t  i32;
    uint16_t u16;
    int16_t  i16;
    uint8_t  u8;
    int8_t   i8;
    float    f;
    double   d;
    bool     b;
    char     string[96]; //
  };
}ext_united_sensor_t;

typedef struct
{  
  union
  {
    uint32_t u32;
    int32_t  i32;
    uint16_t u16;
    int16_t  i16;
    uint8_t  u8;
    int8_t   i8;
    float    f;
    bool     b;
  };
}l_united_sensor_t;

#ifdef __ICCARM__
typedef enum
#else 
typedef enum : uint8_t
#endif
{
  UNKNOWN_CAN_BUS_TYPE=0,
  MOTOR_CAN_BUS_TYPE=1,
  DIAG_CAN_BUS_TYPE=2,
  DASHBOARD_CAN_BUS_TYPE=3,
  SALOON_CAN_BUS_TYPE=4,
}vehicle_can_bus_t;
   
#ifdef __ICCARM__
typedef enum
#else 
typedef enum : uint8_t
#endif
{
  EMPTY_ID_TYPE=0,   //empty
  STD_ID_TYPE=1,     //11-bits identifiers
  EXT_ID_TYPE=2,     //29-bist identifiers
  EXT_ID_PGN_TYPE=3, //18-bits PGN
}id_type_t;

typedef struct
{
  uint32_t   time;
  uint32_t   id;
  uint8_t    fmi; //filer match index
  uint8_t    rtr:1;
#ifdef __ICCARM__
  id_type_t  id_type:7;
#else
  uint8_t    id_type:7;
#endif
  uint8_t    dlen;
  uint8_t    data[8];
}can_rx_frame_t;

typedef struct
{
  uint32_t   id;
  uint8_t    rtr:1;
#ifdef __ICCARM__
  id_type_t  id_type:7;
#else
  uint8_t    id_type:7;
#endif

  uint8_t    dlen;
  uint8_t    data[8];
}can_tx_frame_t;

typedef struct 
{
  union
  {
    uint64_t uval;
    int64_t  sval;
  };
}val_64;

#ifdef __ICCARM__
typedef enum
#else 
typedef enum : uint8_t
#endif
{
  CAN_SPEED_5KBPS=1,
  CAN_SPEED_10KBPS=2,
  CAN_SPEED_20KBPS=3,
  CAN_SPEED_33333BPS=4,
  CAN_SPEED_47619BPS=5,
  CAN_SPEED_50KBPS=6,
  CAN_SPEED_83333BPS=7,
  CAN_SPEED_95238BPS=8,
  CAN_SPEED_100KBPS=9,
  CAN_SPEED_125KBPS=10,
  CAN_SPEED_250KBPS=11,
  CAN_SPEED_500KBPS=12,
  CAN_SPEED_800KBPS=13,
  CAN_SPEED_1MBPS=14,
}can_speed_t;

#ifdef __ICCARM__
typedef enum
#else 
typedef enum : uint8_t
#endif
{
  OFF_CAN_MODE=0,
  SILENT_CAN_MODE=1,
  NORMAL_CAN_MODE=2,
  LOOPBACK_CAN_MODE=3,
  SILENT_LOOPBACK_CAN_MODE=4,
}can_mode_t;

#ifdef __ICCARM__
typedef enum
#else 
typedef enum : uint8_t
#endif
{
  EMPTY_FILTER_BANK_TYPE=0,
  STD_ID_FILTER_BANK_TYPE=1,
  EXT_ID_FILTER_BANK_TYPE=2,
  STD_ID_W_MASK_FILTER_BANK_TYPE=3,
  EXT_ID_W_MASK_FILTER_BANK_TYPE=4,
}filter_bank_type_t;

#pragma pack(1)

typedef struct
{
  uint8_t reserve:5;
#ifdef __ICCARM__
  id_type_t  can_id_type:3;
#else
  uint8_t    can_id_type:3;
#endif
  
  struct
  {
    uint16_t can_chanel:2;
    uint16_t delay:10; //ms*10
    uint16_t dlen:4;
    uint32_t can_id;
    uint8_t repeats;
    uint8_t data[8];
  }step[CAN_TX_SCRIPT_STEP_COUNT];
  
  uint8_t  script_repeats;
  
  uint8_t reserve_2[4];
}can_tx_script_t;

typedef struct
{
  char name[16];
  can_tx_script_t script;
}can_tx_script_settings_t;

typedef struct 
{
  filter_bank_type_t bank_type;
  
  union
  {
    //STD_ID_FILTER_BANK_TYPE: 4 11-bits id
    struct
    {
      uint16_t value1;
      uint16_t value2;
      uint16_t value3;
      uint16_t value4;
    }std;
    
    //EXT_ID_FILTER_BANK_TYPE: 2 29-bist id
    struct
    {
      uint32_t value1;
      uint32_t value2;
    }ext;
    
    //STD_ID_W_MASK_FILTER_BANK_TYPE: 2 11-bits id with mask
    struct
    {
      uint16_t value1;
      uint16_t mask1;
      uint16_t value2;
      uint16_t mask2;
    }std_w_mask;
    
    //EXT_ID_W_MASK_FILTER_BANK_TYPE: 1 29-bist id with mask
    struct
    {
      uint32_t value1;
      uint32_t mask1;
    }ext_w_mask;
  }bank;
}stm_can_filter_t;

static const uint16_t MAX_FILTER_TABLE_SIZE=CAN_MODUL_FILTER_COUNT*sizeof(stm_can_filter_t);

typedef struct
{
  can_mode_t mode;
  can_speed_t speed;
  uint8_t filter_table_size;
  const stm_can_filter_t* filter_table;
}can_hw_config_t;

#ifndef __ICCARM__
 /** ****************************************************************************
  * @struct CanFiltrerSetting
  * @brief   stm_can_filter_t,  
  *  
  ** ***************************************************************************/
 struct CanFiltrerSetting: public stm_can_filter_t
 {
     CanFiltrerSetting():
     stm_can_filter_t{} //     .
     {

     }
     const CanFiltrerSetting& operator= (const CanFiltrerSetting &source)
     {
         bank_type = source.bank_type;
         bank.ext.value1 = source.bank.ext.value1;
         bank.ext.value2 = source.bank.ext.value2;
         return *this;
     }
     bool operator==(const CanFiltrerSetting &other)
     {
         if(this->bank_type != other.bank_type) return false;
         if(this->bank.ext.value1 != other.bank.ext.value1) return false;
         if(this->bank.ext.value2 != other.bank.ext.value2) return false;
         return true;
     }
     bool operator!=(const CanFiltrerSetting &other)
     {
         return !operator==(other);
     }
 };
#endif



struct stm_can_settings_t
{
#ifdef  __ICCARM__
     stm_can_filter_t filters[CAN_MODUL_FILTER_COUNT];
#else
     CanFiltrerSetting filters[CAN_MODUL_FILTER_COUNT];
#endif
     can_mode_t mode;
     can_speed_t speed;
     uint8_t reserve[10];
};


#ifndef __ICCARM__
/** ****************************************************************************
 * @struct StmCanSettings
 * @brief   stm_can_settings_t,  
 *  
 ** ***************************************************************************/
struct StmCanSettings: public stm_can_settings_t
{
    StmCanSettings():
        stm_can_settings_t{} //   
    {}

    const StmCanSettings& operator= (const StmCanSettings &source)
    {
        int n = CAN_MODUL_FILTER_COUNT;
        while(n--)
        {
            this->filters[n] = source.filters[n];
        }
        this->mode = source.mode;
        this->speed = source.speed;
        memcpy(this->reserve, source.reserve, sizeof(source.reserve));
        return *this;
    }

    bool operator==(const StmCanSettings &other)
    {
        if(this->mode != other.mode) return false;
        if(this->speed != other.speed) return false;
        if(memcmp(this->reserve, other.reserve, sizeof(other.reserve)) !=0)
        {
            return false;
        }
        int n = CAN_MODUL_FILTER_COUNT;
        while(n--)
        {
            if(!(this->filters[n] == other.filters[n]))
            {
                return false;
            }
        }
        return true;
    }
    bool operator!=(const StmCanSettings &other)
    {
        return !operator==(other);
    }
};
#endif

typedef struct
{
  uint32_t rx_drop_counter;   //
  uint8_t reserved1[8];       //
  
  uint8_t reserved2:2;        //
  uint8_t state:2;            //0 - , 1 - , 2 - 
  uint8_t last_err_code:4;    //  
  uint8_t rec;                //Receive Error Counter
  uint8_t tec;                //Transmit Error Counter
  
  union
  {
    uint8_t byte;
    struct
    {
      uint8_t warning:1;      //Error Warning
      uint8_t passive:1;      //Error Passive
      uint8_t bus_off:1;      //Bus Off 
      uint8_t reserved:5;     //
    }bf;
  }err_flags;                 //  CAN  
  
}can_hw_state_t;

inline const char* get_verbose_last_err_code(const uint8_t err)
{
  switch(err)
  {
  case (0):  return "no error";
  case (1):  return "stuff error";
  case (2):  return "form error";
  case (3):  return "acknowledgment error";
  case (4):  return "bit recessive error";
  case (5):  return "bit dominant error";
  case (6):  return "crc error";
  case (7):  return "software set error";
  default:   return "unknown error";
  }
}

typedef struct
{
  uint8_t can_chanel:3;       // CAN: 0, 1, 2
  uint8_t version:5;          //   (0-31), 0 -   
  
  id_type_t can_id_type;      // 
  
  struct
  {
    union
    {
      uint32_t can_id;          //
      uint32_t pgn;             //PGN
    };
    
    united_sensor_t reset_value;//  
    uint8_t reset_timer:4;      //  ,    X 
    uint8_t reset_by_ign_off:1; //     
    uint8_t fast_bool:1;        //   bool (  true)
    uint8_t use_mask_filter:1;  //    ( min max)
    uint8_t is_bf_vega_sensor:1;//  ,     dst_bit_pos
  };
  
  sensor_type_t sensor_type;  //  
  uint16_t sensor_id;         //id  
  char user_sensor_name[30];  // ,    '\0'
  
  uint8_t reserve_1[2];
  
  struct
  {
    union
    {
      //if is_bf_sensor==false
      struct
      {
        uint8_t lenght:7;       //1-64 bits
        uint8_t is_signed:1;    //  
      };
      
      //if is_bf_sensor==true
      struct
      {
        uint8_t dst_bit_pos:7;  //0-63 bits
        uint8_t not_used:1;     //not used 
      };
    };
    
    uint8_t b_order:1;          // , LITTLE_ENDIAN_ORDER  BIG_ENDIAN_ORDER
    uint8_t invert:1;           // ( BOOL)
    uint8_t byte_index:3;       //0-7
    uint8_t start_bit:3;        //0-7
    
    union
    {
      //if use_mask_filter==false
      struct
      {
        val_64 min;             // 
        val_64 max;             // 
      }min_max_filt;
      
      //if use_mask_filter==true
      struct
      {
        uint64_t mask;
        uint64_t value;
      }mask_filt;
    };
    
    double scale;               //val*scale + offset
    double offset;
  };
}can_sensor_setting_v0_t;

typedef struct
{
  uint8_t can_chanel:3;       // CAN: 0, 1, 2
  uint8_t version:5;          //   (0-31), 1 -   
  
#ifdef __ICCARM__
  id_type_t can_id_type:3;    // 
#else
  uint8_t can_id_type:3;      // 
#endif
  uint8_t type:1;             // , 0 -   
  uint8_t reserve_1:3;        //
  uint8_t is_crypted:1;       //  
  
  uint8_t nonce[4];
  
  uint16_t crc;               //
  
  sensor_type_t sensor_type;  //  
  uint16_t sensor_id;         //id  
  char user_sensor_name[30];  // ,    '\0'
  
  uint8_t reserve_2[1];
  
  union
  {
    uint8_t crypt_block[42]; //  
    
    struct
    {
      //uint16_t crc;
      
      union
      {
        uint32_t can_id;          //
        uint32_t pgn;             //PGN
      };
      
      united_sensor_t reset_value;//  
      uint8_t reset_timer:4;      //  ,    X 
      uint8_t reset_by_ign_off:1; //     
      uint8_t fast_bool:1;        //   bool (  true)
      uint8_t use_mask_filter:1;  //    ( min max)
      uint8_t is_bf_vega_sensor:1;//  ,     dst_bit_pos
      
      union
      {
        //if is_bf_sensor==false
        struct
        {
          uint8_t lenght:7;       //1-64 bits
          uint8_t is_signed:1;    //  
        };
        
        //if is_bf_sensor==true
        struct
        {
          uint8_t dst_bit_pos:7;  //0-63 bits
          uint8_t not_used:1;     //not used 
        };
      };
      
      uint8_t b_order:1;          // , LITTLE_ENDIAN_ORDER  BIG_ENDIAN_ORDER
      uint8_t invert:1;           // ( BOOL)
      uint8_t byte_index:3;       //0-7
      uint8_t start_bit:3;        //0-7
      
      union
      {
        //if use_mask_filter==false
        struct
        {
          val_64 min;             // 
          val_64 max;             // 
        }min_max_filt;
        
        //if use_mask_filter==true
        struct
        {
          uint64_t mask;
          uint64_t value;
        }mask_filt;
      };
      
      float scale;               //val*scale + offset
      float offset;
      
      uint8_t reserve_3[3];
    };
  };
}can_sensor_setting_v1_t;


typedef struct
{
  sensor_type_t sensor_type;   //  
  uint16_t sensor_id;          //id  , 0 - 
  char user_sensor_name[16];   // ,    '\0'
}isotp_can_sensor_sensor_subsetting_t;

typedef struct
{
  uint8_t byte_index:7;       //0-127
  uint8_t reserve_4:1;        //
  uint16_t lenght:11;         //in bits 1-1024
  uint16_t start_bit:3;       //0-7
  uint16_t reserve_5:2;       //
  
  uint8_t invert:1;           // ( BOOL)
  uint8_t is_signed:1;        //  
  uint8_t b_order:1;          // , LITTLE_ENDIAN_ORDER  BIG_ENDIAN_ORDER     
  uint8_t reset_if_neg_resp:1; //
  uint8_t reset_if_timeout:1;  //
  uint8_t decode_as_dtc:1;     //  DTC 
  uint8_t reserve_6:2;         //
  
  struct
  {
    val_64 min;             // 
    val_64 max;             // 
  }min_max_filt;
  
  float scale;               //val*scale + offset
  float offset;
  
  uint8_t reserve_7[2];
}isotp_can_sensor_signal_subsetting_t;

typedef struct
{
  uint8_t can_chanel:3;        // CAN: 0, 1, 2
  uint8_t version:5;           //   (0-31), 0 -   
  
#ifdef __ICCARM__
  id_type_t can_id_type:3;    // 
#else
  uint8_t can_id_type:3;      // 
#endif
  uint8_t type:1;             // , 0 -    
  uint8_t reserve_1:3;
  uint8_t is_crypted:1;       //  
  
  uint8_t nonce[4];
  
  uint16_t crc;              //
  
  int32_t  request_condition_treshold;
  uint16_t request_condition_sensor_id;
  uint16_t request_condition_type:3; //0- '>=', 1- '<=', 2 - '=='
  uint16_t request_period:6;         // ,  1  1-63 , 0 -  1 
  uint16_t responce_timeout:6;       // ,  100 , 100-6300 
  uint16_t reserve_2:1;
  
  uint8_t reserve_3;
  
  isotp_can_sensor_sensor_subsetting_t sensor[ISOTP_CAN_SENSOR_TYPE0_SUBSETTINS_COUNT];
  
  union
  {
    uint8_t crypt_block[137];       //  
    
    struct
    {
      //uint16_t crc;
      uint8_t reserve_4:4;
      uint8_t protocol:3;          //
      uint8_t s_tester_present:1;  //
      uint32_t request_id;         //
      uint32_t responce_id;        //
      
      uint8_t request[7];          //sigle frame only
      uint8_t request_len:3;       //bytes, 1-7 
      uint8_t reserve_5:5;          //
      
      isotp_can_sensor_signal_subsetting_t signal[ISOTP_CAN_SENSOR_TYPE0_SUBSETTINS_COUNT];
    };
  };
}isotp_can_sensor_setting_t0_t;

typedef struct
{
  uint8_t can_chanel:3;        // CAN: 0, 1, 2
  uint8_t version:5;           //   (0-31), 0 -   
  
#ifdef __ICCARM__
  id_type_t can_id_type:3;    // 
#else
  uint8_t can_id_type:3;      // 
#endif
  uint8_t type:1;             // , 1 -   
  uint8_t reserve_1:3;
  uint8_t is_crypted:1;       //  
  
  uint8_t nonce[4];
  
  uint16_t crc;               //
  
  struct
  {
    int32_t  request_condition_treshold;
    uint16_t request_condition_sensor_id;
    uint16_t request_condition_type:3; //0- '>=', 1- '<=', 2 - '=='
    uint16_t request_period:6;         // ,  1  1-63 , 0 -  1 
    uint16_t responce_timeout:6;       // ,  100 , 100-6300 
    uint16_t reserve_2:1;
    
    uint8_t reserve_3;
    
    isotp_can_sensor_sensor_subsetting_t sensor;
    
    union
    {
      uint8_t crypt_block[47];       //  
      
      struct
      {
        //uint16_t crc;
        uint8_t reserve_4:4;
        uint8_t protocol:3;          //
        uint8_t s_tester_present:1;  //
        uint32_t requst_id;          //
        uint32_t responce_id;        //
        
        uint8_t request[7];          //sigle frame only
        uint8_t request_len:3;       //bytes, 1-7 
        uint8_t reserve_5:5;          //
        
        isotp_can_sensor_signal_subsetting_t signal;
      };
    };
  }sub_settings[ISOTP_CAN_SENSOR_TYPE1_SUBSETTINS_COUNT];
}isotp_can_sensor_setting_t1_t;

typedef union
{
  isotp_can_sensor_setting_t0_t t0;
  isotp_can_sensor_setting_t1_t t1;
  
  struct
  {
    uint8_t can_chanel:3;        // CAN: 0, 1, 2
    uint8_t version:5;           //   (0-31), 0
    
#ifdef __ICCARM__
    id_type_t can_id_type:3;    // 
#else
    uint8_t can_id_type:3;      // 
#endif
    uint8_t type:1;             // , 0, 1
    uint8_t reserve_1:3;
    uint8_t is_crypted:1;       //  
    
    uint8_t nonce[4];
    
    uint16_t crc;               //
  };
}isotp_can_sensor_setting_t;


typedef struct
{
  uint16_t canlog_cmd;        //  canlog_v4
  
  sensor_type_t sensor_type;  //  
  uint16_t sensor_id;         //id  
  char user_sensor_name[16];  // ,    '\0'  
  
  uint16_t byte_index:7;      //0-127
  uint16_t lenght:6;          //1-32 bits
  uint16_t start_bit:3;       //0-7
  
  float scale;                //val*scale + offset
  float offset;
}canlog_v4_custom_sensor_setting_t;

#pragma pack()

#ifdef __cplusplus
 }
#endif

#endif //__VEGA_CAN_TYPEDEFS
