/**
******************************************************************************
* File Name          : saver2021_lib.c
* Description        :
*
*
******************************************************************************
*/

#include "system_config.h"
#include "saver_lib/saver2021_lib.h"
#include "time_utilities/time_utilities.h"
#include "crc_lib/crc_lib.h"
#include "debug_port.h"
#include "macro_utils.h"
#include <stdlib.h>

extern void BIN_BUFF_LOG(const uint8_t* bin, uint8_t bin_len, const char* string, ...);

NOT_USED(static const char* const dbg_name = "SAVER2021";)

static const uint8_t RAW_RX_LOG_ENABLE = false;

//    ,  .
#define INFO_OPCODE 0x02
static const uint8_t INFO_REQ[] = {0x77, 0x05, 0x02, 0x4E, 0xB3};
static const uint8_t INFO_ANS_LEN = 0x74;

//   
#define REQ1_OPCODE 0x09
static const uint8_t REQ1[] = {0x77, 0x0D, 0x09, 0xF9, 0x52, 0x53, 0xC7, 0x7E, 0x54, 0xC3, 0x62, 0x15, 0xED};
static const uint8_t ANS1_LEN = 0x71;

//   -  , . . , 
#define REQ2_OPCODE 0x0D
static const uint8_t REQ2[] = {0x77, 0x05, 0x0D, 0xBF, 0x5C};
static const uint8_t ANS2_LEN = 0x17;

//   ,  /
#define REQ3_OPCODE 0x04
static const uint8_t REQ3[] = {0x77, 0x06, 0x04, 0x2F, 0x3C, 0x71};
static const uint8_t ANS3_LEN = 0x0F;

//    
#define DTC_REQ1_OPCODE 0x32
static const uint8_t DTC_REQ1[] = {0x77, 0x05, 0x32, 0x78, 0xE0};
static const uint8_t DTC_ANS1_LEN = 0x5A;

/*
//   ?
#define DTC_REQ2_OPCODE 0x35
static const uint8_t DTC_REQ2[] = {0x77, 0x05, 0x35, 0x08, 0x07};
static const uint8_t DTC_ANS2_LEN = 0x50;
static uint8_t DTC_ANS2[DTC_ANS2_LEN] = {0};
*/

// - 
static const uint8_t REQ_COUNTER = 5;

static const uint32_t BAUDRATE = 38000;
static const uint8_t RX_TIMEOUT = 50;

static int int_round_div(const int n, const int d)
{
  return ((n < 0) ^ (d < 0)) ? ((n - d/2)/d) : ((n + d/2)/d);
}

static uint8_t check_rx(const uint8_t* const  rx, const uint16_t rx_len, const uint8_t op_code)
{
  if(rx_len < 4) {return 0;}

  if(rx[0] != 0x66 || rx[1] != rx_len || rx[2] != op_code) {return 0;}

  const uint16_t crc = (rx[rx_len-2] << 8) | (rx[rx_len-1] << 0);

  if(crc != crc16_fast(rx, rx_len-2)) {return 0;}

  return 1;
}

static void parse_packet(saver2021_ctx_t* const ctx, const uint8_t* data, uint8_t len)
{
  switch(data[2])
  {
    case(INFO_OPCODE):
    {
      //  ,  .
      //SN:
      //00 00 16 60 15 01 05728020121
      //11 22 33 44 55 66 2874540200210285
      //11 22 33 44 77 66 28745402002102119
      //00 00 16 99 98 85 0578502133152

      //    ,  !
      sprintf((char*)ctx->pconst->rx_buff, "%u02%02hhu%02hhu", (data[100]<<24)|(data[101]<<16)|(data[102]<<8)|(data[103]<<0), data[105], data[104]);

      ctx->pconst->state_lock_unlock(1);
      ctx->pconst->state->main.sn = strtoull((const char*)ctx->pconst->rx_buff, NULL, 10);
      ctx->pconst->state_lock_unlock(0);

      break;
    }

    case(REQ1_OPCODE):
    {
      // 
      ctx->pconst->state_lock_unlock(1);
      ctx->pconst->state->main.eng_rpm = (data[23]<<8)|(data[24]<<0);
      ctx->pconst->state->main.sens_volt = int_round_div(((data[46]<<8)|(data[47]<<0)), 100);
      ctx->pconst->state->main.map = ((data[27]<<8)|(data[28]<<0));
      ctx->pconst->state->main.p_sys = ((data[29]<<8)|(data[30]<<0));
      ctx->pconst->state->main.t_red = int_round_div((int16_t)((data[31]<<8)|(data[32]<<0)), 10);
      ctx->pconst->state->main.t_gas = int_round_div((int16_t)((data[33]<<8)|(data[34]<<0)), 10);
      ctx->pconst->state->main.t_pcb = int_round_div((int16_t)((data[37]<<8)|(data[38]<<0)), 10);
      ctx->pconst->state->main.load_petrol = data[43];
      ctx->pconst->state->main.load_gas = data[44];
      ctx->pconst->state->main.eng_load = data[45];
      ctx->pconst->state->main.ecu_state = data[51];
      ctx->pconst->state->main.tank_level_avg = int_round_div(data[76]*50, 255); //(val*5)/255
      ctx->pconst->state_lock_unlock(0);

      break;
    }

    case(REQ2_OPCODE):
    {
      //  -  , . . , 
      ctx->pconst->state_lock_unlock(1);
      ctx->pconst->state->main.ecu_reset_counter = data[3];
      ctx->pconst->state->main.emerg_starts_counter = data[4];
      ctx->pconst->state->main.max_pcb_temp = int_round_div(((int16_t)(data[5]<<8)|(data[6]<<0)), 10);
      ctx->pconst->state->main.working_on_petrol = (data[11]<<24)|(data[12]<<16)|(data[13]<<8)|(data[14]<<0);
      ctx->pconst->state->main.working_on_gas = (data[15]<<24)|(data[16]<<16)|(data[17]<<8)|(data[18]<<0);
      ctx->pconst->state->main.distance_left = (data[19]<<8)|(data[20]<<0);
      ctx->pconst->state_lock_unlock(0);

      break;
    }

    case(REQ3_OPCODE):
    {
      //  ,  /

      break;
    }

    case DTC_REQ1_OPCODE:
    {
      //  
      ctx->pconst->state_lock_unlock(1);
      snprintf(ctx->pconst->state->dtc, sizeof(ctx->pconst->state->dtc), "%s", "no DTC");
      uint16_t offset = 0;
      uint8_t dtc_counter = 0;

      for(uint8_t code_num = 1; code_num <= 39; code_num++)
      {
        const uint8_t dtc_count = data[8 + code_num*2] & 0x7F;

        if(dtc_count)
        {
          uint8_t is_active = 0;

          if(code_num <= 8)       {is_active = !!(data[3] & (1<<(code_num-1)));}
          else if(code_num <= 16) {is_active = !!(data[4] & (1<<(code_num-9)));}
          else if(code_num <= 24) {is_active = !!(data[5] & (1<<(code_num-17)));}
          else if(code_num <= 32) {is_active = !!(data[6] & (1<<(code_num-25)));}
          else if(code_num <= 40) {is_active = !!(data[7] & (1<<(code_num-33)));}

          if(is_active) {dtc_counter++;}

          if(sizeof(ctx->pconst->state->dtc) > offset)
          {
            int s_len = snprintf(&ctx->pconst->state->dtc[offset], sizeof(ctx->pconst->state->dtc) - offset, "E%3.3u/%u/%u ", code_num, dtc_count, is_active);

            if(s_len < 0 || s_len >= (sizeof(ctx->pconst->state->dtc) - offset))
            {
              //break;
            }
            else
            {
              offset += s_len;
            }
          }
        }
      }

      ctx->pconst->state->main.dtc_count = dtc_counter;

      ctx->pconst->state_lock_unlock(0);

      break;
    }

    default:
    {
      break;
    }
  }

  return;
}

int saver2121_handler(saver2021_ctx_t* const ctx)
{
  if(ctx->itf == NULL)
  {
    ctx->pconst->state->main.poll_state = 0;

    return 0;
  }

  if(!timeAfter(xTaskGetTickCount(), ctx->poll_timer))
  {
    return 1;
  }

  if(ctx->pconst->rx_buff == NULL ||
     ctx->pconst->rx_buff_size < ANS1_LEN ||
     ctx->pconst->rx_buff_size < INFO_ANS_LEN)
  {
    ctx->pconst->state->main.poll_state = 3;

    return 0;
  }

  if(ctx->itf->lock != NULL) {ctx->itf->lock();}
  ctx->itf->set_param(BAUDRATE, RX_TIMEOUT, "8N1");

  uint8_t* const rx = ctx->pconst->rx_buff;
  const uint8_t* tx;
  uint16_t rx_len, wait_len, tx_len;

  uint8_t rx_counter = 0;

  for(uint8_t i = 0; i < REQ_COUNTER; i++)
  {
    if(i == 0)
    {
      if(ctx->got_sys_info) {continue;}

      tx = INFO_REQ;
      tx_len = sizeof(INFO_REQ);
      wait_len = INFO_ANS_LEN;
    }
    else if(i == 1)
    {
      tx = REQ1;
      tx_len = sizeof(REQ1);
      wait_len = ANS1_LEN;
    }
    else if(i == 2)
    {
      tx = REQ2;
      tx_len = sizeof(REQ2);
      wait_len = ANS2_LEN;
    }
    else if(i == 3)
    {
      tx = REQ3;
      tx_len = sizeof(REQ3);
      wait_len = ANS3_LEN;
    }
    else if(i == 4)
    {
      tx = DTC_REQ1;
      tx_len = sizeof(DTC_REQ1);
      wait_len = DTC_ANS1_LEN;
    }
    /*
    else if(i == 5)
    {
      tx = DTC_REQ2;
      tx_len = sizeof(DTC_REQ2);
      wait_len = DTC_ANS2_LEN;
    }
    */
    else
    {
      continue;
    }

    //__BIN_BUFF_PRINTF(tx, tx_len, "%s, TX %u bytes:\n", dbg_name, tx_len);

    rx_len = ctx->itf->txrx(tx, tx_len, rx, wait_len);

    //__BIN_BUFF_PRINTF(rx, rx_len, "%s, RX %u bytes:\n", dbg_name, rx_len);

    if(RAW_RX_LOG_ENABLE)
    {
      BIN_BUFF_LOG(rx, rx_len, "%s, RX %u bytes:\n", dbg_name, rx_len);
    }

    if(wait_len == rx_len && check_rx(rx, rx_len, tx[2]))
    {
      rx_counter++;

      parse_packet(ctx, rx, rx_len);

      if(i == 0)
      {
        //   ,     
        ctx->got_sys_info = 1;
      }
    }
    else
    {
      __PRINTF("%s: missed a bad packet...\n", dbg_name);
    }
  }

  if(ctx->itf->unlock != NULL) {ctx->itf->unlock();}

  if(rx_counter == 0)
  {
    ctx->pconst->state->main.poll_state = 2;
    ctx->got_sys_info = 0;

    if(RAW_RX_LOG_ENABLE) {ctx->poll_timer = xTaskGetTickCount() + 60000;}
    else                  {ctx->poll_timer = xTaskGetTickCount() + 30000;}
  }
  else
  {
    ctx->pconst->state->main.poll_state = 1;

    if(RAW_RX_LOG_ENABLE) {ctx->poll_timer = xTaskGetTickCount() + 60000;}
    else                  {ctx->poll_timer = xTaskGetTickCount() + 900;}
  }

  return 1;
}
