/**
******************************************************************************
* File Name          : server_notifies.c
* Description        : 
*
*
******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "serv_protocols_lib/server_notifies.h"
#include "serv_protocol_handler_conf.h"

#if defined(NOTIFY_LOCK_EN)
//      
#include "FreeRTOS.h"
#include "semphr.h"
#endif //NOTIFY_LOCK_EN

/* C code---------------------------------------------------------------------*/

/* Global variables-----------------------------------------------------------*/

typedef struct
{
  uint32_t uid;
  uint8_t nonce[8];
  char mess[MAX_NOTIFY_LEN+1];
  notify_state_t state;
}server_notify_t;

#define NOTIFY_SERVERS_COUNT    (MAX_SERVERS_COUNT+1)//+USB

#if defined(EXTERNAL_BLE_BOARD_PRESENT)
#undef NOTIFY_SERVERS_COUNT
#define NOTIFY_SERVERS_COUNT    (MAX_SERVERS_COUNT+1+1)//+USB+BLE
#endif //defined(EXTERNAL_BLE_BOARD_PRESENT)

static_assert(!(MAX_NOTIFIES_COUNT_FOR_EACH_SERVER&(MAX_NOTIFIES_COUNT_FOR_EACH_SERVER-1)), "wrong MAX_NOTIFIES_COUNT_FOR_EACH_SERVER size");
static_assert(!(MAX_NOTIFIES_COUNT_FOR_EACH_SERVER<8), "wrong MAX_NOTIFIES_COUNT_FOR_EACH_SERVER size");

#if defined(NOTIFY_LOCK_EN)
static SemaphoreHandle_t  notify_lock_obj=NULL;
#endif //NOTIFY_LOCK_EN

static const uint8_t server_notifys_idx_mask=MAX_NOTIFIES_COUNT_FOR_EACH_SERVER-1;

static struct server_notifys_sfifo_struct
{
  server_notify_t notify[MAX_NOTIFIES_COUNT_FOR_EACH_SERVER];
  uint8_t in_idx;
  uint8_t out_idx;
}server_notifys_sfifo[NOTIFY_SERVERS_COUNT];

bool add_server_notify(const uint8_t server_id, const uint32_t notify_uid, const uint8_t nonce[8], const notify_state_t notify_state, const char* notify_mess, ...)
{
  bool result=false;
  int s_len;
  
  if(server_id>=NOTIFY_SERVERS_COUNT) return result;
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreTake(notify_lock_obj, portMAX_DELAY);
#endif //NOTIFY_LOCK_EN
  
  if((server_notifys_idx_mask-((server_notifys_sfifo[server_id].in_idx-server_notifys_sfifo[server_id].out_idx)&(server_notifys_idx_mask)))>0)
  {
    server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].uid=notify_uid;
    if(nonce!=NULL) {memcpy(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].nonce, nonce, sizeof(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].nonce));}
    else            {memset(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].nonce, 0, sizeof(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].nonce));}
    server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].state=notify_state;
    
    if(notify_mess!=NULL)
    {
      va_list args;
      va_start(args, notify_mess);
      s_len=vsnprintf(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].mess, sizeof(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].mess), notify_mess, args);
      va_end(args);
    }
    
    if(notify_mess==NULL || s_len<0)
    {
      if(sizeof(server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].mess)) server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].in_idx].mess[0]='\0';
    }
    
    server_notifys_sfifo[server_id].in_idx++;
    server_notifys_sfifo[server_id].in_idx&=server_notifys_idx_mask;
    
    result=true;
  }
  
  if(result)
  {
    extern void add_notify_to_blackbox(const uint8_t server_id, const uint32_t notify_uid, const uint8_t nonce[8], const notify_state_t notify_state, const char* notify_mess);
    add_notify_to_blackbox(server_id, notify_uid, nonce, notify_state, notify_mess);
  }
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreGive(notify_lock_obj);
#endif //NOTIFY_LOCK_EN
    
  return result;
}

bool get_server_notify(const uint8_t server_id, uint32_t* notify_uid, const uint8_t* nonce[8], notify_state_t* notify_state, const char** notify_mess)
{
  bool result=false;
  
  if(server_id>=NOTIFY_SERVERS_COUNT) return result;
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreTake(notify_lock_obj, portMAX_DELAY);
#endif //NOTIFY_LOCK_EN
  
  if((server_notifys_sfifo[server_id].in_idx-server_notifys_sfifo[server_id].out_idx)&(server_notifys_idx_mask))
  {    
    if(notify_uid!=NULL)   *notify_uid=server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].out_idx].uid;
    if(nonce!=NULL)        *nonce=server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].out_idx].nonce;
    if(notify_state!=NULL) *notify_state=server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].out_idx].state;
    if(notify_mess!=NULL)  *notify_mess=server_notifys_sfifo[server_id].notify[server_notifys_sfifo[server_id].out_idx].mess;
    
    result=true;
  }
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreGive(notify_lock_obj);
#endif //NOTIFY_LOCK_EN
  
  return result;
}

bool delete_server_notify(const uint8_t server_id)
{
  bool result=false;
  
  if(server_id>=NOTIFY_SERVERS_COUNT) return result;
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreTake(notify_lock_obj, portMAX_DELAY);
#endif //NOTIFY_LOCK_EN
  
  if((server_notifys_sfifo[server_id].in_idx-server_notifys_sfifo[server_id].out_idx)&(server_notifys_idx_mask))
  {
    server_notifys_sfifo[server_id].out_idx++;
    server_notifys_sfifo[server_id].out_idx&=server_notifys_idx_mask;
    
    result=true;
  }
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreGive(notify_lock_obj);
#endif //NOTIFY_LOCK_EN
  
  return result;
}

void clear_server_notifies(const uint8_t server_id)
{
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreTake(notify_lock_obj, portMAX_DELAY);
#endif //NOTIFY_LOCK_EN
  
  server_notifys_sfifo[server_id].in_idx=0;
  server_notifys_sfifo[server_id].out_idx=(server_notifys_sfifo[server_id].in_idx)&(server_notifys_idx_mask);
  
#if defined(NOTIFY_LOCK_EN)
  xSemaphoreGive(notify_lock_obj);
#endif //NOTIFY_LOCK_E
}

void init_server_notifies(void)
{
#if defined(NOTIFY_LOCK_EN)
  if(notify_lock_obj==NULL)
  {
    notify_lock_obj=xSemaphoreCreateMutex();
  }
#endif //NOTIFY_LOCK_EN

  for(uint8_t server_id=0; server_id<NOTIFY_SERVERS_COUNT; server_id++)
  {
    server_notifys_sfifo[server_id].in_idx=0;
    server_notifys_sfifo[server_id].out_idx=(server_notifys_sfifo[server_id].in_idx)&(server_notifys_idx_mask);
  }
}
