/**
******************************************************************************
* File Name          : vega_protocol.c
* Description        : Протокол обмена данными VEGA
*
*
******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "serv_protocols_lib/vega/vega_protocol.h"
#include "debug_port.h"
#include "macro_utils.h"
#if defined(VEGA_USE_NONCE_AUTH)
#include "tea_crypt/tea.h"
#endif //defined(VEGA_USE_NONCE_AUTH)

#if defined(CAN_VEGA_FWD_PRESENT)
#include "serv_protocols_lib/can_vega_fwd.h"
extern bool bad_mutex_take(volatile uint8_t* const mutex, const bool is_wait_for_taking);
extern void bad_mutex_give(volatile uint8_t* const mutex);
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
#include "serv_protocols_lib/fast_data_upload.h"
#endif //defined(FASTDATA_PRESENT)

#if defined(VEGA_SECURE_PRESENT) && defined(VEGA_ESECURE_PRESENT)
#error
#endif //defined(VEGA_SECURE_PRESENT) && defined(VEGA_ESECURE_PRESENT)

static const char* get_pheader(const uint8_t conn_id)
{
#if !defined(TINY_SERV_PROTOCOLS)
  switch (conn_id)
  {
  case 0:  return "vegaP_ctx1:";
  case 1:  return "vegaP_ctx2:";
  case 2:  return "vegaP_ctx3:";
  case 3:  return "vegaP_ctx4:";
  case 4:  return "vegaP_ctx5:";
  case 5:  return "vegaP_ctx6:";
  default: return "vegaP_ctxX:";
  }
#else
  return "";
#endif //!defined(TINY_SERV_PROTOCOLS)
}

NOT_USED(static const char* get_verbose_vega_packet_type(const EVega_message_type type))
{
  switch (type)
  {
  case COMMAND:                 return "COMMAND";
  case COMMAND_ANS:             return "COMMAND_ANS";
  case FILE_CHUNK:              return "FILE_CHUNK";
  case FILE_CHUNK_ANS:          return "FILE_CHUNK_ANS";
  case AUTORIZATION:            return "AUTORIZATION";
  case AUTORIZATION_ANS:        return "AUTORIZATION_ANS";
  case GET_FILE:                return "GET_FILE";
  case GET_FILE_ANS:            return "GET_FILE_ANS";
  case GET_DEV_INF:             return "GET_DEV_INF";
  case GET_DEV_INF_ANS:         return "GET_DEV_INF_ANS";
  case GET_DEV_LIST:            return "GET_DEV_LIST";
  case GET_DEV_LIST_ANS:        return "GET_DEV_LIST_ANS";
  case CONNECT_TO_DEV:          return "CONNECT_TO_DEV";
  case CONNECT_TO_DEV_ANS:      return "CONNECT_TO_DEV_ANS";
  case DISCONNECT_FROM_DEV:     return "DISCONNECT_FROM_DEV";
  case BBOX_MESSAGES:           return "BBOX_MESSAGES";
  case BBOX_MESSAGES_ANS:       return "BBOX_MESSAGES_ANS";
  case FILE_WHOLLY:             return "FILE_WHOLLY";
  case FILE_WHOLLY_ANS:         return "FILE_WHOLLY_ANS";
  case FILE_CLEAN:              return "FILE_CLEAN";
  case FILE_CLEAN_ANS:          return "FILE_CLEAN_ANS";
  case GET_FILE_REPORT:         return "GET_FILE_REPORT";
  case GET_FILE_REPORT_ANS:     return "GET_FILE_REPORT_ANS";
  case GET_STATUS_REPORT:       return "GET_STATUS_REPORT";
  case GET_STATUS_REPORT_ANS:   return "GET_STATUS_REPORT_ANS";
  case TRANSP_DATA_TO_ITF:      return "TRANSP_DATA_TO_ITF";
  case TRANSP_DATA_FROM_ITF:    return "TRANSP_DATA_FROM_ITF";
  case FWD_DATA_TO_CAN:         return "FWD_DATA_TO_CAN";
  case FWD_DATA_FROM_CAN:       return "FWD_DATA_FROM_CAN";
  case FAST_DATA_MESS:          return "FAST_DATA_MESS";
  case COMMAND_W_NONCE:         return "COMMAND_W_NONCE";
  case COMMAND_ANS_W_SINGN:     return "COMMAND_ANS_W_SINGN";
  default:                      return "unknown:";
  }
}

#if defined(VEGA_USE_NONCE_AUTH)
//
#endif //defined(VEGA_USE_NONCE_AUTH)

void VegaResetCtxState(vega_ctx_t* ctx)
{
  ctx->is_authenticated=0;
#if defined(VEGA_USE_NONCE_AUTH)
  ctx->nonce_pass=0;
#endif //defined(VEGA_USE_NONCE_AUTH)
  //ctx->is_wait_ping_ack=0;

  ctx->wait_out_len=0;

  ctx->is_bbox_upload_enable=0;
  ctx->bbox_pack_id_counter=0;
  ctx->bbox.is_wait_ack=0;

  ctx->streaming_dwn.wait_data_len=0;
  //т.к. дозагрузку сервер не умеет, в новой сессии сбрасываем ожидание кусков
  ctx->dwn_wait_chunk_num=0;//не ждем куски файлов

#if defined(CAN_VEGA_FWD_PRESENT)
  clean_rx_can_fwd_sfifo_from_reader();
#endif //CAN_VEGA_FWD_PRESENT
}

static int16_t socket_write(vega_ctx_t* ctx, uint8_t* buff, const uint16_t len)
{
#if defined(VEGA_ESECURE_PRESENT)
  if(ctx->esecure!=NULL)
  {
    int16_t ret=vega_esecure_tx_prepare(ctx->esecure, buff, len);
    if(ret<0) return ret;

    //отправляем из ctx->esecure->buff с длинной и подписью
    return ctx->common_const->socket_write(ctx->conn_ctx_id, ctx->esecure->tx.buff, ret);
  }
#endif //defined(VEGA_ESECURE_PRESENT)

#if defined(VEGA_SECURE_PRESENT)
  if(ctx->secure!=NULL)
  {
    int16_t ret=vega_secure_tx_prepare(ctx->secure, buff, len);
    if(ret<0) return ret;
  }
#endif //defined(VEGA_SECURE_PRESENT)

  return ctx->common_const->socket_write(ctx->conn_ctx_id, buff, len);
}

/*
Разбираем buff, длинной buff_len. Ещем максимум 1 пакет транспортного уровня.
return:
ret < 0
один из пакетов имеет неправильный заголовок, либо содержит не поддерживаемые поля, либо не сошлась crc в заголовке либо в данных
:
длина найденного транспортного пакета
*/
static int16_t VegaParseSinglePacket(vega_ctx_t* ctx)
{
  if(!ctx->parse_len) return 0;

  const TVega_message* header=(const TVega_message*)&ctx->parse_buff[0];

  if(ctx->parse_len>=sizeof(TVega_message))
  {//можем читать заголовок

    if(ctx->rx_mem_size<VEGA_PACKET_LEN(header->len))
    {
      if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s rx_mem_size is too small for packet reception\n", get_pheader(ctx->conn_ctx_id));
      return PROTOCOL_CTX_MEM_ERR;
    }

    if(ctx->parse_len>=VEGA_PACKET_LEN(header->len))
    {//пакет полностью принят
      uint32_t packet_crc;
      memcpy(&packet_crc, &ctx->parse_buff[sizeof(TVega_message)+header->len], sizeof(uint32_t));
      if(packet_crc==crc32_fast_full(&ctx->parse_buff[0], VEGA_PACKET_CRC_CALC_LEN(header->len)))
      {
        uint8_t* recv_pload;
        recv_pload=&ctx->parse_buff[VEGA_PACKET_PAYLOAD_OFFSET()];

        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->parse_buff, VEGA_PACKET_LEN(header->len), "%s got %s(%u), plen: %u\n", get_pheader(ctx->conn_ctx_id), get_verbose_vega_packet_type(header->type), header->type, header->len);
        if(VEGA_DEBUG_LEVEL==MDEBUG_L) __PRINTF("%s got %s(%u), plen: %u\n", get_pheader(ctx->conn_ctx_id), get_verbose_vega_packet_type(header->type), header->type, header->len);

        if(header->type!=COMMAND && ctx->wait_out_len)
        {
          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, ctx->wait_out_len, "%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);
          if(VEGA_DEBUG_LEVEL==MDEBUG_L) __PRINTF("%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);

          //если заполняли буфер ответами на команды и пришла не команда, вычищаем буфер, иначе он затрется другими ответами
          if(0>socket_write(ctx, ctx->tx_mem, ctx->wait_out_len)) return PROTOCOL_CTX_CONN_ERR;

          ctx->wait_out_len=0;
        }

        if(header->type==GET_DEV_INF)
        {
          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          TVega_get_dev_inf_ans* ans=(TVega_get_dev_inf_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          memcpy(ans->imei, ctx->common_const->get_device_imei(), 15);
          ans->imei[15]='\0';
          ans->dev_type=ctx->dev_type;

#if defined(VEGA_USE_NONCE_AUTH)
          //
#endif //defined(VEGA_USE_NONCE_AUTH)

          out_header->type=GET_DEV_INF_ANS;
          out_header->len=sizeof(TVega_get_dev_inf_ans);

          uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
          memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

          if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
        }
        else if(header->type==DISCONNECT_FROM_DEV)
        {
          VegaResetCtxState(ctx);
          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s configurator disconnect from device\n", get_pheader(ctx->conn_ctx_id));
        }
        else if(header->type==AUTORIZATION)
        {

          if(!(header->len==sizeof(TVega_auth_mess_legacy) || header->len==sizeof(TVega_auth_mess) || header->len==sizeof(TVega_hard_auth_mess)))
          {//неправильно сформирован пакет
            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }

          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          uint8_t* ans=&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          if(header->len==sizeof(TVega_hard_auth_mess))
          {
            TVega_hard_auth_mess* auth_mess=(TVega_hard_auth_mess*)recv_pload;

            uint8_t sys_pwd_len=strnlen(ctx->passwd, VEGA_HARD_PASS_LEN);
            uint8_t recv_pwd_len=strnlen(auth_mess->pass, sizeof(auth_mess->pass));

            if(!(*ctx->use_passwd_opt) \
              || (sys_pwd_len==recv_pwd_len && memcmp(auth_mess->pass, ctx->passwd, recv_pwd_len)==0)
#if defined(VEGA_USE_NONCE_AUTH)
                  || VegaVerefyNonceAuthPass(ctx, auth_mess)
#endif //defined(VEGA_USE_NONCE_AUTH)
                    )
            {
              if(!ctx->is_authenticated)
              {
                ctx->is_authenticated=1;

                if(auth_mess->is_bbox_upload_enable)
                  ctx->is_bbox_upload_enable=1;
                else
                  ctx->is_bbox_upload_enable=0;

                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s server/host is authorized, bbox_upload_enable=%hhu\n", get_pheader(ctx->conn_ctx_id), ctx->is_bbox_upload_enable);
              }
              else
              {
                if(auth_mess->is_bbox_upload_enable)
                  ctx->is_bbox_upload_enable=1;
                else
                  ctx->is_bbox_upload_enable=0;

                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s server/host already authorized, bbox_upload_enable=%hhu\n", get_pheader(ctx->conn_ctx_id), ctx->is_bbox_upload_enable);
              }
            }
            else
            {
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s autorization fail, wrong passwd\n", get_pheader(ctx->conn_ctx_id));
              ctx->is_authenticated=0;
              ctx->is_bbox_upload_enable=0;
            }
          }
          else
          {
            TVega_auth_mess* auth_mess=(TVega_auth_mess*)recv_pload;

            uint8_t sys_pwd_len=strnlen(ctx->passwd, VEGA_HARD_PASS_LEN);
            uint8_t recv_pwd_len=strnlen(auth_mess->pass, sizeof(auth_mess->pass));

            if(!(*ctx->use_passwd_opt) || (sys_pwd_len==recv_pwd_len && memcmp(auth_mess->pass, ctx->passwd, recv_pwd_len)==0))
            {
              if(!ctx->is_authenticated)
              {
                ctx->is_authenticated=1;

                if(header->len==sizeof(TVega_auth_mess) && auth_mess->is_bbox_upload_enable)
                  ctx->is_bbox_upload_enable=1;
                else
                  ctx->is_bbox_upload_enable=0;

                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s server/host is authorized, bbox_upload_enable=%hhu\n", get_pheader(ctx->conn_ctx_id), ctx->is_bbox_upload_enable);
              }
              else
              {
                if(header->len==sizeof(TVega_auth_mess) && auth_mess->is_bbox_upload_enable)
                  ctx->is_bbox_upload_enable=1;
                else
                  ctx->is_bbox_upload_enable=0;

                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s server/host already authorized, bbox_upload_enable=%hhu\n", get_pheader(ctx->conn_ctx_id), ctx->is_bbox_upload_enable);
              }
            }
            else
            {
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s autorization fail, wrong passwd\n", get_pheader(ctx->conn_ctx_id));
              ctx->is_authenticated=0;
              ctx->is_bbox_upload_enable=0;
            }
          }

          out_header->len=sizeof(uint8_t);
          out_header->type=AUTORIZATION_ANS;
          *ans=ctx->is_authenticated;
          uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
          memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

          if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
        }
        else if(!ctx->is_authenticated)
        {
          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          uint8_t* ans=&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          if(VEGA_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s server/host not authorized!!!\n", get_pheader(ctx->conn_ctx_id));

          out_header->len=sizeof(uint8_t);
          out_header->type=AUTORIZATION_ANS;
          *ans=ctx->is_authenticated;
          uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
          memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

          if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
        }
        else if(header->type==COMMAND || header->type==COMMAND_W_NONCE)
        {
          uint8_t command_ans_header_size;

          if(header->type==COMMAND)
          {
            command_ans_header_size=sizeof(TVega_command_ans);
          }
          else //if(header->type==COMMAND_W_SINGN)
          {
            command_ans_header_size=sizeof(TVega_command_ans_w_sign);
          }

          for(uint8_t i=0; ; i++)
          {
            if(i>=2)
            {//после высвобождения буфера, всеравно не хватает места
              ctx->wait_out_len=0;
              if(VEGA_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s tx_mem_size is too small for packet creation\n", get_pheader(ctx->conn_ctx_id));
              break;
            }
            uint16_t out_len=0;
            if(ctx->tx_mem_size <= ctx->wait_out_len+8 || 0>ctx->got_vega_packet_callback(ctx->conn_ctx_id, ctx->parse_buff, ctx->tx_mem+ctx->wait_out_len, ctx->tx_mem_size-ctx->wait_out_len, &out_len))
            {//недостаточно места для формирования ответа
              if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, ctx->wait_out_len, "%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);
              if(VEGA_DEBUG_LEVEL==MDEBUG_L) __PRINTF("%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);

              if(0>socket_write(ctx, ctx->tx_mem, ctx->wait_out_len)) return PROTOCOL_CTX_CONN_ERR;

              ctx->wait_out_len=0;
              continue;
            }
            //проверяем на запроса VEGA_INPUT_BUFFER_SIZE и VEGA_OUTPUT_BUFFER_SIZE, и подменяем на актуальный
            //так сделано из-за того, что syscom ничего незнает о буферах (они разные, для разных подключений)
            else if(out_len==sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_param_ans_args)+sizeof(uint16_t)+sizeof(uint32_t))
            {
              EVega_command_code req_comm_code;

              if(header->type==COMMAND)
              {
                const TVega_command* const req_comm=(const TVega_command* const)(ctx->parse_buff+sizeof(TVega_message));
		req_comm_code=req_comm->code;
              }
              else //if(header->type==COMMAND_W_SINGN)
              {
                const TVega_command_w_nonce* const req_comm=(const TVega_command_w_nonce* const)(ctx->parse_buff+sizeof(TVega_message));
		req_comm_code=req_comm->code;
              }

              if(req_comm_code==GET_PARAM)
              {
                TVega_command_ans* comm_ans=(TVega_command_ans*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message));
                if(comm_ans->result==DONE)
                {
                  TVega_get_param_ans_args* comm_ans_args=(TVega_get_param_ans_args*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size);
                  if((comm_ans_args->id==VEGA_INPUT_BUFFER_SIZE || comm_ans_args->id==VEGA_OUTPUT_BUFFER_SIZE) && comm_ans_args->len==sizeof(uint16_t))
                  {
                    uint16_t* arg=(uint16_t*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_param_ans_args));
                    if(comm_ans_args->id==VEGA_INPUT_BUFFER_SIZE) arg[0]=ctx->rx_mem_size;
                    else                                          arg[0]=ctx->tx_mem_size;
                    uint32_t* crc=(uint32_t*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_param_ans_args)+sizeof(arg[0]));
                    crc[0]=crc32_fast_full(ctx->tx_mem+ctx->wait_out_len, sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_param_ans_args)+sizeof(arg[0]));
                  }
                }
              }
            }
            else if(out_len==sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_ext_param_ans_args)+sizeof(uint16_t)+sizeof(uint32_t))
            {
              EVega_command_code req_comm_code;

              if(header->type==COMMAND)
              {
                const TVega_command* const req_comm=(const TVega_command* const)(ctx->parse_buff+sizeof(TVega_message));
		req_comm_code=req_comm->code;
              }
              else //if(header->type==COMMAND_W_SINGN)
              {
                const TVega_command_w_nonce* const req_comm=(const TVega_command_w_nonce* const)(ctx->parse_buff+sizeof(TVega_message));
		req_comm_code=req_comm->code;
              }

              if(req_comm_code==GET_EXT_PARAM)
              {
                TVega_command_ans* comm_ans=(TVega_command_ans*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message));
                if(comm_ans->result==DONE)
                {
                  TVega_get_ext_param_ans_args* comm_ans_args=(TVega_get_ext_param_ans_args*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size);
                  if((comm_ans_args->id==VEGA_INPUT_BUFFER_SIZE || comm_ans_args->id==VEGA_OUTPUT_BUFFER_SIZE) && comm_ans_args->len==sizeof(uint16_t))
                  {
                    uint16_t* arg=(uint16_t*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_ext_param_ans_args));
                    if(comm_ans_args->id==VEGA_INPUT_BUFFER_SIZE) arg[0]=ctx->rx_mem_size;
                    else                                          arg[0]=ctx->tx_mem_size;
                    uint32_t* crc=(uint32_t*)(ctx->tx_mem+ctx->wait_out_len+sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_ext_param_ans_args)+sizeof(arg[0]));
                    crc[0]=crc32_fast_full(ctx->tx_mem+ctx->wait_out_len, sizeof(TVega_message)+command_ans_header_size+sizeof(TVega_get_ext_param_ans_args)+sizeof(arg[0]));
                  }
                }
              }
            }
            ctx->wait_out_len+=out_len;
            break;
          }
        }
        else if(header->type==GET_FILE)
        {
          TVega_get_file* req=(TVega_get_file*)recv_pload;
          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          TVega_get_file_ans* ans=(TVega_get_file_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          memcpy(ans->filename, req->filename, sizeof(ans->filename));
          req->filename[sizeof(req->filename)-1]='\0';

          //если запрошенный кусок не вместится в tx_mem_size
          if((sizeof(TVega_message)+sizeof(TVega_get_file_ans)+req->data_len+sizeof(uint32_t)) > ctx->tx_mem_size)
          {
            ans->result=GET_FILE_DATA_SIZE_ERROR;
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%s\") out buff err\n", get_pheader(ctx->conn_ctx_id), req->filename);
          }
          else
          {
            int8_t res;
            res=ctx->common_const->file_read_chunk(ctx->conn_ctx_id, req->filename, &ans->filesize, &ctx->tx_mem[sizeof(TVega_message)+sizeof(TVega_get_file_ans)], &ans->data_size, req->data_len, req->offset);
            if(res==0)
            {
              ans->result=GET_FILE_OK;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got req bin file (\"%s\"), offset=%lu, req_len=%lu, fsize=%lu, upload=%.1f%%\n", get_pheader(ctx->conn_ctx_id), req->filename, req->offset, ans->data_size, ans->filesize, ((req->offset+ans->data_size)*100.0f)/ans->filesize);
            }
            else if(res==-4)
            {
              ans->result=GET_FILE_FILENAME_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%s\") fname err\n", get_pheader(ctx->conn_ctx_id), req->filename);
            }
            else if(res==-3)
            {
              ans->result=GET_FILE_OUT_OF_DATA;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%s\") out of data err\n", get_pheader(ctx->conn_ctx_id), req->filename);
            }
            else if(res==-2)
            {
              ans->result=GET_FILE_BUSY;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%s\") is busy\n", get_pheader(ctx->conn_ctx_id), req->filename);
            }
            else //if(res==-1)
            {
              ans->result=GET_FILE_MEMORY_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%s\") mem err\n", get_pheader(ctx->conn_ctx_id), req->filename);
            }
          }
          if(ans->result!=GET_FILE_OK) ans->data_size=0;

          out_header->type=GET_FILE_ANS;
          out_header->len=sizeof(TVega_get_file_ans)+ans->data_size;

          uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
          memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));
          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

          if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
        }
        else if(header->type==FILE_CHUNK)
        {
          TVega_file_chunk* req=(TVega_file_chunk*)recv_pload;
          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          TVega_file_chunk_ans* ans=(TVega_file_chunk_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          out_header->len=sizeof(TVega_file_chunk_ans);
          out_header->type=FILE_CHUNK_ANS;

          if(header->len!=sizeof(TVega_file_chunk)+req->chunk_len)
          {//неправильно сформирован пакет, докачка не возможна?
            //todo
            ctx->dwn_wait_chunk_num=0;
            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }

          uint8_t is_first_chunk=0;
          int8_t res;
          if(ctx->dwn_wait_chunk_num==0)
          {//пришел кусок и нет активной загрузки, начинаем новую загрузку
            ctx->dwn_wait_chunk_num=1;
            ctx->dwn_chunk_write_offset=0;
            ctx->dwn_fname="";
            is_first_chunk=1;
          }

          if(ctx->dwn_wait_chunk_num==req->chunk_num)
          {//получили ожидаемый кусок
            res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &ctx->dwn_fname, recv_pload+sizeof(TVega_file_chunk), req->chunk_len, is_first_chunk, 0);
            if(0>res)
            {
              if(res==-2)
              {//не допустимое имя файла
                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s \"%.*s\" file_write_chunk fail, wrong file name\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname);
                //todo
                memcpy(ans->filename, req->filename, sizeof(ans->filename));
                ans->chunk_num=req->chunk_num;
                ans->result=CHUNK_ANS_FILENAME_ERROR;
                ctx->dwn_wait_chunk_num=0;

                uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
                memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

                if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

                if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
              }
              else
              {//другая ошибка
                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s \"%.*s\" file_write_chunk fail\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname);
                //todo
                memcpy(ans->filename, req->filename, sizeof(ans->filename));
                ans->chunk_num=req->chunk_num;
                ans->result=CHUNK_ANS_MEMORY_ERROR;
                ctx->dwn_wait_chunk_num=0;

                uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
                memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

                if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

                if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
              }
            }
            else //без ошибок
            {
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got bin chunk (\"%.*s\"), len=%hu, %.1f%%\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname, req->chunk_len, (req->chunk_num*100.0f)/req->chunks_total);
            if(req->chunk_num==req->chunks_total)
            {//загрузка завершена

              //проверяем подпись файла
              res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &ctx->dwn_fname, NULL, NULL, NULL, 1);
              if(res<0)
              {//подпись не сошлась
                if(res==-3)
                {
                  if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%.*s\") wrong signature\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname);
                  ans->result=CHUNK_ANS_SIGNATURE_ERROR;
                }
                else
                {
                  if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%.*s\") save err\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname);
                  ans->result=CHUNK_ANS_MEMORY_ERROR;
                }
                memcpy(ans->filename, req->filename, sizeof(ans->filename));
                ans->chunk_num=req->chunk_num;
                ctx->dwn_wait_chunk_num=0;

                uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
                memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

                if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

                if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
              }
              else
              {
                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%.*s\") successfully downloaded and verified\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), ctx->dwn_fname);

                //todo
                memcpy(ans->filename, req->filename, sizeof(ans->filename));
                ans->chunk_num=req->chunk_num;
                ans->result=CHUNK_ANS_OK;
                ctx->dwn_wait_chunk_num=0;

                uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
                memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

                if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

                if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;

                //шлем сигнал о получении файла
                ctx->common_const->file_downloaded_signal(ctx->conn_ctx_id, &ctx->dwn_fname);
              }
            }
            else
            {
              memcpy(ans->filename, req->filename, sizeof(ans->filename));
              ans->chunk_num=req->chunk_num;
              ans->result=CHUNK_ANS_OK;

              uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
              memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

              if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

              if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;

              ctx->dwn_wait_chunk_num++;
              ctx->dwn_chunk_write_offset+=req->chunk_len;
            }
            }
          }
          else
          {//нарушена последовательность кусков, докачка не возможна
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bin file (\"%.*s\"), unexpected chunk number\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), req->filename);

            memcpy(ans->filename, req->filename, sizeof(ans->filename));
            ans->chunk_num=req->chunk_num;
            ans->result=CHUNK_ANS_SEQUENCE_ERROR;
            ctx->dwn_wait_chunk_num=0;

            uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
            memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

            if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

            if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
          }
        }
        else if(header->type==BBOX_MESSAGES_ANS)
        {
          if(header->len!=sizeof(TVega_bbox_pack_ans))
          {//неправильно сформирован пакет
            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }

          TVega_bbox_pack_ans* ans=(TVega_bbox_pack_ans*)recv_pload;

          if(ctx->bbox.is_wait_ack && ans->pack_id==ctx->bbox.waiting_pack_id_counter && ans->result==BBOX_ANS_OK)
          {
            if(VEGA_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s confirmed %hhu bbox mess\n", get_pheader(ctx->conn_ctx_id), ctx->bbox.mess_cnt_out);

            ctx->bbox.is_wait_ack=0;

            if(ctx->bbox_receiver!=0)
            {
              ctx->common_const->grab_mess(NULL);
              if(ctx->common_const->delete_messages(NULL, ctx->bbox_receiver,  ctx->bbox.mess_cnt_out)<0)
              {//если это произошло то все очень плохо
                ctx->common_const->release_mess(NULL);
                return PROTOCOL_CTX_BBOX_ERR;
              }
              ctx->common_const->release_mess(NULL);

              if(VEGA_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s deleted %hhu mess in bbox\n", get_pheader(ctx->conn_ctx_id), ctx->bbox.mess_cnt_out);
            }
            if(ctx->bbox.find_wrong_mess_fmt_mode)
            {
              if(ctx->bbox.mess_counter_in_wrong_fmt_mode) ctx->bbox.mess_counter_in_wrong_fmt_mode--;
              ctx->bbox.not_ack_counter=0;

              //если все сообщения отправлены, переходим к режиу отправки пакетами
              if(!ctx->bbox.mess_counter_in_wrong_fmt_mode)
              {
                if(VEGA_DEBUG_LEVEL>=LDEBUG_L) LOG("%s out of \"find wrong mess fmt mode\"\n", get_pheader(ctx->conn_ctx_id));
                ctx->bbox.find_wrong_mess_fmt_mode=0;
              }
            }
          }
          else//не ждем ack или пришел не тот ack или результат не BBOX_ANS_OK
          {
            if(!ctx->bbox.is_wait_ack)
            {//не ждем ack
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bbox ack err, we don't wait ack\n", get_pheader(ctx->conn_ctx_id));
              return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
            }
            else if(ans->pack_id!=ctx->bbox.waiting_pack_id_counter)
            {//не тот ack
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bbox ack err, unexpected counter: %hu, we wait: %hu\n", get_pheader(ctx->conn_ctx_id), ans->pack_id, ctx->bbox.waiting_pack_id_counter);
              return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
            }
            else
            {//сервер ответил ошибкой
              ctx->bbox.end_wait_ack_time=xTaskGetTickCount();
              ctx->bbox.is_wait_ack=2;//признак ответа с ошибкой
            }
          }
        }
        else if(header->type==FILE_WHOLLY)
        {
          if(header->len!=sizeof(TVega_file_wholly))
          {//неправильно сформирован пакет
            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }

#if defined(VEGA_ESECURE_PRESENT)
          if(ctx->esecure!=NULL)
          {
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s FILE_WHOLLY not support in esecure mode\n", get_pheader(ctx->conn_ctx_id));}

            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }
#endif //defined(VEGA_ESECURE_PRESENT)

          TVega_file_wholly* file=(TVega_file_wholly*)recv_pload;

          ctx->streaming_dwn.wait_data_len=file->filesize;
          ctx->streaming_dwn.data_size=file->filesize;
          memcpy(&ctx->streaming_dwn.fname, file->filename, sizeof(ctx->streaming_dwn.fname));
          ctx->streaming_dwn.recv_crc=file->filecrc;
          crc32_fast_init(&ctx->streaming_dwn.calc_crc);
          ctx->streaming_dwn.last_recv_chunk_time=xTaskGetTickCount();

          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s start file %.*s download, size %lu bytes\n", get_pheader(ctx->conn_ctx_id), sizeof(file->filename), file->filename, file->filesize);
        }
        else if(header->type==FILE_CLEAN)
        {
          if(header->len!=sizeof(TVega_file_clean))
          {//неправильно сформирован пакет
            return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
          }

          TVega_file_clean* req=(TVega_file_clean*)recv_pload;
          TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
          TVega_file_clean_ans* ans=(TVega_file_clean_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];
          int8_t res;

          out_header->len=sizeof(TVega_file_clean_ans);
          out_header->type=FILE_CLEAN_ANS;

          memcpy(ans->filename, req->filename, sizeof(ans->filename));
          req->filename[sizeof(req->filename)-1]='\0';

          res=ctx->common_const->file_clean(req->filename);

          if(res<0)
          {
            if(res==-2)
            {
              ans->result=FILE_CLEAN_FILENAME_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s clean \"%.*s\" file err, unauthorized name\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), req->filename);
            }
            else if(res==-3)
            {
              ans->result=FILE_CLEAN_FILE_BUSY;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s clean \"%.*s\" file err, file busy\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), req->filename);
            }
            else
            {
              ans->result=FILE_CLEAN_MEMORY_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s clean \"%.*s\" file err, memmory or other error\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), req->filename);
            }
          }
          else
          {
            ans->result=FILE_CLEAN_ANS_OK;
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s clean file \"%.*s\" was done\n", get_pheader(ctx->conn_ctx_id), sizeof(ans->filename), req->filename);
          }

          uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
          memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

          if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

          if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
        }

#if defined(REPORT_GEN_PRESENT)
        //
#endif //defined(REPORT_GEN_PRESENT)

#if defined(ITF_TRANSPORT_PRESENT)
        else if(header->type==TRANSP_DATA_TO_ITF)
        {
          const TVega_itf_data* to_itf_data=(const TVega_itf_data*)recv_pload;
          TVega_itf_data* from_itf_data=(TVega_itf_data*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

          if(NULL==ctx->common_const->itf_tx_rx_callback)
          {
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s recv itf packet but itf callback is NULL, drop it...\n", get_pheader(ctx->conn_ctx_id));}
          }
          else if(header->len!=(sizeof(*to_itf_data)+to_itf_data->data_len))
          {
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s recv itf packet with wrong len, drop it...\n", get_pheader(ctx->conn_ctx_id));}
          }
          else
          {
            memset(from_itf_data, 0, sizeof(*from_itf_data));
            from_itf_data->itf_id=to_itf_data->itf_id;
            from_itf_data->rs.baudrate=to_itf_data->rs.baudrate;
            from_itf_data->rs.timeout=to_itf_data->rs.timeout;
            from_itf_data->data_len=ctx->tx_mem_size-sizeof(TVega_message)-sizeof(*from_itf_data)-sizeof(uint32_t);
            from_itf_data->data_len=ctx->common_const->itf_tx_rx_callback(to_itf_data, &recv_pload[sizeof(*to_itf_data)],
                                                                          &ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()+sizeof(*from_itf_data)], from_itf_data->data_len);

            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s recv itf packet, itf: %hhu, tx_len: %hu, rx_len: %hu\n", get_pheader(ctx->conn_ctx_id), to_itf_data->itf_id, to_itf_data->data_len, from_itf_data->data_len);}

            if(from_itf_data->data_len>0)
            {
              TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
              out_header->type=TRANSP_DATA_FROM_ITF;
              out_header->len=sizeof(*from_itf_data)+from_itf_data->data_len;
              uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
              memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

              if(VEGA_DEBUG_LEVEL>=HDEBUG_L) {__BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));}

              if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
            }
          }
        }
#endif //defined(ITF_TRANSPORT_PRESENT)

#if defined(CAN_VEGA_FWD_PRESENT)
        else if(header->type==FWD_DATA_TO_CAN)
        {
          static volatile uint8_t mutex=0;

          TVega_can_data_header* can_data_header=(TVega_can_data_header*)recv_pload;

          if(header->len==(sizeof(TVega_can_data_header)+can_data_header->frame_count*sizeof(TVega_tx_can_frame)))
          {
            bad_mutex_take(&mutex, true);
            for(uint16_t i=0; i<can_data_header->frame_count; i++)
            {
              TVega_tx_can_frame* can_frame=(TVega_tx_can_frame*)&recv_pload[sizeof(TVega_can_data_header)+i*sizeof(TVega_tx_can_frame)];

              write_to_tx_can_fwd_sfifo(can_frame);
            }
            bad_mutex_give(&mutex);

            if(VEGA_DEBUG_LEVEL>=MDEBUG_L) {__PRINTF("%s got %u can messages\n", get_pheader(ctx->conn_ctx_id), can_data_header->frame_count);}
          }
          else
          {
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s wrong frame_count or message len in FWD_DATA_TO_CAN, drop packet...\n", get_pheader(ctx->conn_ctx_id));}
          }
        }
#endif //defined(CAN_VEGA_FWD_PRESENT)

        else
        {//неподдерживаемый тип сообщения, пропускаем
          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s recv packet with unsupport type (%hhu), drop it...\n", get_pheader(ctx->conn_ctx_id), header->type);
          //return VEGA_PACKET_LEN(header->len);
        }
      }
      else
      {//crc не совпало, пропускаем пакет
        if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s recv packet with wrong crc, drop it...\n", get_pheader(ctx->conn_ctx_id));
        //return VEGA_PACKET_LEN(header->len);
      }
    }
    else
    {//нужные ещё данные
      return 0;
    }
  }
  else
  {//нужные ещё данные
    return 0;
  }

#if defined(CHANGE_SIM_BY_NO_VALID_VEGA_PACKET_PRESENT)
  ctx->common_const->got_valid_vega_packet(ctx->conn_ctx_id, header);
#endif //defined(CHANGE_SIM_BY_NO_VALID_VEGA_PACKET_PRESENT)

  //возвращаем длину найденного транспортного пакета
  return VEGA_PACKET_LEN(header->len);
}

/*
Разбираем buff, длинной buff_len, на пакеты.
return:
ret < 0
один из пакетов имеет неправильный заголовок, либо содержит не поддерживаемые поля, либо не сошлась crc в заголовке либо в данных
:
длина разоброранных данных
*/
int16_t VegaRxProcessing(vega_ctx_t* ctx)
{
  int16_t processed_len=0;
  int16_t total_processed_len=0;
  uint16_t total_parse_len=ctx->parse_len;

  for(;;)
  {
    ctx->parse_buff+=processed_len;
    ctx->parse_len-=processed_len;

    if(ctx->streaming_dwn.wait_data_len==0)     processed_len=VegaParseSinglePacket(ctx);
    else                                        processed_len=0;

    if(processed_len<0)
    {
      ctx->streaming_dwn.wait_data_len=0;
      return processed_len;
    }
    else if(ctx->streaming_dwn.wait_data_len>0)
    {
      uint16_t curr_bin_data_len;
      uint8_t is_first_chunk=0;

      ctx->streaming_dwn.last_recv_chunk_time=xTaskGetTickCount();

      if(ctx->streaming_dwn.wait_data_len==ctx->streaming_dwn.data_size)
      {
        is_first_chunk=1;
      }

      total_processed_len+=processed_len;

      if(ctx->streaming_dwn.wait_data_len>=total_parse_len-total_processed_len) curr_bin_data_len=total_parse_len-total_processed_len;
      else                                                                      curr_bin_data_len=ctx->streaming_dwn.wait_data_len;

      total_processed_len+=curr_bin_data_len;
      ctx->streaming_dwn.wait_data_len-=curr_bin_data_len;

      crc32_fast_update(&ctx->streaming_dwn.calc_crc, &ctx->parse_buff[processed_len], curr_bin_data_len);
      const char* fname="";
      int8_t res;
      res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &fname, &ctx->parse_buff[processed_len], curr_bin_data_len, is_first_chunk, 0);
      if(0>res)
      {
        TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
        TVega_file_wholly_ans* ans=(TVega_file_wholly_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];
        out_header->len=sizeof(TVega_file_wholly_ans);
        out_header->type=FILE_WHOLLY_ANS;
        memcpy(ans->filename, ctx->streaming_dwn.fname, sizeof(ans->filename));

        if(res==-2)
        {
          ans->result=FILE_WHOLLY_FILENAME_ERROR;
          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", wrong file name\n", get_pheader(ctx->conn_ctx_id), fname);
        }
        else
        {
          ans->result=FILE_WHOLLY_MEMORY_ERROR;
          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", device memory error\n", get_pheader(ctx->conn_ctx_id), fname);
        }

        *(uint32_t*)&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)]=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));

        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

        if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;

        //тут нужно сделать таймаут, чтобы ответ ушел
        return PROTOCOL_CTX_BBOX_ERR;
      }
      if(VEGA_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s got bin chunk (\"%s\"), len=%hu, more wait=%lu(%.1f%%)\n", get_pheader(ctx->conn_ctx_id), fname, curr_bin_data_len, ctx->streaming_dwn.wait_data_len, ((ctx->streaming_dwn.data_size-ctx->streaming_dwn.wait_data_len)*100.0f)/ctx->streaming_dwn.data_size);

      if(ctx->streaming_dwn.wait_data_len==0)
      {
        //загрузка завершена
        TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];
        TVega_file_wholly_ans* ans=(TVega_file_wholly_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];
        out_header->len=sizeof(TVega_file_wholly_ans);
        out_header->type=FILE_WHOLLY_ANS;
        memcpy(ans->filename, ctx->streaming_dwn.fname, sizeof(ans->filename));

        crc32_fast_finish(&ctx->streaming_dwn.calc_crc);
        if(ctx->streaming_dwn.calc_crc==ctx->streaming_dwn.recv_crc)
        {
          //проверяем подпись файла
          res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &fname, NULL, NULL, NULL, 1);
          if(res<0)
          {
            if(res==-3)
            {
              ans->result=FILE_WHOLLY_SIGNATURE_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", wrong signature\n", get_pheader(ctx->conn_ctx_id), fname);
            }
            else
            {
              ans->result=FILE_WHOLLY_MEMORY_ERROR;
              if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", device memory error\n", get_pheader(ctx->conn_ctx_id), fname);
            }
          }
          else
          {
            ans->result=FILE_WHOLLY_ANS_OK;
            if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\" successfully downloaded\n", get_pheader(ctx->conn_ctx_id), fname);

            //шлем сигнал о получении файла
            ctx->common_const->file_downloaded_signal(ctx->conn_ctx_id, &fname);
          }
        }
        else
        {
          ans->result=FILE_WHOLLY_CRC_ERROR;
          if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", crc err\n", get_pheader(ctx->conn_ctx_id), fname);
        }

        *(uint32_t*)&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)]=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));

        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));

        if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;

        if(total_parse_len>total_processed_len)
        {
          //processed_len=total_processed_len;
          processed_len+=curr_bin_data_len;
          continue;// после бинарных данных ещё есть ещё пакет, разбираем...
        }
      }
      return total_processed_len;
    }
    else if(processed_len==0)
    {
      if(ctx->wait_out_len)
      {//вычищаем буфер
        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(ctx->tx_mem, ctx->wait_out_len, "%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);
        if(VEGA_DEBUG_LEVEL==MDEBUG_L) __PRINTF("%s we sent %hu bytes\n", get_pheader(ctx->conn_ctx_id), ctx->wait_out_len);

        if(0>socket_write(ctx, ctx->tx_mem, ctx->wait_out_len)) return PROTOCOL_CTX_CONN_ERR;

        ctx->wait_out_len=0;
      }
      return total_processed_len;
    }
    else
    {
      total_processed_len+=processed_len;
    }
  }
}

void VegaDeInitCtx(vega_ctx_t* ctx)
{
  memset(ctx, 0, sizeof(vega_ctx_t));

  ctx->streaming_dwn.wait_data_len=0;
  ctx->dwn_wait_chunk_num=0;//не ждем куски файлов
  ctx->bbox.find_wrong_mess_fmt_mode=0;
  ctx->bbox.not_ack_counter=0;
  ctx->is_bbox_upload_enable=0;
}

int16_t VegaBboxUploadHandle(vega_ctx_t* ctx)
{
  uint8_t bb_mess_out_cnt;
  uint32_t available_mess;

  if(ctx->bbox_receiver==0 || ctx->is_bbox_upload_enable==0) return 0;

  ctx->common_const->grab_mess(NULL);
  if(ctx->common_const->get_mess_count(NULL, ctx->bbox_receiver, &available_mess)<0)
  {
    ctx->common_const->release_mess(NULL);
    return PROTOCOL_CTX_BBOX_ERR;
  }
  if(!available_mess)
  {
    //ничего нет на отправку
    ctx->common_const->release_mess(NULL);
    return 0;
  }

  if(available_mess>UINT8_MAX) available_mess=UINT8_MAX;

  //если режим отправки по одному сообщению
  if(ctx->bbox.find_wrong_mess_fmt_mode) available_mess=1;

  TVega_message* header=(TVega_message*)&ctx->tx_mem[0];
  TVega_bbox_pack_header* bbox_header=(TVega_bbox_pack_header*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];
  uint16_t total_mess_size=0;

  for(bb_mess_out_cnt=0; bb_mess_out_cnt < available_mess; bb_mess_out_cnt++)
  {
    uint8_t* mess_ptr;
    mess_ptr=ctx->common_const->read_message(NULL, ctx->bbox_receiver, bb_mess_out_cnt);
    if(mess_ptr==NULL)
    {
      ctx->common_const->release_mess(NULL);
      if(VEGA_DEBUG_LEVEL>=LDEBUG_L) LOG("%s read_message error (available_mess=%lu, read_offset=%hhu)\n", get_pheader(ctx->conn_ctx_id), available_mess, bb_mess_out_cnt);
      return PROTOCOL_CTX_BBOX_ERR;
    }

    TVega_bb_message* vpos_data=(TVega_bb_message*)&mess_ptr[0];
    uint16_t curr_mess_size=sizeof(TVega_bb_message)+vpos_data->parameters_len;

    //поместится ли считанная запись?
    if(ctx->tx_mem_size <= VEGA_PACKET_LEN(sizeof(TVega_bbox_pack_header)+total_mess_size+curr_mess_size)) break;

    memcpy(&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()+sizeof(TVega_bbox_pack_header)+total_mess_size], vpos_data, curr_mess_size);
    total_mess_size+=curr_mess_size;
  }

  ctx->common_const->release_mess(NULL);

  if(!bb_mess_out_cnt)
  {
    //не хватает места чтобы создать пакет с хотябы одним сообщением координат
    if(VEGA_DEBUG_LEVEL>=LDEBUG_L) LOG("%s can't creat single bbox packet, deleting last mess...\n", get_pheader(ctx->conn_ctx_id));

    ctx->common_const->grab_mess(NULL);
    if(ctx->common_const->delete_messages(NULL, ctx->bbox_receiver, 1)<0)
    {//если это произошло то все очень плохо
      ctx->common_const->release_mess(NULL);
      return PROTOCOL_CTX_BBOX_ERR;
    }
    ctx->common_const->release_mess(NULL);

    return 0;//PROTOCOL_CTX_MEM_ERR;
  }

  header->len=sizeof(TVega_bbox_pack_header)+total_mess_size;
  header->type=BBOX_MESSAGES;
  bbox_header->pack_id=ctx->bbox_pack_id_counter;
  bbox_header->mess_cnt=bb_mess_out_cnt;

  *(uint32_t*)&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(header->len)]=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(header->len));

  ctx->bbox.waiting_pack_id_counter=ctx->bbox_pack_id_counter;
  ctx->bbox.is_wait_ack=1;
  ctx->bbox.end_wait_ack_time=VEGA_BBOX_WAIT_ACK_TIMEOUT_MS-1+xTaskGetTickCount();
  ctx->bbox_pack_id_counter++;
  ctx->bbox.mess_cnt_out=bb_mess_out_cnt;

  if(VEGA_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bbox packet sent, pack_len=%hu bytes, bbox cnt=%hhu\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(header->len), ctx->bbox.mess_cnt_out);

  if(0>socket_write(ctx, &ctx->tx_mem[0], VEGA_PACKET_LEN(header->len))) return PROTOCOL_CTX_CONN_ERR;

  return VEGA_PACKET_LEN(header->len);
}

#if defined(USE_SERVER_NOTIFY_IN_VEGA)
EVega_command_ans_result Convert_notify_state_from_cmd_result(const notify_state_t notify_state)
{
  switch(notify_state)
  {
  case IN_PROCESS_NOTIFY_STATE:   {return IN_PROCESS;}
  case IS_LAST_NOTIFY_STATE:      {return DONE;}
  case IS_BUSY_NOTIFY_STATE:      {return BUSY;}
  case UNKNOWN_ARGS_NOTIFY_STATE: {return UNKNOWN_PARAMETER;}
  default:                        {return COMM_ERROR;}
  }
}

int16_t VegaNotifiesUploadHandle(vega_ctx_t* ctx)
{
    uint32_t notify_uid;
    notify_state_t notify_state;
    const char* notify_mess;
    const uint8_t* nonce;

    if(get_server_notify(ctx->conn_ctx_id, &notify_uid, &nonce, &notify_state, &notify_mess))
    {
      TVega_message* out_header=(TVega_message*)&ctx->tx_mem[0];

      if(memcmp(nonce, "\x00\x00\x00\x00\x00\x00\x00\x00", 8)==0)
      {
        TVega_command_ans* ans=(TVega_command_ans*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

        out_header->len=sizeof(TVega_command_ans);
        out_header->type=COMMAND_ANS;

        ans->uid=notify_uid;
        ans->result=Convert_notify_state_from_cmd_result(notify_state);
        ans->arg_len=0;

        int s_len;
        s_len=snprintf((char*)&ctx->tx_mem[sizeof(TVega_message)+sizeof(TVega_command_ans)], (__UINT8_T_MAX__)-sizeof(TVega_message)-sizeof(TVega_command_ans), notify_mess);
        if(s_len>=0)
        {
          ans->arg_len=s_len+1;
        }

        out_header->len+=ans->arg_len;

        uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
        memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) {__BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));}

        if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
      }
      else
      {
        TVega_command_ans_w_sign* ans=(TVega_command_ans_w_sign*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

        out_header->len=sizeof(TVega_command_ans_w_sign);
        out_header->type=COMMAND_ANS_W_SINGN;

        ans->uid=notify_uid;
        ans->result=Convert_notify_state_from_cmd_result(notify_state);
        memcpy(ans->nonce, nonce, sizeof(ans->nonce));
        ans->arg_len=0;

        int s_len;
        s_len=snprintf((char*)&ctx->tx_mem[sizeof(TVega_message)+sizeof(TVega_command_ans_w_sign)], (__UINT8_T_MAX__)-sizeof(TVega_message)-sizeof(TVega_command_ans_w_sign), notify_mess);
        if(s_len>=0)
        {
          ans->arg_len=s_len+1;
        }

        out_header->len+=ans->arg_len;

        mbedtls_poly1305_mac(get_cmd_ans_mac_key(), &ans->nonce[0], out_header->len-sizeof(ans->mac), ans->mac);

        uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(out_header->len));
        memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(out_header->len)], &out_crc, sizeof(uint32_t));

        if(VEGA_DEBUG_LEVEL>=HDEBUG_L) {__BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(out_header->len), "%s we sent %hu bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(out_header->len));}

        if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(out_header->len))) return PROTOCOL_CTX_CONN_ERR;
      }

      delete_server_notify(ctx->conn_ctx_id);
    }

   return 0;
}
#endif //defined(USE_SERVER_NOTIFY_IN_VEGA)

#if defined(CAN_VEGA_FWD_PRESENT)
int16_t VegaCanMessUploadHandle(vega_ctx_t* ctx)
{
  static volatile uint8_t mutex=0;
  uint32_t max_naclen=0;

  if(NULL==ctx->common_const->is_can_fwd_enabled_callback || !ctx->common_const->is_can_fwd_enabled_callback(ctx->conn_ctx_id, &max_naclen)) return 0;

  if(max_naclen)
  {
    uint32_t naclen=0;
    if(0>ctx->common_const->socket_get_nack_len(ctx->conn_ctx_id, &naclen)) return PROTOCOL_CTX_CONN_ERR;

    if(naclen>max_naclen) return 0;
  }

  bad_mutex_take(&mutex, true);
  uint16_t avl_mess_count=filled_count_in_rx_can_fwd_sfifo();

  if(avl_mess_count)
  {
    TVega_message* header=(TVega_message*)&ctx->tx_mem[0];
    TVega_can_data_header* can_data_header=(TVega_can_data_header*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

    header->type=FWD_DATA_FROM_CAN;
    header->len=sizeof(TVega_can_data_header);
    can_data_header->frame_count=0;
    memset(can_data_header->reserved, 0x00, sizeof(can_data_header->reserved));

    uint16_t can_frames=(ctx->tx_mem_size-(sizeof(TVega_message)+sizeof(TVega_can_data_header)+sizeof(uint32_t)))/sizeof(TVega_rx_can_frame);

    if(can_frames>avl_mess_count) can_frames=avl_mess_count;
    if(can_frames>UINT8_MAX)      can_frames=UINT8_MAX;

    for(uint16_t i=0; i<can_frames; i++)
    {
      TVega_rx_can_frame* can_frame=(TVega_rx_can_frame*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()+sizeof(TVega_can_data_header)+i*sizeof(TVega_rx_can_frame)];

      read_from_rx_can_fwd_sfifo(can_frame);

      header->len+=sizeof(TVega_rx_can_frame);
      can_data_header->frame_count++;
    }
    bad_mutex_give(&mutex);

    uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(header->len));
    memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(header->len)], &out_crc, sizeof(uint32_t));

    if(VEGA_DEBUG_LEVEL>=HDEBUG_L) {__BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(header->len), "%s we sent %u bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(header->len));}
    else if(VEGA_DEBUG_LEVEL>=MDEBUG_L) {__PRINTF("%s we sent %u bytes with %u can messages\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(header->len), can_data_header->frame_count);}

    if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(header->len))) return PROTOCOL_CTX_CONN_ERR;
  }
  else
  {
    bad_mutex_give(&mutex);
  }

  return 0;
}
#endif //defined(CAN_VEGA_FWD_PRESENT)

#if defined(FASTDATA_PRESENT)
int16_t VegaFastDataUploadHandle(vega_ctx_t* ctx)
{
  //static volatile uint8_t mutex=0;
  uint32_t max_naclen=0;

  if(NULL==ctx->common_const->is_fastdata_fwd_enabled_callback || !ctx->common_const->is_fastdata_fwd_enabled_callback(ctx->conn_ctx_id, &max_naclen)) return 0;

  if(max_naclen)
  {
    uint32_t naclen=0;
    if(0>ctx->common_const->socket_get_nack_len(ctx->conn_ctx_id, &naclen)) return PROTOCOL_CTX_CONN_ERR;

    if(naclen>max_naclen) return 0;
  }

  //bad_mutex_take(&mutex, true);
  uint16_t avl_mess_count=filled_count_in_fast_data_sfifo();

  if(avl_mess_count)
  {
    TVega_message* header=(TVega_message*)&ctx->tx_mem[0];
    TVega_fast_data_header* fast_data_header=(TVega_fast_data_header*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()];

    header->type=FAST_DATA_MESS;
    header->len=sizeof(TVega_fast_data_header);
    //fast_data_header->version=0;
    fast_data_header->count=0;

    uint16_t max_count=(ctx->tx_mem_size-(sizeof(TVega_message)+sizeof(TVega_fast_data_header)+sizeof(uint32_t)))/sizeof(TVega_fast_data);

    if(max_count>avl_mess_count) max_count=avl_mess_count;
    if(max_count>UINT8_MAX)      max_count=UINT8_MAX;

    uint32_t last_drop_counter;
    for(uint16_t i=0; i<max_count; i++)
    {
      TVega_fast_data* data=(TVega_fast_data*)&ctx->tx_mem[VEGA_PACKET_PAYLOAD_OFFSET()+sizeof(TVega_fast_data_header)+i*sizeof(TVega_fast_data)];

      read_from_fast_data_sfifo(data);

      header->len+=sizeof(TVega_fast_data);
      fast_data_header->count++;
      last_drop_counter=data->drop_counter;
    }
    //bad_mutex_give(&mutex);

    uint32_t out_crc=crc32_fast_full(ctx->tx_mem, VEGA_PACKET_CRC_CALC_LEN(header->len));
    memcpy(&ctx->tx_mem[VEGA_PACKET_CRC_OFFSET(header->len)], &out_crc, sizeof(uint32_t));

    if(VEGA_DEBUG_LEVEL>=HDEBUG_L) {__BIN_BUFF_PRINTF(ctx->tx_mem, VEGA_PACKET_LEN(header->len), "%s we sent %u bytes:\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(header->len));}
    else if(VEGA_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s we sent %u bytes with %u fast data messages (last_drop_counter=%u)\n", get_pheader(ctx->conn_ctx_id), VEGA_PACKET_LEN(header->len), fast_data_header->count, last_drop_counter);}

    if(0>socket_write(ctx, ctx->tx_mem, VEGA_PACKET_LEN(header->len))) return PROTOCOL_CTX_CONN_ERR;
  }
  else
  {
    //bad_mutex_give(&mutex);
  }

  return 0;
}
#endif //defined(FASTDATA_PRESENT)

#if defined(BLE_SENSORS_PRESENT)
//
#endif //defined(BLE_SENSORS_PRESENT)
