/**
******************************************************************************
* File Name          : wialon_binary_protocol.c
* Description        :    wialon binary
*                      
*                      
******************************************************************************
*/
#include <stdbool.h>
#include "serv_protocols_lib/wialon_binary/wialon_binary_protocol.h"

#if defined(USE_SERVER_NOTIFY_IN_WCOMBINE)
#include "serv_protocols_lib/server_notifies.h"
#endif //USE_SERVER_NOTIFY_IN_WCOMBINE

#if defined(NO_PACK_DATA_TO_LOWER_DATA_TYPE_IN_WCOMBINE)
static const bool NO_PACK_DATA_TO_LOWER_DATA_TYPE=true;
#else
static const bool NO_PACK_DATA_TO_LOWER_DATA_TYPE=false;
#endif //NO_PACK_DATA_TO_LOWER_DATA_TYPE_IN_WCOMBINE

static const bool LBS_SEPARATE_SUBRECORD=true;

static const char* get_pheader(const uint8_t conn_id)
{
#if !defined(TINY_SERV_PROTOCOLS)
  switch (conn_id)
  {
  case 0:  return "wcombP_ctx1:";
  case 1:  return "wcombP_ctx2:";
  case 2:  return "wcombP_ctx3:";
  case 3:  return "wcombP_ctx4:";
  default: return "wcombP_ctxX:";
  }
#else
  return "";
#endif //!defined(TINY_SERV_PROTOCOLS)
}

static void wialon_bin_crc16_init(wialon_bin_crc_t* crc)
{
  *crc=0;
}

static void wialon_bin_crc16(uint8_t *buf, uint16_t len, wialon_bin_crc_t* crc)
{
  uint16_t var=*crc;
  
  for (uint16_t pos = 0; pos < len; pos++)
  {
    var ^= (uint16_t)buf[pos];                  // XOR byte into least sig. byte of crc
    
    for (uint8_t i = 8; i != 0; i--)             // Loop over each bit
    {
      if ((var & 0x0001) != 0)                  // If the LSB is set
      {                
        var >>= 1;                              // Shift right and XOR 0xA001
        var ^= 0xA001;
      }
      else                                       // Else LSB is not set
        var >>= 1;                              // Just shift right
    }
  }
  
  *crc=var;
  return;
}

static int16_t WialonBinSendDriverMessIpsFormat(wialon_bin_ctx_t* ctx, const char* mess, ...)
{
#if defined(TELTONIKA_CMD_ANS_MODE_PRESENT)
  return 0;
#endif //defined(TELTONIKA_CMD_ANS_MODE_PRESENT)
  
  int s_len;
  wialon_bin_crc_t crc;
  char* out_ptr;
  uint16_t available_out_mem;
  
  out_ptr=(char*)ctx->tx_mem;
  available_out_mem=ctx->tx_mem_size;
  
  s_len=snprintf(out_ptr, available_out_mem, "#M#");
  if(s_len<0 || s_len>=available_out_mem)
  {
    return 0;
    //return PROTOCOL_CTX_MEM_ERR;
  }
  out_ptr+=s_len; available_out_mem-=s_len;
  
  va_list args;
  va_start(args, mess);
  s_len=vsnprintf(out_ptr, available_out_mem, mess, args);
  va_end(args);
  if(s_len<0 || s_len>=available_out_mem) 
  {
    return 0;
    //return PROTOCOL_CTX_MEM_ERR;
  }
  out_ptr+=s_len; available_out_mem-=s_len;
  
  s_len=snprintf(out_ptr, available_out_mem, ";");
  if(s_len<0 || s_len>=available_out_mem) 
  {
    return 0;
    //return PROTOCOL_CTX_MEM_ERR;
  }
  out_ptr+=s_len; available_out_mem-=s_len;
  
  wialon_bin_crc16_init(&crc);
  wialon_bin_crc16(ctx->tx_mem+(sizeof("#M#")-1), (out_ptr-(char*)ctx->tx_mem)-(sizeof("#M#")-1), &crc);
  
  s_len=snprintf(out_ptr, available_out_mem, "%4.4hX\r\n", crc); 
  if(s_len<0 || s_len>=available_out_mem) 
  {
    return 0;
    //return PROTOCOL_CTX_MEM_ERR;
  }
  
  out_ptr+=s_len; available_out_mem-=s_len;
  
  if(0>ctx->common_const->socket_write(ctx->conn_ctx_id, ctx->tx_mem, out_ptr-(char*)ctx->tx_mem)) return PROTOCOL_CTX_CONN_ERR;
  
  return (int16_t)(out_ptr-(char*)ctx->tx_mem);
}

static int16_t WialonBinSendDriverMess(wialon_bin_ctx_t* ctx, const char* mess, ...)
{
#if defined(TELTONIKA_CMD_ANS_MODE_PRESENT)
  return 0;
#endif //defined(TELTONIKA_CMD_ANS_MODE_PRESENT)
  
  if(ctx->tx_mem_size<=(sizeof(wialon_bin_header_t)+sizeof(wialon_bin_data_header_t)+sizeof(uint8_t)+sizeof(wialon_bin_crc_t)))
  {
    //   
    return 0;
  }
     
  const uint16_t available_out_mem_for_text=ctx->tx_mem_size-(sizeof(wialon_bin_header_t)+sizeof(wialon_bin_data_header_t)+sizeof(uint8_t)+sizeof(wialon_bin_crc_t));
  
  //     
  wialon_bin_header_t* header=(wialon_bin_header_t*)&ctx->tx_mem[0];
  header->head=WIALON_BIN_TX_HEAD;
  header->type=DATA_WIALON_BIN_PACKET_TYPE;
  header->seq=ctx->tx_seq;
  header->len=0;
  
  wialon_bin_data_header_t* bin_data_header=(wialon_bin_data_header_t*)&ctx->tx_mem[sizeof(wialon_bin_header_t)];
  bin_data_header->time=ctx->common_const->get_unix_time();
  bin_data_header->count=1;
  header->len+=sizeof(wialon_bin_data_header_t);
  
  ctx->tx_mem[sizeof(wialon_bin_header_t)+sizeof(wialon_bin_data_header_t)]=WIALON_BIN_DRIVER_MESS_SUBRECORD_TYPE;
  header->len+=sizeof(uint8_t);
  
  va_list args;
  va_start(args, mess);
  int s_len=vsnprintf((char*)&ctx->tx_mem[sizeof(wialon_bin_header_t)+sizeof(wialon_bin_data_header_t)+sizeof(uint8_t)], available_out_mem_for_text, mess, args);
  va_end(args);
  if(s_len<0 || s_len>=available_out_mem_for_text)
  {
    //  
    return 0;
  }
  header->len+=s_len+1;//  + '\0'
  
  wialon_bin_crc_t* crc=(wialon_bin_crc_t*)&ctx->tx_mem[WIALON_BIN_TX_CRC_OFFSET(header->len)];
  wialon_bin_crc16_init(crc);  
  wialon_bin_crc16(ctx->tx_mem, WIALON_BIN_TX_CRC_OFFSET(header->len), crc);
  
  if(0>ctx->common_const->socket_write(ctx->conn_ctx_id, ctx->tx_mem, WIALON_BIN_TX_PACKET_SIZE(header->len))) return PROTOCOL_CTX_CONN_ERR;
  
  //ctx->wait_driver_mess_ack_seq=ctx->tx_seq;
  ctx->tx_seq++;
  //ctx->is_wait_driver_mess_ack=1;
  
  return WIALON_BIN_TX_PACKET_SIZE(header->len);
}

void WialonBinResetCtxState(wialon_bin_ctx_t* ctx)
{
  ctx->is_authenticated=0;
  ctx->is_wait_ping_ack=0;
  ctx->dwn.wait_data_len=0;
  ctx->bbox.is_wait_ack=0;
  ctx->tx_seq=0;
  
  ctx->upl.is_wait_ack=0;
  ctx->upl.last_chunk=0;
  ctx->upl.curr_chunk=1;
  
  //ctx->is_wait_driver_mess_ack=0;
}

void WialonBinDeInitCtx(wialon_bin_ctx_t* ctx)
{
  memset(ctx, 0, sizeof(wialon_bin_ctx_t));
  
  ctx->bbox.find_wrong_mess_fmt_mode=0;
  ctx->bbox.not_ack_counter=0; 
}


static uint16_t parse_custom_commands(wialon_bin_ctx_t* ctx, char* pl, uint16_t parse_len, int16_t (*SendDriverMess)(wialon_bin_ctx_t* ctx, const char* mess, ...))
{
  //    '#'   '/' (   UUID)
  if(pl[0]!='#' && pl[0]!='/') return 0;
    
  if(1)
  {
    //   
    uint16_t i;
    for(i=0; i<parse_len; i++)
    {
      if(pl[i]=='\r') break;
    }
    if(i>=parse_len) return 0;
  }
  
#if !defined(TINY_SERV_PROTOCOLS)
  uint8_t can_cmd_idx;
  uint8_t command_uuid_str_len;
  uint32_t command_uuid;
#endif //!defined(TINY_SERV_PROTOCOLS)
  
  if(parse_len>=(sizeof("#tofactory#\r\n")-1) && strncmp(pl, "#tofactory#\r\n", sizeof("#tofactory#\r\n")-1)==0)
  {
    SendDriverMess(ctx, "tofactory command received\n");
    
    ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, TOFACTORY_CMD_TYPE, 10, NULL);
    
    return (sizeof("#tofactory#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#bboxclear#\r\n")-1) && strncmp(pl, "#bboxclear#\r\n", sizeof("#bboxclear#\r\n")-1)==0)
  {
    SendDriverMess(ctx, "bboxclear command received\n");
    
    ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, BBOX_CLEAR_CMD_TYPE, 10, NULL);
    
    return (sizeof("#bboxclear#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#rebootmodem#\r\n")-1) && strncmp(pl, "#rebootmodem#\r\n", sizeof("#rebootmodem#\r\n")-1)==0)
  {
    SendDriverMess(ctx, "rebootmodem command received\n");
    
    ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, REBOOT_MODEM_CMD_TYPE, 10, NULL);
    
    return (sizeof("#rebootmodem#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#rebootgnss#\r\n")-1) && strncmp(pl, "#rebootgnss#\r\n", sizeof("#rebootgnss#\r\n")-1)==0)
  {
    if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, REBOOT_GNSS_CMD_TYPE, 0, NULL))
      SendDriverMess(ctx, "rebootgnss command not support\n");
    else
      SendDriverMess(ctx, "rebootgnss command received\n");
    
    return (sizeof("#rebootgnss#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#reboot#\r\n")-1) && strncmp(pl, "#reboot#\r\n", sizeof("#reboot#\r\n")-1)==0)
  {
    SendDriverMess(ctx, "reboot command received\n");
    
    ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, REBOOT_CMD_TYPE, 10, NULL);
    
    return (sizeof("#reboot#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#getlog#\r\n")-1) && strncmp(pl, "#getlog#\r\n", sizeof("#getlog#\r\n")-1)==0)
  {
    return (sizeof("#getlog#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#makephoto#\r\n")-1) && strncmp(pl, "#makephoto#\r\n", sizeof("#makephoto#\r\n")-1)==0)
  {
    if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, MAKE_PHOTO_CMD_TYPE, 0, NULL))
      SendDriverMess(ctx, "makephoto command not support\n");
    else 
      SendDriverMess(ctx, "makephoto command received\n");
    
    return (sizeof("#makephoto#\r\n")-1);
  }
  else if(parse_len>=(sizeof("#server?#\r\n")-1) && strncmp(pl, "#server?#\r\n", sizeof("#server?#\r\n")-1)==0)
  {
    if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, SERVER_CFG_INFO, 0, &ctx->tx_mem[ctx->tx_mem_size/2]))
      SendDriverMess(ctx, "server? command err\n");
    else 
      SendDriverMess(ctx, "server config:\n%s\n", (char*)&ctx->tx_mem[ctx->tx_mem_size/2]);
    
    return (sizeof("#server?#\r\n")-1);
  }
  else if((parse_len>=(sizeof("#setoutX=Y#\r\n")-1) \
    && strncmp(pl, "#setout", sizeof("#setout")-1)==0 \
      && strncmp(pl+sizeof("#setoutX")-1, "=", sizeof("=")-1)==0 \
        && strncmp(pl+sizeof("#setoutX=Y#")-1, "\r\n", sizeof("\r\n")-1)==0) \
          || 
            (parse_len>=(sizeof("#setoutXX=Y#\r\n")-1) \
              && strncmp(pl, "#setout", sizeof("#setout")-1)==0 \
                && strncmp(pl+sizeof("#setoutXX")-1, "=", sizeof("=")-1)==0 \
                  && strncmp(pl+sizeof("#setoutXX=Y#")-1, "\r\n", sizeof("\r\n")-1)==0)
              )
  {
    ext_cmd_set_out_param_t param;
    uint16_t c_len;
    
    param.pin=(uint8_t)strtol(pl+sizeof("#setout")-1, NULL, 10);
    if(pl[sizeof("#setoutX")-1]=='=')
    {
      param.state=(uint8_t)strtol(pl+sizeof("#setoutX=")-1, NULL, 10);
      c_len=(sizeof("#setoutX=Y#\r\n")-1);
    }
    else
    {
      param.state=(uint8_t)strtol(pl+sizeof("#setoutXX=")-1, NULL, 10);
      c_len=(sizeof("#setoutXX=Y#\r\n")-1);
    }
    
    if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, SET_OUT_STATE, 0, &param))
    {
      SendDriverMess(ctx, "setout%hhu=%hhu command err\n", param.pin, param.state);
    }
    else 
    {
      SendDriverMess(ctx, "setout%hhu=%hhu command received\n", param.pin, param.state);
    }
    
    return c_len;
  }
  else if((parse_len>=(sizeof("#setextoutX=Y#\r\n")-1) \
    && strncmp(pl, "#setextout", sizeof("#setextout")-1)==0 \
      && strncmp(pl+sizeof("#setextoutX")-1, "=", sizeof("=")-1)==0 \
        && strncmp(pl+sizeof("#setextoutX=Y#")-1, "\r\n", sizeof("\r\n")-1)==0) \
          || 
            (parse_len>=(sizeof("#setextoutXX=Y#\r\n")-1) \
              && strncmp(pl, "#setextout", sizeof("#setextout")-1)==0 \
                && strncmp(pl+sizeof("#setextoutXX")-1, "=", sizeof("=")-1)==0 \
                  && strncmp(pl+sizeof("#setextoutXX=Y#")-1, "\r\n", sizeof("\r\n")-1)==0)
              )
  {
    ext_cmd_set_out_param_t param;
    uint16_t c_len;
    
    param.pin=(uint8_t)strtol(pl+sizeof("#setextout")-1, NULL, 10);
    if(pl[sizeof("#setextoutX")-1]=='=')
    {
      param.state=(uint8_t)strtol(pl+sizeof("#setextoutX=")-1, NULL, 10);
      c_len=(sizeof("#setextoutX=Y#\r\n")-1);
    }
    else
    {
      param.state=(uint8_t)strtol(pl+sizeof("#setextoutXX=")-1, NULL, 10);
      c_len=(sizeof("#setextoutXX=Y#\r\n")-1);
    }
    
    if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, SET_EXT_OUT_STATE, 0, &param))
    {
      SendDriverMess(ctx, "setextout%hhu=%hhu command err\n", param.pin, param.state);
    }
    else 
    {
      SendDriverMess(ctx, "setextout%hhu=%hhu command received\n", param.pin, param.state);
    }
    
    return c_len;
  }
  else if((parse_len>=(sizeof("#runcanscriptX#\r\n")-1) \
    && strncmp(pl, "#runcanscript", sizeof("#runcanscript")-1)==0 \
      && strncmp(pl+sizeof("#runcanscriptX")-1, "#\r\n", sizeof("#\r\n")-1)==0) \
        || \
          (parse_len>=(sizeof("#runcanscriptXX#\r\n")-1) \
            && strncmp(pl, "#runcanscript", sizeof("#runcanscript")-1)==0 \
              && strncmp(pl+sizeof("#runcanscriptXX")-1, "#\r\n", sizeof("#\r\n")-1)==0)
            || \
              (parse_len>=(sizeof("#stopcanscripts#\r\n")-1) \
                && strncmp(pl, "#stopcanscripts#\r\n", sizeof("#stopcanscripts#\r\n")-1)==0)
                )
  {
    uint16_t c_len;
    
    if(strncmp(pl, "#stopcanscripts#\r\n", sizeof("#stopcanscripts#\r\n")-1)==0)
    {
      c_len=(sizeof("#stopcanscripts#\r\n")-1);
      
      int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, STOP_CAN_SCRIPTS, 0, NULL);
      
      if(cmd_ret<0)
      {
        SendDriverMess(ctx, "stopcanscripts command err%s\n", (cmd_ret==-2)?(", busy"):(""));
      }
      else 
      {
        SendDriverMess(ctx, "stopcanscripts command received\n");
      }
    }
    else
    {
      uint8_t param=(uint8_t)strtol(pl+sizeof("#runcanscript")-1, NULL, 10);
      
      if(pl[sizeof("#runcanscriptX")-1]=='#')
      {
        c_len=(sizeof("#runcanscriptX#\r\n")-1);
      }
      else
      {
        c_len=(sizeof("#runcanscriptXX#\r\n")-1);
      }
      
      int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, RUN_CAN_SCRIPT, 0, &param);
      
      if(cmd_ret<0)
      {
        SendDriverMess(ctx, "runcanscript%u command err%s\n", param, (cmd_ret==-2)?(", busy"):(""));
      }
      else 
      {
        SendDriverMess(ctx, "runcanscript%u command received\n", param);
      }
    }
    
    return c_len;
  }
  else if(parse_len>=(sizeof("#changesim#\r\n")-1) && strncmp(pl, "#changesim#\r\n", sizeof("#changesim#\r\n")-1)==0 ||
          parse_len>=(sizeof("#changesim:1#\r\n")-1) && strncmp(pl, "#changesim:1#\r\n", sizeof("#changesim:1#\r\n")-1)==0 ||
          parse_len>=(sizeof("#changesim:2#\r\n")-1) && strncmp(pl, "#changesim:2#\r\n", sizeof("#changesim:2#\r\n")-1)==0)
  {
    const char* verbose_cmd="changesim";
    const uint8_t* desire_sim_id_from_cmd=NULL;
    uint16_t c_len=(sizeof("#changesim#\r\n")-1);
    
    if(strncmp(pl, "#changesim:1#\r\n", sizeof("#changesim:1#\r\n")-1)==0)
    {
      static const uint8_t SIM_1_ID=0;
      desire_sim_id_from_cmd=&SIM_1_ID;
      verbose_cmd="changesim:1";
      c_len=(sizeof("#changesim:1#\r\n")-1);
    }
    else if(strncmp(pl, "#changesim:2#\r\n", sizeof("#changesim:2#\r\n")-1)==0)
    {
      static const uint8_t SIM_2_ID=1;
      desire_sim_id_from_cmd=&SIM_2_ID;
      verbose_cmd="changesim:2";
      c_len=(sizeof("#changesim:2#\r\n")-1);
    }
    
    int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, CHANGE_SIM_CARD_CMD_TYPE, 0, desire_sim_id_from_cmd);
    
    if(cmd_ret<0)
    {
      const char* verbose_err="";
      
      if(cmd_ret==-2)      verbose_err=", busy";
      else if(cmd_ret==-3) verbose_err=", already in use";
      
      SendDriverMess(ctx, "%s command err%s\n", verbose_cmd, verbose_err);
    }
    else
    {
      SendDriverMess(ctx, "%s command received\n", verbose_cmd);
    }
    
    return c_len;
  }
  else if(parse_len>=(sizeof("#updbalance#\r\n")-1) && strncmp(pl, "#updbalance#\r\n", sizeof("#updbalance#\r\n")-1)==0)
  {
    const char* verbose_cmd="updbalance";
    const uint16_t c_len=(sizeof("#updbalance#\r\n")-1);
    
    int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, UPD_BALANCE_CMD_TYPE, 0, NULL);
    
    if(cmd_ret<0)
    {
      SendDriverMess(ctx, "%s command not support\n", verbose_cmd);
    }
    else
    {
      SendDriverMess(ctx, "%s command received\n", verbose_cmd);
    }
    
    return c_len;
  }
#if !defined(TINY_SERV_PROTOCOLS)
  else if(parse_len>=(sizeof("#makeddd#\r\n")-1) && strncmp(pl, "#makeddd#\r\n", sizeof("#makeddd#\r\n")-1)==0 ||
          parse_len>=(sizeof("#makeddd:1#\r\n")-1) && strncmp(pl, "#makeddd:1#\r\n", sizeof("#makeddd:1#\r\n")-1)==0 ||
          parse_len>=(sizeof("#makeddd:2#\r\n")-1) && strncmp(pl, "#makeddd:2#\r\n", sizeof("#makeddd:2#\r\n")-1)==0)
  {
    const char* verbose_cmd="makeddd";
    const uint8_t* desire_slot_id_from_cmd=NULL;
    uint16_t c_len=(sizeof("#makeddd#\r\n")-1);
    
    if(strncmp(pl, "#makeddd#\r\n", sizeof("#makeddd#\r\n")-1)==0 ||
       strncmp(pl, "#makeddd:1#\r\n", sizeof("#makeddd:1#\r\n")-1)==0)
    {
      static const uint8_t SLOT1_ID=0;
      desire_slot_id_from_cmd=&SLOT1_ID;
      
      if(strncmp(pl, "#makeddd#\r\n", sizeof("#makeddd#\r\n")-1)==0)
      {
        verbose_cmd="makeddd";
        c_len=(sizeof("#makeddd#\r\n")-1);
      }
      else
      {
        verbose_cmd="makeddd:1";
        c_len=(sizeof("#makeddd:1#\r\n")-1);
      }
    }
    else if(strncmp(pl, "#makeddd:2#\r\n", sizeof("#makeddd:2#\r\n")-1)==0)
    {
      static const uint8_t SLOT2_ID=1;
      desire_slot_id_from_cmd=&SLOT2_ID;
      verbose_cmd="makeddd:2";
      c_len=(sizeof("#makeddd:2#\r\n")-1);
    }
    
    int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, MAKE_DDD_FILE_CMD_TYPE, 0, desire_slot_id_from_cmd);
    
    if(cmd_ret<0)
    {
      const char* verbose_err="";
      
      if(cmd_ret==-2) verbose_err=", busy";
      else if(cmd_ret==-3) verbose_err=", communication failed";
      else if(cmd_ret==-4) verbose_err=", not available";
      else if(cmd_ret==-5) verbose_err=", not configured";
      
      SendDriverMess(ctx, "%s command err%s\n", verbose_cmd, verbose_err);
    }
    else
    {
      SendDriverMess(ctx, "%s command received\n", verbose_cmd);
    }
    
    return c_len;
  }
  else if(parse_len>=(sizeof("#ftp://")-1) && 
          strncmp(pl, "#ftp://", sizeof("#ftp://")-1)==0 &&
          strnstr(pl, "#\r\n", parse_len)!=NULL)
  {
    const char* verbose_cmd="ftp";
    char* end=strnstr(pl, "#\r\n", parse_len);
    const uint16_t c_len=end-pl+sizeof("#\r\n")-1;
    end[0]='\0';
    
    SendDriverMess(ctx, "got link: %s\r\n", &pl[1]);
        
    int8_t cmd_ret=ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, FTP_LINK_CMD_TYPE, 0, &pl[1]);
    
    if(cmd_ret<0)
    {
      SendDriverMess(ctx, "%s download err(%hhi)\r\n", verbose_cmd, cmd_ret);
    }
    else
    {
      SendDriverMess(ctx, "%s download ok\r\n", verbose_cmd);
    }
    
    return c_len;
  }
  else if((can_cmd_idx=get_wialon_can_cmd_table_idx(0, pl, parse_len, &command_uuid, &command_uuid_str_len))!=UNKNOWN_CAN_CMD_TABLE_IDX)
  {
    uint8_t cmd_str_len;
    const char* verbose;
    uint8_t cmd[6];
    //[0]-command
    //[1-4]-uuid
    //[5]-server_id
    
    cmd_str_len=get_wialon_can_cmd_params(0, can_cmd_idx, &cmd[0], &verbose);
        
    if(cmd_str_len)
    {
      if(command_uuid_str_len)
      {
        memcpy(&cmd[1], &command_uuid, sizeof(uint32_t));
        cmd[5]=ctx->conn_ctx_id;
        
        if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, CAN_CONTROL_CMD_W_UUID_TYPE, 0, &cmd[0]))
        {
          SendDriverMess(ctx, "/%lX %s command err\n", command_uuid, verbose);
        }
        else
        {
          SendDriverMess(ctx, "/%lX %s command received\n", command_uuid, verbose);
        }
      }
      else
      {
        if(0>ctx->common_const->common_cmd_callback(ctx->conn_ctx_id, CAN_CONTROL_CMD_TYPE, 0, &cmd[0]))
        {
          SendDriverMess(ctx, "%s command err\n", verbose);
        }
        else
        {
          SendDriverMess(ctx, "%s command received\n", verbose);
        }
      }
    }
    
    return cmd_str_len+command_uuid_str_len;
  }
#endif //!defined(TINY_SERV_PROTOCOLS)
  
  return 0;
}


/*
 buff,  buff_len.   1 .
return:
ret < 0  
     ,     ,    crc     
:
   
*/
static int16_t WialonBinParseSinglePacket(wialon_bin_ctx_t* ctx)
{ 
  if(!ctx->parse_len) return 0;
  
  uint16_t custom_commands_pased_len;
  
  char* pl=(char*)ctx->parse_buff;
  if(ctx->parse_len>4 && (strncmp(pl, "#US#", 4)==0 || strncmp(pl, "#UC#", 4)==0))
  {//        
    char* end_ptr=strnstr((char*)ctx->parse_buff, "\r\n", ctx->parse_len);
    
    if(end_ptr==NULL) return 0;//  
    
    end_ptr+=strlen("\r\n");
    
    if(*(pl+2)=='S') ctx->dwn.data_type=0;
    else             ctx->dwn.data_type=1;
    
    pl+=4;
    
    ctx->dwn.data_size=strtoul(pl, NULL, 10);
    if(!ctx->dwn.data_size) 
    {
      return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
    }
    ctx->dwn.wait_data_len=ctx->dwn.data_size;
    
    pl=strchr(pl, ';');
    if(pl==NULL) return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
    pl++;
    
    ctx->dwn.recv_crc=(uint16_t)strtoul(pl, NULL, 16);
    if(!ctx->dwn.recv_crc)
    {
      ctx->dwn.wait_data_len=0;
      return  PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
    }
    
    wialon_bin_crc_t crc;
    wialon_bin_crc16_init(&crc);
    ctx->dwn.calc_crc=crc;
    //    
    
    return end_ptr-(char*)ctx->parse_buff;
  }
  else if((custom_commands_pased_len=parse_custom_commands(ctx, pl, ctx->parse_len, WialonBinSendDriverMessIpsFormat))>0)
  {
    return custom_commands_pased_len;
  }
  else if(ctx->parse_len>=(sizeof("#AM#X\r\n")-1) && (strncmp(pl, "#AM#1\r\n", 7)==0 || strncmp(pl, "#AM#0\r\n", 7)==0 ))//   
  {
    return (sizeof("#AM#X\r\n")-1);
  }
  else// 
  {
    wialon_bin_server_packet_header_t* header=(wialon_bin_server_packet_header_t*)&ctx->parse_buff[0];
    
    if(ctx->parse_len>=sizeof(wialon_bin_server_packet_header_t))
    {//  
      
      if(header->head!=WIALON_BIN_RX_HEAD)
      {
        if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s can't find correct head\n", get_pheader(ctx->conn_ctx_id));
        return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
      }
      
      if(header->code==WIALON_BIN_CMD_TO_DEVICE_CODE)
      {
        //todo:  ,  
        uint32_t pl_len;
        uint8_t  pl_len_field_size;
        
        //   
        if((ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)])&(0x80))
        {
          pl_len_field_size=4;

          pl_len =(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+0])<<24;
          pl_len|=(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+1])<<16;
          pl_len|=(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+2])<<8;
          pl_len|=(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+3])<<0;
          pl_len&=~0x80000000;
        }
        else
        {
          pl_len_field_size=2;
          
          pl_len =(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+0])<<8;
          pl_len|=(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+1])<<0;
        }
        
        if(ctx->rx_mem_size<sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len+sizeof(wialon_bin_crc_t))
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s rx_mem_size is too small for packet reception\n", get_pheader(ctx->conn_ctx_id));}
          return PROTOCOL_CTX_MEM_ERR;
        }
        
        if(ctx->parse_len>=sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len+sizeof(wialon_bin_crc_t))
        {//   
          wialon_bin_crc_t command_crc;
          wialon_bin_crc16_init(&command_crc);  
          wialon_bin_crc16(&ctx->parse_buff[0], sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len, &command_crc);

          if(__REV16(command_crc)==*(wialon_bin_crc_t*)&ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len])
          {
            if(WIALON_BIN_DEBUG_LEVEL>=MDEBUG_L) {__PRINTF("%s got command\n", get_pheader(ctx->conn_ctx_id));}
            
            //        5 
            if(pl_len>sizeof(uint32_t)+sizeof(uint8_t))
            {
              //  sizeof(Type)==1  Custom command
              if(ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+sizeof(uint32_t)]==WIALON_BIN_CUSTOM_COMMAND_TYPE)
              {
                uint16_t custom_command_len=pl_len-(sizeof(uint32_t)+sizeof(uint8_t));
                char* custom_comand=(char*)&ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+sizeof(uint32_t)+sizeof(uint8_t)];
                
                //  crc   "\r\n"
                ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len+0]='\r';
                ctx->parse_buff[sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len+1]='\n';
                  
                if(parse_custom_commands(ctx, custom_comand, custom_command_len+sizeof("\r\n")-1, WialonBinSendDriverMess))
                {
                  if(WIALON_BIN_DEBUG_LEVEL>=HDEBUG_L) {__PRINTF("%s got known custom cmd:%.*s\n", get_pheader(ctx->conn_ctx_id), custom_command_len, custom_comand);}
                }
                else
                {
                  if(WIALON_BIN_DEBUG_LEVEL>=HDEBUG_L) {__PRINTF("%s got unknown custom cmd:%.*s\n", get_pheader(ctx->conn_ctx_id), custom_command_len, custom_comand);}
                }
              }
            }
            else
            {
              //   Time  Type
              if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s payload size err in command packet, drop it...\n", get_pheader(ctx->conn_ctx_id));}
            }
          }
          else
          {
            if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) {__PRINTF("%s bad crc in command packet, drop it...\n", get_pheader(ctx->conn_ctx_id));}
          }
          
          //crc  ,   
          return (sizeof(wialon_bin_server_packet_header_t)+pl_len_field_size+pl_len+sizeof(wialon_bin_crc_t));   
        }
      }
      else
      {
        wialon_bin_serv_ack_t* ack=(wialon_bin_serv_ack_t*)&ctx->parse_buff[0];
        
        if(ack->code==WIALON_BIN_PACKET_OK_RESP_CODE || ack->code==WIALON_BIN_INVALID_PACKET_RESP_CODE)
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got ans, seq=%hu\n", get_pheader(ctx->conn_ctx_id), ack->seq);
          
          if(ctx->is_authenticated==0 && ack->code==WIALON_BIN_PACKET_OK_RESP_CODE)
          {
            if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s auth success\n", get_pheader(ctx->conn_ctx_id));
            ctx->is_authenticated=1;
          }
          else
          {
            if(ctx->bbox.is_wait_ack && ack->seq==ctx->bbox.waiting_rx_seq)
            {
              if(ack->code==WIALON_BIN_PACKET_OK_RESP_CODE)
              {
                if(WIALON_BIN_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s confirmed %hhu bbox mess\n", get_pheader(ctx->conn_ctx_id), ctx->bbox.mess_cnt_out);
                
                ctx->bbox.is_wait_ack=0;
                
                if(ctx->bbox_receiver!=0)
                {
                  ctx->common_const->grab_mess(NULL);
                  if(ctx->common_const->delete_messages(NULL, ctx->bbox_receiver,  ctx->bbox.mess_cnt_out)<0)
                  {//      
                    ctx->common_const->release_mess(NULL);
                    return PROTOCOL_CTX_BBOX_ERR;
                  }
                  ctx->common_const->release_mess(NULL); 
                  
                  if(WIALON_BIN_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s deleted %hhu mess in bbox\n", get_pheader(ctx->conn_ctx_id), ctx->bbox.mess_cnt_out);
                }
                if(ctx->bbox.find_wrong_mess_fmt_mode)
                {
                  if(ctx->bbox.mess_counter_in_wrong_fmt_mode) ctx->bbox.mess_counter_in_wrong_fmt_mode--;
                  ctx->bbox.not_ack_counter=0;
                  
                  //   ,     
                  if(!ctx->bbox.mess_counter_in_wrong_fmt_mode)
                  {
                    if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s out of \"find wrong mess fmt mode\"\n", get_pheader(ctx->conn_ctx_id));
                    ctx->bbox.find_wrong_mess_fmt_mode=0;
                  }
                }
              }
              else
              {
                if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s bbox ack err, ack_code: %hhu\n", get_pheader(ctx->conn_ctx_id), ack->code);
                ctx->bbox.end_wait_ack_time=xTaskGetTickCount();
                ctx->bbox.is_wait_ack=2;//   
              }
            }
#if !defined(TINY_SERV_PROTOCOLS)
            else if(ctx->upl.is_wait_ack && ctx->upl.waiting_rx_seq==ack->seq)
            {
              if(ack->code==WIALON_BIN_PACKET_OK_RESP_CODE)
              {// 
                if(WIALON_BIN_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s confirmed %hu file chunk\n", get_pheader(ctx->conn_ctx_id), ctx->upl.curr_chunk-1);
                
                if(ctx->upl.curr_chunk==ctx->upl.last_chunk+1)
                {
                  //    
                  if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s %s was fully uploaded\n", get_pheader(ctx->conn_ctx_id), (ctx->upl.file_type==0)?(ctx->upl.name):("ddd file"));
                  
                  ctx->upl.is_wait_ack=0;
                  //ctx->photo_tx_last_chunk=0;
                  //ctx->photo_tx_curr_chunk=1;
                  
                  if(0>ctx->common_const->photo_delete(ctx->conn_ctx_id, ctx->upl.name))
                  {
                    if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s \"%s\" photo_delete fail\n", get_pheader(ctx->conn_ctx_id), ctx->upl.name);
                    return PROTOCOL_CTX_BBOX_ERR;
                  }
                }
                else
                {
                  ctx->upl.is_wait_ack=0;
                }
              }
              else
              {//    WIALON_BIN_PACKET_OK_RESP_CODE   ,  (???)
                
                if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s \"%s\" will be deleted (wrong ack resp)\n", get_pheader(ctx->conn_ctx_id), ctx->upl.name);
                
                if(0>ctx->common_const->photo_delete(ctx->conn_ctx_id, ctx->upl.name))
                {
                  if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s \"%s\" photo_delete fail\n", get_pheader(ctx->conn_ctx_id), ctx->upl.name);
                  return PROTOCOL_CTX_BBOX_ERR;
                }
                
                ctx->upl.is_wait_ack=0;
                ctx->upl.last_chunk=0;
                ctx->upl.curr_chunk=1;
                
                //return PROTOCOL_CTX_SENT_DATA_FMT_ERR;
              }
            }
#endif //!defined(TINY_SERV_PROTOCOLS)
            else if(ctx->is_wait_ping_ack && ctx->wait_ping_ack_seq==ack->seq)//    ,   
            {
              extern void got_ping_ans(const uint8_t conn_id);
              got_ping_ans(ctx->conn_ctx_id);
              
              ctx->is_wait_ping_ack=0;
            }
            //     , ..    ,     ,    ack->seq    ctx->wait_driver_mess_ack_seq
            //else if(ctx->is_wait_driver_mess_ack && ctx->wait_driver_mess_ack_seq==ack->seq)//      ,   
            //{
            //  ctx->is_wait_driver_mess_ack=0;
            //}
            else
            {
              if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s ack err, unexpected ack_seq: %hu, we wait: bbox:%hu(%hhu) or ping:%hu(%hhu) or file:%hu(%hhu)\n", get_pheader(ctx->conn_ctx_id), ack->seq, ctx->bbox.waiting_rx_seq, ctx->bbox.is_wait_ack, ctx->wait_ping_ack_seq, ctx->is_wait_ping_ack, ctx->upl.waiting_rx_seq, ctx->upl.is_wait_ack);
              // , ..          
              //return PROTOCOL_CTX_PARSE_DATA_FMT_ERR;
            }
          }
        }
        else if(ack->code==WIALON_BIN_INVALID_CRC_RESP_CODE)
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got ans, invalid crc\n", get_pheader(ctx->conn_ctx_id));
          
          //     bbox
          if(ctx->bbox.is_wait_ack) 
          {
            ctx->bbox.end_wait_ack_time=xTaskGetTickCount();
            ctx->bbox.is_wait_ack=2;//   
          }
        }
        else if(ack->code==WIALON_BIN_AUTH_ERR_RESP_CODE)
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got ans, auth err\n", get_pheader(ctx->conn_ctx_id));
          return PROTOCOL_CTX_AUTH_ERR;
        }
        else if(ack->code==WIALON_BIN_INVALID_PWD_RESP_CODE)
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got ans, invalid pwd\n", get_pheader(ctx->conn_ctx_id));
          return PROTOCOL_CTX_AUTH_ERR;
        }
        else
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s got ans, unknown code (%hu)\n", get_pheader(ctx->conn_ctx_id), ack->seq);
        }
        return  sizeof(wialon_bin_serv_ack_t);
      }
    }
    else
    {//  
      return 0;
    }
  }
  //   
  return 0;
}

int16_t WialonBinRxProcessing(wialon_bin_ctx_t* ctx)
{
  int16_t processed_len=0;
  int16_t total_processed_len=0;
  uint16_t total_parse_len=ctx->parse_len;
  
  for(;;)
  {
    ctx->parse_buff+=processed_len;
    ctx->parse_len-=processed_len;
    
    if(ctx->dwn.wait_data_len==0) processed_len=WialonBinParseSinglePacket(ctx);
    else                          processed_len=0;
    
    if(processed_len<0) 
    {
      ctx->dwn.wait_data_len=0; 
      return processed_len;
    }
    else if(ctx->dwn.wait_data_len>0)
    {
      uint16_t curr_bin_data_len;
      uint8_t is_first_chunk=0;
      
      if(ctx->dwn.wait_data_len==ctx->dwn.data_size) 
      {
        WialonBinSendDriverMess(ctx, "file download command received");
        is_first_chunk=1;
      }
      
      total_processed_len+=processed_len;
      
      if(ctx->dwn.wait_data_len>=total_parse_len-total_processed_len) curr_bin_data_len=total_parse_len-total_processed_len;
      else                                                            curr_bin_data_len=ctx->dwn.wait_data_len;
      
      total_processed_len+=curr_bin_data_len;
      ctx->dwn.wait_data_len-=curr_bin_data_len;
      
      wialon_bin_crc_t crc;
      crc=ctx->dwn.calc_crc;
      wialon_bin_crc16(&ctx->parse_buff[processed_len], curr_bin_data_len, &crc);  
      ctx->dwn.calc_crc=crc;
      const char* fname="";
      int8_t res;
      res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &fname, &ctx->parse_buff[processed_len], curr_bin_data_len, is_first_chunk, 0);
      if(0>res)
      {
        if(res==-2)
        {
          WialonBinSendDriverMess(ctx, "file \"%s\", wrong file name\n", fname);
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", wrong file name\n", get_pheader(ctx->conn_ctx_id), fname);
        }
        else
        {
          WialonBinSendDriverMess(ctx, "file \"%s\", device memory error\n", fname);
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", device memory error\n", get_pheader(ctx->conn_ctx_id), fname);
        }
        return PROTOCOL_CTX_BBOX_ERR;
      }
      if(WIALON_BIN_DEBUG_LEVEL>=MDEBUG_L) __PRINTF("%s got bin chunk (type=%hhu, \"%s\"), len=%hu, more wait=%lu(%.1f%%)\n", get_pheader(ctx->conn_ctx_id), ctx->dwn.data_type, fname, curr_bin_data_len, ctx->dwn.wait_data_len, ((ctx->dwn.data_size-ctx->dwn.wait_data_len)*100.0f)/ctx->dwn.data_size);
      
      if(ctx->dwn.wait_data_len==0)
      {
        // 
        if(ctx->dwn.calc_crc==ctx->dwn.recv_crc)
        {
          //  
          res=ctx->common_const->file_write_chunk(ctx->conn_ctx_id, &fname, NULL, NULL, NULL, 1);
          if(res<0)
          {
            if(res==-3)
            {
              WialonBinSendDriverMess(ctx, "file \"%s\", wrong signature\n", fname);
              if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", wrong signature\n", get_pheader(ctx->conn_ctx_id), fname);
            }
            else
            {
              WialonBinSendDriverMess(ctx, "file \"%s\", device memory error\n", fname);
              if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", device memory error\n", get_pheader(ctx->conn_ctx_id), fname);
            }
          }
          else
          {
            if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\" successfully downloaded\n", get_pheader(ctx->conn_ctx_id), fname);
            WialonBinSendDriverMess(ctx, "file \"%s\" successfully downloaded\n", fname);
            
            //    
            ctx->common_const->file_downloaded_signal(ctx->conn_ctx_id, &fname);
          }
        }
        else
        {
          if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s file \"%s\", crc err\n", get_pheader(ctx->conn_ctx_id), fname);
          WialonBinSendDriverMess(ctx, "file \"%s\", crc err\n", fname);
        }
        
        if(total_parse_len>total_processed_len) 
        {
          //processed_len=total_processed_len;
          processed_len+=curr_bin_data_len;
          continue;//       , ...
        }
      }
      return total_processed_len;
    }
    else if(processed_len==0)
    {
      return total_processed_len;
    }
    else 
    {
      total_processed_len+=processed_len;
    }
  }
}

int16_t WialonBinSendAuthRequest(wialon_bin_ctx_t* ctx)
{
  uint8_t use_passwd=*ctx->use_passwd_opt;
  
  wialon_bin_header_t* header=(wialon_bin_header_t*)&ctx->tx_mem[0];
  header->head=WIALON_BIN_TX_HEAD;
  header->type=LOGIN_WIALON_BIN_PACKET_TYPE;
  header->seq=ctx->tx_seq;
  header->len=sizeof(wialon_bin_login_t);
  ctx->tx_seq++;
  
  wialon_bin_login_t* data=(wialon_bin_login_t*)&ctx->tx_mem[WIALON_BIN_TX_DATA_OFFSET()];
    
  data->version=WIALON_BIN_PROTOCOL_VERSION;
  if(use_passwd)
  {
    uint8_t pwd_len=strnlen(ctx->passwd, WIALON_BIN_MAX_PASSWD_LEN);
    char* pwd=(char*)&ctx->tx_mem[WIALON_BIN_TX_DATA_OFFSET()+sizeof(wialon_bin_login_t)];
    
    data->flags=WIALON_BIN_UINT64_ID_STRING_PWD;
    memcpy(pwd, ctx->passwd, pwd_len);
    pwd[pwd_len]='\0';
    header->len+=(pwd_len+1);//     
  }
  else
  {
    data->flags=WIALON_BIN_UINT64_ID_NO_PWD;
  }
  data->id=ctx->terminal_id;
  
  wialon_bin_crc_t* crc=(wialon_bin_crc_t*)&ctx->tx_mem[WIALON_BIN_TX_CRC_OFFSET(header->len)];
  wialon_bin_crc16_init(crc);  
  wialon_bin_crc16(ctx->tx_mem, WIALON_BIN_TX_CRC_OFFSET(header->len), crc);
  
  if(0>ctx->common_const->socket_write(ctx->conn_ctx_id, &ctx->tx_mem[0], WIALON_BIN_TX_PACKET_SIZE(header->len))) return PROTOCOL_CTX_CONN_ERR;
  
  if(WIALON_BIN_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(&ctx->tx_mem[0], WIALON_BIN_TX_PACKET_SIZE(header->len), "%s sent auth packet, pack_len=%hu bytes\n", get_pheader(ctx->conn_ctx_id), WIALON_BIN_TX_PACKET_SIZE(header->len));
  if(WIALON_BIN_DEBUG_LEVEL==MDEBUG_L || WIALON_BIN_DEBUG_LEVEL==LDEBUG_L) __PRINTF("%s sent auth packet, pack_len=%hu bytes\n", get_pheader(ctx->conn_ctx_id), WIALON_BIN_TX_PACKET_SIZE(header->len));
  
  return WIALON_BIN_TX_PACKET_SIZE(header->len);
}

int16_t WialonBinBboxUploadHandle(wialon_bin_ctx_t* ctx)
{
  uint8_t bb_mess_out_cnt;
  uint32_t available_mess;
  uint16_t subrecord_offset=0;
  
  if(ctx->bbox_receiver==0) return 0;
  
  ctx->common_const->grab_mess(NULL);
  if(ctx->common_const->get_mess_count(NULL, ctx->bbox_receiver, &available_mess)<0)
  {
    ctx->common_const->release_mess(NULL);
    return PROTOCOL_CTX_BBOX_ERR;
  }
  if(!available_mess)
  {
    //   
    ctx->common_const->release_mess(NULL);
    return 0;
  }
  
  if(available_mess>UINT8_MAX) available_mess=UINT8_MAX;
  
  //     
  if(ctx->bbox.find_wrong_mess_fmt_mode) available_mess=1;
  
  //    
  subrecord_offset+=sizeof(wialon_bin_data_t);
  
  uint8_t pack_err=0;
  
  uint16_t backup_subrecord_offset;
  
  for(bb_mess_out_cnt=0; bb_mess_out_cnt < available_mess; bb_mess_out_cnt++)
  {     
    backup_subrecord_offset=subrecord_offset;   
    
    //     Position Data?
    if(ctx->tx_mem_size < subrecord_offset+\
      (sizeof(wialon_bin_data_header_t)+sizeof(wialon_bin_pos_subrecord_t))+sizeof(wialon_bin_crc_t)) 
    {
      pack_err=1; 
      break;
    }
    
    uint8_t* mess_ptr;
    mess_ptr=ctx->common_const->read_message(NULL, ctx->bbox_receiver, bb_mess_out_cnt);
    if(mess_ptr==NULL)
    {
      ctx->common_const->release_mess(NULL);
      if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s read_message error (available_mess=%lu, read_offset=%hhu)\n", get_pheader(ctx->conn_ctx_id), available_mess, bb_mess_out_cnt);
      return PROTOCOL_CTX_BBOX_ERR;
    }
    
    TVega_bb_message* vpos_data=(TVega_bb_message*)&mess_ptr[0];
    
    wialon_bin_data_header_t* bin_data_header=(wialon_bin_data_header_t*)&ctx->tx_mem[subrecord_offset];
    if(vpos_data->time>=SERV_TIME_BACK_S) vpos_data->time-=SERV_TIME_BACK_S;
    bin_data_header->time=vpos_data->time;
    bin_data_header->count=1;//  Position Data
    subrecord_offset+=sizeof(wialon_bin_data_header_t);
    
    wialon_bin_pos_subrecord_t* pos_subrecord=(wialon_bin_pos_subrecord_t*)&ctx->tx_mem[subrecord_offset];
    pos_subrecord->subrecord_type=WIALON_BIN_POS_DATA_SUBRECORD_TYPE;
    pos_subrecord->lat=(int32_t)(vpos_data->lat*1000000.0f);
    pos_subrecord->lon=(int32_t)(vpos_data->lon*1000000.0f);
    pos_subrecord->speed=vpos_data->speed;
    pos_subrecord->course=vpos_data->direction;
    pos_subrecord->height=vpos_data->alt;
    pos_subrecord->sats=vpos_data->sat_count;
    pos_subrecord->hdop=0;//    wcombine hdop        VEGA
    subrecord_offset+=sizeof(wialon_bin_pos_subrecord_t);
    
    if(vpos_data->parameters_len)//  
    {
      uint8_t* read_param_pos;
      
      uint16_t param_id;
      uint8_t  param_len;
      param_types_t param_type;
      void*   param_value;
      uint8_t param_type_field_len=0;
      const uint8_t* param_type_in_pack=NULL;
      
      //c    param type
      if(vpos_data->pack_with_type) param_type_field_len=1;
      
      if(vpos_data->parameters_len >= (4+param_type_field_len))
      {
        TVega_lbs_data lbs={0};
        
        read_param_pos=(uint8_t*)vpos_data + sizeof(TVega_bb_message);
        
        wialon_bin_custom_param_subrecord_header_t* custom_subrecord_header=(wialon_bin_custom_param_subrecord_header_t*)&ctx->tx_mem[subrecord_offset];
        //custom_subrecord_header->subrecord_type=WIALON_BIN_CUSTOM_PARAM_SUBRECORD_TYPE;
        //custom_subrecord_header->count=X|0x8000;
        subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_header_t);//,          
        
        //  
        uint8_t sub_count=0;
        
        while(vpos_data->parameters_len)
        {
          memcpy(&param_id, read_param_pos, sizeof(uint16_t));
          read_param_pos+=sizeof(uint16_t);
          if(param_type_field_len)
          {
            param_type_in_pack=read_param_pos;
            read_param_pos+=param_type_field_len;
          }
          memcpy(&param_len, read_param_pos, sizeof(uint8_t));
          read_param_pos+=sizeof(uint8_t);
          param_value=read_param_pos;
          read_param_pos+=param_len;
                    
          if(sizeof(param_id)+param_type_field_len+sizeof(param_len)+param_len>vpos_data->parameters_len)
          {
            //       
            pack_err=2;
            break;
          }
          
          uint16_t trans_param_id;
          if(0>ctx->common_const->get_message_param_name_type(param_id, &trans_param_id, param_type_in_pack, NULL, &param_type, NULL))
          {
            //  
            pack_err=2;
            break;
          }
          
          if(param_type==FLOAT_PARAM_TYPE && param_id==VEGA_HDOP)
          {
            //    
            pos_subrecord->hdop=(uint16_t)(*(float*)param_value*100.0f);
          }
          else if(LBS_SEPARATE_SUBRECORD && param_type==ARRAY_PARAM_TYPE && param_id==VEGA_GSM_LBS)
          {
            memcpy(&lbs, param_value, sizeof(lbs));
          }
          else if(param_type==BOOL_PARAM_TYPE || param_type==UINT8_PARAM_TYPE || param_type==UINT16_PARAM_TYPE || param_type==UINT32_PARAM_TYPE || param_type==UINT64_PARAM_TYPE || param_type==FLOAT_PARAM_TYPE || param_type==STRING_PARAM_TYPE \
            || param_type==INT8_PARAM_TYPE || param_type==INT16_PARAM_TYPE || param_type==INT32_PARAM_TYPE || param_type==INT64_PARAM_TYPE || param_type==DOUBLE_PARAM_TYPE)
          {
            if(trans_param_id) {param_id=trans_param_id;}
            
            uint8_t custom_subrecord_size;
            if(param_id>127)    custom_subrecord_size=sizeof(wialon_bin_custom_param_subrecord_ext_t);
            else                custom_subrecord_size=sizeof(wialon_bin_custom_param_subrecord_t);
            
            if(param_type==UINT8_PARAM_TYPE || param_type==BOOL_PARAM_TYPE)
            {
              //    wialon_bin_u8_t?
              //  wialon_bin_custom_param_subrecord_header_t  
              if(ctx->tx_mem_size < subrecord_offset +
                 (custom_subrecord_size+sizeof(wialon_bin_u8_t)) +
                   sizeof(wialon_bin_crc_t)) 
              {
                pack_err=1;
                break;
              }
              
              if(param_id>127)
              {
                wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id|0x8000;
                custom_subrecord->type=WIALON_BIN_U8_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
              }
              else
              {
                wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id;
                custom_subrecord->type=WIALON_BIN_U8_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
              }
              
              wialon_bin_u8_t* u8_data=(wialon_bin_u8_t*)&ctx->tx_mem[subrecord_offset];
              *u8_data=*(uint8_t*)param_value;
              subrecord_offset+=sizeof(wialon_bin_u8_t);
            }
            else if(param_type==INT8_PARAM_TYPE)
            {
              //    wialon_bin_i8_t?
              //  wialon_bin_custom_param_subrecord_header_t  
              if(ctx->tx_mem_size < subrecord_offset +
                 (custom_subrecord_size+sizeof(wialon_bin_i8_t)) +
                   sizeof(wialon_bin_crc_t))
              {
                pack_err=1;
                break;
              }
              
              if(param_id>127)
              {
                wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id|0x8000;
                custom_subrecord->type=WIALON_BIN_I8_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
              }
              else
              {
                wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id;
                custom_subrecord->type=WIALON_BIN_I8_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
              }
              
              wialon_bin_i8_t* i8_data=(wialon_bin_i8_t*)&ctx->tx_mem[subrecord_offset];
              *i8_data=*(int8_t*)param_value;
              subrecord_offset+=sizeof(wialon_bin_i8_t);
            }
            else if(param_type==UINT16_PARAM_TYPE) 
            {
              if(/*NO_PACK_DATA_TO_LOWER_DATA_TYPE ||*/ *(uint16_t*)param_value>UINT8_MAX)
              {
                //    wialon_bin_u16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u16_t* u16_data=(wialon_bin_u16_t*)&ctx->tx_mem[subrecord_offset];
                *u16_data=*(uint16_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_u16_t);
              }
              else
              {
                //    wialon_bin_u8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u8_t* u8_data=(wialon_bin_u8_t*)&ctx->tx_mem[subrecord_offset];
                *u8_data=(uint8_t)(*(uint16_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_u8_t);
              }
            }
            else if(param_type==INT16_PARAM_TYPE) 
            {
              if(NO_PACK_DATA_TO_LOWER_DATA_TYPE || *(int16_t*)param_value>INT8_MAX || *(int16_t*)param_value<INT8_MIN)
              {
                //    wialon_bin_i16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i16_t* i16_data=(wialon_bin_i16_t*)&ctx->tx_mem[subrecord_offset];
                *i16_data=*(int16_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_i16_t);
              }
              else
              {
                //    wialon_bin_i8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i8_t* i8_data=(wialon_bin_i8_t*)&ctx->tx_mem[subrecord_offset];
                *i8_data=(int8_t)(*(int16_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_i8_t);
              }
            }
            else if(param_type==UINT32_PARAM_TYPE)
            {
              if(/*NO_PACK_DATA_TO_LOWER_DATA_TYPE ||*/ *(uint32_t*)param_value>UINT16_MAX)
              {
                //    wialon_bin_u32_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u32_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u32_t* u32_data=(wialon_bin_u32_t*)&ctx->tx_mem[subrecord_offset];
                *u32_data=*(uint32_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_u32_t);
              }
              else if(*(uint32_t*)param_value>UINT8_MAX)
              {
                //    wialon_bin_u16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u16_t* u16_data=(wialon_bin_u16_t*)&ctx->tx_mem[subrecord_offset];
                *u16_data=(uint16_t)(*(uint32_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_u16_t);
              }
              else
              {
                //    wialon_bin_u8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u8_t* u8_data=(wialon_bin_u8_t*)&ctx->tx_mem[subrecord_offset];
                *u8_data=(uint8_t)(*(uint32_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_u8_t);
              }
            }
            else if(param_type==INT32_PARAM_TYPE)
            {
              if(NO_PACK_DATA_TO_LOWER_DATA_TYPE || *(int32_t*)param_value>INT16_MAX || *(int32_t*)param_value<INT16_MIN)
              {
                //    wialon_bin_i32_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i32_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i32_t* i32_data=(wialon_bin_i32_t*)&ctx->tx_mem[subrecord_offset];
                *i32_data=*(int32_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_i32_t);
              }
              else if(*(int32_t*)param_value>INT8_MAX || *(int32_t*)param_value<INT8_MIN)
              {
                //    wialon_bin_i16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i16_t* i16_data=(wialon_bin_i16_t*)&ctx->tx_mem[subrecord_offset];
                *i16_data=(int16_t)(*(int32_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_i16_t);
              }
              else
              {
                //    wialon_bin_i8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i8_t* i8_data=(wialon_bin_i8_t*)&ctx->tx_mem[subrecord_offset];
                *i8_data=(int8_t)(*(int32_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_i8_t);
              }
            }
            else if(param_type==UINT64_PARAM_TYPE)
            {
              if(/*NO_PACK_DATA_TO_LOWER_DATA_TYPE ||*/ *(__packed uint64_t*)param_value>UINT32_MAX)
              {
                //    wialon_bin_u64_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u64_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U64_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U64_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u64_t* u64_data=(wialon_bin_u64_t*)&ctx->tx_mem[subrecord_offset];
                *u64_data=*(__packed uint64_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_u64_t);
              }
              else if(*(__packed uint64_t*)param_value>UINT16_MAX)
              {
                //    wialon_bin_u32_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u32_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u32_t* u32_data=(wialon_bin_u32_t*)&ctx->tx_mem[subrecord_offset];
                *u32_data=(uint32_t)*(__packed uint64_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_u32_t);
              }
              else if(*(__packed uint64_t*)param_value>UINT8_MAX)
              {
                //    wialon_bin_u16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u16_t* u16_data=(wialon_bin_u16_t*)&ctx->tx_mem[subrecord_offset];
                *u16_data=(uint16_t)(*(__packed uint64_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_u16_t);
              }
              else
              {
                //    wialon_bin_u8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_u8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_U8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_u8_t* u8_data=(wialon_bin_u8_t*)&ctx->tx_mem[subrecord_offset];
                *u8_data=(uint8_t)(*(__packed uint64_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_u8_t);
              }
            }
            else if(param_type==INT64_PARAM_TYPE)
            {
              if(NO_PACK_DATA_TO_LOWER_DATA_TYPE || *(__packed int64_t*)param_value>INT32_MAX || *(__packed int64_t*)param_value<INT32_MIN)
              {
                //    wialon_bin_i64_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i64_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I64_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I64_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i64_t* i64_data=(wialon_bin_i64_t*)&ctx->tx_mem[subrecord_offset];
                *i64_data=*(__packed int64_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_i64_t);
              }
              else if(*(__packed int64_t*)param_value>INT16_MAX || *(__packed int64_t*)param_value<INT16_MIN)
              {
                //    wialon_bin_i32_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i32_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I32_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i32_t* i32_data=(wialon_bin_i32_t*)&ctx->tx_mem[subrecord_offset];
                *i32_data=(int32_t)*(__packed int64_t*)param_value;
                subrecord_offset+=sizeof(wialon_bin_i32_t);
              }
              else if(*(__packed int64_t*)param_value>INT8_MAX || *(__packed int64_t*)param_value<INT8_MIN)
              {
                //    wialon_bin_i16_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i16_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I16_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i16_t* i16_data=(wialon_bin_i16_t*)&ctx->tx_mem[subrecord_offset];
                *i16_data=(int16_t)(*(__packed int64_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_i16_t);
              }
              else
              {
                //    wialon_bin_i8_t?
                //  wialon_bin_custom_param_subrecord_header_t  
                if(ctx->tx_mem_size < subrecord_offset +
                   (custom_subrecord_size+sizeof(wialon_bin_i8_t)) +
                     sizeof(wialon_bin_crc_t)) 
                {
                  pack_err=1;
                  break;
                }
                
                if(param_id>127)
                {
                  wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id|0x8000;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
                }
                else
                {
                  wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                  custom_subrecord->number=param_id;
                  custom_subrecord->type=WIALON_BIN_I8_TYPE;
                  subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
                }
                
                wialon_bin_i8_t* i8_data=(wialon_bin_i8_t*)&ctx->tx_mem[subrecord_offset];
                *i8_data=(int8_t)(*(__packed int64_t*)param_value);
                subrecord_offset+=sizeof(wialon_bin_i8_t);
              }
            }
            else if(param_type==FLOAT_PARAM_TYPE) 
            {
              /*
              // HDOP
              if(param_id==VEGA_HDOP)
              {
                pos_subrecord->hdop=(uint16_t)(*(float*)param_value*100.0f);
              }
              */
              
              //    wialon_bin_float_t?
              //  wialon_bin_custom_param_subrecord_header_t  
              if(ctx->tx_mem_size < subrecord_offset +
                 (custom_subrecord_size+sizeof(wialon_bin_float_t)) +
                   sizeof(wialon_bin_crc_t)) 
              {
                pack_err=1;
                break;
              }
              
              if(param_id>127)
              {
                wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id|0x8000;
                custom_subrecord->type=WIALON_BIN_FLOAT_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
              }
              else
              {
                wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id;
                custom_subrecord->type=WIALON_BIN_FLOAT_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
              }
              
              wialon_bin_float_t* float_data=(wialon_bin_float_t*)&ctx->tx_mem[subrecord_offset];
              *float_data=*(float*)param_value;
              subrecord_offset+=sizeof(wialon_bin_float_t);
            }
            else if(param_type==DOUBLE_PARAM_TYPE) 
            {
              //    wialon_bin_double_t?
              //  wialon_bin_custom_param_subrecord_header_t  
              if(ctx->tx_mem_size < subrecord_offset +
                 (custom_subrecord_size+sizeof(wialon_bin_double_t)) +
                   sizeof(wialon_bin_crc_t)) 
              {
                pack_err=1;
                break;
              }
              
              if(param_id>127)
              {
                wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id|0x8000;
                custom_subrecord->type=WIALON_BIN_DOUBLE_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
              }
              else
              {
                wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id;
                custom_subrecord->type=WIALON_BIN_DOUBLE_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
              }
              
              wialon_bin_double_t* double_data=(wialon_bin_double_t*)&ctx->tx_mem[subrecord_offset];
              *double_data=*(__packed double*)param_value;
              subrecord_offset+=sizeof(wialon_bin_double_t);
            }
            else//STRING_PARAM_TYPE
            {
              uint8_t slen=strnlen((char*)param_value, WIALON_BIN_STRING_TYPE_MAX_SIZE-1);//  WIALON_BIN_STRING_TYPE_MAX_LEN   '\0'
              
              //      slen?
              //  wialon_bin_custom_param_subrecord_header_t  
              if(ctx->tx_mem_size < subrecord_offset +
                 (custom_subrecord_size+(slen+1)) +
                   sizeof(wialon_bin_crc_t)) 
              {
                pack_err=1;
                break;
              }
              
              if(param_id>127)
              {
                wialon_bin_custom_param_subrecord_ext_t* custom_subrecord=(wialon_bin_custom_param_subrecord_ext_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id|0x8000;
                custom_subrecord->type=WIALON_BIN_STRING_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_ext_t);
              }
              else
              {
                wialon_bin_custom_param_subrecord_t* custom_subrecord=(wialon_bin_custom_param_subrecord_t*)&ctx->tx_mem[subrecord_offset];
                custom_subrecord->number=param_id;
                custom_subrecord->type=WIALON_BIN_STRING_TYPE;
                subrecord_offset+=sizeof(wialon_bin_custom_param_subrecord_t);
              }
              
              char* string_data=(char*)&ctx->tx_mem[subrecord_offset];
              memcpy(string_data, (char*)param_value, slen);
              string_data[slen]='\0';
              subrecord_offset+=slen+1;
            }
            sub_count++;
          }
          else
          {
            //
          }
          vpos_data->parameters_len-=sizeof(param_id)+param_type_field_len+sizeof(param_len)+param_len;
        }
        
        if(sub_count && !pack_err)
        {
          custom_subrecord_header->subrecord_type=WIALON_BIN_CUSTOM_PARAM_SUBRECORD_TYPE;
          custom_subrecord_header->count=sub_count|0x8000;
          bin_data_header->count+=1;//     VEGA
          
          // ,    
          //     LBS
          if(LBS_SEPARATE_SUBRECORD && lbs.mcc && lbs.mnc && lbs.lac && lbs.cell_id)
          {
            if(ctx->tx_mem_size < subrecord_offset+\
              sizeof(wialon_bin_lbs_subrecord_t)+sizeof(wialon_bin_crc_t)) 
            {
              pack_err=1; 
              break;
            }
            
            wialon_bin_lbs_subrecord_t* lbs_subrecord=(wialon_bin_lbs_subrecord_t*)&ctx->tx_mem[subrecord_offset];
            
            memset((void*)lbs_subrecord, 0, sizeof(wialon_bin_lbs_subrecord_t));
            lbs_subrecord->subrecord_type=WIALON_BIN_LBS_SUBRECORD_TYPE;
            lbs_subrecord->count=1;
            
            lbs_subrecord->mcc=lbs.mcc;
            lbs_subrecord->mnc=lbs.mnc;
            lbs_subrecord->lac=lbs.lac;
            lbs_subrecord->cell_id=lbs.cell_id;
            lbs_subrecord->rx_level=lbs.rx_level;
            lbs_subrecord->ta=0;
            
            bin_data_header->count++;
            subrecord_offset+=sizeof(wialon_bin_lbs_subrecord_t);
          }
        }
        
        if(pack_err)
        {
          break;
        }
      }
      else
      {
        //   
        pack_err=2;
      }
    }
  }
  ctx->common_const->release_mess(NULL);
  
  
  //  
  if(!bb_mess_out_cnt)
  {
    //          
    //if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s can't creat EgtsPos\n", get_pheader(ctx->conn_ctx_id));
    
    if(pack_err==1)
    {
      if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s tx_mem_size is too small for packing single bbox mess, deleting it...\n", get_pheader(ctx->conn_ctx_id));
    }
    else
    { 
      if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s invalid bbox mess, deleting it...\n", get_pheader(ctx->conn_ctx_id));
    }
    
    ctx->common_const->grab_mess(NULL);
    if(0>ctx->common_const->delete_messages(NULL, ctx->bbox_receiver,  1))
    {//      
      ctx->common_const->release_mess(NULL);
      return PROTOCOL_CTX_BBOX_ERR;
    }
    ctx->common_const->release_mess(NULL);
    
    if(pack_err==1) return 0;//PROTOCOL_CTX_MEM_ERR;
    else            return 0;
  }
  //  ,     
  else if(pack_err)
  {
    // subrecord_offset
    subrecord_offset=backup_subrecord_offset;
  }
  
  if(pack_err==2)
  {
    if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s detect invalid bbox mess, delete it later\n", get_pheader(ctx->conn_ctx_id));
  }
  
  
  wialon_bin_data_t* header=(wialon_bin_data_t*)&ctx->tx_mem[0];
  header->head=WIALON_BIN_TX_HEAD;
  header->type=DATA_WIALON_BIN_PACKET_TYPE;
  header->seq=ctx->tx_seq;
  header->len=subrecord_offset-WIALON_BIN_TX_DATA_OFFSET();
  
  wialon_bin_crc_t* crc=(wialon_bin_crc_t*)&ctx->tx_mem[WIALON_BIN_TX_CRC_OFFSET(header->len)];
  wialon_bin_crc16_init(crc);  
  wialon_bin_crc16(ctx->tx_mem, WIALON_BIN_TX_CRC_OFFSET(header->len), crc);
  
  if(WIALON_BIN_TX_PACKET_SIZE(header->len)>ctx->tx_mem_size)
  {
    if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) LOG("%s fatal err, we out of tx_mem_size (len=%hu)\n", get_pheader(ctx->conn_ctx_id), WIALON_BIN_TX_PACKET_SIZE(header->len));
  }
  
  if(0>ctx->common_const->socket_write(ctx->conn_ctx_id, &ctx->tx_mem[0], WIALON_BIN_TX_PACKET_SIZE(header->len))) return PROTOCOL_CTX_CONN_ERR;
  
  ctx->bbox.waiting_rx_seq=ctx->tx_seq;
  ctx->tx_seq++;
  ctx->bbox.is_wait_ack=1;
  ctx->bbox.end_wait_ack_time=WIALON_BIN_BBOX_WAIT_ACK_TIMEOUT_MS-1+xTaskGetTickCount();
  ctx->bbox.mess_cnt_out=bb_mess_out_cnt;
  
  //if(WIALON_BIN_DEBUG_LEVEL>=HDEBUG_L) __BIN_BUFF_PRINTF(&ctx->tx_mem[0], WIALON_BIN_TX_PACKET_SIZE(header->len), "%s sent bbox packet, pack_len=%hu bytes\n", get_pheader(ctx->conn_ctx_id), WIALON_BIN_TX_PACKET_SIZE(header->len));
  if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s sent bbox packet, pack_len=%hu bytes\n", get_pheader(ctx->conn_ctx_id), WIALON_BIN_TX_PACKET_SIZE(header->len));
  
  return WIALON_BIN_TX_PACKET_SIZE(header->len);
}

int16_t WialonBinSendPing(wialon_bin_ctx_t* ctx)
{
  wialon_keepalive_t* keepalive=(wialon_keepalive_t*)&ctx->tx_mem[0];
  
  keepalive->head=WIALON_BIN_TX_HEAD;
  keepalive->type=KEEP_ALIVE_WIALON_BIN_PACKET_TYPE;
  keepalive->seq=ctx->tx_seq;
  ctx->wait_ping_ack_seq=ctx->tx_seq;
  ctx->tx_seq++;
  
  if(0>ctx->common_const->socket_write(ctx->conn_ctx_id, &ctx->tx_mem[0], sizeof(wialon_keepalive_t))) return PROTOCOL_CTX_CONN_ERR;
  
  if(WIALON_BIN_DEBUG_LEVEL>=LDEBUG_L) __PRINTF("%s sent ping packet, seq=%hu\n", get_pheader(ctx->conn_ctx_id), keepalive->seq);
  
  return sizeof(wialon_keepalive_t);
}

int16_t WialonBinFileUploadHandle(wialon_bin_ctx_t* ctx)
{
  return 0;
}


#if defined(USE_SERVER_NOTIFY_IN_WCOMBINE)
int16_t WialonBinNotifiesUploadHandle(wialon_bin_ctx_t* ctx)
{
    uint32_t notify_uid;
    const char* notify_mess;
    
    if(get_server_notify(ctx->conn_ctx_id, &notify_uid, NULL, NULL, &notify_mess))
    {
      if(notify_mess[0]!='\0')
      {
        int16_t res=WialonBinSendDriverMess(ctx, "/%lX %s\n", notify_uid, notify_mess);
        
        if(res<0)  return res;
      }

      delete_server_notify(ctx->conn_ctx_id);
    }

   return 0;
}
#endif //USE_SERVER_NOTIFY_IN_WCOMBINE