/**
******************************************************************************
* File Name          : time_utilities.c
* Description        :    
* Source	     : https://ru.wikipedia.org/wiki/UNIX-%D0%B2%D1%80%D0%B5%D0%BC%D1%8F                     
*                      
******************************************************************************
*/
#include <stdbool.h>
/* Includes ------------------------------------------------------------------*/
#include "time_utilities/time_utilities.h"

/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
#define TBIAS_DAYS		((70 * (uint32_t)365) + 17)
#define TBIAS_SECS		(TBIAS_DAYS * (uint32_t)86400)
#define	TBIAS_YEAR		1900
#define __MONTAB(year)		((((year) & 03) || ((year) == 0)) ? mos : lmos)
#define	__DAY_TO_32(year, mon)	(((year - 1) / 4) + __MONTAB(year)[mon])

/* Private variables ---------------------------------------------------------*/
static const int16_t	lmos[] = {0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335};
static const int16_t	mos[] = {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/
/* 
convert time structure to scalar time 
*/
uint32_t conv_time_to_unixtime(const time_struct_t *time)
{	
  int32_t days;
  uint32_t secs;
  int32_t mon, year;
  
  /* Calculate number of days. */
  mon = time->mon - 1;
  year = time->year - TBIAS_YEAR;
  days  = __DAY_TO_32(year, mon) - 1;
  days += 365 * year;
  days += time->day;
  days -= TBIAS_DAYS;
  
  /* Calculate number of seconds. */
  secs  = 3600 * time->hour;
  secs += 60 * time->min;
  secs += time->sec;
  
  secs += (days * (uint32_t)86400);
  
  return (secs);//unix_time
}

/* 
convert scalar time to time structure 
*/
void conv_unixtime_to_time(time_struct_t *time, uint32_t unix_time)
{
  uint32_t secs;
  int32_t days;
  int32_t mon;
  int32_t year;
  int32_t i;
  const int16_t* pm;
  
  
  secs = unix_time;
  days = TBIAS_DAYS;
  
  /* days, hour, min, sec */
  days += secs / 86400;		        secs = secs % 86400;
  time->hour = secs / 3600;	        secs %= 3600;
  time->min = secs / 60;		time->sec = secs % 60;
  
  /* determine year */
  for (year = days / 365; days < (i = __DAY_TO_32(year, 0) + 365*year); ) { --year; }
  days -= i;
  time->year = year + TBIAS_YEAR;
  
  /* determine month */
  pm = __MONTAB(year);
  for (mon = 12; days < pm[--mon]; );
  time->mon = mon + 1;
  time->day = days - pm[mon] + 1;
}

/* 
return egts time
*/
uint32_t conv_unixtime_to_egtstime(uint32_t unix_time)
{
  if(unix_time<EGTS_UNIX_TIME_OFFSET) return 0;
  else return unix_time-EGTS_UNIX_TIME_OFFSET;
}

/* 
return true if time_now > time_alarm
*/
bool timeAfter(uint32_t time_now, uint32_t time_alarm) {
  bool res = false;
  uint32_t val = (time_alarm - time_now);
  if (val & 0x80000000UL) {
    res = true;
  }
  return res;
}