/**
******************************************************************************
* File Name          : debug_uart.c
* Description        : debug uart for stm32f10x
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "debug_port.h"

/* Private variables ---------------------------------------------------------*/
#if defined(USE_FREERTOS)
SemaphoreHandle_t DebugUartMutex=NULL;
#endif //USE_FREERTOS

#if defined(USE_USB_DEBUG_PORT)
static uint8_t u_buff[64];
static uint8_t u_pos=0;
#endif //USE_USB_DEBUG_PORT

#if (defined(USE_USB_DEBUG_PORT) && defined(USE_UART_DEBUG_PORT))  
//#error USB_DEBUG and UART_DEBUG can't be used both
#endif 

#if (defined(USE_USB_DEBUG_PORT) && (!defined(USE_FREERTOS)))  
#error USB_DEBUG can't be used witout FreeRtos
#endif 

#if (defined(USE_USB_DEBUG_PORT) && defined(BOOTLOADER))  
#error USB_DEBUG can't be used in bootloader
#endif 

/* Private functions ---------------------------------------------------------*/
void DebugUartInit(void)
{
#if (defined(USE_USB_DEBUG_PORT) || defined(USE_UART_DEBUG_PORT))  
  #if defined(USE_FREERTOS)
    if(DebugUartMutex==NULL) DebugUartMutex=xSemaphoreCreateMutex();
  #endif //USE_FREERTOS
#endif //USE_USB_DEBUG_PORT || USE_UART_DEBUG_PORT
  
#if (defined(USE_USB_DEBUG_PORT))
  USB_CDC_Init();
#endif //USE_USB_DEBUG_PORT
  
#if defined(USE_UART_DEBUG_PORT)
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  
  /* Enable USART clock */
  __DEBUG_UART_CLOCK_ENABLE();
  
  /* Remap_Tx if macro defined*/
  __DEBUG_UART_REMAP_ENABLE();
  
  /* Configure USART Tx and as alternate function push-pull */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = DEBUG_UART_TX_PIN;
  GPIO_Init(DEBUG_UART_TX_PORT, &GPIO_InitStructure);
  
  USART_InitStructure.USART_BaudRate = DEBUG_UART_BAUDRATE;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode =  USART_Mode_Tx;
  USART_InitStructure.USART_Parity = USART_Parity_No;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_Init(DEBUG_UART, &USART_InitStructure);
  
  USART_Cmd(DEBUG_UART, ENABLE);
#endif //USE_UART_DEBUG_PORT
}


void DebugUartDeInit(void)
{
#if defined(USE_USB_DEBUG_PORT)
  USB_CDC_Deinit();
#endif //USE_USB_DEBUG_PORT

#if defined(USE_UART_DEBUG_PORT)
  GPIO_InitTypeDef GPIO_InitStructure;
  
  USART_Cmd(DEBUG_UART, DISABLE);
  
  __DEBUG_UART_CLOCK_DISABLE();
  
  /* Configure UART pins as analog */
  GPIO_InitStructure.GPIO_Pin = DEBUG_UART_TX_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_Init(DEBUG_UART_TX_PORT, &GPIO_InitStructure);
#endif //USE_UART_DEBUG_PORT
}

void pflush(void)
{
  #if defined(USE_USB_DEBUG_PORT)
  if(u_pos!=0)
  {
    USB_CDC_Send(u_buff, u_pos, 2);
    u_pos=0;
  }
  #endif //USE_USB_DEBUG_PORT
  
  #if defined(USE_UART_DEBUG_PORT)
  //none
  #endif //USE_UART_DEBUG_PORT
}

#if _DLIB_FILE_DESCRIPTOR > 0
int fputc(int ch, FILE *f)
#else
int putchar(int ch)
#endif //_DLIB_FILE_DESCRIPTOR > 0
{
#if defined(USE_USB_DEBUG_PORT)
  if(u_pos<sizeof(u_buff))
  {
    u_buff[u_pos]=(uint8_t)ch;
    u_pos++;
  }
  if(u_pos>=sizeof(u_buff))
  {
    pflush();
  }
#endif //USE_USB_DEBUG_PORT
  
#if (defined(USE_UART_DEBUG_PORT))
  USART_SendData(DEBUG_UART, (uint8_t)ch);
  while (USART_GetFlagStatus(DEBUG_UART, USART_FLAG_TC) == RESET);
#endif //USE_UART_DEBUG_PORT
  return ch;
}