/**
******************************************************************************
* File Name          : debug_uart.h
* Description        : debug uart for stm32f10x
*                      
*                      
******************************************************************************
*/
#ifndef __DEBUG_UART_H
#define __DEBUG_UART_H

#ifdef __cplusplus
extern "C" {
#endif 

  /* Includes ----------------------------------------------------------------*/
#include "stm32f2xx.h"
#include <stdio.h>
  
#if (!defined(BOOTLOADER))
  #define USE_UART_DEBUG_PORT
  //#define USE_USB_DEBUG_PORT
#else
  #define USE_UART_DEBUG_PORT
#endif //!BOOTLOADER
  
#if defined(USE_FREERTOS)  
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h" 
#endif //USE_FREERTOS
  
#if defined(USE_UART_DEBUG_PORT)
#include "stm32f2xx_usart.h"
#endif //USE_UART_DEBUG_PORT
  
#if defined(USE_USB_DEBUG_PORT)
#include "USB_CDC_STM32/usbd_cdc_interface.h"
#endif //USE_USB_DEBUG_PORT
  
#if defined(USE_UART_DEBUG_PORT)
  /* Private define ----------------------------------------------------------*/
#define DEBUG_UART                        UART4
#define DEBUG_UART_BAUDRATE               921600
#define DEBUG_UART_TX_PIN                 GPIO_Pin_10
#define DEBUG_UART_TX_PORT                GPIOC
//#define DEBUG_UART_RX_PIN                GPIO_Pin_11
//#define DEBUG_UART_RX_PORT               GPIOC
  
//#define DEBUG_UART_IRQn                 UART5_IRQn
//#define DEBUG_UART_IRQHandler           UART5_IRQHandler
//#define DEBUG_UART_IRQn_PRIO            11 //highest priority with use FreeRTOS API
  
  /* Private macro -------------------------------------------------------------*/
#define __DEBUG_UART_CLOCK_ENABLE()      (RCC_APB1PeriphClockCmd(RCC_APB1Periph_UART4, ENABLE))
#define __DEBUG_UART_CLOCK_DISABLE()     (RCC_APB1PeriphClockCmd(RCC_APB1Periph_UART4, DISABLE))  
  
#define __DEBUG_UART_REMAP_ENABLE()      (GPIO_PinAFConfig(DEBUG_UART_TX_PORT, GPIO_PinSource10, GPIO_AF_UART4))
#endif //USE_UART_DEBUG_PORT
  
#if defined(USE_FREERTOS)  
#if defined(USE_UART_DEBUG_PORT) || defined(USE_USB_DEBUG_PORT)
#define __DEBUG_UART_GRAB()              (xSemaphoreTake(DebugUartMutex, portMAX_DELAY))
#define __DEBUG_UART_RELEASE()           (xSemaphoreGive(DebugUartMutex))
#else
#define __DEBUG_UART_GRAB()              //none
#define __DEBUG_UART_RELEASE()           //none
#endif
#else
#define __DEBUG_UART_GRAB()              //none
#define __DEBUG_UART_RELEASE()           //none
#endif //USE_FREERTOS
  
#define __PRINTF(...)                           \
{                                               \
  __DEBUG_UART_GRAB();                          \
  printf(__VA_ARGS__);                          \
  pflush();                                     \
  __DEBUG_UART_RELEASE();                       \
}    

#define __BIN_BUFF_PRINTF(bin, len, ...)        \
{                                               \
  __DEBUG_UART_GRAB();                          \
  printf(__VA_ARGS__);                          \
  for(uint16_t __idx=0; __idx<len; __idx++)     \
  {                                             \
    printf("%02hhX", *(bin+__idx));             \
  }                                             \
  printf("\n");                                 \
  pflush();                                     \
  __DEBUG_UART_RELEASE();                       \
}

#define __CHAR_BUFF_PRINTF(ch, len, ...)       \
{                                              \
  __DEBUG_UART_GRAB();                         \
  printf(__VA_ARGS__);                         \
  uint16_t __idx;                              \
  for(__idx=0; __idx<len; __idx++)             \
  {                                            \
    printf("%c", *(ch+__idx));                 \
  }                                            \
  if(*(ch+__idx)!='\n')                        \
  {                                            \
    printf("\n");                              \
  }                                            \
  pflush();                                    \
  __DEBUG_UART_RELEASE();                      \
}

/* Exported functions ----------------------------------------------------- */   
void DebugUartInit(void);
void DebugUartDeInit(void);
void pflush(void);
/* External variables --------------------------------------------------------*/ 
#if defined(USE_FREERTOS)  
extern SemaphoreHandle_t DebugUartMutex;
#endif //USE_FREERTOS

#ifdef __cplusplus
}
#endif 

#endif /* __DEBUG_UART_H */