/**
******************************************************************************
* File Name          :
* Description        :
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "stm32f2xx.h"
#include "gps_uart.h"

static gps_uart_driver_t gps_uart={0};

void gps_uart_init(uint32_t baud_rate, const char* rx_sync_string)
{
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  cbuff_init(&gps_uart.rx_cbuff, gps_uart.rx_mem, sizeof(gps_uart.rx_mem));
  gps_uart.rx_ore_flag=0;
  gps_uart.rx_full_flag=0;
  gps_uart.max_filled_size=0;
  gps_uart.rx_sync_string=rx_sync_string;
  gps_uart.rx_sync_string_len=strlen(rx_sync_string);
  
  /* Create UART sync objects */
  if(gps_uart.tx_sem==NULL) {vSemaphoreCreateBinary(gps_uart.tx_sem);}
  xSemaphoreTake(gps_uart.tx_sem, 0); 
  
  if(gps_uart.rx_sem==NULL) {vSemaphoreCreateBinary(gps_uart.rx_sem);}
  xSemaphoreTake(gps_uart.rx_sem, 0);
  
  /* Enable AFIO clock */
  __GPS_AFIO_CLOCK_ENABLE();
  
  /* UART clock enable */
  __GPS_UART_CLOCK_ENABLE();
  
  /* UART remap if enabled */
  __GPS_UART_REMAP(); 
    
  /* Configure UART pins  */ 
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = GPS_UART_TX_PIN;
  GPIO_Init(GPS_UART_TX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;
  GPIO_InitStructure.GPIO_Pin = GPS_UART_RX_PIN;
  GPIO_Init(GPS_UART_RX_PORT, &GPIO_InitStructure);
      
  USART_InitStructure.USART_BaudRate = baud_rate;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No ;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(GPS_UART, &USART_InitStructure);
    
  NVIC_InitStructure.NVIC_IRQChannel = GPS_UART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = GPS_UART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
 
  USART_ClearITPendingBit(GPS_UART, USART_IT_RXNE);
  USART_ClearITPendingBit(GPS_UART, USART_IT_TC);
  USART_ITConfig(GPS_UART, USART_IT_RXNE, ENABLE);
  USART_Cmd(GPS_UART, ENABLE);
 
  gps_uart.is_init=1;
}

void gps_uart_deinit(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;
  
  gps_uart.is_init=0;
  
  /* Disable interrupts */
  USART_ITConfig(GPS_UART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(GPS_UART, USART_IT_TC, DISABLE);
  
  /* Disable the UART */
  USART_Cmd(GPS_UART, DISABLE);
  
  /* UART clock disable */
  __GPS_UART_CLOCK_DISABLE();
  
  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = GPS_UART_RX_PIN;
  GPIO_Init(GPS_UART_RX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = GPS_UART_TX_PIN;
  GPIO_Init(GPS_UART_TX_PORT, &GPIO_InitStructure);
}


void gps_uart_write(uint8_t *src, uint16_t len)
{
  if(!len) return;
    
  gps_uart.tx_data=src;
  gps_uart.tx_data_end=src+len-1;
  
  USART_SendData(GPS_UART, *gps_uart.tx_data);
  
  USART_ITConfig(GPS_UART, USART_IT_TC, ENABLE);
  xSemaphoreTake(gps_uart.tx_sem, portMAX_DELAY);
}


int16_t gps_uart_read(uint8_t *dst, uint16_t len)
{
  uint16_t read_len;
  
  if(gps_uart.rx_ore_flag || gps_uart.rx_full_flag)
  {
    int16_t err;
    if(gps_uart.rx_full_flag) err=-1;
    else                      err=-2;
    gps_uart_flush_rx();
    return err;
  }
  
  read_len=filled_cbuff_len(&gps_uart.rx_cbuff);
  
  if(!read_len) return 0;
  
  if(read_len>len) read_len=len;
  
  read_from_cbuff(&gps_uart.rx_cbuff, dst, read_len);
  return (int16_t)read_len;
}

int16_t gps_uart_avl_len(void)
{
  return filled_cbuff_len(&gps_uart.rx_cbuff);
}

uint8_t gps_uart_wait_sync(uint32_t time_mS)
{
  if(xSemaphoreTake(gps_uart.rx_sem, time_mS))
    return 1;
  else 
    return 0;
}

void gps_uart_flush_rx(void)
{
  USART_ITConfig(GPS_UART, USART_IT_RXNE, DISABLE);
  clear_cbuff_from_reader(&gps_uart.rx_cbuff);
  gps_uart.rx_ore_flag=0;
  gps_uart.rx_full_flag=0;
  xSemaphoreTake(gps_uart.rx_sem, 0);
  USART_ITConfig(GPS_UART, USART_IT_RXNE, ENABLE);
}

#ifdef __cplusplus
extern "C" {
#endif
  #pragma optimize=speed
  void GPS_UART_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    
    if(USART_GetITStatus(GPS_UART, USART_IT_TC) != RESET)
    {
      if(gps_uart.tx_data>=gps_uart.tx_data_end)
      {//  
        USART_ITConfig(GPS_UART, USART_IT_TC, DISABLE);
        xSemaphoreGiveFromISR(gps_uart.tx_sem, &xHigherPriorityTaskWoken);
      }
      else
      {
        gps_uart.tx_data++;
        USART_SendData(GPS_UART, *gps_uart.tx_data);
      }
    }
    else if(USART_GetITStatus(GPS_UART, USART_IT_RXNE) != RESET)
    {
      uint8_t rx_val;
      rx_val=USART_ReceiveData(GPS_UART);
      
      if(free_cbuff_len(&gps_uart.rx_cbuff)>0)//    
      {
        uint16_t filled_len;
        write_byte_to_cbuff(&gps_uart.rx_cbuff, rx_val);
        filled_len=filled_cbuff_len(&gps_uart.rx_cbuff);
        
        if(gps_uart.rx_sync_string==NULL || is_cbuf_ends_with_ref_buff(&gps_uart.rx_cbuff, (uint8_t*)gps_uart.rx_sync_string, gps_uart.rx_sync_string_len))
        {
          xSemaphoreGiveFromISR(gps_uart.rx_sem, &xHigherPriorityTaskWoken);
        }
        
        if(gps_uart.max_filled_size<filled_len) gps_uart.max_filled_size=filled_len;
       }
       else// 
       {
         gps_uart.rx_full_flag=1;
         xSemaphoreGiveFromISR(gps_uart.rx_sem, &xHigherPriorityTaskWoken);
       }
    }
    else if(USART_GetITStatus(GPS_UART, USART_IT_ORE) != RESET ||  USART_GetFlagStatus(GPS_UART, USART_FLAG_ORE) != RESET) // 
    { 
      USART_ReceiveData(GPS_UART); 
      gps_uart.rx_ore_flag=1;
    }
    
    if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
  }
#ifdef __cplusplus
}
#endif
