/**
******************************************************************************
* File Name          : modem_uart.c
* Description        : modem uart for stm32l1xx
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "modem_uart.h"

#if USE_MODEM_UART_DMA_RX > 0
#error this driver not support uart dma rx
#endif

#if USE_MODEM_UART_DMA_TX > 0
#error this driver not support uart dma tx
#endif

#pragma data_alignment=4
static uint8_t modem_rx_mem[MODEM_RX_MEM_SIZE];
static modem_uart_rx_t modem_uart_rx={0};
static SemaphoreHandle_t modem_tx_bin_sem=NULL;
static SemaphoreHandle_t modem_rx_bin_sem=NULL;
static uint8_t* modem_tx_data;
static uint8_t* modem_tx_data_end;
static uint32_t time_from_last_recv; 

#if EN_MODEM_DEBUG > 0
#warning modem debug is on
__IO uint8_t modem_tx_log_out_en=1;
__IO uint8_t modem_rx_log_out_en=1;
#endif

void ModemUartInit(void)
{
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  cbuff_init(&modem_uart_rx.cbuff, modem_rx_mem, sizeof(modem_rx_mem));
  modem_uart_rx.sync_obj=&modem_rx_bin_sem;
  modem_uart_rx.ore_flag=0;
  modem_uart_rx.full_flag=0;
  
  /* Create UART sync objects */
  if(modem_tx_bin_sem==NULL) {vSemaphoreCreateBinary(modem_tx_bin_sem);}
  xSemaphoreTake(modem_tx_bin_sem, 0); 
  if(modem_rx_bin_sem==NULL)  {vSemaphoreCreateBinary(modem_rx_bin_sem);}
  xSemaphoreTake(modem_rx_bin_sem, 0); 
    
  /* UART clock enable */
  __MODEM_UART_CLOCK_ENABLE();
  
  /* UART remap if enabled */
  __MODEM_UART_REMAP(); 
    
  /* Configure UART pins  */
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_OType = MODEM_UART_TX_PIN_MODE;
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_TX_PIN;
  GPIO_Init(MODEM_UART_TX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_RX_PIN;
  GPIO_Init(MODEM_UART_RX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_CTS_PIN;
  GPIO_Init(MODEM_UART_CTS_PORT, &GPIO_InitStructure);
    
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_OType = MODEM_UART_SOFT_RTS_PIN_MODE;
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_SOFT_RTS_PIN;
  GPIO_Init(MODEM_UART_SOFT_RTS_PORT, &GPIO_InitStructure);
  __MODEM_UART_SOFT_RTS_PIN_HI();
    
  USART_InitStructure.USART_BaudRate = MODEM_UART_BAUD_RATE;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No ;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_CTS;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(MODEM_UART, &USART_InitStructure);
    
  NVIC_InitStructure.NVIC_IRQChannel = MODEM_UART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MODEM_UART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
 
  USART_ClearITPendingBit(MODEM_UART, USART_IT_RXNE);
  USART_ClearITPendingBit(MODEM_UART, USART_IT_TXE);
  USART_ITConfig(MODEM_UART, USART_IT_RXNE, ENABLE);
  USART_Cmd(MODEM_UART, ENABLE);
 __MODEM_UART_SOFT_RTS_PIN_LOW();
 
  modem_uart_rx.is_init=1;
}


void ModemUartDeInit(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;
  
  modem_uart_rx.is_init=0;
  
  /* Disable interrupts */
  USART_ITConfig(MODEM_UART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
  
  /* Disable the UART */
  USART_Cmd(MODEM_UART, DISABLE);
  
  /* UART clock disable */
  __MODEM_UART_CLOCK_DISABLE();
  
  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_TX_PIN;
  GPIO_Init(MODEM_UART_TX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_RX_PIN;
  GPIO_Init(MODEM_UART_RX_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_CTS_PIN;
  GPIO_Init(MODEM_UART_CTS_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_SOFT_RTS_PIN;
  GPIO_Init(MODEM_UART_SOFT_RTS_PORT, &GPIO_InitStructure);
}

#define SEND_TIMEOUT_AFTER_LAST_RECV       (50)
#define MAX_TIME_WAIT_AFTER_LAST_RECV      (200)
int16_t WriteToModemUart(uint8_t *out_data, uint16_t num_of_bytes, TickType_t time_wait)
{
  if(num_of_bytes==STRING_LEN) 
  {
    num_of_bytes=strlen((const char*)out_data);
  }
  if(num_of_bytes==0)  return MODEM_UART_TX_OK;
  
  ///*
  uint8_t TimeOutCount=MAX_TIME_WAIT_AFTER_LAST_RECV/SEND_TIMEOUT_AFTER_LAST_RECV;
  uint32_t CurrTickCount;
  uint32_t AlarmTickCount;
  while(TimeOutCount--)
  {
    CurrTickCount = xTaskGetTickCount();
    AlarmTickCount = time_from_last_recv+SEND_TIMEOUT_AFTER_LAST_RECV;
    
    if(timeAfter(CurrTickCount, AlarmTickCount)) break;
    if(AlarmTickCount>CurrTickCount) vTaskDelay(AlarmTickCount-CurrTickCount);
  }
  //*/
  
  TimeOutCount=0;//todo:      time_wait.   1000ms
  while (USART_GetFlagStatus(MODEM_UART, USART_FLAG_TXE) == RESET && TimeOutCount<100)
  {
    vTaskDelay(10);
    TimeOutCount++;
  }
  if(TimeOutCount>=100) return MODEM_UART_TX_TIMEOUT_ERR;
  
  xSemaphoreTake(modem_tx_bin_sem, 0);// ,       ,   modem_tx_bin_sem 

  modem_tx_data=out_data;
  modem_tx_data_end=out_data+num_of_bytes-1;
  
  USART_SendData(MODEM_UART, *modem_tx_data);
  USART_ITConfig(MODEM_UART, USART_IT_TXE, ENABLE);

  if( pdTRUE == xSemaphoreTake(modem_tx_bin_sem, time_wait) )
  {
#if EN_MODEM_DEBUG > 0
    if(modem_tx_log_out_en)
    {
      __CHAR_BUFF_PRINTF(out_data, num_of_bytes, "ModemTxLog, len: %hu\n", num_of_bytes);
    }
#endif //EN_GSM_DEBUG
    return MODEM_UART_TX_OK;
  }
  else
  {
    USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
    return MODEM_UART_TX_TIMEOUT_ERR;
  }
}

modem_uart_rx_t* GetModemUartRxStructPtr(void)
{
  return &modem_uart_rx;
}

#ifdef __cplusplus
extern "C" {
#endif
  
#pragma optimize=speed
void MODEM_UART_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    
    if(USART_GetITStatus(MODEM_UART, USART_IT_TXE) != RESET)
    {
      if(modem_tx_data>=modem_tx_data_end)
      {//  
        USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
        xSemaphoreGiveFromISR(modem_tx_bin_sem, &xHigherPriorityTaskWoken);
        if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
      }
      else
      {
        modem_tx_data++;
        USART_SendData(MODEM_UART, *modem_tx_data);
      }
    }
    
    else if(USART_GetITStatus(MODEM_UART, USART_IT_RXNE) != RESET)
    {
      uint8_t rx_val;
      rx_val=USART_ReceiveData(MODEM_UART);
      if (modem_uart_rx.full_flag>0){//   GsmRxBuf,   '\n'
        if('\n'==rx_val)//check '\n'
        {
          if(modem_uart_rx.full_flag==2)//     
          {
            clear_cbuff_from_writter(&modem_uart_rx.cbuff);
          }
        }     
      }
      else if(free_cbuff_len(&modem_uart_rx.cbuff)>0)//     
      {
        write_byte_to_cbuff(&modem_uart_rx.cbuff, rx_val);
        if(is_cbuf_ends_with_ref_buff(&modem_uart_rx.cbuff, "\r\n", 2))
        {
          xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);//       (   ,      (  ?))
        }       
        if(free_cbuff_len(&modem_uart_rx.cbuff)<=MODEM_RX_BUFF_WM)
        {
          __MODEM_UART_SOFT_RTS_PIN_HI();
          xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);
        }
      }
      else// 
      {
        modem_uart_rx.full_flag=1;
        xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);// ,      
        __MODEM_UART_SOFT_RTS_PIN_LOW();//  RTS      '\n'
      }
      
      time_from_last_recv=xTaskGetTickCountFromISR();
      
      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
    }
    
    else if(USART_GetITStatus(MODEM_UART, USART_IT_ORE) != RESET ||  USART_GetFlagStatus(MODEM_UART, USART_FLAG_ORE) != RESET) // 
    { 
      USART_ReceiveData(MODEM_UART); 
      modem_uart_rx.ore_flag=1;
    }
  }
#ifdef __cplusplus
}
#endif 
