/**
******************************************************************************
* File Name          : rs485_driver.c
* Description        : RS485 driver    
**                       ST32F10x
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include "rs485_driver.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Defines -------------------------------------------------------------------*/
/* Global variables-----------------------------------------------------------*/
/* Private constants ---------------------------------------------------------*/
static const USART_InitTypeDef USART_InitStructure_def = 
{
  .USART_BaudRate = RS485_USART_BAUD_DEFAULT,
  .USART_WordLength = USART_WordLength_8b,
  .USART_StopBits = USART_StopBits_1,
  .USART_Parity = USART_Parity_No,
  .USART_Mode = USART_Mode_Rx | USART_Mode_Tx,
  .USART_HardwareFlowControl = USART_HardwareFlowControl_None,
};

/* Private variables ---------------------------------------------------------*/
static enum { IDLE, TX, RX, END } usart_state;
static SemaphoreHandle_t txrx_end_sem=NULL;
static uint8_t *tx_buf, *rx_buf;
static uint16_t tx_cnt, rx_cnt, rx_num;
static int txrx_timewait, de_out_wait=10;
/* Public functions ----------------------------------------------------------*/

void RS485_Init(void)
{
  GPIO_InitTypeDef GPIO_InitStructure; 
  
  //- Create USART sync objects 
  if(txrx_end_sem==NULL) {vSemaphoreCreateBinary(txrx_end_sem);}
  //  xSemaphoreTake(smf_txrx_end, 0);
  
  __RS485_USART_CLOCK_ENABLE();	//  USART 
  __RS485_USART_PIN_REMAP();	// remap     .h
  
  /* Configure RS485_RE/DE as push-pull and init */
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_25MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = RS485_RE_Pin;
  GPIO_Init(RS485_RE_GPIO_Port, &GPIO_InitStructure);
  GPIO_InitStructure.GPIO_Pin = RS485_DE_Pin;
  GPIO_Init(RS485_DE_GPIO_Port, &GPIO_InitStructure);
  RS485_DE_Z();			//   Z-state
  
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_25MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  
  /* Configure USART Rx as pull-up input */
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;
  GPIO_InitStructure.GPIO_Pin =  RS485_USART_RX_PIN;
  GPIO_Init(RS485_USART_PORT, &GPIO_InitStructure);
  
  /* Configure USART Tx as alternate function push-pull */
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = RS485_USART_TX_PIN;
  GPIO_Init(RS485_USART_PORT, &GPIO_InitStructure);
  
  USART_DeInit(RS485_USART);	//     USART
  
  USART_InitTypeDef USART_InitStructure; 
  memcpy(&USART_InitStructure, &USART_InitStructure_def, sizeof(USART_InitTypeDef));
  USART_Init(RS485_USART, &USART_InitStructure);
  
  //- NVIC
  NVIC_InitTypeDef NVIC_InitStructure; 
  NVIC_InitStructure.NVIC_IRQChannel = RS485_USART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = RS485_USART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  // State
  usart_state = IDLE; //  no interrupt pending
  USART_ReceiveData(RS485_USART);
  USART_ClearITPendingBit(RS485_USART, USART_IT_RXNE);
  USART_ClearITPendingBit(RS485_USART, USART_IT_TXE);
  USART_ClearITPendingBit(RS485_USART, USART_IT_TC);
  //USART_ITConfig(RS485_USART, USART_IT_RXNE, ENABLE);  
  
  USART_Cmd(RS485_USART, ENABLE);
}

//- Set Param
void RS485_SetParam(int baudrate, int timeout, char *param_string)
{
  if(baudrate) 
  {
    USART_InitTypeDef USART_InitStructure; 
    memcpy(&USART_InitStructure,&USART_InitStructure_def,sizeof(USART_InitTypeDef));
    USART_InitStructure.USART_BaudRate = baudrate; // 
    USART_Init(RS485_USART, &USART_InitStructure);
  }
  if(timeout) txrx_timewait = timeout;
}

//- TxRx on IT
static uint16_t RS485_TxRxIT(uint8_t *sbuf, uint16_t slen, uint8_t *rbuf, uint16_t rlen)
{
  xSemaphoreTake(txrx_end_sem,0);
  usart_state = IDLE;
  if(slen) 
  {
    // Tx - mybeRX
    vTaskDelay(de_out_wait);    // wait before DE 
    RS485_DE_Out(); 
    tx_buf=sbuf; tx_cnt=slen-1; rx_buf=rbuf; rx_cnt=0; rx_num=rlen;
    USART_ClearITPendingBit(RS485_USART,USART_IT_TC);
    USART_SendData(RS485_USART, *tx_buf++);
    usart_state = TX;
    USART_ITConfig(RS485_USART, USART_IT_TC, ENABLE); // enable Transmition Complete IT
    if( xSemaphoreTake(txrx_end_sem,txrx_timewait) != pdTRUE ) 
    { 
      USART_ITConfig(RS485_USART, USART_IT_TC, DISABLE); // disable Transmition Complete IT
      usart_state = IDLE;
      return rx_cnt;
    }
    // DE!!!
    RS485_DE_In();
    USART_ITConfig(RS485_USART, USART_IT_TC, DISABLE); // disable Transmition Complete IT
    USART_ITConfig(RS485_USART,USART_IT_RXNE,DISABLE); USART_ITConfig(RS485_USART,USART_IT_TXE,DISABLE);
    usart_state = IDLE;
    return rx_cnt;
  }
  else 
  {
    // Rx only
  }
  RS485_DE_In();
  return rx_cnt;
}

uint16_t RS485_TxRx(uint8_t *sbuf, uint16_t slen, uint8_t *rbuf, uint16_t rlen)
{
  return RS485_TxRxIT(sbuf,slen,rbuf,rlen);
}

static void stop_txrxIT(void)
{
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  USART_ITConfig(RS485_USART,USART_IT_TXE,DISABLE); 
  USART_ITConfig(RS485_USART,USART_IT_TC,DISABLE);
  USART_ITConfig(RS485_USART,USART_IT_RXNE,DISABLE);
  usart_state=IDLE;
  xSemaphoreGiveFromISR(txrx_end_sem, &xHigherPriorityTaskWoken);
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
}

void RS485_DeInit(void)
{
  GPIO_InitTypeDef GPIO_InitStructure; 
  
  //- NVIC
  USART_ITConfig(RS485_USART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(RS485_USART, USART_IT_TXE, DISABLE);  
  USART_ITConfig(RS485_USART, USART_IT_TC, DISABLE);  
  
  RS485_DE_Z(); //   Z-state
  
  GPIO_InitStructure.GPIO_Pin = RS485_RE_Pin | RS485_DE_Pin;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = RS485_RE_Pin;
  GPIO_Init(RS485_RE_GPIO_Port, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = RS485_DE_Pin;
  GPIO_Init(RS485_DE_GPIO_Port, &GPIO_InitStructure);
  
  /* Configure USART Rx and Tx as analog input */
  GPIO_InitStructure.GPIO_Pin =  RS485_USART_RX_PIN | RS485_USART_TX_PIN;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_Init(RS485_USART_PORT, &GPIO_InitStructure);
  
  __RS485_USART_CLOCK_DISABLE(); //  USART
  USART_DeInit(RS485_USART);	 //     USART
}

#ifdef __cplusplus
extern "C" {
#endif 
#pragma optimize=speed
void RS485_USART_IRQHandler(void)
{
  ITStatus tmpstatus=USART_GetITStatus(RS485_USART, USART_IT_TC); 
  // Check state
  switch(usart_state) 
  {
  case TX: USART_ClearITPendingBit(RS485_USART,USART_IT_TC);
  if(tx_cnt--){USART_SendData(RS485_USART, *tx_buf++);} // send next byte
  else 
  { 
    tx_cnt=0;  RS485_DE_In(); USART_ITConfig(RS485_USART, USART_IT_TC, DISABLE); // disable Transmition Complete IT
    usart_state=IDLE;
    if(rx_num) 
    {
      RS485_DE_In();
      usart_state = RX; USART_ReceiveData(RS485_USART);
      USART_ITConfig(RS485_USART, USART_IT_RXNE, ENABLE);
    }
    else {stop_txrxIT(); return;}
  } 
  break;
  case RX: *rx_buf++=USART_ReceiveData(RS485_USART); rx_cnt++; 
  if(rx_cnt>=rx_num) stop_txrxIT();  
  break;
  default:  stop_txrxIT();
  }
}    
#ifdef __cplusplus
}
#endif 
