/**
******************************************************************************
* File Name          : rtc_driver.
* Description        :  rtc stm32f2xx
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "rtc_driver.h"
#include "time_utilities/time_utilities.h"
#include "system_config.h"

#if RTC_LOCK_EN > 0
#include "FreeRTOS.h"
#include "semphr.h"
#endif

/* Private variables ---------------------------------------------------------*/
#if RTC_LOCK_EN > 0
static SemaphoreHandle_t  RTC_mutex;
#endif

/* Masks Definition */
#define RTC_TR_RESERVED_MASK    ((uint32_t)0x007F7F7F)
#define RTC_DR_RESERVED_MASK    ((uint32_t)0x00FFFF3F) 
#define RTC_INIT_MASK           ((uint32_t)0xFFFFFFFF)  
#define RTC_RSF_MASK            ((uint32_t)0xFFFFFF5F)
#define RTC_FLAGS_MASK          ((uint32_t)(RTC_FLAG_TSOVF | RTC_FLAG_TSF | RTC_FLAG_WUTF | \
                                            RTC_FLAG_ALRBF | RTC_FLAG_ALRAF | RTC_FLAG_INITF | \
                                            RTC_FLAG_RSF | RTC_FLAG_INITS | RTC_FLAG_WUTWF | \
                                            RTC_FLAG_ALRBWF | RTC_FLAG_ALRAWF | RTC_FLAG_TAMP1F ))

/* Private function prototypes -----------------------------------------------*/

/* Private functions ---------------------------------------------------------*/
static uint8_t RTC_ByteToBcd2(uint8_t Value)
{
  uint8_t bcdhigh = 0;
  while (Value >= 10)
  {
    bcdhigh++;
    Value -= 10;
  }
  return  ((uint8_t)(bcdhigh << 4) | Value);
}

static uint8_t RTC_Bcd2ToByte(uint8_t Value)
{
  uint8_t tmp = 0;
  tmp = ((uint8_t)(Value & (uint8_t)0xF0) >> (uint8_t)0x4) * 10;
  return (tmp + (Value & (uint8_t)0x0F));
}

static ErrorStatus RTC_SetBinTimeDate(RTC_TimeTypeDef* RTC_TimeStruct, RTC_DateTypeDef* RTC_DateStruct)
{
  uint32_t tmpreg;
  ErrorStatus status = ERROR;
    
  /* Check the parameters */
  if ((RTC->CR & RTC_CR_FMT) != (uint32_t)RESET)
  {
    assert_param(IS_RTC_HOUR12(RTC_TimeStruct->RTC_Hours));
    assert_param(IS_RTC_H12(RTC_TimeStruct->RTC_H12));
  } 
  else
  {
    RTC_TimeStruct->RTC_H12 = 0x00;
    assert_param(IS_RTC_HOUR24(RTC_TimeStruct->RTC_Hours));
  }
  assert_param(IS_RTC_MINUTES(RTC_TimeStruct->RTC_Minutes));
  assert_param(IS_RTC_SECONDS(RTC_TimeStruct->RTC_Seconds));
  
  if (((RTC_DateStruct->RTC_Month & 0x10) == 0x10))
  {
    RTC_DateStruct->RTC_Month = (RTC_DateStruct->RTC_Month & (uint32_t)~(0x10)) + 0x0A;
  }
  assert_param(IS_RTC_YEAR(RTC_DateStruct->RTC_Year));
  assert_param(IS_RTC_MONTH(RTC_DateStruct->RTC_Month));
  assert_param(IS_RTC_DATE(RTC_DateStruct->RTC_Date));
  assert_param(IS_RTC_WEEKDAY(RTC_DateStruct->RTC_WeekDay));
    
  /* Disable the write protection for RTC registers */
  RTC->WPR = 0xCA;
  RTC->WPR = 0x53;

  /* Set Initialization mode */
  if (RTC_EnterInitMode() == ERROR)
  {
    status = ERROR;
  }
  else
  {
    /* Set the RTC_TR register */
    tmpreg = (uint32_t)(((uint32_t)RTC_ByteToBcd2(RTC_TimeStruct->RTC_Hours) << 16) | \
      ((uint32_t)RTC_ByteToBcd2(RTC_TimeStruct->RTC_Minutes) << 8) | \
        ((uint32_t)RTC_ByteToBcd2(RTC_TimeStruct->RTC_Seconds)) | \
          (((uint32_t)RTC_TimeStruct->RTC_H12) << 16));
    RTC->TR = (uint32_t)(tmpreg & RTC_TR_RESERVED_MASK);
    
    /* Set the RTC_DR register */
    tmpreg = (((uint32_t)RTC_ByteToBcd2(RTC_DateStruct->RTC_Year) << 16) | \
      ((uint32_t)RTC_ByteToBcd2(RTC_DateStruct->RTC_Month) << 8) | \
        ((uint32_t)RTC_ByteToBcd2(RTC_DateStruct->RTC_Date)) | \
          ((uint32_t)RTC_DateStruct->RTC_WeekDay << 13));
    RTC->DR = (uint32_t)(tmpreg & RTC_DR_RESERVED_MASK);
    
    /* Exit Initialization mode */
    RTC_ExitInitMode(); 

    if(RTC_WaitForSynchro() == ERROR)
    {
      status = ERROR;
    }
    else
    {
      status = SUCCESS;
    }
  }
  /* Enable the write protection for RTC registers */
  RTC->WPR = 0xFF; 
    
  return status;
}

static void RTC_GetBinTimeDate(RTC_TimeTypeDef* RTC_TimeStruct, RTC_DateTypeDef* RTC_DateStruct)
{
  uint32_t tmpreg;

  /* Get the RTC_TR register */
  tmpreg = (uint32_t)(RTC->TR & RTC_TR_RESERVED_MASK); 
  
  /* Fill the structure fields with the read parameters */
  RTC_TimeStruct->RTC_Hours = (uint8_t)((tmpreg & (RTC_TR_HT | RTC_TR_HU)) >> 16);
  RTC_TimeStruct->RTC_Minutes = (uint8_t)((tmpreg & (RTC_TR_MNT | RTC_TR_MNU)) >>8);
  RTC_TimeStruct->RTC_Seconds = (uint8_t)(tmpreg & (RTC_TR_ST | RTC_TR_SU));
  RTC_TimeStruct->RTC_H12 = (uint8_t)((tmpreg & (RTC_TR_PM)) >> 16);  

  /* Convert the structure parameters to Binary format */
  RTC_TimeStruct->RTC_Hours = (uint8_t)RTC_Bcd2ToByte(RTC_TimeStruct->RTC_Hours);
  RTC_TimeStruct->RTC_Minutes = (uint8_t)RTC_Bcd2ToByte(RTC_TimeStruct->RTC_Minutes);
  RTC_TimeStruct->RTC_Seconds = (uint8_t)RTC_Bcd2ToByte(RTC_TimeStruct->RTC_Seconds);   

  /* Get the RTC_TR register */
  tmpreg = (uint32_t)(RTC->DR & RTC_DR_RESERVED_MASK); 
  
  /* Fill the structure fields with the read parameters */
  RTC_DateStruct->RTC_Year = (uint8_t)((tmpreg & (RTC_DR_YT | RTC_DR_YU)) >> 16);
  RTC_DateStruct->RTC_Month = (uint8_t)((tmpreg & (RTC_DR_MT | RTC_DR_MU)) >> 8);
  RTC_DateStruct->RTC_Date = (uint8_t)(tmpreg & (RTC_DR_DT | RTC_DR_DU));
  RTC_DateStruct->RTC_WeekDay = (uint8_t)((tmpreg & (RTC_DR_WDU)) >> 13);  

  /* Convert the structure parameters to Binary format */
  RTC_DateStruct->RTC_Year = (uint8_t)RTC_Bcd2ToByte(RTC_DateStruct->RTC_Year);
  RTC_DateStruct->RTC_Month = (uint8_t)RTC_Bcd2ToByte(RTC_DateStruct->RTC_Month);
  RTC_DateStruct->RTC_Date = (uint8_t)RTC_Bcd2ToByte(RTC_DateStruct->RTC_Date); 
}

static inline void _delay_loops(uint32_t loops) 
{
  asm volatile (
                "1: SUBS %[loops], %[loops], #1 \n"
                  "   BNE 1b \n"
                    : [loops] "+r"(loops)
                      );
}
#define __DELAY_MS(MS, F_CPU) _delay_loops((uint32_t)((double)MS * F_CPU / 3000.0))
#define __DELAY_US(US, F_CPU) _delay_loops((uint32_t)((double)US * F_CPU / 3000000.0))

void RtcClockInit(void)
{
#if RTC_LOCK_EN > 0
  RTC_mutex=xSemaphoreCreateMutex();
  
  if(RTC_mutex == NULL)
  {
#if (!defined(BOOTLOADER))  
    printf("RTC: can't creat RTC_mutex...\n\r");
#endif
    for(;;);
  }
#endif
  
  /* Enable the PWR clock */
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_PWR, ENABLE);
  
  /* Allow access to RTC */
  PWR_BackupAccessCmd(ENABLE);
  
  if (RTC_ReadBackupRegister(RTC_BKP_DR0) != 0x32F2)
  {
    /* Enable the LSE OSC */
    RCC_LSEConfig(RCC_LSE_ON);
    
    /* Wait till LSE is ready */
    uint32_t ms_count=0;
    while(RCC_GetFlagStatus(RCC_FLAG_LSERDY) == RESET)
    {
      __DELAY_MS(1, TARGET_CPU_FREQ_HZ);
      ms_count++;
      if(ms_count>=(5*60*1000))
      {
        NVIC_SystemReset();
      }
    }
    
    /* Select the RTC Clock Source */
    RCC_RTCCLKConfig(RCC_RTCCLKSource_LSE);
        
    /* Enable the RTC Clock */
    RCC_RTCCLKCmd(ENABLE);
    
    /* Wait for RTC APB registers synchronisation */
    RTC_WaitForSynchro();
    
    /* Configure the RTC data register and RTC prescaler */
    RTC_InitTypeDef RTC_InitStructure;
    RTC_InitStructure.RTC_AsynchPrediv = 0x7F;
    RTC_InitStructure.RTC_SynchPrediv = 0xFF;
    RTC_InitStructure.RTC_HourFormat = RTC_HourFormat_24;
    
    /* Check on RTC init */
    if(RTC_Init(&RTC_InitStructure) == ERROR)
    {
#if (!defined(BOOTLOADER)) 
      printf("RTC Prescaler Config failed...\n");
#endif
    }
   
    RTC_DateTypeDef date;
    RTC_TimeTypeDef time;
    RTC_DateStructInit(&date);
    RTC_TimeStructInit(&time);
    
    RTC_SetBinTimeDate(&time, &date);
    
    /* Indicator for the RTC configuration */
    RTC_WriteBackupRegister(RTC_BKP_DR0, 0x32F2);
  }
  else
  {
    /* Wait for RTC APB registers synchronisation */
    RTC_WaitForSynchro();
    
    /* Clear the RTC Alarm Flag */
    RTC_ClearFlag(RTC_FLAG_ALRAF);
    
    /* Clear the EXTI Line 17 Pending bit (Connected internally to RTC Alarm) */
    EXTI_ClearITPendingBit(EXTI_Line17);
  }
}

void SetUnixTime(uint32_t utime)
{
  RTC_DateTypeDef date;
  RTC_TimeTypeDef time;
  time_struct_t vega_time;
  
  conv_unixtime_to_time(&vega_time, utime);
  
  if(!(vega_time.year>=2000 && vega_time.year<=2099 && IS_RTC_MONTH(vega_time.mon) && IS_RTC_DATE(vega_time.day) && IS_RTC_HOUR24(vega_time.hour) && IS_RTC_MINUTES(vega_time.min) && IS_RTC_SECONDS(vega_time.sec)))
  {
    return;
  }

#if RTC_LOCK_EN > 0
  xSemaphoreTake( RTC_mutex, portMAX_DELAY );
#endif
  
  date.RTC_Year=(uint8_t)(vega_time.year-2000);
  date.RTC_Month=vega_time.mon;
  date.RTC_Date=vega_time.day;
  date.RTC_WeekDay=RTC_Weekday_Monday;
  time.RTC_Hours=vega_time.hour;
  time.RTC_Minutes=vega_time.min;
  time.RTC_Seconds=vega_time.sec;
  time.RTC_H12=RTC_H12_AM;
  
  //RTC_WaitForSynchro();  
  RTC_SetBinTimeDate(&time, &date);

#if RTC_LOCK_EN > 0
  xSemaphoreGive( RTC_mutex );
#endif
}

uint32_t GetUnixTime(void)
{
  RTC_TimeTypeDef time;
  RTC_DateTypeDef date;
  time_struct_t vega_time;
  uint32_t utime;
  
#if RTC_LOCK_EN > 0
  xSemaphoreTake( RTC_mutex, portMAX_DELAY );
#endif
  
  //RTC_WaitForSynchro();  
  RTC_GetBinTimeDate(&time, &date);

  vega_time.day=date.RTC_Date;
  vega_time.mon=date.RTC_Month;
  vega_time.year=2000+date.RTC_Year;
  vega_time.hour=time.RTC_Hours;
  vega_time.min=time.RTC_Minutes;
  vega_time.sec=time.RTC_Seconds;
  
  utime=conv_time_to_unixtime(&vega_time);
  
#if RTC_LOCK_EN > 0
  xSemaphoreGive( RTC_mutex );
#endif
  return utime;
}