/**
  ******************************************************************************
  * File Name          : system_startup.c
  * Description        :      main
  *                       , NVIC, watchdog, RTC, BKP  ..  
  *                         system_stm32f10x.c
  *                             
  ******************************************************************************
*/

/* Includes  -----------------------------------------------------------------*/
#include "stm32f2xx.h"
#include "boot_config.h"
#include "rtc_driver.h"
#include "debug_port.h"
#include "System/system.h"
#include "signal_manager/signal_manager.h"
#include "leds_driver.h"

// /////////////////////////////////////////////////////////////////////
#if defined(BOOTLOADER)  
#if CHECK_PROT_BITS == 0
#warning   CHECK_PROT_BITS   > 0
#endif
#endif
#if (!defined(BOOTLOADER))  
#if IWDG_ENABLE == 0
#warning   IWDG_ENABLE   > 0
#endif
#endif

#if (!defined(BOOTLOADER))
static_assert(!(TARGET_CPU_FREQ_HZ != configCPU_CLOCK_HZ), "TARGET_CPU_FREQ_HZ != configCPU_CLOCK_HZ");
#endif //(!defined(BOOTLOADER))

#if (defined(BOOTLOADER_MODEL_2))
#if (!defined(BOOTLOADER))
__root const char __flashed_marker[8]                           @ (APP_START_ADDRESS+DEVICE_INT_VECT_SIZE)         = "ST-Link";
__root const char __device_type_marker[DEVICE_MARKER_MAX_SIZE]  @ (APP_START_ADDRESS+DEVICE_INT_VECT_SIZE+8)       = DEVICE_MARKER;
__root const char __fw_version[48]                              @ (APP_START_ADDRESS+DEVICE_INT_VECT_SIZE+8+DEVICE_MARKER_MAX_SIZE) = APLICATION_VERSION_STRING;
__root const uint8_t* const __board_hw_version                                                                     = ((const uint8_t* const)(BOOT_START_ADDR+32));
__root const uint8_t* const __fw_key                                                                               = ((const uint8_t* const)(BOOT_START_ADDR+48));
__root const uint32_t* const __vega_signat_key                                                                     = ((const uint32_t* const)(BOOT_START_ADDR+48+16));
#else
__root const char __botloader_version[32]                       @ (BOOT_START_ADDR+0)                              =  BOOTLOADER_VERSION_STRING;
__root const uint8_t __board_hw_version[1]                      @ (BOOT_START_ADDR+32)                             = {BOARD_HW_VERSION};
__root const char __boot_reserve[15]                            @ (BOOT_START_ADDR+32+1)                           = {};
__root const uint8_t __fw_key[16]                               @ (BOOT_START_ADDR+48)                             = {0x45,0x3f,0x26,0x48,0x97,0xfe,0x9a,0x45,0x11,0x98,0x8A,0xB6,0x47,0xD1,0xE6,0x72};
__root const uint32_t __vega_signat_key[4]                      @ (BOOT_START_ADDR+48+16)                          = {0x6021528a, 0x47f50670, 0x8d99AAfB, 0xC6aD9EFa};
__root const char* const __device_type_marker                                                                      = ((const char*)(APP_START_ADDRESS+DEVICE_INT_VECT_SIZE+8));
__root const char* const __flashed_marker                                                                          = ((const char*)(APP_START_ADDRESS+DEVICE_INT_VECT_SIZE));
#endif
#elif (defined(BOOTLOADER_MODEL_1))
#if (!defined(BOOTLOADER))
__root const char __flashed_marker[8]                          @ (APP_END_ADDRESS+1-8)                            = "ST-Link";
__root const char __device_type_marker[DEVICE_MARKER_MAX_SIZE] @ (APP_END_ADDRESS+1-8-DEVICE_MARKER_MAX_SIZE)     = DEVICE_MARKER;
__root const char __fw_version[48]                             @ (APP_END_ADDRESS+1-8-DEVICE_MARKER_MAX_SIZE-48)  = APLICATION_VERSION_STRING;
__root const uint8_t* const __board_hw_version                                                                    = ((const uint8_t* const)(BOOT_START_ADDR+32));
__root const uint8_t* const __fw_key                                                                              = ((const uint8_t* const)(BOOT_START_ADDR+48));
__root const uint32_t* const __vega_signat_key                                                                    = ((const uint32_t* const)(BOOT_START_ADDR+48+16));
#else
__root const char __botloader_version[32]                      @ (BOOT_START_ADDR+0)                              = BOOTLOADER_VERSION_STRING;
__root const uint8_t __board_hw_version[1]                     @ (BOOT_START_ADDR+32)                             = {BOARD_HW_VERSION};
__root const char __boot_reserve[15]                           @ (BOOT_START_ADDR+32+1)                           = {};
__root const uint8_t __fw_key[16]                              @ (BOOT_START_ADDR+48)                             = {0x45,0x3f,0x26,0x48,0x97,0xfe,0x9a,0x45,0x11,0x98,0x8A,0xB6,0x47,0xD1,0xE6,0x72};
__root const uint32_t __vega_signat_key[4]                     @ (BOOT_START_ADDR+48+16)                          = {0x6021528a, 0x47f50670, 0x8d99AAfB, 0xC6aD9EFa};
__root const char* const __device_type_marker                                                                     = ((const char*)(APP_END_ADDRESS+1-8-DEVICE_MARKER_MAX_SIZE));
__root const char* const __flashed_marker                                                                         = ((const char*)(APP_END_ADDRESS+1-8));
#endif
#else
#error not defined BOOTLOADER_MODEL
#endif //(defined(BOOTLOADER_MODEL_2))

#if (!defined(BOOTLOADER))
const char* __get_fw_version(void) {return __fw_version;}
#else
const char* __get_botloader_version(void) {return __botloader_version;}
#endif
const char* __get_flashed_marker(void) {return __flashed_marker;}
const char* __get_device_type_marker(void) {return __device_type_marker;}
const uint8_t* __get_fw_key(void) {return __fw_key;}
const uint32_t* __get_vega_signat_key(void) {return __vega_signat_key;}
uint8_t __get_board_hw_version(void) {return *(volatile uint8_t*)__board_hw_version;}

/* C code---------------------------------------------------------------------*/
/* PLL_VCO = (HSE_VALUE or HSI_VALUE / PLL_M) * PLL_N */
#define PLL_M      16
#define PLL_N      240

/* SYSCLK = PLL_VCO / PLL_P */
#if (TARGET_CPU_FREQ_HZ == 120000000)
#define PLL_P      2
#elif (TARGET_CPU_FREQ_HZ == 60000000)
#define PLL_P      4
#else
#error
#endif

/* USB OTG FS, SDIO and RNG Clock =  PLL_VCO / PLLQ */
#define PLL_Q      5

//     120/60 
static void SetSysClock(void)
{
  __IO uint32_t StartUpCounter = 0, HSEStatus = 0;
  
  /* Enable HSE */
  RCC->CR |= ((uint32_t)RCC_CR_HSEON);
  
  /* Wait till HSE is ready and if Time out is reached exit */
  do
  {
    HSEStatus = RCC->CR & RCC_CR_HSERDY;
    StartUpCounter++;
  } while((HSEStatus == 0) && (StartUpCounter != HSE_STARTUP_TIMEOUT));
  
  if ((RCC->CR & RCC_CR_HSERDY) != RESET)
  {
    HSEStatus = (uint32_t)0x01;
  }
  else
  {
    HSEStatus = (uint32_t)0x00;
  }
  
  if (HSEStatus == (uint32_t)0x01)
  {
#if (TARGET_CPU_FREQ_HZ == 120000000)
    /* HCLK = SYSCLK / 1*/
    RCC->CFGR |= RCC_CFGR_HPRE_DIV1;
    
    /* PCLK2 = HCLK / 2*/
    RCC->CFGR |= RCC_CFGR_PPRE2_DIV2;
    
    /* PCLK1 = HCLK / 4*/
    RCC->CFGR |= RCC_CFGR_PPRE1_DIV4;
#elif (TARGET_CPU_FREQ_HZ == 60000000)
    /* HCLK = SYSCLK / 1*/
    RCC->CFGR |= RCC_CFGR_HPRE_DIV1;
    
    /* PCLK2 = HCLK / 2*/
    RCC->CFGR |= RCC_CFGR_PPRE2_DIV2;
    
    /* PCLK1 = HCLK / 2*/
    RCC->CFGR |= RCC_CFGR_PPRE1_DIV2;
#else
#error
#endif

    /* Configure the main PLL */
    RCC->PLLCFGR = PLL_M | (PLL_N << 6) | (((PLL_P >> 1) -1) << 16) |
      (RCC_PLLCFGR_PLLSRC_HSE) | (PLL_Q << 24);
    
    /* Enable the main PLL */
    RCC->CR |= RCC_CR_PLLON;
    
    /* Wait till the main PLL is ready */
    while((RCC->CR & RCC_CR_PLLRDY) == 0)
    {
    }
    
    /* Configure Flash prefetch, Instruction cache, Data cache and wait state */
    FLASH->ACR = FLASH_ACR_PRFTEN | FLASH_ACR_ICEN | FLASH_ACR_DCEN | FLASH_ACR_LATENCY_3WS;
    
    /* Select the main PLL as system clock source */
    RCC->CFGR &= (uint32_t)((uint32_t)~(RCC_CFGR_SW));
    RCC->CFGR |= RCC_CFGR_SW_PLL;
    
    /* Wait till the main PLL is used as system clock source */
    while ((RCC->CFGR & (uint32_t)RCC_CFGR_SWS ) != RCC_CFGR_SWS_PLL);
    {
    }
  }
  else
  { /* If HSE fails to start-up, the application will have wrong clock
    configuration. User can add here some code to deal with this error */
  }
  
#if defined(BOOTLOADER)
  SysTick_Config(TARGET_CPU_FREQ_HZ/1000);
  __enable_interrupt();
#endif
}

//    ( , .  ,  , RTC, Watchdog, BKP)
void SystemInit(void)
{
  //  
  __disable_interrupt();
  
#if defined(BOOTLOADER)
  if(OB_BOR_LEVEL3!=FLASH_OB_GetBOR())
  {
    FLASH_OB_Unlock();
    FLASH_ClearFlag(FLASH_FLAG_EOP | FLASH_FLAG_OPERR | FLASH_FLAG_WRPERR | FLASH_FLAG_PGAERR | FLASH_FLAG_PGPERR | FLASH_FLAG_PGSERR
#if defined(STM32F413_423xx)
                    | FLASH_FLAG_RDERR                
#endif //defined(STM32F413_423xx)
                      );
    FLASH_OB_BORConfig(OB_BOR_LEVEL3);
    FLASH_OB_Launch();
    FLASH_OB_Lock();
  }
#endif
    
  //    
  extern  int use_bootloader;  extern  int __ICFEDIT_region_ROM_start__;  extern  int __ICFEDIT_region_ROM_end__; extern  int __ICFEDIT_intvec_start__;
  //Reset the RCC clock configuration to the default reset state
  RCC->CR |= (uint32_t)0x00000001;//Set HSION bit
  RCC->CFGR = 0x00000000;//Reset CFGR register
  RCC->CR &= (uint32_t)0xFEF6FFFF;//Reset HSEON, CSSON and PLLON bits
  RCC->PLLCFGR = 0x24003010;//Reset PLLCFGR register
  RCC->CR &= (uint32_t)0xFFFBFFFF;//Reset HSEBYP bit
  RCC->CIR = 0x00000000;//Disable all interrupts
  //  
  SetSysClock();                            
  //             
#if (!defined(BOOTLOADER))  
  if( (unsigned long)&__ICFEDIT_region_ROM_start__ != APP_START_ADDRESS || (unsigned long)&__ICFEDIT_region_ROM_end__ != APP_END_ADDRESS ) while(1);
#else
  if( (unsigned long)&__ICFEDIT_region_ROM_end__ != APP_START_ADDRESS-1 ) while(1);
#endif
  //       
  NVIC_SetVectorTable(NVIC_VectTab_FLASH, ((unsigned long)&__ICFEDIT_intvec_start__) - FLASH_BASE);
}

static void UnusedPinConfig(void)
{
  //none
}

static void PVD_Init(void)
{
  EXTI_InitTypeDef EXTI_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  // Configure EXTI Line16(PVD Output) to generate an interrupt on rising edges
  EXTI_ClearITPendingBit(EXTI_Line16);
  EXTI_InitStructure.EXTI_Line = EXTI_Line16;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Rising;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);
  
  // Enable the PVD Interrupt
  NVIC_InitStructure.NVIC_IRQChannel = PVD_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  // Configure the PVD Level to 2.9V
  PWR_PVDLevelConfig(PWR_CR_PLS_LEV6);
  
  // Enable the PVD Output
  PWR_PVDCmd(ENABLE);
  
  //         
  if(PWR_GetFlagStatus(PWR_FLAG_PVDO))
  {
    EXTI_GenerateSWInterrupt(EXTI_Line16);
  }
}

void ProgramInit(void)
{
  uint8_t no_optimaze;
#if (!defined(BOOTLOADER))  
  no_optimaze=__flashed_marker[0]; no_optimaze=__device_type_marker[0]; no_optimaze=__fw_version[0]; no_optimaze=(uint8_t)__vega_signat_key[0];
#else
  no_optimaze=__botloader_version[0]; no_optimaze=__fw_key[0]; no_optimaze=__vega_signat_key[0];
#endif
  no_optimaze++;
  
  //   
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_4); 
  //   GPIO
  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA | RCC_AHB1Periph_GPIOB | RCC_AHB1Periph_GPIOC | RCC_AHB1Periph_GPIOD | RCC_AHB1Periph_GPIOE, ENABLE);
  //
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_SYSCFG, ENABLE);
  //swj deinit
  GPIO_InitTypeDef GPIO_InitStructure;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_15;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
#if (!defined(BOOTLOADER))  
  //    ADC
  ADC_CommonInitTypeDef ADC_CommonInitStructure = {ADC_Mode_Independent, ADC_Prescaler_Div4, ADC_DMAAccessMode_Disabled, ADC_TwoSamplingDelay_15Cycles};
  ADC_CommonInit(&ADC_CommonInitStructure);
#endif
  //  JTAG
  //GPIO_PinRemapConfig(GPIO_Remap_SWJ_JTAGDisable, ENABLE);
#if defined(INTERNAL_AKB_PRESENT)
#if defined(INTERNAL_AKB_ON_OFF_CONTROL_PRESENT)
  //                    
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz; 
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = LIPO_ON_PIN;
  GPIO_Init(LIPO_ON_PORT, &GPIO_InitStructure);
  __LIPO_ON_EN();
#endif //INTERNAL_AKB_ON_OFF_CONTROL_PRESENT
#endif //INTERNAL_AKB_PRESENT
  //   PWR
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_PWR, ENABLE);
  //    BKP domain
  PWR_BackupAccessCmd(ENABLE);
  //  IWDG
#if (!defined(BOOTLOADER))  
#if IWDG_ENABLE > 0  
  /*Enable IWDG */
  DBGMCU_APB1PeriphConfig(DBGMCU_IWDG_STOP, ENABLE);
  IWDG_WriteAccessCmd(IWDG_WriteAccess_Enable);
  IWDG_SetPrescaler(IWDG_Prescaler_256);
  IWDG_SetReload((uint16_t)IWDG_RELOAD_VAL);
  IWDG_ReloadCounter();
  IWDG_Enable();
#endif
#endif
  //   
  DebugUartInit();
  //    
  RtcClockInit();
  //
  PVD_Init();
  //
  UnusedPinConfig();
}

#ifdef __cplusplus
extern "C" {
#endif     
void PVD_IRQHandler(void)
{
  if(EXTI_GetITStatus(EXTI_Line16) != RESET)
  {
    /* Clear the Key Button EXTI line pending bit */
    EXTI_ClearITPendingBit(EXTI_Line16);
    
    //    BKP 
    RTC_WriteBackupRegister(RTC_BKP_DR2, 0xA5A5);
    
    NVIC_SystemReset();
  }
}
#ifdef __cplusplus
}
#endif 
