/**
******************************************************************************
* File Name          : main.cpp
* Description        : Main program body
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "debug_port.h"
#include "leds_driver.h"
#include "crc_lib/crc_lib.h"
#include "mbedtls-aes_tiny_lib/mbedtls-aes_tiny.h"
#include "MX25L6435E/MX25L6435E.h"
#include "NOR_FTL/NOR_FTL.h"
#include "Black_box/Black_box.h"
#include "memory_spi.h"
#include "boot_config.h"
#include "rtc_driver.h"

#include "string.h"
#include "stdarg.h"

/* Extern functions ----------------------------------------------------------*/
extern void ProgramInit(void);

/* Private typedef -----------------------------------------------------------*/
typedef  void (*pFunction)(void);

/* Private define ------------------------------------------------------------*/

pFunction Jump_To_Application;
uint32_t JumpAddress;

boot_status update_prog(const char* f_name, const uint8_t* key);

//  SPI
CMemory_spi Memory_spi;
//     SPI     macronix
static void SPI_csup(void) {Memory_spi.CS_up();}
static void SPI_csdown(void) {Memory_spi.CS_down();}
static void SPI_senddata(const uint8_t* data, uint16_t len) {Memory_spi.Send_data(data, len);}
static void SPI_sendbyte(uint8_t byte) {Memory_spi.Send_byte(byte);}
static void SPI_getdata(uint8_t* data, uint16_t len) {Memory_spi.Get_data(data, len);}
static uint8_t SPI_getbyte(void) {return Memory_spi.Get_byte();}
static void SPI_init(void) {Memory_spi.Init();}
static void SPI_deinit(void) {Memory_spi.Deinit();}
//   macronix
CMX25L64 MX25_memory(&SPI_csup, &SPI_csdown, &SPI_senddata, &SPI_sendbyte, &SPI_getdata, &SPI_getbyte, &SPI_init, &SPI_deinit);
//      macronix  FTL
static uint8_t MX25_memory_init(void){return MX25_memory.Init();}
static void MX25_memory_deinit(void){MX25_memory.Deinit();}
static void MX25_memory_write(uint16_t sector, uint16_t address, const uint8_t* data, uint16_t size){MX25_memory.Write_data(sector, address, data, size);}
static void MX25_memory_read(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){MX25_memory.Read_data(sector, address, data, size);}
static void MX25_memory_erase_sector(uint16_t sector){MX25_memory.Erase_sector(sector);}
static void MX25_memory_erase_chip(void){MX25_memory.Erase_chip();}
// NOR flash translation layer (    )
CNOR_ftl NOR_ftl(&MX25_memory_init, &MX25_memory_deinit, &MX25_memory_write, &MX25_memory_read, &MX25_memory_erase_sector, &MX25_memory_erase_chip);
//    FTL   
static uint8_t mem_init(void){return (uint8_t)NOR_ftl.Init();}
static void    mem_deinit(void){NOR_ftl.Deinit();}
static uint8_t mem_format(void){return (uint8_t)NOR_ftl.Format_memory();}
static uint8_t mem_write(uint32_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Write_data(address, data, size);}
static uint8_t mem_read(uint32_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Read_data(address, data, size);}
static uint8_t mem_direct_write(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Direct_write_data(sector, address, data, size);}
static uint8_t mem_direct_read(uint16_t sector, uint16_t address, uint8_t* data, uint16_t size){return (uint8_t)NOR_ftl.Direct_read_data(sector, address, data, size);}
static uint8_t mem_direct_erase_sector(uint16_t sector){return (uint8_t)NOR_ftl.Direct_erase_sector(sector);}
//  
CBlack_box Black_box(mem_init, mem_deinit, mem_format, mem_write, mem_read, mem_direct_write, mem_direct_read, mem_direct_erase_sector, NOR_FTL_DIRECT_SECTOR_COUNT, NOR_FTL_SECTOR_SIZE, NOR_ftl.Get_memory_size());
static void LOG(uint8_t stdout_en, const char* string, ...)
{
  static uint8_t buff[256];
  static uint8_t first_start=1;
    
  //       
  memset(buff, 0x00, sizeof(buff));
  uint32_t unixtime = GetUnixTime();
  time_struct_t time; 
  conv_unixtime_to_time(&time, unixtime);
  sprintf((char*)buff, "%s[%d/%d/%d][%d:%d:%d]", first_start ? "\n" : "", time.day, time.mon, time.year, time.hour, time.min, time.sec);
  first_start=0;
  uint16_t buff_offset = strlen((char*)buff);
  //        
  va_list ap;
  va_start(ap, string);
  int retval = vsnprintf((char*)buff + buff_offset, sizeof(buff) - buff_offset, string, ap);
  va_end(ap);  
  uint16_t size = strlen((char*)buff);
  
  if(stdout_en) printf("%s", (char*)(buff+buff_offset));
  
  //    
  if(Black_box.fopen("LOG")!= BB_OPERATION_OK)  return;
  if(Black_box.fwrite("LOG", buff, size)!= BB_OPERATION_OK)  return;
  if(Black_box.fsave()!= BB_OPERATION_OK)  return;
  Black_box.fclose("LOG");
}

static void delay_ms(uint32_t ms)
{
  extern uint32_t SysTickReg;
  uint32_t end_wait_time=SysTickReg+ms-1; 
  while(!timeAfter(SysTickReg, end_wait_time));
}

static void check_boot_err_led_script(void)
{
  LedB_On(); LedG_On(); LedR_On();
  delay_ms(1500);
  LedB_Off(); LedG_Off(); LedR_Off();
  delay_ms(1000);
  LedB_On(); LedG_On(); LedR_On();
  delay_ms(200);
  LedB_Off(); LedG_Off(); LedR_Off();
  delay_ms(200);
  LedB_On(); LedG_On(); LedR_On();
  delay_ms(200);
  LedB_Off(); LedG_Off(); LedR_Off();
  delay_ms(200);
  LedB_On(); LedG_On(); LedR_On();
  delay_ms(200);
  LedB_Off(); LedG_Off(); LedR_Off();
  delay_ms(2000);
}

static void shifting_led_script(uint8_t mull)
{
  static uint8_t cnt=0;
  
  if(cnt<1*mull)
  {
    LedB_On(); LedR_Off(); LedG_Off();
    cnt++;
  }
  else if(cnt<2*mull)
  {
    LedB_Off(); LedR_On(); LedG_Off();
    cnt++;
  }
  else if(cnt<3*mull)
  {
    LedB_Off(); LedR_Off(); LedG_On();
    cnt++;
  }
  else
  {
   cnt=0; 
  }
}

static void start_led_script(void)
{
  LedR_Off(); LedG_Off(); LedB_Off();
  
  LedB_On(); delay_ms(100); LedR_On(); delay_ms(100); LedG_On(); delay_ms(200);
  LedG_Off(); delay_ms(100); LedR_Off();  delay_ms(100); LedB_Off();
}

static uint32_t GetSector(uint32_t Address)
{
  uint32_t sector = 0;
  
  if((Address < ADDR_FLASH_SECTOR_1) && (Address >= ADDR_FLASH_SECTOR_0))
  {
    sector = FLASH_Sector_0;
  }
  else if((Address < ADDR_FLASH_SECTOR_2) && (Address >= ADDR_FLASH_SECTOR_1))
  {
    sector = FLASH_Sector_1;
  }
  else if((Address < ADDR_FLASH_SECTOR_3) && (Address >= ADDR_FLASH_SECTOR_2))
  {
    sector = FLASH_Sector_2;
  }
  else if((Address < ADDR_FLASH_SECTOR_4) && (Address >= ADDR_FLASH_SECTOR_3))
  {
    sector = FLASH_Sector_3;
  }
  else if((Address < ADDR_FLASH_SECTOR_5) && (Address >= ADDR_FLASH_SECTOR_4))
  {
    sector = FLASH_Sector_4;
  }
  else if((Address < ADDR_FLASH_SECTOR_6) && (Address >= ADDR_FLASH_SECTOR_5))
  {
    sector = FLASH_Sector_5;
  }
  else if((Address < ADDR_FLASH_SECTOR_7) && (Address >= ADDR_FLASH_SECTOR_6))
  {
    sector = FLASH_Sector_6;
  }
  else if((Address < ADDR_FLASH_SECTOR_8) && (Address >= ADDR_FLASH_SECTOR_7))
  {
    sector = FLASH_Sector_7;
  }
  else if((Address < ADDR_FLASH_SECTOR_9) && (Address >= ADDR_FLASH_SECTOR_8))
  {
    sector = FLASH_Sector_8;
  }
  else if((Address < ADDR_FLASH_SECTOR_10) && (Address >= ADDR_FLASH_SECTOR_9))
  {
    sector = FLASH_Sector_9;
  }
  else if((Address < ADDR_FLASH_SECTOR_11) && (Address >= ADDR_FLASH_SECTOR_10))
  {
    sector = FLASH_Sector_10;  
  }
  else/*(Address < FLASH_END_ADDR) && (Address >= ADDR_FLASH_SECTOR_11))*/
  {
    sector = FLASH_Sector_11;  
  }
  
  return sector;
}

int main(void)
{
  ProgramInit();
  Leds_Init();
  start_led_script(); delay_ms(200);
  start_led_script(); delay_ms(200);
  start_led_script();
  
  __PRINTF("\n");
  __PRINTF("board hw: %hhu, bootloader version: %s (%s %s)\n", __get_board_hw_version(), __get_botloader_version(), __DATE__, __TIME__);
  
  if(false)
  {
    bool is_erase_enable=false;
    static uint8_t buff[MX25L64_SECTOR_SIZE];
    
    MX25_memory_init();
    
    for(uint32_t sector=0; sector<(FlashSize/MX25L64_SECTOR_SIZE); sector++)
    {
      MX25_memory_read(sector, 0, buff, sizeof(buff));
      
      for(uint32_t i=0; i<sizeof(buff); i++)
      {
        if(buff[i]!=0xff) {is_erase_enable=true; break;}
      }
      if(!(sector%16)) LedG_Toggle();
    }
    LedG_Off();
    
    if(is_erase_enable)
    {
      LedG_On();
      MX25_memory_erase_chip();
      LedG_Off();
    }
    
    for(;;)
    {
      LedB_On(); LedG_On(); LedR_On();
      delay_ms(300);
      LedB_Off(); LedG_Off(); LedR_Off();
      delay_ms(1000);
    }
  }
  
  if(BB_OPERATION_OK!=Black_box.Init())
  {
    __PRINTF("bbox init fail\n");
    for(;;) check_boot_err_led_script();
  }
  else
  {
    __PRINTF("bbox init ok\n");
  }
  
  boot_cmd_t boot_cmd;
  Black_box.Register_in_base("boot_cmd", sizeof(boot_cmd), 0, &boot_cmd);
  boot_cmd.fname[sizeof(boot_cmd.fname)-1]='\0';
  if(boot_cmd.cmd==0xff &&  boot_cmd.fname[0]==0xff) 
  {
    LOG(1, "it's first system start...\n");
    boot_cmd.cmd=NORMAL_SYSTEM_START;
    boot_cmd.fname[0]='\0';
    Black_box.Save_base("boot_cmd");
  }

  
  LOG(0, "board hw: %hhu, bootloader version: %s (%s %s)\n", __get_board_hw_version(), __get_botloader_version(), __DATE__, __TIME__);
  //LOG(0, "device sn: %llu", sn);
  
#if CHECK_PROT_BITS > 0
  __PRINTF("check protection bits...\n");
  if(RESET == FLASH_OB_GetRDP())
  {
    LOG(1, "protection bits are not set\n");
    for(;;) check_boot_err_led_script();
  }
  else
  {
    __PRINTF("protection bits are set\n");
  }
#endif //CHECK_PROT_BITS
  
  if(boot_cmd.cmd==UPGRADE_FW_CMD)
  {
    LOG(1, "need to update the firmware, fname \"%s\"\n", boot_cmd.fname);
    
    boot_cmd.cmd = update_prog(boot_cmd.fname, __get_fw_key());
    Black_box.Save_base("boot_cmd");
    
    if(boot_cmd.cmd == RES_FLASHED_OK)
    {
      //LOG(1, "flashed ok\n");
    }
    else if(boot_cmd.cmd == RES_IMAGE_ERR)
    {
      LOG(1, "image file err\n");
    }
    else if(boot_cmd.cmd == RES_FLASH_DEVICE_ERR)
    {
      LOG(1, "err in flashed device memory\n");
    }
    
    else if(boot_cmd.cmd == RES_EXT_MEM_ERR)
    {
      LOG(1, "ext memory err\n");
    }
    else if(boot_cmd.cmd == RES_CRYPT_LIB_ERR)
    {
      LOG(1, "err in crypt lib\n");
    }
    else //unknown
    {
      LOG(1, "unknown err\n");
    }
  }
  else
  {
    LOG(1, "no need to update the firmware\n");
  }
  
  if(memcmp(DEVICE_MARKER, __get_device_type_marker(), sizeof(DEVICE_MARKER)) != 0)//     
  {
    LOG(1, "not find \"%s\" marker\n", DEVICE_MARKER);
    for(;;) check_boot_err_led_script();
  }
  
  if(memcmp("ST-Link", __get_flashed_marker(), sizeof("ST-Link")) == 0)//  ST_link
  {
    LOG(1, "application flashed by ST-Link\n");
  }
  else
  {
#if (defined(BOOTLOADER_MODEL_2))
    //  + 4     (   ) + 4  crc (.  + 4   +  ) +   (  -  .  - 4   - 4  crc)
    uint32_t image_len=*(uint32_t*)(APP_START_ADDRESS+DEVICE_INT_VECT_SIZE);
    uint32_t image_crc=*(uint32_t*)(APP_START_ADDRESS+DEVICE_INT_VECT_SIZE+sizeof(uint32_t));
    uint32_t actiual_crc;
    
    if(image_len>((APP_END_ADDRESS+1)-APP_START_ADDRESS) || image_len<=(DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t)))
    {
      LOG(1, "wrong application image len\n");
      for(;;) check_boot_err_led_script();
    }
    
    LOG(1, "check application image crc...\n");
   
    crc32_fast_init(&actiual_crc);
    crc32_fast_update(&actiual_crc, (uint8_t*)(APP_START_ADDRESS), DEVICE_INT_VECT_SIZE+sizeof(uint32_t));
    crc32_fast_update(&actiual_crc, (uint8_t*)(APP_START_ADDRESS+DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t)), image_len-(DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t)));
    crc32_fast_finish(&actiual_crc);

    if(image_crc == actiual_crc)
    {
      LOG(1, "application image crc is ok\n");
    }
    else
    {
      LOG(1, "wrong image application crc\n");
      for(;;) check_boot_err_led_script();
    }
#elif (defined(BOOTLOADER_MODEL_1))
    /* Check fw crc */
    /*
    image (image_len bytes)  -|image_crc 
    image_len (4 bytes)      -|
    image_crc (4 bytes) 
    */
    uint32_t image_crc=*(uint32_t*)(APP_END_ADDRESS+1-sizeof(uint32_t));
    uint32_t image_len=*(uint32_t*)(APP_END_ADDRESS+1-sizeof(uint32_t)-sizeof(uint32_t));
    uint32_t actiual_crc;
    
    if(image_len>((APP_END_ADDRESS+1)-APP_START_ADDRESS-sizeof(uint32_t)))
    {
      LOG(1, "wrong application image len\n");
      for(;;) check_boot_err_led_script();
    }
    
    LOG(1, "check application image crc...\n");
    
    actiual_crc = crc32_fast_full((uint8_t*)(APP_START_ADDRESS), image_len);

    if(image_crc == actiual_crc)
    {
      LOG(1, "application image crc is ok\n");
    }
    else
    {
      LOG(1, "wrong image application crc\n");
      for(;;) check_boot_err_led_script();
    }
#else
#error not defined BOOTLOADER_MODEL
#endif //(defined(BOOTLOADER_MODEL_2))
  }
  
  //  ,  crc   
  /* Check if valid stack address (RAM address) then jump to user application */
  __PRINTF("check if valid stack address (RAM address)\n");

  /*  crc  len   8  */
  if (((*(__IO uint32_t*)(APP_START_ADDRESS)) & 0x2FFE0000 ) == 0x20000000)
  {
    LOG(1, "jump to application...\n");
    
    __disable_interrupt();
    
    /* Jump to user application */
    JumpAddress = *(__IO uint32_t*) (APP_START_ADDRESS + 4);
    Jump_To_Application = (pFunction) JumpAddress;
    /* Initialize user application's Stack Pointer */
    __set_MSP(*(__IO uint32_t*)(APP_START_ADDRESS));
    Jump_To_Application();
  }
  
  __enable_interrupt();
  LOG(1, "stack address (RAM address)is not valid\n");
  //  ,  recovery   , ..  ,   .    ,    NAND   . 
  //    .
  for(;;) check_boot_err_led_script();
}


//DataLength in worlds
uint8_t flash_write(uint32_t* FlashAddress, uint32_t* Data ,uint32_t DataLength)
{
  uint32_t i = 0;
  
  for (i = 0; i < DataLength; i++)
  {
    //    ?
    if(*FlashAddress > (APP_END_ADDRESS-3))
    {
      return(3);
    }
    
    /* Device voltage range supposed to be [2.7V to 3.6V], the operation will
    be done by word */ 
    if (FLASH_ProgramWord(*FlashAddress, *(uint32_t*)(Data+i)) == FLASH_COMPLETE)
    {
      /* Check the written value */
      if (*(uint32_t*)*FlashAddress != *(uint32_t*)(Data+i))
      {
        /* Flash content doesn't match SRAM content */
        return(2);
      }
      /* Increment FLASH destination address */
      *FlashAddress += 4;
    }
    else
    {
      /* Error occurred while writing data in Flash memory */
      return (1);
    }
  }
  return (0);
}

#define FILE_HEADER_LEN         32 //iv+crc32+fw_len+fw_ver
/*
iv (16 bytes)
crc32 (4 bytes)                  -|
fw_len(4 bytes)           -|      |crypted
fw_ver(8 bytes)            |crc   |data
fw_data (fw_len bytes)    -|     -|
*/

boot_status update_prog(const char* f_name, const uint8_t* key)
{
  static uint8_t enc_buf[2048];
  static uint8_t dec_buf[2048];
  static uint8_t iv[16];
  static uint8_t fw_ver[8];
  static uint8_t dmarker[DEVICE_MARKER_MAX_SIZE];
  static mbedtls_aes_context aes_ctx;
  
  static_assert(!(sizeof(enc_buf)!=sizeof(dec_buf)), "sizeof(enc_buf)!=sizeof(dec_buf)");
  static_assert(!(sizeof(enc_buf)<(DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t))), "buff size is too small");
    
  uint32_t read_offset, fw_len, f_size, image_len;
  uint32_t crc32_in_header, crc32_reading, crc32_to_write;

  //HwLedOn(BOOT_LED1);
      
  if(Black_box.fsize(f_name, &f_size) != BB_OPERATION_OK) 
  { 
    return RES_EXT_MEM_ERR;
  }
  
  if(f_size<(FILE_HEADER_LEN+512) || f_size>(FILE_HEADER_LEN+((APP_END_ADDRESS+1)-APP_START_ADDRESS)) || 0!=((f_size-16)%16))
  {
    //      16
    return RES_IMAGE_ERR;
  }
  
  if(Black_box.fopen(f_name) != BB_OPERATION_OK)
  {
    return RES_EXT_MEM_ERR;
  }
      
  read_offset=0;
  if(Black_box.fread(f_name, read_offset, iv, 16) != BB_OPERATION_OK)
  {
    return RES_EXT_MEM_ERR;
  }
  read_offset+=16;
  
  mbedtls_aes_init(&aes_ctx);
  if(mbedtls_aes_setkey_dec(&aes_ctx, key, 128) != 0)
  {
    return RES_CRYPT_LIB_ERR;
  }
  
  if(Black_box.fread(f_name, read_offset, enc_buf, 16) != BB_OPERATION_OK)
  {
    return RES_EXT_MEM_ERR;
  }
  read_offset+=16;
  
  if(mbedtls_aes_crypt_cbc(&aes_ctx, MBEDTLS_AES_DECRYPT, 16, iv, enc_buf, dec_buf) != 0)
  {
    return RES_CRYPT_LIB_ERR;
  }
  
  memcpy((uint8_t*)&crc32_in_header, &dec_buf[0], sizeof(uint32_t));
  memcpy((uint8_t*)&fw_len, &dec_buf[4], sizeof(uint32_t));
  memcpy(&fw_ver[0], &dec_buf[8], 8);
  
  if(fw_len>((APP_END_ADDRESS+1)-APP_START_ADDRESS) || fw_len%16)
  {
    return RES_IMAGE_ERR;
  }
  
  crc32_fast_init(&crc32_reading);
  crc32_fast_update(&crc32_reading, &dec_buf[4], 16-4);
  
  crc32_fast_init(&crc32_to_write);
  
#if (defined(BOOTLOADER_MODEL_2))
  image_len=fw_len;
#elif (defined(BOOTLOADER_MODEL_1))
  image_len=fw_len-4;//   (   crc)
#endif //(defined(BOOTLOADER_MODEL_2))
  
  LOG(1, "check firmware file...\n");
  
#if (defined(BOOTLOADER_MODEL_2))
  //
#elif (defined(BOOTLOADER_MODEL_1))
  //        
  const char* dmarker_addr_presumably=(const char*)APP_START_ADDRESS+(f_size-FILE_HEADER_LEN-8-DEVICE_MARKER_MAX_SIZE);
  
  if(dmarker_addr_presumably!=__get_device_type_marker())
  {
    LOG(1, "unexpected device type marker marker position\n");
    return RES_IMAGE_ERR;
  }
#endif //(defined(BOOTLOADER_MODEL_2))
  
  for(uint32_t i=0, read_len; i<=(f_size-FILE_HEADER_LEN)/sizeof(enc_buf); i++)
  {
    
    if(i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf)) 
      read_len=(f_size-FILE_HEADER_LEN)%sizeof(enc_buf);
    else
      read_len=sizeof(enc_buf);
    
    if(read_len)
    {
      if(Black_box.fread(f_name, read_offset, enc_buf, read_len) != BB_OPERATION_OK)
      {
        return RES_EXT_MEM_ERR;
      }
      read_offset+=read_len;
      
      if(mbedtls_aes_crypt_cbc(&aes_ctx, MBEDTLS_AES_DECRYPT, read_len, iv, enc_buf, dec_buf) != 0)
      {
        return RES_CRYPT_LIB_ERR;
      }
      
      crc32_fast_update(&crc32_reading, dec_buf, read_len);
      
#if (defined(BOOTLOADER_MODEL_2))
      if(i==0)
      {
        memcpy(dmarker, &dec_buf[DEVICE_INT_VECT_SIZE+8], DEVICE_MARKER_MAX_SIZE);
        memcpy(&dec_buf[DEVICE_INT_VECT_SIZE], &image_len, sizeof(uint32_t));
        // 4   crc
        
        crc32_fast_update(&crc32_to_write, &dec_buf[0], DEVICE_INT_VECT_SIZE+sizeof(uint32_t));// crc     4  
        crc32_fast_update(&crc32_to_write, &dec_buf[DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t)], read_len-(DEVICE_INT_VECT_SIZE+sizeof(uint32_t)+sizeof(uint32_t)));// crc  4   crc
      }
      else
      {
        crc32_fast_update(&crc32_to_write, &dec_buf[0], read_len);
      }
#elif (defined(BOOTLOADER_MODEL_1))
      if( ((i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf) - 1) && 0==((f_size-FILE_HEADER_LEN)%sizeof(enc_buf)))
         || (i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf)) )// 
      {
        memcpy(dmarker, &dec_buf[read_len-8-DEVICE_MARKER_MAX_SIZE], DEVICE_MARKER_MAX_SIZE);
        memcpy(&dec_buf[read_len-8], (uint8_t*)&image_len, 4);
        crc32_fast_update(&crc32_to_write, &dec_buf[0], read_len-4);
      }
      else
      {
        crc32_fast_update(&crc32_to_write, dec_buf, read_len);
      }
#endif //(defined(BOOTLOADER_MODEL_2))
      
    }
    shifting_led_script(10);
  }
  LedR_Off(); LedB_Off(); LedG_Off();
  
  mbedtls_aes_free(&aes_ctx);
  
  crc32_fast_finish(&crc32_to_write);
  crc32_fast_finish(&crc32_reading);
  if(crc32_in_header!=crc32_reading)
  {
    return RES_IMAGE_ERR;
  }
  
  if(memcmp(DEVICE_MARKER, dmarker, sizeof(DEVICE_MARKER)) != 0)
  {
    LOG(1, "not find \"%s\" marker in file\n", DEVICE_MARKER);
    return RES_IMAGE_ERR;
  }
  
  LOG(1, "firmware file is valid\n");
  //HwLedOn(BOOT_LED2);  
    
  FLASH_Unlock();
  
  //Clear pending flags (if any)  
  FLASH_ClearFlag(FLASH_FLAG_EOP | FLASH_FLAG_OPERR | FLASH_FLAG_WRPERR | FLASH_FLAG_PGAERR | FLASH_FLAG_PGPERR | FLASH_FLAG_PGSERR
#if defined(STM32F413_423xx)
                  | FLASH_FLAG_RDERR                
#endif //defined(STM32F413_423xx)
                    );
  
#if (defined(BOOTLOADER_MODEL_2))
  LOG(1, "start erasing mcu flash, start addr=0x%lX, sector=%hhu, end addr=0x%lX, sector=%hhu\n", APP_START_ADDRESS, GetSector(APP_START_ADDRESS)/8, APP_START_ADDRESS+image_len, GetSector(APP_START_ADDRESS+image_len-1)/8);
  
  for(uint32_t erase_sector=GetSector(APP_START_ADDRESS); erase_sector <= GetSector(APP_START_ADDRESS+image_len-1); erase_sector+=8)
  {
    /* Device voltage range supposed to be [2.7V to 3.6V], the operation will be done by word */ 
    if(FLASH_EraseSector(erase_sector, VoltageRange_3) != FLASH_COMPLETE)
    {
      FLASH_Lock();
      return RES_FLASH_DEVICE_ERR;
    }
    shifting_led_script(8);
  }
#elif (defined(BOOTLOADER_MODEL_1))
  LOG(1, "start erasing mcu flash, start addr=0x%lX, end addr=0x%lX\n", APP_START_ADDRESS, APP_END_ADDRESS);
  
  for(uint32_t erase_sector=GetSector(APP_START_ADDRESS); erase_sector <= GetSector(APP_END_ADDRESS); erase_sector+=8)
  {
    /* Device voltage range supposed to be [2.7V to 3.6V], the operation will be done by word */ 
    if (FLASH_EraseSector(erase_sector, VoltageRange_3) != FLASH_COMPLETE)
    {
      FLASH_Lock();
      return RES_FLASH_DEVICE_ERR;
    }
    shifting_led_script(8);
  }
#endif //(defined(BOOTLOADER_MODEL_2))

  LedR_Off(); LedB_Off(); LedG_Off();
   
  LOG(1, "erase mcu flash ok\n");
  //HwLedOn(BOOT_LED3);  
  
  read_offset=0;
  if(Black_box.fread(f_name, read_offset, iv, 16)!= BB_OPERATION_OK)
  {
    FLASH_Lock();
    return RES_EXT_MEM_ERR;
  }
  read_offset+=16;
  
  mbedtls_aes_init(&aes_ctx);
  if(mbedtls_aes_setkey_dec(&aes_ctx, key, 128) != 0)
  {
    FLASH_Lock();
    return RES_CRYPT_LIB_ERR;
  }
  
    if(Black_box.fread(f_name, read_offset, enc_buf, FILE_HEADER_LEN-16) != BB_OPERATION_OK)
  {
    FLASH_Lock();
    return RES_EXT_MEM_ERR;
  }
  read_offset+=FILE_HEADER_LEN-16;
  
  if(mbedtls_aes_crypt_cbc(&aes_ctx, MBEDTLS_AES_DECRYPT, (FILE_HEADER_LEN-16), iv, enc_buf, dec_buf) != 0)
  {
    FLASH_Lock();
    return RES_CRYPT_LIB_ERR;
  }
  
  uint32_t Flash_Write_Address=APP_START_ADDRESS;
  LOG(1, "start mcu memory flashing...\n");

  for(uint32_t i=0, read_len; i<=(f_size-FILE_HEADER_LEN)/sizeof(enc_buf); i++)
  {
    
    if(i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf)) 
      read_len=(f_size-FILE_HEADER_LEN)%sizeof(enc_buf);
    else
      read_len=sizeof(enc_buf);
    
    if(read_len)
    {
      
      if(Black_box.fread(f_name, read_offset, enc_buf, read_len) != BB_OPERATION_OK)
      {
        FLASH_Lock();
        return RES_EXT_MEM_ERR;
      }
      read_offset+=read_len;
      
      if(mbedtls_aes_crypt_cbc(&aes_ctx, MBEDTLS_AES_DECRYPT, read_len, iv, enc_buf, dec_buf) != 0)
      {
        FLASH_Lock();
        return RES_CRYPT_LIB_ERR;
      }
      
      //crc32_fast_update(&crc32_reading, dec_buf, read_len);
      
#if (defined(BOOTLOADER_MODEL_2))
      if(i==0)
      {
        memcpy(&dec_buf[DEVICE_INT_VECT_SIZE], &image_len, sizeof(uint32_t));
        memcpy(&dec_buf[DEVICE_INT_VECT_SIZE+sizeof(uint32_t)], &crc32_to_write, sizeof(uint32_t));
      }
#elif (defined(BOOTLOADER_MODEL_1))
      if( ((i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf) - 1) && 0==((f_size-FILE_HEADER_LEN)%sizeof(enc_buf)))
         || (i==(f_size-FILE_HEADER_LEN)/sizeof(enc_buf)) )// 
      {
        memcpy(&dec_buf[read_len-4], (uint8_t*)&crc32_to_write, 4);
        memcpy(&dec_buf[read_len-8], (uint8_t*)&image_len, 4);
      }
#endif //(defined(BOOTLOADER_MODEL_2))
      
      //  
      if(0!=flash_write(&Flash_Write_Address, (uint32_t*)&dec_buf[0] ,read_len/4))
      {
        FLASH_Lock();
        return RES_FLASH_DEVICE_ERR;
      }
    }
    shifting_led_script(1);
  }
  LedR_Off(); LedB_Off(); LedG_Off();
  
  FLASH_Lock();
  
  LOG(1, "mcu memory flashing ok\n");
  
  //HwLedOn(BOOT_LED4);
  
  mbedtls_aes_free(&aes_ctx);
  return RES_FLASHED_OK;
}
