/**
******************************************************************************
* File Name          :
* Description        :
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "stm32f4xx.h"
#include "external_uart.h"

static uart_driver_t uart_driver[EXTERNAL_UART_CTX_COUNT]={0};

static inline USART_TypeDef* get_uart_pherph_from_ctx(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return NULL;
  }
}

static inline uint16_t get_uart_tx_pin(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_TX_PIN;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_TX_PIN;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return 0;
  }
}

static inline uint16_t get_uart_rx_pin(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_RX_PIN;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_RX_PIN;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return 0;
  }
}

static inline GPIOPuPd_TypeDef get_uart_rx_pin_pupd(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_TX_PIN_PUPD;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_TX_PIN_PUPD;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return GPIO_PuPd_NOPULL;
  }
}

static inline GPIO_TypeDef* get_uart_tx_port(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_TX_PORT;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_TX_PORT;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return NULL;
  }
}

static inline GPIO_TypeDef* get_uart_rx_port(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_RX_PORT;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_RX_PORT;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return 0;
  }
}

static inline IRQn_Type get_uart_irqn(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: return EXTERNAL_UART1_IRQn;
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: return EXTERNAL_UART2_IRQn;
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 return UsageFault_IRQn;
  }
}

static inline void afio_clock(const uint8_t ctx, const bool is_enable)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: {if(is_enable){__EXTERNAL_UART1_AFIO_CLOCK_ENABLE();} else{__EXTERNAL_UART1_AFIO_CLOCK_DISABLE();} break;}
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: {if(is_enable){__EXTERNAL_UART2_AFIO_CLOCK_ENABLE();} else{__EXTERNAL_UART2_AFIO_CLOCK_DISABLE();} break;}
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 {break;}
  }
  
}

static inline void uart_clock(const uint8_t ctx, const bool is_enable)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: {if(is_enable){__EXTERNAL_UART1_CLOCK_ENABLE();} else{__EXTERNAL_UART1_CLOCK_DISABLE();} break;}
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: {if(is_enable){__EXTERNAL_UART2_CLOCK_ENABLE();} else{__EXTERNAL_UART2_CLOCK_DISABLE();} break;}
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 {break;}
  }
}

static inline void uart_remap(const uint8_t ctx)
{
  switch(ctx)
  {
#if (EXTERNAL_UART_CTX_COUNT>0)
  case EXTERNAL_UART_CTX1: {__EXTERNAL_UART1_REMAP(); break;}
#endif //(EXTERNAL_UART_CTX_COUNT>0)
  
#if (EXTERNAL_UART_CTX_COUNT>1)
  case EXTERNAL_UART_CTX2: {__EXTERNAL_UART2_REMAP(); break;}
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
  default:                 {break;}
  }
}

void external_uart_init(const uint8_t ctx, const uint32_t baud_rate, void* const rx_heap, const uint16_t rx_heap_size, bool (*rx_it_callback)(cbuff_t* const))
{
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return;
  
  uart_driver[ctx].rx_mem=rx_heap;
  uart_driver[ctx].rx_mem_size=rx_heap_size;
  
  cbuff_init(&uart_driver[ctx].rx_cbuff, uart_driver[ctx].rx_mem, uart_driver[ctx].rx_mem_size);
  uart_driver[ctx].rx_ore_flag=0;
  uart_driver[ctx].rx_full_flag=0;
  
  uart_driver[ctx].rx_it_callback=rx_it_callback;
  
  /* Create UART sync objects */
  //if(uart_driver[ctx].tx_sem==NULL) {vSemaphoreCreateBinary(uart_driver[ctx].tx_sem);}
  //xSemaphoreTake(uart_driver[ctx].tx_sem, 0);
  
  if(uart_driver[ctx].rx_sem==NULL) {vSemaphoreCreateBinary(uart_driver[ctx].rx_sem);}
  xSemaphoreTake(uart_driver[ctx].rx_sem, 0);
  
  uart_driver[ctx].wait_rx_len=0;
    
  /* Enable AFIO clock */
  afio_clock(ctx, true);
  
  /* UART clock enable */
  uart_clock(ctx, true);
  
  /* UART remap if enabled */
  uart_remap(ctx); 
    
  /* Configure UART pins  */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = get_uart_tx_pin(ctx);
  GPIO_Init(get_uart_tx_port(ctx), &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_PuPd  = get_uart_rx_pin_pupd(ctx);
  GPIO_InitStructure.GPIO_Pin = get_uart_rx_pin(ctx);
  GPIO_Init(get_uart_rx_port(ctx), &GPIO_InitStructure);
    
  USART_InitStructure.USART_BaudRate = baud_rate;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No ;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(get_uart_pherph_from_ctx(ctx), &USART_InitStructure);
    
  NVIC_InitStructure.NVIC_IRQChannel = get_uart_irqn(ctx);
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = EXTERNAL_UART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
 
  USART_ClearITPendingBit(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE);
  USART_ClearITPendingBit(get_uart_pherph_from_ctx(ctx), USART_IT_TXE);
  USART_ClearITPendingBit(get_uart_pherph_from_ctx(ctx), USART_IT_TC);

  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, ENABLE);
  USART_Cmd(get_uart_pherph_from_ctx(ctx), ENABLE);
 
  uart_driver[ctx].is_init=1;
}

void external_uart_deinit(const uint8_t ctx)
{
  GPIO_InitTypeDef GPIO_InitStructure;
  
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return;
  
  uart_driver[ctx].is_init=0;
  
  /* Disable interrupts */
  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, DISABLE);
  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_TXE, DISABLE);
  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_TC, DISABLE);
  
  /* Disable the UART */
  USART_Cmd(get_uart_pherph_from_ctx(ctx), DISABLE);
  
  /* UART clock disable */
  uart_clock(ctx, false);
  
  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  
  GPIO_InitStructure.GPIO_Pin = get_uart_tx_pin(ctx);
  GPIO_Init(get_uart_tx_port(ctx), &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = get_uart_rx_pin(ctx);
  GPIO_Init(get_uart_rx_port(ctx), &GPIO_InitStructure);
}

void external_uart_write(const uint8_t ctx, const void* src, const uint16_t len)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return;
  
  USART_ClearFlag(get_uart_pherph_from_ctx(ctx), USART_FLAG_TC);
  
  for(uint16_t i=0; i<len; i++)
  {
    USART_SendData(get_uart_pherph_from_ctx(ctx), *((uint8_t*)src+i));
    while (USART_GetFlagStatus(get_uart_pherph_from_ctx(ctx), USART_FLAG_TC) == RESET);
  }
}

int16_t external_uart_read(const uint8_t ctx, void* const dst, const uint16_t len)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return -2;
  
  uint16_t read_len;
  
  if(uart_driver[ctx].rx_ore_flag || uart_driver[ctx].rx_full_flag)
  {
    USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, DISABLE);
    clear_cbuff_from_reader(&uart_driver[ctx].rx_cbuff);
    uart_driver[ctx].rx_ore_flag=0;
    uart_driver[ctx].rx_full_flag=0;
    USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, ENABLE);
    return -1;
  }
  
  read_len=filled_cbuff_len(&uart_driver[ctx].rx_cbuff);
  
  if(!read_len) return 0;
  
  if(read_len>len) read_len=len;
  
  read_from_cbuff(&uart_driver[ctx].rx_cbuff, dst, read_len);
  return (int16_t)read_len;
}

int16_t external_uart_set_wait_sync_len(const uint8_t ctx, const uint16_t len)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return -2;
  
  if(uart_driver[ctx].rx_cbuff.buff_mask>=len)
  {
    uart_driver[ctx].wait_rx_len=len;
    return (int16_t)len;
  }
  else
  {
    uart_driver[ctx].wait_rx_len=0;
    return -1;
  }
}

void external_uart_flush_rx(const uint8_t ctx)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return;
  
  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, DISABLE);
  clear_cbuff_from_reader(&uart_driver[ctx].rx_cbuff);
  uart_driver[ctx].rx_ore_flag=0;
  uart_driver[ctx].rx_full_flag=0;
  uart_driver[ctx].wait_rx_len=0;
  xSemaphoreTake(uart_driver[ctx].rx_sem, 0);
  USART_ITConfig(get_uart_pherph_from_ctx(ctx), USART_IT_RXNE, ENABLE);
}

int16_t external_uart_txrx(const uint8_t ctx, const void* const tx, const uint16_t tx_size, void* const rx, const uint16_t rx_size, const uint32_t timeout, const bool is_rx_flush)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return -2;
  
  if(rx!=NULL)
  {
    if(is_rx_flush) external_uart_flush_rx(ctx);
    external_uart_set_wait_sync_len(ctx, rx_size);
  }
  
  if(tx!=NULL)
  {
    external_uart_write(ctx, tx, tx_size);
  }
  
  if(rx!=NULL)
  {
    external_uart_wait_sync(ctx, timeout);
    return external_uart_read(ctx, rx, rx_size);
  }
  else
  {
    return 0;
  }
}

uint8_t external_uart_wait_sync(const uint8_t ctx, const uint32_t time_mS)
{
  if(ctx>=EXTERNAL_UART_CTX_COUNT) return 1;
  
  if(uart_driver[ctx].wait_rx_len && filled_cbuff_len(&uart_driver[ctx].rx_cbuff)==uart_driver[ctx].wait_rx_len)
  {
    xSemaphoreTake(uart_driver[ctx].rx_sem, 0);
    return 1;
  }
  else if(xSemaphoreTake(uart_driver[ctx].rx_sem, time_mS))
  {
    return 1;
  }
  else
  {
    return 0;
  }
}
  
#pragma optimize=speed
static void external_uart_it_handler(const uint8_t ctx)
{
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  
  USART_TypeDef* const USARTx=get_uart_pherph_from_ctx(ctx);
  
  if(USART_GetITStatus(USARTx, USART_IT_RXNE) != RESET)
  {
    volatile uint8_t rx_val;
    
    rx_val=USART_ReceiveData(USARTx);
    
    if(free_cbuff_len(&uart_driver[ctx].rx_cbuff)>0)//     
    {
      write_byte_to_cbuff(&uart_driver[ctx].rx_cbuff, rx_val);
      
      if(uart_driver[ctx].rx_it_callback!=NULL)
      {
        if(uart_driver[ctx].rx_it_callback(&uart_driver[ctx].rx_cbuff))
        {
          xSemaphoreGiveFromISR(uart_driver[ctx].rx_sem, &xHigherPriorityTaskWoken);
        }
      }
      
      if(uart_driver[ctx].wait_rx_len && filled_cbuff_len(&uart_driver[ctx].rx_cbuff)==uart_driver[ctx].wait_rx_len)
      {
        xSemaphoreGiveFromISR(uart_driver[ctx].rx_sem, &xHigherPriorityTaskWoken);
      }
    }
    else// 
    {
      if(!uart_driver[ctx].rx_full_flag)
      {
        uart_driver[ctx].rx_full_flag=1;
        
        xSemaphoreGiveFromISR(uart_driver[ctx].rx_sem, &xHigherPriorityTaskWoken);
      }
    }
  }
  else if(USART_GetITStatus(USARTx, USART_IT_ORE) != RESET ||  USART_GetFlagStatus(USARTx, USART_FLAG_ORE) != RESET) // 
  {
    USART_ReceiveData(USARTx); 
    uart_driver[ctx].rx_ore_flag=1;
  }
  
  if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
}

#ifdef __cplusplus
extern "C" {
#endif
  
#pragma optimize=speed
void EXTERNAL_UART1_IRQHandler(void)
{
  external_uart_it_handler(EXTERNAL_UART_CTX1);
}

#if (EXTERNAL_UART_CTX_COUNT>1)
#pragma optimize=speed
void EXTERNAL_UART2_IRQHandler(void)
{
  external_uart_it_handler(EXTERNAL_UART_CTX2);
}
#endif //(EXTERNAL_UART_CTX_COUNT>1)
  
#if (EXTERNAL_UART_CTX_COUNT>2)
#error
#endif //(EXTERNAL_UART_CTX_COUNT>2)
  
#ifdef __cplusplus
}
#endif 
