/**
******************************************************************************
* File Name          :
* Description        :
*
*
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "stm32f4xx.h"
#include "lin_uart.h"

static lin_uart_driver_t lin_uart={0};

void lin_uart_init(uint32_t baud_rate)
{
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;

  if(!baud_rate) return;

  lin_uart.rx_ore_flag=0;
  lin_uart.rx_pos=0xff;

  /* Enable AFIO clock */
  __LIN_AFIO_CLOCK_ENABLE();

  /* UART clock enable */
  __LIN_UART_CLOCK_ENABLE();

  /* UART remap if enabled */
  __LIN_UART_REMAP();

  /* Configure UART pins  */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = LIN_UART_TX_PIN;
  GPIO_Init(LIN_UART_TX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_UP;
  GPIO_InitStructure.GPIO_Pin = LIN_UART_RX_PIN;
  GPIO_Init(LIN_UART_RX_PORT, &GPIO_InitStructure);

  USART_InitStructure.USART_BaudRate = baud_rate;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No ;
  USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(LIN_UART, &USART_InitStructure);

  USART_LINBreakDetectLengthConfig(LIN_UART, USART_LINBreakDetectLength_11b);

  NVIC_InitStructure.NVIC_IRQChannel = LIN_UART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = LIN_UART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  USART_ClearITPendingBit(LIN_UART, USART_IT_LBD | USART_IT_RXNE | USART_IT_RXNE);
  USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(LIN_UART, USART_IT_TC, DISABLE);
  USART_ITConfig(LIN_UART, USART_IT_LBD, ENABLE);
  USART_LINCmd(LIN_UART, ENABLE);
  USART_Cmd(LIN_UART, ENABLE);

  lin_uart.is_init=true;
}

void lin_uart_deinit(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;

  lin_uart.is_init=false;

  /* Disable interrupts */
  USART_ITConfig(LIN_UART, USART_IT_LBD, DISABLE);
  USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(LIN_UART, USART_IT_TC, DISABLE);

  /* Disable the LIN mode */
  USART_LINCmd(LIN_UART, DISABLE);

  /* Disable the UART */
  USART_Cmd(LIN_UART, DISABLE);

  /* UART clock disable */
  __LIN_UART_CLOCK_DISABLE();

  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_Pin = LIN_UART_RX_PIN;
  GPIO_Init(LIN_UART_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = LIN_UART_TX_PIN;
  GPIO_Init(LIN_UART_TX_PORT, &GPIO_InitStructure);

  lin_uart.rx_ore_flag=0;
  lin_uart.rx_pos=0xff;
}

void lin_uart_send_break(void)
{
  if(!lin_uart.is_init) return;

  USART_SendBreak(LIN_UART);
}

void lin_uart_write(const uint8_t *src, uint8_t len)
{
  if(!lin_uart.is_init) return;

  if(!len || len>sizeof(lin_uart.tx_buff)) return;

  USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);

  lin_uart.tx_pos=0;
  lin_uart.tx_len=len;
  memcpy(lin_uart.tx_buff, src, lin_uart.tx_len);

  USART_SendData(LIN_UART, lin_uart.tx_buff[lin_uart.tx_pos]);

  USART_ITConfig(LIN_UART, USART_IT_TC, ENABLE);
}

//lin_uart_write почему то шлет len+1 байт в режиме мастера, нужно разбираться
void lin_uart_write_2(const uint8_t *src, uint8_t len)
{
  if(!lin_uart.is_init) return;

  if(len<1 || len>sizeof(lin_uart.tx_buff)) return;

  USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);

  lin_uart.tx_pos=0;
  lin_uart.tx_len=len;
  memcpy(lin_uart.tx_buff, src, lin_uart.tx_len);
  lin_uart.tx_len--;//!!!

  USART_SendData(LIN_UART, lin_uart.tx_buff[lin_uart.tx_pos]);

  USART_ITConfig(LIN_UART, USART_IT_TC, ENABLE);
}

bool is_lin_uart_write_busy(void)
{
  if(!lin_uart.is_init) return false;

  //если идет переадча Break или USART_IT_TC разрешено, то идет передача
  static const uint16_t IT_MASK=((uint16_t)0x001F);

  if(LIN_UART->CR1 & USART_CR1_SBK) return true; //в передаче break

  uint32_t usartreg = 0x00, itpos = 0x00, itmask = 0x00;
  uint32_t usartxbase = 0x00;

  usartxbase = (uint32_t)LIN_UART;

  /* Get the USART register index */
  usartreg = (((uint8_t)USART_IT_TC) >> 0x05);

  /* Get the interrupt position */
  itpos = USART_IT_TC & IT_MASK;
  itmask = (((uint32_t)0x01) << itpos);

  if (usartreg == 0x01) /* The IT is in CR1 register */
  {
    usartxbase += 0x0C;
  }
  else if (usartreg == 0x02) /* The IT is in CR2 register */
  {
    usartxbase += 0x10;
  }
  else /* The IT is in CR3 register */
  {
    usartxbase += 0x14;
  }

  if((*(__IO uint32_t*)usartxbase) & itmask) return true;
  else                                       return false;
}

uint8_t lin_get_pid_from_ident(uint8_t ident)
{
  uint8_t p[2];

  p[0]=0x01&((ident>>0)^(ident>>1)^(ident>>2)^(ident>>4));
  p[1]=0x01&(~((ident>>1)^(ident>>3)^(ident>>4)^(ident>>5)));

  return ( (ident) | (p[1]<<7) | ( p[0]<<6) );
}

uint8_t lin_check_parity(uint8_t pid)
{
  if(lin_get_pid_from_ident(0x3F&pid)==pid) return 1;
  else                                      return 0;
}

uint8_t lin_calc_checksum(uint8_t ident, uint8_t is_enhanced, const uint8_t* data, uint8_t dlen)
{
  uint16_t calc = 0;

  //Frame identifiers 60 (0x3C) to 61 (0x3D) shall always use classic checksum
  if(is_enhanced && !(ident >= 0x3C && ident <= 0x3D))
  {
    calc = lin_get_pid_from_ident(ident);
  }

  for(uint8_t i = 0; i < dlen; i++)
  {
    calc = calc + data[i];
  }

  if (calc > 0xFF)
  {
    calc -= 0xFF;
  }

  return (uint8_t)(0xFF - calc);
}

#ifdef __cplusplus
extern "C" {
#endif

#pragma optimize=speed
  void LIN_UART_IRQHandler(void)
  {
    //BaseType_t xHigherPriorityTaskWoken = pdFALSE;

    if(USART_GetITStatus(LIN_UART, USART_IT_LBD) != RESET)
    {
      USART_ClearITPendingBit(LIN_UART, USART_IT_LBD);
      USART_ReceiveData(LIN_UART);
      USART_ITConfig(LIN_UART, USART_IT_RXNE, ENABLE);
      lin_uart.rx_pos=0;
    }

    if(USART_GetITStatus(LIN_UART, USART_IT_TC) != RESET)
    {
      if(lin_uart.tx_pos>=lin_uart.tx_len)
      {//все данные отправлены
        USART_ITConfig(LIN_UART, USART_IT_TC, DISABLE);
        USART_ReceiveData(LIN_UART);
        USART_ITConfig(LIN_UART, USART_IT_RXNE, ENABLE);
        lin_uart.rx_pos=0xff;//не принимаем до получения LBD
      }
      else
      {
        //ошибка? при работе мастером отправляется лишний байт. Нужно проверить в slave.
        USART_SendData(LIN_UART, lin_uart.tx_buff[++lin_uart.tx_pos]);
      }
    }
    else if(USART_GetITStatus(LIN_UART, USART_IT_RXNE) != RESET)
    {
      uint8_t rx_val=USART_ReceiveData(LIN_UART);

      if((lin_uart.rx_pos>=sizeof(lin_uart.rx_buff)) || (lin_uart.rx_pos==0 && rx_val!=0x55) || (lin_uart.rx_pos==1 && !lin_check_parity(rx_val)))
      {
        lin_uart.rx_pos=0xff;//не принимаем до получения LBD
        USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);
      }
      else
      {
        lin_uart.rx_buff[lin_uart.rx_pos++]=rx_val;

        if(lin_uart.rx_pos==2)
        {
          lin_rx_callback(lin_uart.rx_buff, lin_uart.rx_pos);
        }

        if(lin_uart.rx_pos>=2)
        {
          lin_uart.rx_pos=0xff;//не принимаем до получения LBD
          USART_ITConfig(LIN_UART, USART_IT_RXNE, DISABLE);
        }
      }
    }
    else if((USART_GetITStatus(LIN_UART, USART_IT_ORE) != RESET) ||  (USART_GetFlagStatus(LIN_UART, USART_FLAG_ORE) != RESET))
    {
      USART_ReceiveData(LIN_UART);
      lin_uart.rx_ore_flag=1;
    }

    //if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
  }
#ifdef __cplusplus
}
#endif
