/**
******************************************************************************
* File Name          : ll_rs_driver.c
* Description        : 
*                      
*                      
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include "ll_rs_driver.h"

/* Defines -------------------------------------------------------------------*/

/* Global variables-----------------------------------------------------------*/

/* Private constants ---------------------------------------------------------*/
static const USART_InitTypeDef USART_InitStructure_def = 
{
  .USART_BaudRate = 115200,
  .USART_WordLength = USART_WordLength_8b,
  .USART_StopBits = USART_StopBits_1,
  .USART_Parity = USART_Parity_No,
  .USART_Mode = USART_Mode_Rx | USART_Mode_Tx,
  .USART_HardwareFlowControl = USART_HardwareFlowControl_None,
};

/* Private variables ---------------------------------------------------------*/

/* Public functions ----------------------------------------------------------*/
static bool is_rede_pins_present(const ll_uart_hw_t* const hw)
{
  if(hw->re_port != NULL && hw->de_port != NULL) {return true;}
  else                                           {return false;}
}

static void set_z_mode(const ll_uart_hw_t* const hw)
{
  if(is_rede_pins_present(hw))
  {
    GPIO_SetBits(hw->re_port, hw->re_pin);
    GPIO_ResetBits(hw->de_port, hw->de_pin);
  }
}

static void set_out_mode(const ll_uart_hw_t* const hw)
{
  if(is_rede_pins_present(hw))
  {
    GPIO_SetBits(hw->de_port, hw->de_pin);
    GPIO_SetBits(hw->re_port, hw->re_pin);
  }
}

static void set_in_mode(const ll_uart_hw_t* const hw)
{
  if(is_rede_pins_present(hw))
  {
    GPIO_ResetBits(hw->de_port, hw->de_pin);
    GPIO_ResetBits(hw->re_port, hw->re_pin);
  }
}

static void stop_txrx_it(ll_uart_driver_t* const driver)
{
  BaseType_t xHigherPriorityTaskWoken = pdFALSE;
  
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_TXE,DISABLE); 
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC,DISABLE);
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE,DISABLE);
  driver->uart_state=IDLE;
  
  xSemaphoreGiveFromISR(driver->txrx_end_sem, &xHigherPriorityTaskWoken);
  
  if(xHigherPriorityTaskWoken == pdTRUE) {portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);}
}

void ll_rs_init(ll_uart_driver_t* const driver)
{
  GPIO_InitTypeDef GPIO_InitStructure; 
  
  //- Create USART sync objects 
  if(driver->txrx_end_sem==NULL) {vSemaphoreCreateBinary(driver->txrx_end_sem);}
  //xSemaphoreTake(driver->txrx_end_sem, 0);
  
  if(driver->hw->clock) {driver->hw->clock(true);}
  if(driver->hw->remap) {driver->hw->remap();}
  
  /* Configure RS485_RE/DE as push-pull and init */
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  if(is_rede_pins_present(driver->hw))
  {
    GPIO_InitStructure.GPIO_Pin = driver->hw->re_pin;
    GPIO_Init(driver->hw->re_port, &GPIO_InitStructure);
    GPIO_InitStructure.GPIO_Pin = driver->hw->de_pin;
    GPIO_Init(driver->hw->de_port, &GPIO_InitStructure);
    set_z_mode(driver->hw); //   Z-state
  }
  
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_25MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = driver->hw->tx_pin_otype;
  
  /* Configure USART Rx as pull-up input */
  GPIO_InitStructure.GPIO_PuPd  = driver->hw->rx_pin_pupd;
  GPIO_InitStructure.GPIO_Pin = driver->hw->rx_pin;
  GPIO_Init(driver->hw->rx_port, &GPIO_InitStructure);
  
  /* Configure USART Tx as alternate function push-pull */
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = driver->hw->tx_pin;
  GPIO_Init(driver->hw->tx_port, &GPIO_InitStructure);
  
  USART_DeInit(driver->hw->uart_pherph); //     USART
  
  USART_InitTypeDef USART_InitStructure; 
  memcpy(&USART_InitStructure, &USART_InitStructure_def, sizeof(USART_InitStructure));
  USART_Init(driver->hw->uart_pherph, &USART_InitStructure);
  
  //- NVIC
  NVIC_InitTypeDef NVIC_InitStructure; 
  NVIC_InitStructure.NVIC_IRQChannel = driver->hw->irqn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = driver->hw->irq_prio;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  // State
  driver->uart_state = IDLE; //  no interrupt pending
  USART_ReceiveData(driver->hw->uart_pherph);
  USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_RXNE);
  USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TXE);
  USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TC);
  //USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, ENABLE);  
  
  USART_Cmd(driver->hw->uart_pherph, ENABLE);
}

void ll_rs_deinit(ll_uart_driver_t* const driver)
{
  GPIO_InitTypeDef GPIO_InitStructure; 
  
  //- NVIC
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_TXE, DISABLE);  
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, DISABLE);  
  
  set_z_mode(driver->hw);
  
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  /* Configure USART Rx and Tx as analog input */
  GPIO_InitStructure.GPIO_Pin = driver->hw->rx_pin;
  GPIO_Init(driver->hw->rx_port, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = driver->hw->tx_pin;
  GPIO_Init(driver->hw->tx_port, &GPIO_InitStructure);
  
  if(driver->hw->clock) {driver->hw->clock(false);} //  USART
  USART_DeInit(driver->hw->uart_pherph); //     USART
}

void ll_rs_set_param(ll_uart_driver_t* const driver, const uint32_t baudrate, uint32_t timeout, const char *param_string)
{
  ll_rs_set_ext_param(driver, baudrate, timeout, param_string, NULL);
}

void ll_rs_set_ext_param(ll_uart_driver_t* const driver, const uint32_t baudrate, uint32_t timeout, const char *param_string, bool (*data_recv_it_callback)(const uint8_t* data, const uint16_t len))
{
  if(baudrate)
  {
    driver->data_recv_it_callback = data_recv_it_callback;
    
    const char* param = strstr(param_string, "--stopbyte ");
    
    if(param != NULL)
    {
      driver->enable_stop_byte = 1;
      
      driver->stop_byte = strtol(param+sizeof("--stopbyte ")-1, NULL, 16);
    }
    else
    {
      driver->enable_stop_byte = 0;
    }
    
    driver->echo_disable_option = 0;
    
    if(strstr(param_string, "--echo disable") != NULL)
    {
      driver->echo_disable_option = 1;
    }
    
    if(strstr(param_string, "--assync_mode") != NULL)
    {
      set_in_mode(driver->hw);
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, DISABLE); // disable Transmition Complete IT
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_TXE, DISABLE);
      driver->uart_state = IDLE;
      driver->rx_num = 0;
      
      USART_ReceiveData(driver->hw->uart_pherph);
      USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_RXNE);
      USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TXE);
      USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TC);
      
      driver->assync_mode = 1;
    }
    else
    {
      driver->assync_mode = 0;
    }
    
    if(USART_GetFlagStatus(driver->hw->uart_pherph, USART_FLAG_ORE))
    {
      USART_ReceiveData(driver->hw->uart_pherph);
    }
    
    USART_InitTypeDef USART_InitStructure;
    memcpy(&USART_InitStructure, &USART_InitStructure_def, sizeof(USART_InitStructure));
    USART_InitStructure.USART_BaudRate = baudrate;
    
    if(strstr(param_string, "--parity even") != NULL)
    {
      USART_InitStructure.USART_Parity = USART_Parity_Even;
      USART_InitStructure.USART_WordLength = USART_WordLength_9b;
    }
    else if(strstr(param_string, "--parity odd") != NULL)
    {
      USART_InitStructure.USART_Parity = USART_Parity_Odd;
      USART_InitStructure.USART_WordLength = USART_WordLength_9b;
    }
    else
    {
      USART_InitStructure.USART_Parity = USART_Parity_No;
      USART_InitStructure.USART_WordLength = USART_WordLength_8b;
    }
    
    if(strstr(param_string, "--stopbits 1.5") != NULL)
    {
      USART_InitStructure.USART_StopBits = USART_StopBits_1_5;
    }
    else if(strstr(param_string, "--stopbits 2") != NULL)
    {
      USART_InitStructure.USART_StopBits = USART_StopBits_2;
    }
    else
    {
      USART_InitStructure.USART_StopBits = USART_StopBits_1;
    }
    
    USART_Init(driver->hw->uart_pherph, &USART_InitStructure);
  }
    
  if(timeout)
  {
    driver->txrx_timewait = timeout;
  }
}

void ll_rs_assync_rx(ll_uart_driver_t* const driver, uint8_t *rbuf, const uint16_t rbuf_size)
{
  if(!driver->assync_mode)
  {
    return;
  }
  
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
  
  driver->rx_buf = rbuf;
  driver->rx_cnt = 0;
  driver->rx_num = rbuf_size;
  
  USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, ENABLE);
  
  return;
}

uint16_t ll_rs_assync_get_rx_count(ll_uart_driver_t* const driver)
{
  if(!driver->assync_mode)
  {
    return 0;
  }
  
  return driver->rx_cnt;
}

uint16_t ll_rs_assync_tx(ll_uart_driver_t* const driver, const uint8_t *sbuf, const uint16_t slen)
{
  if(!driver->assync_mode)
  {
    return 0;
  }
  
  USART_ClearFlag(driver->hw->uart_pherph, USART_FLAG_TC);
  
  if(is_rede_pins_present(driver->hw))
  {
    vTaskDelay(driver->hw->de_out_wait); // wait before DE
    set_out_mode(driver->hw);
  }
  
  if(driver->echo_disable_option)
  {
    driver->hw->uart_pherph->CR1 &=~ USART_Mode_Rx;
  }
  
  for(uint16_t i=0; i<slen; i++)
  {
    USART_SendData(driver->hw->uart_pherph, *((uint8_t*)sbuf+i));
    while (USART_GetFlagStatus(driver->hw->uart_pherph, USART_FLAG_TC) == RESET);
  }
  
  if(driver->echo_disable_option)
  {
    driver->hw->uart_pherph->CR1 |= USART_Mode_Rx;
  }
  
  if(is_rede_pins_present(driver->hw))
  {
    set_in_mode(driver->hw);
  }
  
  return slen;
}

uint16_t ll_rs_txrx(ll_uart_driver_t* const driver, const uint8_t *sbuf, const uint16_t slen, uint8_t* const rbuf, uint16_t const rlen)
{
  if(driver->assync_mode)
  {
    return 0;
  }
  
  xSemaphoreTake(driver->txrx_end_sem, 0);
    
  driver->uart_state = IDLE;
  
  if(driver->echo_disable_option)
  {
    //   
    USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
    __IO uint8_t rx_val = USART_ReceiveData(driver->hw->uart_pherph);
    
    //      ,        , 
    //..           
  }
  
  if(slen)
  {
    if(driver->echo_disable_option)
    {
      driver->hw->uart_pherph->CR1 &=~ USART_Mode_Rx;
    }
    
    // Tx - mybeRX
    if(is_rede_pins_present(driver->hw))
    {
      vTaskDelay(driver->hw->de_out_wait); // wait before DE
      set_out_mode(driver->hw);
    }
        
    driver->tx_buf = sbuf;
    driver->tx_cnt = slen-1;
    driver->rx_buf = rbuf;
    driver->rx_cnt = 0;
    driver->rx_num = rlen;
    
    USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TC);
    driver->tx_buf++;
    USART_SendData(driver->hw->uart_pherph, driver->tx_buf[-1]);
    driver->uart_state = TX;
    USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, ENABLE); // enable Transmition Complete IT
    
    if(xSemaphoreTake(driver->txrx_end_sem, driver->txrx_timewait) != pdTRUE)
    { 
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, DISABLE); // disable Transmition Complete IT
      driver->uart_state = IDLE;
      
      return driver->rx_cnt;
    }
    
    // DE!!!
    set_in_mode(driver->hw);
    USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, DISABLE); // disable Transmition Complete IT
    USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
    USART_ITConfig(driver->hw->uart_pherph, USART_IT_TXE, DISABLE);
    driver->uart_state = IDLE;
    
    return driver->rx_cnt;
  }
  else 
  {
    if(driver->echo_disable_option)
    {
      driver->hw->uart_pherph->CR1 |= USART_Mode_Rx;
    }
    
    // Rx only
    ///*
    //warning
    if(driver->rx_num)
    {
      set_in_mode(driver->hw);
      driver->uart_state = RX; 
      
      driver->rx_buf = rbuf;
      driver->rx_cnt = 0;
      driver->rx_num = rlen;
      
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, ENABLE);
      xSemaphoreTake(driver->txrx_end_sem, driver->txrx_timewait);
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, DISABLE);
      
      driver->uart_state = IDLE;
    }
    //*/
  }
  
  set_in_mode(driver->hw);
  
  return driver->rx_cnt;
}

#pragma optimize=speed
void ll_rs_irq_handler(ll_uart_driver_t* const driver)
{
  if(driver->assync_mode)
  {
    __IO uint16_t SR = driver->hw->uart_pherph->SR;
    
    const uint8_t var = USART_ReceiveData(driver->hw->uart_pherph);
    
    if(driver->rx_cnt < driver->rx_num)
    {
      driver->rx_buf[driver->rx_cnt] = var;
      driver->rx_cnt++;
    }
    
    if(driver->data_recv_it_callback != NULL)
    {
      driver->data_recv_it_callback(&var, 1);
    }
    
    return;
  }
  
  const ITStatus tmpstatus = USART_GetITStatus(driver->hw->uart_pherph, USART_IT_TC);
  
  // Check state
  switch(driver->uart_state) 
  {
  case TX:
    USART_ClearITPendingBit(driver->hw->uart_pherph, USART_IT_TC);
    
    if(driver->tx_cnt--)
    {
      USART_SendData(driver->hw->uart_pherph, driver->tx_buf[0]); //send next byte
      driver->tx_buf++;
    }
    else
    {
      driver->tx_cnt = 0;
      set_in_mode(driver->hw);
      USART_ITConfig(driver->hw->uart_pherph, USART_IT_TC, DISABLE); //disable Transmition Complete IT
      driver->uart_state = IDLE;
      
      if(driver->rx_num) 
      {
        set_in_mode(driver->hw);
        driver->uart_state = RX;
        USART_ReceiveData(driver->hw->uart_pherph);
        USART_ITConfig(driver->hw->uart_pherph, USART_IT_RXNE, ENABLE);
        
        if(driver->echo_disable_option)
        {
          driver->hw->uart_pherph->CR1 |= USART_Mode_Rx;
        }
      }
      else
      {
        stop_txrx_it(driver);
        return;
      }
    }
    break;
    
  case RX:
    driver->rx_buf[0] = USART_ReceiveData(driver->hw->uart_pherph);
    driver->rx_buf++;
    driver->rx_cnt++; 
    if(driver->data_recv_it_callback != NULL && driver->data_recv_it_callback(&driver->rx_buf[-driver->rx_cnt], driver->rx_cnt)) {stop_txrx_it(driver);}
    else if(driver->rx_cnt >= driver->rx_num) {stop_txrx_it(driver);}
    else if(driver->enable_stop_byte && driver->stop_byte == driver->rx_buf[-1]) {stop_txrx_it(driver);}
    
    break;
    
  default:
    stop_txrx_it(driver);
  }
}