/**
  ******************************************************************************
  * File Name          : memory_spi.cpp
  * Description        :  SPI   Macronix
  *     
  *                      
  ******************************************************************************
*/

#include "memory_spi.h"
#include "system_config.h"

#ifndef USE_FREERTOS
#include "time_utilities/time_utilities.h"
#endif //USE_FREERTOS

/* Global variables-----------------------------------------------------------*/
#ifdef USE_FREERTOS
SemaphoreHandle_t  CMemory_spi::MEMORY_SPI_Semaphore;
#else
volatile uint8_t   CMemory_spi::MEMORY_SPI_Semaphore;
extern  volatile uint32_t  SysTickReg;
#endif // USE_FREERTOS

/* C++ code-------------------------------------------------------------------*/
// 
CMemory_spi::CMemory_spi(void)
{
  #ifdef USE_FREERTOS
  vSemaphoreCreateBinary( this->MEMORY_SPI_Semaphore );
  #ifdef DEBUG
  vQueueAddToRegistry(this->MEMORY_SPI_Semaphore, "mem_spi");
  #endif //DEBUG
  xSemaphoreTake(MEMORY_SPI_Semaphore, portMAX_DELAY); 
  #else
  MEMORY_SPI_Semaphore = 0;
  #endif // USE_FREERTOS
}

// 
void CMemory_spi::Init(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
  SPI_InitTypeDef   SPI_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  MEMORY_SPI_CLOCK(ENABLE);
  MEMORY_SPI_REMAP(); 
  MEMORY_SPI_Tx_DMAx_CLOCK(ENABLE);
  MEMORY_SPI_Rx_DMAx_CLOCK(ENABLE); 
    
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  
  /* Configure SPIx MOSI as alternate function */
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_MOSI_PIN;
  GPIO_Init(MEMORY_SPI_MOSI_PORT, &GPIO_InitStructure);
  
  /* Configure SPIx SCK as alternate function */
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_SCK_PIN;
  GPIO_Init(MEMORY_SPI_SCK_PORT, &GPIO_InitStructure);
  
  /* Configure SPIx MISO as alternate function */
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_MISO_PIN;
  GPIO_Init(MEMORY_SPI_MISO_PORT, &GPIO_InitStructure);
  
  /* Configure CS as push-pull */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_CS_PIN;
  GPIO_Init(MEMORY_SPI_CS_PORT, &GPIO_InitStructure);
  MEMORY_SPI_CS_HI();
  
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_1Edge;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
#if (TARGET_CPU_FREQ_HZ == 96000000)
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_4;//96/4=24MHz
#elif (TARGET_CPU_FREQ_HZ == 60000000)
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_2;//60/2=30MHz
#else
#error
#endif
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_Init(MEMORY_SPI, &SPI_InitStructure);
  
  DMA_InitStructure.DMA_PeripheralBaseAddr=(uint32_t)&MEMORY_SPI->DR;
  DMA_InitStructure.DMA_PeripheralInc=DMA_PeripheralInc_Disable;
  DMA_InitStructure.DMA_PeripheralDataSize=DMA_PeripheralDataSize_Byte;
  DMA_InitStructure.DMA_MemoryDataSize=DMA_MemoryDataSize_Byte;
  DMA_InitStructure.DMA_Mode=DMA_Mode_Normal;  
  DMA_InitStructure.DMA_Priority=DMA_Priority_Low;          
  DMA_InitStructure.DMA_FIFOMode=DMA_FIFOMode_Disable; 
  DMA_InitStructure.DMA_FIFOThreshold=DMA_FIFOThreshold_Full;
  DMA_InitStructure.DMA_MemoryBurst=DMA_MemoryBurst_Single;
  DMA_InitStructure.DMA_PeripheralBurst=DMA_PeripheralBurst_Single;
      
  DMA_DeInit(MEMORY_SPI_Tx_DMAy_Streamx);
  while(DMA_GetCmdStatus(MEMORY_SPI_Tx_DMAy_Streamx) != DISABLE);
  DMA_ITConfig(MEMORY_SPI_Tx_DMAy_Streamx, DMA_IT_TC|DMA_IT_HT|DMA_IT_TE, DISABLE);
  
  DMA_DeInit(MEMORY_SPI_Rx_DMAy_Streamx);
  while(DMA_GetCmdStatus(MEMORY_SPI_Rx_DMAy_Streamx) != DISABLE);
  DMA_ITConfig(MEMORY_SPI_Rx_DMAy_Streamx, DMA_IT_TC|DMA_IT_HT|DMA_IT_TE, DISABLE);
  
  NVIC_InitStructure.NVIC_IRQChannel = MEMORY_SPI_Tx_DMAy_Streamx_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MEMORY_SPI_Tx_DMAy_Streamx_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  NVIC_InitStructure.NVIC_IRQChannel = MEMORY_SPI_Rx_DMAy_Streamx_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MEMORY_SPI_Rx_DMAy_Streamx_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  SPI_I2S_DMACmd(MEMORY_SPI, SPI_I2S_DMAReq_Rx, ENABLE);
  SPI_I2S_DMACmd(MEMORY_SPI, SPI_I2S_DMAReq_Tx, ENABLE);
      
  /* SPIx enable */
  SPI_Cmd(MEMORY_SPI, ENABLE);
}

// 
void CMemory_spi::Deinit(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
    
  DMA_Cmd(MEMORY_SPI_Rx_DMAy_Streamx, DISABLE);
  DMA_ITConfig(MEMORY_SPI_Rx_DMAy_Streamx, DMA_IT_TC|DMA_IT_HT|DMA_IT_TE, DISABLE);
  DMA_ClearITPendingBit(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
  SPI_I2S_DMACmd(MEMORY_SPI, SPI_I2S_DMAReq_Rx, DISABLE);
    
  DMA_Cmd(MEMORY_SPI_Tx_DMAy_Streamx, DISABLE);
  DMA_ITConfig(MEMORY_SPI_Tx_DMAy_Streamx, DMA_IT_TC|DMA_IT_HT|DMA_IT_TE, DISABLE);
  DMA_ClearITPendingBit(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
  SPI_I2S_DMACmd(MEMORY_SPI, SPI_I2S_DMAReq_Tx, DISABLE);
    
  //   SPI
  SPI_I2S_ITConfig(MEMORY_SPI, SPI_I2S_IT_RXNE, DISABLE);
  //  SPI
  SPI_Cmd(MEMORY_SPI, DISABLE);

  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_MOSI_PIN;
  GPIO_Init(MEMORY_SPI_MOSI_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_SCK_PIN;
  GPIO_Init(MEMORY_SPI_SCK_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_MISO_PIN;
  GPIO_Init(MEMORY_SPI_MISO_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = MEMORY_SPI_CS_PIN;
  GPIO_Init(MEMORY_SPI_CS_PORT, &GPIO_InitStructure);
    
  MEMORY_SPI_CLOCK(DISABLE); 
  //MEMORY_SPI_Tx_DMAx_CLOCK(DISABLE);
  //MEMORY_SPI_Rx_DMAx_CLOCK(DISABLE);
}

//  
void CMemory_spi::CS_up(void)
{
  MEMORY_SPI_CS_HI();
}

//   
void CMemory_spi::CS_down(void)
{
  #ifdef USE_FREERTOS
  xSemaphoreTake(MEMORY_SPI_Semaphore, 0);
  #else
  MEMORY_SPI_Semaphore = 0;
  #endif
  MEMORY_SPI_CS_LOW();
}

//  
void CMemory_spi::Send_data(const uint8_t* data, uint16_t len)
{  
  if(!len) return;

  DMA_InitStructure.DMA_Channel=MEMORY_SPI_Tx_DMA_Chanel;
  DMA_InitStructure.DMA_Memory0BaseAddr=(uint32_t)data;
  DMA_InitStructure.DMA_BufferSize=len;
  DMA_InitStructure.DMA_DIR=DMA_DIR_MemoryToPeripheral;
  DMA_InitStructure.DMA_MemoryInc=DMA_MemoryInc_Enable; 
  DMA_Init(MEMORY_SPI_Tx_DMAy_Streamx, &DMA_InitStructure);
  
  SPI_I2S_ReceiveData(MEMORY_SPI);
  DMA_ClearFlag(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
  DMA_ClearFlag(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
  DMA_ITConfig(MEMORY_SPI_Tx_DMAy_Streamx, DMA_IT_TC, ENABLE);
    
  DMA_Cmd(MEMORY_SPI_Tx_DMAy_Streamx, ENABLE);

  #ifdef USE_FREERTOS
  if(!xSemaphoreTake(MEMORY_SPI_Semaphore, MEMORY_SPI_MAX_TX_RX_TIMEOUT_MS))
  {
    RTC_WriteBackupRegister(RTC_BKP_DR4, 0xA5A5);
    NVIC_SystemReset();
  }
  #else
  uint32_t end_wait_time=SysTickReg+MEMORY_SPI_MAX_TX_RX_TIMEOUT_MS-1; 
  while(MEMORY_SPI_Semaphore == 0)
  {
    if(timeAfter(SysTickReg, end_wait_time))
    {
      RTC_WriteBackupRegister(RTC_BKP_DR4, 0xA4A4);
      NVIC_SystemReset();
    }
  }
  MEMORY_SPI_Semaphore = 0;  
  #endif
  
  // DMA_IT_TC spi     , ...
  while(SPI_I2S_GetFlagStatus(MEMORY_SPI, SPI_I2S_FLAG_TXE) == RESET);
  while(SPI_I2S_GetFlagStatus(MEMORY_SPI, SPI_I2S_FLAG_BSY) == SET);
  
  DMA_ITConfig(MEMORY_SPI_Tx_DMAy_Streamx, DMA_IT_TC, DISABLE);
  DMA_Cmd(MEMORY_SPI_Tx_DMAy_Streamx, DISABLE);
}


//  
void CMemory_spi::Send_byte(uint8_t byte)
{
  Send_data(&byte, 1);
}

//     
void CMemory_spi::Get_data(uint8_t* data, uint16_t len)
{  
  static const uint8_t dummy=0;
    
  if(!len) return;

  DMA_InitStructure.DMA_Channel=MEMORY_SPI_Rx_DMA_Chanel;
  DMA_InitStructure.DMA_Memory0BaseAddr=(uint32_t)data;
  DMA_InitStructure.DMA_BufferSize=len;
  DMA_InitStructure.DMA_DIR=DMA_DIR_PeripheralToMemory;
  DMA_InitStructure.DMA_MemoryInc=DMA_MemoryInc_Enable;       
  DMA_Init(MEMORY_SPI_Rx_DMAy_Streamx, &DMA_InitStructure);
  
  SPI_I2S_ReceiveData(MEMORY_SPI);
  DMA_ClearFlag(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
  DMA_ClearFlag(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
  DMA_ITConfig(MEMORY_SPI_Rx_DMAy_Streamx, DMA_IT_TC, ENABLE);
  
  DMA_InitStructure.DMA_Channel=MEMORY_SPI_Tx_DMA_Chanel;
  DMA_InitStructure.DMA_Memory0BaseAddr=(uint32_t)&dummy;
  DMA_InitStructure.DMA_BufferSize=len;
  DMA_InitStructure.DMA_DIR=DMA_DIR_MemoryToPeripheral;
  DMA_InitStructure.DMA_MemoryInc=DMA_MemoryInc_Disable;      
  DMA_Init(MEMORY_SPI_Tx_DMAy_Streamx, &DMA_InitStructure);
  
  DMA_ITConfig(MEMORY_SPI_Tx_DMAy_Streamx, DMA_IT_TC|DMA_IT_HT|DMA_IT_TE, DISABLE);
    
  DMA_Cmd(MEMORY_SPI_Rx_DMAy_Streamx, ENABLE);
  DMA_Cmd(MEMORY_SPI_Tx_DMAy_Streamx, ENABLE);
 
  #ifdef USE_FREERTOS
  if(!xSemaphoreTake(MEMORY_SPI_Semaphore, MEMORY_SPI_MAX_TX_RX_TIMEOUT_MS))
  {
    RTC_WriteBackupRegister(RTC_BKP_DR4, 0xA5A5);
    NVIC_SystemReset();
  }
  #else
  uint32_t end_wait_time=SysTickReg+MEMORY_SPI_MAX_TX_RX_TIMEOUT_MS-1; 
  while(MEMORY_SPI_Semaphore == 0)
  {
    if(timeAfter(SysTickReg, end_wait_time))
    {
      RTC_WriteBackupRegister(RTC_BKP_DR4, 0xA4A4);
      NVIC_SystemReset();
    }
  }
  MEMORY_SPI_Semaphore = 0;  
  #endif
  
  DMA_ITConfig(MEMORY_SPI_Rx_DMAy_Streamx, DMA_IT_TC, DISABLE);
  DMA_Cmd(MEMORY_SPI_Rx_DMAy_Streamx, DISABLE);
  DMA_Cmd(MEMORY_SPI_Tx_DMAy_Streamx, DISABLE); 
}

//   
uint8_t CMemory_spi::Get_byte(void)
{
  uint8_t data;
  Get_data(&data, 1);
  return data;
}

#ifdef __cplusplus
extern "C" {
#endif 
  void MEMORY_SPI_Tx_DMAy_Streamx_IRQHandler(void)
  {
    #ifdef USE_FREERTOS
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    #endif
    if(DMA_GetITStatus(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx) != RESET)
    {
      DMA_ClearITPendingBit(MEMORY_SPI_Tx_DMAy_Streamx, MEMORY_SPI_Tx_DMAy_FLAG_TCx);
      #ifdef USE_FREERTOS
      xSemaphoreGiveFromISR(CMemory_spi::MEMORY_SPI_Semaphore, &xHigherPriorityTaskWoken);
      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
      #else
      CMemory_spi::MEMORY_SPI_Semaphore = 1;
      #endif
    }
    else
    {//    
      for(;;);
    }
  }
  
  void MEMORY_SPI_Rx_DMAy_Streamx_IRQHandler(void)
  {
    #ifdef USE_FREERTOS
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;
    #endif
    if(DMA_GetITStatus(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx) != RESET)
    {
      DMA_ClearITPendingBit(MEMORY_SPI_Rx_DMAy_Streamx, MEMORY_SPI_Rx_DMAy_FLAG_TCx);
      #ifdef USE_FREERTOS
      xSemaphoreGiveFromISR(CMemory_spi::MEMORY_SPI_Semaphore, &xHigherPriorityTaskWoken);
      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
      #else
      CMemory_spi::MEMORY_SPI_Semaphore = 1;
      #endif
    }
    else
    {//    
      for(;;);
    }
  }
  
#ifdef __cplusplus
}
#endif 
