/**
******************************************************************************
* File Name          : modem_uart.c
* Description        : modem uart for stm32f4xx
*
*
******************************************************************************
*/
/* Includes ------------------------------------------------------------------*/
#include "modem_uart.h"

#if defined(USE_MODEM_UART_DMA_RX)
#warning USE_MODEM_UART_DMA_RX
#warning В режиме DMA RTS не проверен
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX)
#warning USE_MODEM_UART_DMA_TX
#endif //defined(USE_MODEM_UART_DMA_TX)

#if defined(USE_MODEM_UART_DMA_RX)
#pragma data_alignment=4
static uint8_t modem_dma_rx_buff[MODEM_DMA_RX_MEM_SIZE];
static uint16_t dma_rx_it_old_pos;
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX)
static DMA_InitTypeDef DMA_TxStructure;
#endif //defined(USE_MODEM_UART_DMA_TX)

#pragma data_alignment=4
static uint8_t modem_rx_mem[MODEM_RX_MEM_SIZE];
static modem_uart_rx_t modem_uart_rx={0};
static SemaphoreHandle_t modem_tx_bin_sem=NULL;
static SemaphoreHandle_t modem_rx_bin_sem=NULL;

#if !defined(USE_MODEM_UART_DMA_TX)
static const  uint8_t* modem_tx_data;
static const uint8_t* modem_tx_data_end;
#endif //!defined(USE_MODEM_UART_DMA_TX)

#if EN_MODEM_DEBUG > 0
#warning modem debug is on
__IO uint8_t modem_tx_log_out_en=1;
__IO uint8_t modem_rx_log_out_en=1;
#endif

void ModemUartInit(uint32_t baudrate, uint8_t hw_flow_control)
{
  USART_InitTypeDef USART_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;

  cbuff_init(&modem_uart_rx.cbuff, modem_rx_mem, sizeof(modem_rx_mem));
  modem_uart_rx.sync_obj=&modem_rx_bin_sem;
  modem_uart_rx.ore_flag=0;
  modem_uart_rx.full_flag=0;

  /* Create UART sync objects */
  if(modem_tx_bin_sem==NULL) {vSemaphoreCreateBinary(modem_tx_bin_sem);}
  xSemaphoreTake(modem_tx_bin_sem, 0);
  if(modem_rx_bin_sem==NULL)  {vSemaphoreCreateBinary(modem_rx_bin_sem);}
  xSemaphoreTake(modem_rx_bin_sem, 0);

  /* UART clock enable */
  __MODEM_UART_CLOCK_ENABLE();

  /* UART remap if enabled */
  __MODEM_UART_REMAP();

  /* Configure UART pins  */
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = MODEM_UART_TX_PIN_MODE;
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_TX_PIN;
  GPIO_Init(MODEM_UART_TX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_RX_PIN;
  GPIO_Init(MODEM_UART_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_CTS_PIN;
  GPIO_Init(MODEM_UART_CTS_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_OType = MODEM_UART_SOFT_RTS_PIN_MODE;
  GPIO_InitStructure.GPIO_Pin = MODEM_UART_SOFT_RTS_PIN;
  GPIO_Init(MODEM_UART_SOFT_RTS_PORT, &GPIO_InitStructure);
  __MODEM_UART_SOFT_RTS_PIN_HI();

  USART_InitStructure.USART_BaudRate = baudrate;//MODEM_UART_BAUD_RATE;
  USART_InitStructure.USART_WordLength = USART_WordLength_8b;
  USART_InitStructure.USART_StopBits = USART_StopBits_1;
  USART_InitStructure.USART_Parity = USART_Parity_No ;
  if(hw_flow_control) {USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_CTS;}
  else                {USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;}
  USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
  USART_Init(MODEM_UART, &USART_InitStructure);

#if defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)
  __MODEM_UART_TxRx_DMAx_CLOCK_ENABLE();
#endif //defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_RX)
  /* Configure the DMA */
  DMA_InitTypeDef DMA_RxStructure;
  DMA_DeInit(MODEM_UART_Rx_DMAy_Streamx);

  DMA_RxStructure.DMA_Channel=MODEM_UART_Rx_DMA_Chanel;
  DMA_RxStructure.DMA_PeripheralBaseAddr=(uint32_t)&(MODEM_UART->DR);
  DMA_RxStructure.DMA_Memory0BaseAddr=(uint32_t)&modem_dma_rx_buff[0];
  DMA_RxStructure.DMA_DIR=DMA_DIR_PeripheralToMemory;;
  DMA_RxStructure.DMA_BufferSize=sizeof(modem_dma_rx_buff);
  DMA_RxStructure.DMA_PeripheralInc=DMA_PeripheralInc_Disable;
  DMA_RxStructure.DMA_MemoryInc=DMA_MemoryInc_Enable;;
  DMA_RxStructure.DMA_PeripheralDataSize=DMA_PeripheralDataSize_Byte;
  DMA_RxStructure.DMA_MemoryDataSize=DMA_MemoryDataSize_Byte;
  DMA_RxStructure.DMA_Mode=DMA_Mode_Circular;
  DMA_RxStructure.DMA_Priority=DMA_Priority_High;
  DMA_RxStructure.DMA_FIFOMode=DMA_FIFOMode_Disable;
  DMA_RxStructure.DMA_FIFOThreshold=DMA_FIFOThreshold_Full;
  DMA_RxStructure.DMA_MemoryBurst=DMA_MemoryBurst_Single;
  DMA_RxStructure.DMA_PeripheralBurst=DMA_PeripheralBurst_Single;
  DMA_Init(MODEM_UART_Rx_DMAy_Streamx, &DMA_RxStructure);

  /* DMA IRQ Channel configuration */
  NVIC_InitStructure.NVIC_IRQChannel = MODEM_UART_Rx_DMAy_Streamx_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MODEM_UART_Rx_DMAy_Streamx_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; /* Not used as 4 bits are used for the pre-emption priority. */
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  dma_rx_it_old_pos=0;

  DMA_ClearITPendingBit(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_HTx);

  DMA_ITConfig(MODEM_UART_Rx_DMAy_Streamx, DMA_IT_HT, ENABLE);
  DMA_ITConfig(MODEM_UART_Rx_DMAy_Streamx, DMA_IT_TC, ENABLE);

  /* Enable USARTy DMA Rx request */
  USART_DMACmd(MODEM_UART, USART_DMAReq_Rx, ENABLE);
  DMA_Cmd(MODEM_UART_Rx_DMAy_Streamx, ENABLE);
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX)
  /* Configure the DMA */
  DMA_DeInit(MODEM_UART_Tx_DMAy_Streamx);

  DMA_TxStructure.DMA_Channel=MODEM_UART_Tx_DMA_Chanel;
  DMA_TxStructure.DMA_PeripheralBaseAddr=(uint32_t)&(MODEM_UART->DR);
  DMA_TxStructure.DMA_Memory0BaseAddr=0;
  DMA_TxStructure.DMA_DIR=DMA_DIR_MemoryToPeripheral;
  DMA_TxStructure.DMA_BufferSize=0;
  DMA_TxStructure.DMA_PeripheralInc=DMA_PeripheralInc_Disable;
  DMA_TxStructure.DMA_MemoryInc=DMA_MemoryInc_Enable;
  DMA_TxStructure.DMA_PeripheralDataSize=DMA_PeripheralDataSize_Byte;
  DMA_TxStructure.DMA_MemoryDataSize=DMA_MemoryDataSize_Byte;
  DMA_TxStructure.DMA_Mode=DMA_Mode_Normal;
  DMA_TxStructure.DMA_Priority=DMA_Priority_Low;
  DMA_TxStructure.DMA_FIFOMode=DMA_FIFOMode_Disable;;
  DMA_TxStructure.DMA_FIFOThreshold=DMA_MemoryBurst_Single;
  DMA_TxStructure.DMA_MemoryBurst=DMA_MemoryBurst_Single;
  DMA_TxStructure.DMA_PeripheralBurst=DMA_PeripheralBurst_Single;
  DMA_Init(MODEM_UART_Tx_DMAy_Streamx, &DMA_TxStructure);

  /* Enable USARTy DMA Tx request */
  USART_DMACmd(MODEM_UART, USART_DMAReq_Tx, ENABLE);

  /* DMA IRQ Channel configuration */
  NVIC_InitStructure.NVIC_IRQChannel = MODEM_UART_Tx_DMAy_Streamx_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MODEM_UART_Tx_DMAy_Streamx_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; /* Not used as 4 bits are used for the pre-emption priority. */
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  DMA_ClearITPendingBit(MODEM_UART_Tx_DMAy_Streamx, MODEM_UART_Tx_DMAy_FLAG_TCx);
  DMA_ClearITPendingBit(MODEM_UART_Tx_DMAy_Streamx, MODEM_UART_Tx_DMAy_FLAG_HTx);

  DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_HT, DISABLE);
  DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_TC, DISABLE);
#endif //defined(USE_MODEM_UART_DMA_TX)

  NVIC_InitStructure.NVIC_IRQChannel = MODEM_UART_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = MODEM_UART_IRQn_PRIO;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0; //Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  USART_ClearITPendingBit(MODEM_UART, USART_IT_RXNE);
  USART_ClearITPendingBit(MODEM_UART, USART_IT_TXE);
  USART_ClearITPendingBit(MODEM_UART, USART_IT_TC);
  USART_ClearITPendingBit(MODEM_UART, USART_IT_IDLE);

#if defined(USE_MODEM_UART_DMA_RX)
  USART_ITConfig(MODEM_UART, USART_IT_IDLE, ENABLE);
#else
  USART_ITConfig(MODEM_UART, USART_IT_RXNE, ENABLE);
#endif //defined(USE_MODEM_UART_DMA_RX)

  USART_Cmd(MODEM_UART, ENABLE);
  __MODEM_UART_SOFT_RTS_PIN_LOW();

#if EN_MODEM_DEBUG > 0
  __PRINTF("MODEM_UART_BUFF_FILLING_TIME_1_2: %u\n", MODEM_UART_BUFF_FILLING_TIME_1_2);
#endif //EN_MODEM_DEBUG > 0

  modem_uart_rx.is_init=1;
}

void ModemUartDeInit(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;

  modem_uart_rx.is_init=0;

  /* Disable interrupts */
  USART_ITConfig(MODEM_UART, USART_IT_RXNE, DISABLE);
  USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
  USART_ITConfig(MODEM_UART, USART_IT_TC, DISABLE);
  USART_ITConfig(MODEM_UART, USART_IT_IDLE, DISABLE);

  /* Disable the UART */
  USART_Cmd(MODEM_UART, DISABLE);

#if defined(USE_MODEM_UART_DMA_RX)
  DMA_ITConfig(MODEM_UART_Rx_DMAy_Streamx, DMA_IT_HT, DISABLE);
  DMA_ITConfig(MODEM_UART_Rx_DMAy_Streamx, DMA_IT_TC, DISABLE);
  DMA_Cmd(MODEM_UART_Rx_DMAy_Streamx, DISABLE);
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX)
  DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_HT, DISABLE);
  DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_TC, DISABLE);
  DMA_Cmd(MODEM_UART_Tx_DMAy_Streamx, DISABLE);
#endif //defined(USE_MODEM_UART_DMA_TX)

#if defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)
  __MODEM_UART_TxRx_DMAx_CLOCK_DISABLE();
#endif //defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)

  /* UART clock disable */
  __MODEM_UART_CLOCK_DISABLE();

  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_TX_PIN;
  GPIO_Init(MODEM_UART_TX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_RX_PIN;
  GPIO_Init(MODEM_UART_RX_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_CTS_PIN;
  GPIO_Init(MODEM_UART_CTS_PORT, &GPIO_InitStructure);

  GPIO_InitStructure.GPIO_Pin = MODEM_UART_SOFT_RTS_PIN;
  GPIO_Init(MODEM_UART_SOFT_RTS_PORT, &GPIO_InitStructure);
}

#define SEND_TIMEOUT_AFTER_LAST_RECV       (50)
#define MAX_TIME_WAIT_AFTER_LAST_RECV      (200)
int16_t WriteToModemUart(const uint8_t *out_data, uint16_t num_of_bytes, uint32_t time_wait)
{
  if(num_of_bytes==0)
  {
    return MODEM_UART_TX_OK;
  }

  if(num_of_bytes==STRING_LEN)
  {
    num_of_bytes=strlen((const char*)out_data);
  }

#if defined(USE_MODEM_UART_DMA_TX)
  DMA_TxStructure.DMA_Memory0BaseAddr=(uint32_t)out_data;
  DMA_TxStructure.DMA_BufferSize=(uint32_t)num_of_bytes;
  DMA_Init(MODEM_UART_Tx_DMAy_Streamx, &DMA_TxStructure);

  DMA_ClearITPendingBit(MODEM_UART_Tx_DMAy_Streamx, MODEM_UART_Tx_DMAy_FLAG_TCx);
  DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_TC, ENABLE);

  xSemaphoreTake(modem_tx_bin_sem, 0);//вычитываем семафор, если в прошлый раз вышли по таймауту, не дождавшись modem_tx_bin_sem
  DMA_Cmd(MODEM_UART_Tx_DMAy_Streamx, ENABLE);

  if(xSemaphoreTake(modem_tx_bin_sem, time_wait))
  {
    DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_TC, DISABLE);
    DMA_Cmd(MODEM_UART_Tx_DMAy_Streamx, DISABLE);

#if EN_MODEM_DEBUG > 0
    if(modem_tx_log_out_en)
    {
      __CHAR_BUFF_PRINTF(out_data, num_of_bytes, "ModemTxLog, len: %hu\n", num_of_bytes);
    }
#endif //EN_GSM_DEBUG

    return MODEM_UART_TX_OK;
  }
  else
  {
    DMA_ITConfig(MODEM_UART_Tx_DMAy_Streamx, DMA_IT_TC, DISABLE);
    DMA_Cmd(MODEM_UART_Tx_DMAy_Streamx, DISABLE);

    __PRINTF("ModemTxLog: timeout error\n")

    return MODEM_UART_TX_TIMEOUT_ERR;
  }
#else
  uint8_t TimeOutCount=MAX_TIME_WAIT_AFTER_LAST_RECV/SEND_TIMEOUT_AFTER_LAST_RECV;

  TimeOutCount=0;//todo: сделать вермя ожидания с учетом time_wait. Пока ждем 1000ms
  while (USART_GetFlagStatus(MODEM_UART, USART_FLAG_TXE) == RESET && TimeOutCount<100)
  {
    vTaskDelay(10);
    TimeOutCount++;
  }
  if(TimeOutCount>=100) return MODEM_UART_TX_TIMEOUT_ERR;

  xSemaphoreTake(modem_tx_bin_sem, 0);//вычитываем семафор, если в прошлый раз вышли по таймауту, не дождавшись modem_tx_bin_sem

  modem_tx_data=out_data;
  modem_tx_data_end=out_data+num_of_bytes-1;

  USART_SendData(MODEM_UART, *modem_tx_data);
  USART_ITConfig(MODEM_UART, USART_IT_TXE, ENABLE);

  if( pdTRUE == xSemaphoreTake(modem_tx_bin_sem, time_wait) )
  {
#if EN_MODEM_DEBUG > 0
    if(modem_tx_log_out_en)
    {
      __CHAR_BUFF_PRINTF(out_data, num_of_bytes, "ModemTxLog, len: %hu\n", num_of_bytes);
    }
#endif //EN_GSM_DEBUG
    return MODEM_UART_TX_OK;
  }
  else
  {
    USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
    return MODEM_UART_TX_TIMEOUT_ERR;
  }
#endif //defined(USE_MODEM_UART_DMA_TX)
}

modem_uart_rx_t* GetModemUartRxStructPtr(void)
{
  return &modem_uart_rx;
}

#ifdef __cplusplus
extern "C" {
#endif

#if defined(USE_MODEM_UART_DMA_RX)
#pragma optimize=speed
  static uint8_t uart_rx_check(uint8_t* is_ovf)
  {
    /* Calculate current position in buffer */
    uint16_t pos=sizeof(modem_dma_rx_buff)-DMA_GetCurrDataCounter(MODEM_UART_Rx_DMAy_Streamx);

    if(pos!=dma_rx_it_old_pos)
    {
      /* Check change in received data */
      if(pos>dma_rx_it_old_pos)
      {
        /* Current position is over previous one */
        /* We are in "linear" mode */
        /* Process data directly by subtracting "pointers" */
        if(free_cbuff_len(&modem_uart_rx.cbuff)>=(pos-dma_rx_it_old_pos))
        {
          write_to_cbuff(&modem_uart_rx.cbuff, &modem_dma_rx_buff[dma_rx_it_old_pos], pos-dma_rx_it_old_pos);
        }
        else
        {
          is_ovf[0]=1;
        }
      }
      else
      {
        /* We are in "overflow" mode */
        /* First process data to the end of buffer */
        if(free_cbuff_len(&modem_uart_rx.cbuff)>=sizeof(modem_dma_rx_buff)-dma_rx_it_old_pos)
        {
          write_to_cbuff(&modem_uart_rx.cbuff, &modem_dma_rx_buff[dma_rx_it_old_pos], sizeof(modem_dma_rx_buff)-dma_rx_it_old_pos);
        }
        else
        {
          is_ovf[0]=1;
        }

        /* Check and continue with beginning of buffer */
        if(pos>0)
        {
          if(free_cbuff_len(&modem_uart_rx.cbuff)>=pos)
          {
            write_to_cbuff(&modem_uart_rx.cbuff, &modem_dma_rx_buff[0], pos);
          }
          else
          {
            is_ovf[0]=1;
          }
        }
      }
    }
    dma_rx_it_old_pos=pos; /* Save current position as old */

    /* Check and manually update if we reached end of buffer */
    if(dma_rx_it_old_pos==sizeof(modem_dma_rx_buff))
    {
      dma_rx_it_old_pos=0;
    }

    if(filled_cbuff_len(&modem_uart_rx.cbuff)>MODEM_RX_MEM_SIZE/4) return 1;
    else                                                           return 0;
  }
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX)
  #pragma optimize=speed
  void MODEM_UART_Tx_DMAy_Streamx_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;

    //GPIO_SetBits(GPIOD, GPIO_Pin_2);
    if(DMA_GetITStatus(MODEM_UART_Tx_DMAy_Streamx, MODEM_UART_Tx_DMAy_FLAG_TCx) != RESET)
    {
      DMA_ClearITPendingBit(MODEM_UART_Tx_DMAy_Streamx, MODEM_UART_Tx_DMAy_FLAG_TCx);
      xSemaphoreGiveFromISR(modem_tx_bin_sem, &xHigherPriorityTaskWoken);
    }

    if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
    //GPIO_ResetBits(GPIOD, GPIO_Pin_2);
  }
#endif //defined(USE_MODEM_UART_DMA_TX)

#if defined(USE_MODEM_UART_DMA_RX)
#pragma optimize=speed
  void MODEM_UART_Rx_DMAy_Streamx_IRQHandler(void)
  {
    //GPIO_SetBits(GPIOD, GPIO_Pin_2);
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;

    if(DMA_GetITStatus(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_TCx) != RESET)
    {
      DMA_ClearITPendingBit(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_TCx);
    }
    else if(DMA_GetITStatus(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_HTx) != RESET)
    {
      DMA_ClearITPendingBit(MODEM_UART_Rx_DMAy_Streamx, MODEM_UART_Rx_DMAy_FLAG_HTx);
    }

    uint8_t is_ovf=0;

    if(uart_rx_check(&is_ovf))
    {
      if(is_ovf)
      {
        clear_cbuff_from_writter(&modem_uart_rx.cbuff);
        __MODEM_UART_SOFT_RTS_PIN_LOW();
      }

      xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);
      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
    }

    if(free_cbuff_len(&modem_uart_rx.cbuff)<=MODEM_RX_BUFF_WM)
    {
      __MODEM_UART_SOFT_RTS_PIN_HI();
    }
  }
#endif //defined(USE_MODEM_UART_DMA_RX)

#pragma optimize=speed
void MODEM_UART_IRQHandler(void)
  {
    BaseType_t xHigherPriorityTaskWoken = pdFALSE;

    if(0){}

#if !defined(USE_MODEM_UART_DMA_TX)
    else if(USART_GetITStatus(MODEM_UART, USART_IT_TXE) != RESET)
    {
      if(modem_tx_data>=modem_tx_data_end)
      {//все данные отправлены
        USART_ITConfig(MODEM_UART, USART_IT_TXE, DISABLE);
        xSemaphoreGiveFromISR(modem_tx_bin_sem, &xHigherPriorityTaskWoken);
        if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
      }
      else
      {
        modem_tx_data++;
        USART_SendData(MODEM_UART, modem_tx_data[0]);
      }
    }
#endif //!defined(USE_MODEM_UART_DMA_TX)

#if defined(USE_MODEM_UART_DMA_RX)
    else if(USART_GetITStatus(MODEM_UART, USART_IT_IDLE) != RESET)
    {
      USART_ReceiveData(MODEM_UART);

      uint8_t is_ovf=0;
      uart_rx_check(&is_ovf);

      if(is_ovf)
      {
        clear_cbuff_from_writter(&modem_uart_rx.cbuff);
        __MODEM_UART_SOFT_RTS_PIN_LOW();
      }
      ///*
      else if(free_cbuff_len(&modem_uart_rx.cbuff)<=MODEM_RX_BUFF_WM)
      {
        __MODEM_UART_SOFT_RTS_PIN_HI();
      }
      //*/

      xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);
      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
    }
#else //USE_MODEM_UART_DMA_RX == 0
    else if(USART_GetITStatus(MODEM_UART, USART_IT_RXNE) != RESET)
    {
      uint8_t rx_val=USART_ReceiveData(MODEM_UART);
      if(modem_uart_rx.full_flag>0)
      {//если было переполение GsmRxBuf, ждём символ '\n'
        if('\n'==rx_val)//check '\n'
        {
          if(modem_uart_rx.full_flag==2)//что было в буфере уже обработалось
          {
            clear_cbuff_from_writter(&modem_uart_rx.cbuff);
          }
        }
      }
      else if(free_cbuff_len(&modem_uart_rx.cbuff)>0)// если есть место для записи
      {
        write_byte_to_cbuff(&modem_uart_rx.cbuff, rx_val);

        if(is_cbuf_ends_with_ref_buff(&modem_uart_rx.cbuff, "\r\n", 2))
        {
          xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);//шлем семафор для форсированного запуска разбора буфера (если идут бинарные данные, то этого делать не нужно (как это отследить?))
        }
        if(free_cbuff_len(&modem_uart_rx.cbuff)<=MODEM_RX_BUFF_WM)
        {
          __MODEM_UART_SOFT_RTS_PIN_HI();
          xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);
        }
      }
      else//буфер заполнен
      {
        modem_uart_rx.full_flag=1;
        xSemaphoreGiveFromISR(modem_rx_bin_sem, &xHigherPriorityTaskWoken);//шлем семафор, чтобы разобрать что есть в буфере
        __MODEM_UART_SOFT_RTS_PIN_LOW();// опускаем RTS для того чтобы смогли отловить '\n'
      }

      if(xHigherPriorityTaskWoken==pdTRUE) portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
    }
    else if(USART_GetITStatus(MODEM_UART, USART_IT_ORE) != RESET ||  USART_GetFlagStatus(MODEM_UART, USART_FLAG_ORE) != RESET) //байт потерян
    {
      USART_ReceiveData(MODEM_UART);
      modem_uart_rx.ore_flag=1;
    }
#endif //defined(USE_MODEM_UART_DMA_RX)
  }
#ifdef __cplusplus
}
#endif
