/**
******************************************************************************
* File Name          : modem_uart.h
* Description        : modem uart for stm32l1xx
*
*
******************************************************************************
*/
#ifndef __MODEM_UART_H
#define __MODEM_UART_H

#ifdef __cplusplus
extern "C" {
#endif

  /* Includes ----------------------------------------------------------------*/
#include "stm32f4xx.h"
#include "string.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "time_utilities/time_utilities.h"
#include "cbuff/cbuff.h"
#include "debug_port.h"

  /* Exported types -----------------------------------------------------------*/
  typedef struct
  {
    uint8_t is_init;
    cbuff_t cbuff;
    SemaphoreHandle_t* sync_obj;
    uint8_t full_flag;
    uint8_t ore_flag;
  }modem_uart_rx_t;

/* Private define ----------------------------------------------------------*/
#define EN_MODEM_DEBUG                                   0

#define STRING_LEN                                       (0xFFFF)

#define MODEM_UART_TX_OK                                 1
#define MODEM_UART_TX_TIMEOUT_ERR                        0

//#define USE_MODEM_UART_DMA_RX
//#define USE_MODEM_UART_DMA_TX

#define MODEM_RX_MEM_SIZE                                (256)
#define MODEM_RX_BUFF_WM                                 (MODEM_RX_MEM_SIZE/4)

#define MODEM_UART                                       USART2
#define MODEM_UART_IRQn                                  USART2_IRQn
#define MODEM_UART_IRQHandler                            USART2_IRQHandler
#define MODEM_UART_IRQn_PRIO                             12 //highest-1 priority with use FreeRTOS API
#define MODEM_UART_BAUD_RATE                             ((uint32_t)(115200))

#define MODEM_UART_TX_PIN                                GPIO_Pin_5
#define MODEM_UART_TX_PORT                               GPIOD
#define MODEM_UART_TX_PIN_MODE                           GPIO_OType_PP

#define MODEM_UART_RX_PIN                                GPIO_Pin_6
#define MODEM_UART_RX_PORT                               GPIOD

#define MODEM_UART_SOFT_RTS_PIN                          GPIO_Pin_4
#define MODEM_UART_SOFT_RTS_PORT                         GPIOD
#define MODEM_UART_SOFT_RTS_PIN_MODE                     GPIO_OType_PP

#define MODEM_UART_CTS_PIN                               GPIO_Pin_3
#define MODEM_UART_CTS_PORT                              GPIOD

#define __MODEM_UART_REMAP()                             {GPIO_PinAFConfig(MODEM_UART_TX_PORT, GPIO_PinSource5, GPIO_AF_USART2); \
                                                          GPIO_PinAFConfig(MODEM_UART_RX_PORT, GPIO_PinSource6, GPIO_AF_USART2); \
                                                          GPIO_PinAFConfig(MODEM_UART_CTS_PORT, GPIO_PinSource3, GPIO_AF_USART2);}

#define __MODEM_UART_CLOCK_ENABLE()                      (RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART2, ENABLE))
#define __MODEM_UART_CLOCK_DISABLE()                     (RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART2, DISABLE))

#define __MODEM_UART_SOFT_RTS_PIN_HI()                   (GPIO_SetBits(MODEM_UART_SOFT_RTS_PORT, MODEM_UART_SOFT_RTS_PIN))
#define __MODEM_UART_SOFT_RTS_PIN_LOW()                  (GPIO_ResetBits(MODEM_UART_SOFT_RTS_PORT, MODEM_UART_SOFT_RTS_PIN))

#if defined(USE_MODEM_UART_DMA_TX)
#define MODEM_UART_Tx_DMA_Chanel                         DMA_Channel_4
#define MODEM_UART_Tx_DMAy_Streamx                       DMA1_Stream6
#define MODEM_UART_Tx_DMAy_FLAG_TCx                      DMA_IT_TCIF6
#define MODEM_UART_Tx_DMAy_FLAG_HTx                      DMA_IT_HTIF6
#define MODEM_UART_Tx_DMAy_Streamx_IRQn                  DMA1_Stream6_IRQn
#define MODEM_UART_Tx_DMAy_Streamx_IRQHandler            DMA1_Stream6_IRQHandler
#define MODEM_UART_Tx_DMAy_Streamx_IRQn_PRIO             15 //lowest priority
#endif //defined(USE_MODEM_UART_DMA_TX)

#if defined(USE_MODEM_UART_DMA_RX)
#define MODEM_DMA_RX_MEM_SIZE                            96
#define MODEM_UART_Rx_DMA_Chanel                         DMA_Channel_4
#define MODEM_UART_Rx_DMAy_Streamx                       DMA1_Stream5
#define MODEM_UART_Rx_DMAy_FLAG_TCx                      DMA_IT_TCIF5
#define MODEM_UART_Rx_DMAy_FLAG_HTx                      DMA_IT_HTIF5
#define MODEM_UART_Rx_DMAy_Streamx_IRQn                  DMA1_Stream5_IRQn
#define MODEM_UART_Rx_DMAy_Streamx_IRQHandler            DMA1_Stream5_IRQHandler
#define MODEM_UART_Rx_DMAy_Streamx_IRQn_PRIO             MODEM_UART_IRQn_PRIO //должны быть равен приоритету IDLE
#endif //defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)
#define __MODEM_UART_TxRx_DMAx_CLOCK_ENABLE()           (RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_DMA1, ENABLE))
#define __MODEM_UART_TxRx_DMAx_CLOCK_DISABLE()          //(RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_DMA1, DISABLE)) //not used
#endif //defined(USE_MODEM_UART_DMA_TX) || defined(USE_MODEM_UART_DMA_RX)

#if defined(USE_MODEM_UART_DMA_RX)
#define MODEM_UART_BUFF_FILLING_TIME_1_2     (10)
#else
#define MODEM_UART_BUFF_FILLING_TIME_1_2     ((uint32_t)(((1000*MODEM_RX_MEM_SIZE)/2)/((MODEM_UART_BAUD_RATE/10))))//время заполнения половины буфера
#endif //defined(USE_MODEM_UART_DMA_RX)

/* Exported functions ----------------------------------------------------- */
void ModemUartInit(uint32_t baudrate, uint8_t hw_flow_control);
void ModemUartDeInit(void);
int16_t WriteToModemUart(const uint8_t *out_data, uint16_t num_of_bytes, uint32_t time_wait);
modem_uart_rx_t* GetModemUartRxStructPtr(void);

/* External variables --------------------------------------------------------*/
#if EN_MODEM_DEBUG > 0
extern __IO uint8_t modem_tx_log_out_en;
extern __IO uint8_t modem_rx_log_out_en;
#endif

#ifdef __cplusplus
}
#endif

#endif /* __MODEM_UART_H */