/*
** ============================================================================
** FILE            y_eeprom.c
** DESCRIPTION     Contains all EEPROM functions,        
**
** TERGET          PIC18LF26K22
** CREATED         Solovyev A.F. Vega-Absolute Ltd.
** DATA            28.08.2013
** ============================================================================
*/

/*
################################################################################
##            INCLUDE
##############################################################################*/
#include <string.h>
#include "nrf_common.h"
#include "sf_BlackBox.h"
#include "y_eeprom.h"
#include "cph_Encrypt.h"
#include "nrf_GlobalService.h"

#define ADD_FOR_4(x)                    ((((x)+(4-1))/4)*4)//    4
#define EEPROM_EMULATION_BUFF_SIZE      (ADD_FOR_4(LENGTH_SETTINGS+sizeof(uint32_t)))////  + 4  crc

/*
################################################################################
##            VARIABLE
##############################################################################*/

static struct eeprom_emulation_struct
{
  uint8_t is_init;
  uint8_t flash_err;
  uint8_t out_of_mem_err;
  uint32_t flash_sector_addr;
  uint32_t flash_sector_size;
  uint16_t flash_sector_number;
  uint8_t buff[EEPROM_EMULATION_BUFF_SIZE];
}eeprom_emulation={.is_init=0, .flash_sector_addr=BSP_NRF_SETTINGS_SECTOR_ADDR, .flash_sector_size=BSP_NRF_SETTINGS_SECTOR_SIZE, .flash_sector_number=BSP_NRF_SETTINGS_SECTOR_NUMBER};


static void eeprom_emulation_init(void)
{
  memcpy(eeprom_emulation.buff, (uint32_t*)eeprom_emulation.flash_sector_addr, sizeof(eeprom_emulation.buff));
  
  eeprom_emulation.flash_err=0;
  eeprom_emulation.out_of_mem_err=0;
  eeprom_emulation.is_init=1;
  
  //LOG("bb_Init: %hhu, %hhu, %hhu, %hhu, 0x%08lX\n", eeprom_emulation.is_init, eeprom_emulation.out_of_mem_err, eeprom_emulation.flash_err, (uint8_t)FLASH_GetStatus(), FLASH->SR);
}

//    flash
static uint8_t bb_Flush(void)
{
  bb_lock_unlock(1);
  FLASH_Unlock();
  
  //   ,  -      flash
  FLASH_ClearFlag(/*FLASH_FLAG_EOP |*/ FLASH_FLAG_OPERR | FLASH_FLAG_WRPERR | FLASH_FLAG_PGAERR | FLASH_FLAG_PGPERR | FLASH_FLAG_PGSERR
#if defined(STM32F413_423xx)
                  | FLASH_FLAG_RDERR
#endif //defined(STM32F413_423xx)
                    );
    
  if(FLASH_EraseSector(eeprom_emulation.flash_sector_number, VoltageRange_3) == FLASH_COMPLETE)
  {
    for(uint32_t i=0; i<sizeof(eeprom_emulation.buff); i+=4)
    {
      if(FLASH_ProgramWord(eeprom_emulation.flash_sector_addr+i, *(uint32_t*)&eeprom_emulation.buff[i]) == FLASH_COMPLETE)
      {
        if((*(uint32_t*)(eeprom_emulation.flash_sector_addr+i)) != (*(uint32_t*)&eeprom_emulation.buff[i])) 
        {
          eeprom_emulation.flash_err=3;
          break;
        }
      }
      else
      {
        eeprom_emulation.flash_err=2;
        break;
      }
    }
  }
  else
  {
    eeprom_emulation.flash_err=1;
  }
  FLASH_Lock();
  bb_lock_unlock(0);
  
  return eeprom_emulation.flash_err;
}

//==============================================================================
//   
void bb_Write(uint8_t* src, uint16_t len)
{
  //LOG("bb_WriteIn: %hhu, %hhu, %hhu, %hhu, 0x%08lX\n", eeprom_emulation.is_init, eeprom_emulation.out_of_mem_err, eeprom_emulation.flash_err, (uint8_t)FLASH_GetStatus(), FLASH->SR);
  
  if(!eeprom_emulation.is_init) eeprom_emulation_init();
  
  if(len<=(sizeof(eeprom_emulation.buff)-sizeof(uint32_t))) 
  {   
    memcpy(eeprom_emulation.buff, src, len);

    uint32_t crc = nrf_bsp_crc(eeprom_emulation.buff, sizeof(eeprom_emulation.buff)-sizeof(uint32_t));  
    memcpy(&eeprom_emulation.buff[sizeof(eeprom_emulation.buff)-sizeof(uint32_t)], &crc, sizeof(uint32_t));
    
    bb_Flush();
  }
  else
  {
    eeprom_emulation.out_of_mem_err=1;
  }
  
  //LOG("bb_WriteOut: %hhu, %hhu, %hhu, %hhu, 0x%08lX\n", eeprom_emulation.is_init, eeprom_emulation.out_of_mem_err, eeprom_emulation.flash_err, (uint8_t)FLASH_GetStatus(), FLASH->SR);
}

//==============================================================================
//   
bool bb_Read(uint8_t* dst, uint16_t len)
{
  if(!eeprom_emulation.is_init) eeprom_emulation_init();
  
  if(len<=(sizeof(eeprom_emulation.buff)-sizeof(uint32_t))) 
  {
    uint32_t crc;
    memcpy(&crc, &eeprom_emulation.buff[sizeof(eeprom_emulation.buff)-sizeof(uint32_t)], sizeof(uint32_t));
    
    if(crc==nrf_bsp_crc(eeprom_emulation.buff, sizeof(eeprom_emulation.buff)-sizeof(uint32_t)))
    {
      memcpy(dst, eeprom_emulation.buff, len);
      
      return true;
    }
  }
  else
  {
    eeprom_emulation.out_of_mem_err=1;
  }
  
  return false;
}

