/**
  ******************************************************************************
  * File Name          : nrf_bsp_driver.cpp
  * Description        : 
  *     
  *                      
  ******************************************************************************
*/

#include "nrf_bsp_driver.h"
#include "system_config.h"
#include "crc_lib/crc_lib.h"

#ifdef USE_FREERTOS
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#endif //USE_FREERTOS

/* Global variables-----------------------------------------------------------*/
#if defined(IS_NRF_SPI_IT_MODE)
#warning   , .   spi   ,        
static volatile struct spi_it_struct
{
  SemaphoreHandle_t  sync_obj;
  uint8_t*           data;
  uint8_t*           data_end;
  uint8_t            is_read;
}spi_it = {.sync_obj=NULL};
#endif //IS_NRF_SPI_IT_MODE


static void (*nrf_irq_callback)(void)=NULL;

static inline void _delay_loops(uint32_t loops) 
{
  asm volatile (
                "1: SUBS %[loops], %[loops], #1 \n"
                  "   BNE 1b \n"
                    : [loops] "+r"(loops)
                      );
}
#define __DELAY_MS(MS, F_CPU) _delay_loops((uint32_t)((double)MS * F_CPU / 3000.0))
#define __DELAY_US(US, F_CPU) _delay_loops((uint32_t)((double)US * F_CPU / 3000000.0))

static void ll_nrf_spi_cs_enable(void)
{
  /* Clear data port bit */
  NRF_SPI_CS_PORT->BSRRH=NRF_SPI_CS_PIN;
  
#if defined(IS_NRF_SPI_IT_MODE)
  volatile uint8_t spi_dr;
  spi_dr=NRF_SPI->DR;//flush
  SPI_I2S_ITConfig(NRF_SPI, SPI_I2S_IT_RXNE, ENABLE);
#endif //IS_NRF_SPI_IT_MODE
}

static void ll_nrf_spi_cs_disable(void)
{
#if defined(IS_NRF_SPI_IT_MODE)
  SPI_I2S_ITConfig(NRF_SPI, SPI_I2S_IT_RXNE, DISABLE);
#endif //IS_NRF_SPI_IT_MODE
  
  /* Sets data port bit */
  NRF_SPI_CS_PORT->BSRRL=NRF_SPI_CS_PIN;
}

static void ll_nrf_spi_send_data(uint8_t* data, uint16_t len)
{
  if(!len) return;
  
#if defined(IS_NRF_SPI_IT_MODE) 
  spi_it.is_read=0;
  spi_it.data=data;
  spi_it.data_end=data+len-1;

  NRF_SPI->DR=*(spi_it.data);
  xSemaphoreTake(spi_it.sync_obj, portMAX_DELAY);
#else
  volatile uint8_t spi_dr;

  while(len--)
  {
    while(!((NRF_SPI->SR)&SPI_I2S_FLAG_TXE));
    NRF_SPI->DR=*data; data++;
    while(!((NRF_SPI->SR)&SPI_I2S_FLAG_RXNE));
    spi_dr=NRF_SPI->DR;
  }
#endif
  while(((NRF_SPI->SR)&SPI_I2S_FLAG_BSY));
}

static void ll_nrf_spi_read_data(uint8_t* data, uint16_t len)
{
  const uint8_t dummy_byte=0;
  
  if(!len) return;
  
#if defined(IS_NRF_SPI_IT_MODE) 
  spi_it.is_read=1;
  spi_it.data=data;
  spi_it.data_end=data+len-1;
  
  NRF_SPI->DR=dummy_byte;

  xSemaphoreTake(spi_it.sync_obj, portMAX_DELAY);
#else
  while(len--)
  {
    while(!((NRF_SPI->SR)&SPI_I2S_FLAG_TXE));
    NRF_SPI->DR=dummy_byte;
    while(!((NRF_SPI->SR)&SPI_I2S_FLAG_RXNE));
    *data=NRF_SPI->DR; data++;
  }
#endif
  while(((NRF_SPI->SR)&SPI_I2S_FLAG_BSY));
}

static void ll_nrf_spi_init(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
  SPI_InitTypeDef   SPI_InitStructure;
  
#if defined(IS_NRF_SPI_IT_MODE) 
  if(spi_it.sync_obj==NULL) vSemaphoreCreateBinary(spi_it.sync_obj);
  xSemaphoreTake(spi_it.sync_obj, 0); 
#endif
  
  NRF_SPI_CLOCK(ENABLE);
  NRF_SPI_REMAP(); 
    
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_25MHz;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd  = GPIO_PuPd_NOPULL;
  
  /* Configure SPIx MOSI as alternate function */
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_MOSI_PIN;
  GPIO_Init(NRF_SPI_MOSI_PORT, &GPIO_InitStructure);
  
  /* Configure SPIx SCK as alternate function */
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_SCK_PIN;
  GPIO_Init(NRF_SPI_SCK_PORT, &GPIO_InitStructure);
  
  /* Configure SPIx MISO as alternate function */
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_MISO_PIN;
  GPIO_Init(NRF_SPI_MISO_PORT, &GPIO_InitStructure);
  
  /* Configure CS as push-pull */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_CS_PIN;
  GPIO_Init(NRF_SPI_CS_PORT, &GPIO_InitStructure);
  ll_nrf_spi_cs_disable();
  
  SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
  SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
  SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
  SPI_InitStructure.SPI_CPOL = SPI_CPOL_Low;
  SPI_InitStructure.SPI_CPHA = SPI_CPHA_1Edge;
  SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
  //max 10 MHz
#if (TARGET_CPU_FREQ_HZ == 96000000)
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_16;//96/16=6MHz
#elif (TARGET_CPU_FREQ_HZ == 60000000)
  SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_8;//60/8=7.5MHz
#else
#error
#endif
  SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
  SPI_InitStructure.SPI_CRCPolynomial = 7;
  SPI_Init(NRF_SPI, &SPI_InitStructure);

#if defined(IS_NRF_SPI_IT_MODE)
  SPI_I2S_ITConfig(NRF_SPI, SPI_I2S_IT_TXE|SPI_I2S_IT_RXNE|SPI_I2S_IT_ERR, DISABLE);
  
  NVIC_InitTypeDef  NVIC_InitStructure;
  NVIC_InitStructure.NVIC_IRQChannel = NRF_SPI_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = NRF_SPI_IRQn_PRIORITY;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;/* Not used as 4 bits are used for the pre-emption priority. */
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
#endif //IS_NRF_SPI_IT_MODE
  
  /* SPIx enable */
  SPI_Cmd(NRF_SPI, ENABLE);
}

// 
static void ll_nrf_spi_deinit(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
    
  ll_nrf_spi_cs_disable();

#if defined(IS_NRF_SPI_IT_MODE)
  SPI_I2S_ITConfig(NRF_SPI, SPI_I2S_IT_TXE|SPI_I2S_IT_RXNE|SPI_I2S_IT_ERR, DISABLE);
#endif //IS_NRF_SPI_IT_MODE
  
  /* SPIx disable */
  SPI_Cmd(NRF_SPI, DISABLE);

  /* Configure pins as analog */
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_MOSI_PIN;
  GPIO_Init(NRF_SPI_MOSI_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_SCK_PIN;
  GPIO_Init(NRF_SPI_SCK_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_MISO_PIN;
  GPIO_Init(NRF_SPI_MISO_PORT, &GPIO_InitStructure);
  
  GPIO_InitStructure.GPIO_Pin = NRF_SPI_CS_PIN;
  GPIO_Init(NRF_SPI_CS_PORT, &GPIO_InitStructure);
    
  NRF_SPI_CLOCK(DISABLE); 
}

void nrf_chip_enable(void)
{
  //sets data port bit
  NRF_CE_PORT->BSRRL=NRF_CE_PIN;
}

void nrf_chip_disable(void)
{
  //clear data port bit
  NRF_CE_PORT->BSRRH=NRF_CE_PIN;
}

void nrf_bsp_init( void (*nrf_irq)(void) )
{
  GPIO_InitTypeDef  GPIO_InitStructure;
  EXTI_InitTypeDef EXTI_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
  GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = NRF_CE_PIN;
  GPIO_Init(NRF_CE_PORT, &GPIO_InitStructure);
  nrf_chip_disable();
  
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_UP;

  GPIO_InitStructure.GPIO_Pin = NRF_IRQ_PIN;
  GPIO_Init(NRF_IRQ_PORT, &GPIO_InitStructure);
      
  nrf_irq_callback=nrf_irq;
  SYSCFG_EXTILineConfig(NRF_IRQ_EXTI_PORT_SOURCE, NRF_IRQ_EXTI_PIN_SOURCE);
  EXTI_ClearITPendingBit(NRF_IRQ_EXTI_LINE);
    
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_Line = NRF_IRQ_EXTI_LINE;
  EXTI_Init(&EXTI_InitStructure);

  NVIC_InitStructure.NVIC_IRQChannel = NRF_IRQ_EXTI_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = NRF_IRQ_EXTI_PRIORITY;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;//Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  ll_nrf_spi_init();
}

void nrf_bsp_deinit(void)
{
  GPIO_InitTypeDef  GPIO_InitStructure;
  EXTI_InitTypeDef EXTI_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;
  
  NVIC_InitStructure.NVIC_IRQChannel = NRF_IRQ_EXTI_IRQn;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = NRF_IRQ_EXTI_PRIORITY;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;//Not used as 4 bits are used for the pre-emption priority
  NVIC_InitStructure.NVIC_IRQChannelCmd = DISABLE;
  NVIC_Init(&NVIC_InitStructure);
  
  EXTI_ClearITPendingBit(NRF_IRQ_EXTI_LINE);
  
  EXTI_InitStructure.EXTI_LineCmd = DISABLE;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_Line = NRF_IRQ_EXTI_LINE;
  EXTI_Init(&EXTI_InitStructure);

  ll_nrf_spi_deinit();
  
  // Configure pins as analog
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
  
  GPIO_InitStructure.GPIO_Pin = NRF_IRQ_PIN;
  GPIO_Init(NRF_IRQ_PORT, &GPIO_InitStructure);
  
  nrf_chip_disable();
}

//   
void spi_write_block_nRF(uint8_t addr, uint8_t* data, uint8_t length)
{
 ll_nrf_spi_cs_enable();
 
 ll_nrf_spi_send_data(&addr, 1);
 ll_nrf_spi_send_data(data, length);

 ll_nrf_spi_cs_disable();
}

//  
uint8_t spi_write_com_nRF(uint8_t command)
{
 ll_nrf_spi_cs_enable();

 ll_nrf_spi_send_data(&command, 1);
 
 ll_nrf_spi_cs_disable();
 
 return command;
}

//    
void spi_read_block_nRF(uint8_t addr, uint8_t* data, uint8_t length)
{
  ll_nrf_spi_cs_enable();
  
  ll_nrf_spi_send_data(&addr, 1);
  ll_nrf_spi_read_data(data, length);
    
  ll_nrf_spi_cs_disable(); 
}

//   
uint8_t spi_read_byte_nRF(uint8_t addr)
{
 uint8_t byte;
 
 spi_read_block_nRF(addr, &byte, 1);
 
 return byte;
}

//    
void spi_write_byte_nRF(uint8_t addr, uint8_t data)
{
  spi_write_block_nRF(addr, &data, 1);
}

void nrf_bsp_delay_ms(uint32_t ms)
{
  vTaskDelay(ms);
}

void nrf_bsp_delay_us(uint32_t us)
{
  __DELAY_US(us, TARGET_CPU_FREQ_HZ);
}

uint32_t nrf_bsp_get_ms(void)
{
  return xTaskGetTickCount();
}

uint32_t nrf_bsp_crc(uint8_t* buff, uint32_t len)
{
  return crc32_fast_full(buff, len);
}

void nrf_bsp_get_unique_id(uint32_t* unique_id)
{
#if defined(STM32F413_423xx)
  uint8_t* const unique_id_reg_base_addr=(uint8_t*)0x1FFF7A10;
  const uint8_t unique_id_reg_size_in_bytes=12;
#else
#error
#endif
  *unique_id=nrf_bsp_crc(unique_id_reg_base_addr, unique_id_reg_size_in_bytes);
}

#ifdef __cplusplus
extern "C" {
#endif 
  
#if defined(IS_NRF_SPI_IT_MODE)
#pragma optimize=speed
void NRF_SPI_IRQHandler(void)
{
  BaseType_t xHigherPriorityTaskWoken=pdFALSE;
  volatile uint8_t spi_dr;
  
  if((SPI_I2S_GetITStatus(NRF_SPI, SPI_I2S_IT_RXNE) != RESET))
  {
    spi_dr=NRF_SPI->DR;
    
    if(spi_it.is_read) *(spi_it.data)=spi_dr;
    
    if(spi_it.data>=spi_it.data_end)
    {
      xSemaphoreGiveFromISR(spi_it.sync_obj, &xHigherPriorityTaskWoken);
    }
    else
    {
      spi_it.data++;
      
      if(spi_it.is_read) spi_dr=0;
      else               spi_dr=*spi_it.data;
      
      NRF_SPI->DR=spi_dr;
    }
    
    if(xHigherPriorityTaskWoken==pdTRUE) {portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);}
  }
}
#endif //IS_NRF_SPI_IT_MODE

#pragma optimize=speed
void NRF_IRQ_IRQHandler(void)
{
  if(EXTI_GetITStatus(NRF_IRQ_EXTI_LINE) != RESET)
  { 
    EXTI_ClearITPendingBit(NRF_IRQ_EXTI_LINE);
    
    if(nrf_irq_callback!=NULL) nrf_irq_callback();
  }
}
  
#ifdef __cplusplus
}
#endif 